/*! *********************************************************************************
* \addtogroup FRDM-KW40Z Demo
* @{
********************************************************************************** */
/*!
* Copyright (c) 2015, Freescale Semiconductor, Inc.
* All rights reserved.
* \file app.c
* This file is the source file for the Temperature Sensor application
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*
* o Redistributions of source code must retain the above copyright notice, this list
*   of conditions and the following disclaimer.
*
* o Redistributions in binary form must reproduce the above copyright notice, this
*   list of conditions and the following disclaimer in the documentation and/or
*   other materials provided with the distribution.
*
* o Neither the name of Freescale Semiconductor, Inc. nor the names of its
*   contributors may be used to endorse or promote products derived from this
*   software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
* ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/************************************************************************************
*************************************************************************************
* Include
*************************************************************************************
************************************************************************************/
/* Framework / Drivers */
#include "RNG_interface.h"
#include "Keyboard.h"
#include "Led.h"
#include "TimersManager.h"
#include "FunctionLib.h"
#include "fsl_os_abstraction.h"
#include "panic.h"
#include "PWR_Interface.h"
#include "PWR_Configuration.h"
#include "MemManager.h"

/* BLE Host Stack */
#include "gatt_interface.h"
#include "gatt_server_interface.h"
#include "gatt_client_interface.h"
#include "gatt_database.h"
#include "gap_interface.h"
#include "gatt_db_app_interface.h"
#include "gatt_db_handles.h"

/* Profile / Services */
#include "device_info_interface.h"
#include "led_control_interface.h"
#include "input_report_interface.h"
#include "buzzer_control_interface.h"
#include "potentiometer_interface.h"
#include "chip_temperature_interface.h"
#include "accelerometer_interface.h"
#include "compass_interface.h"
#include "remote_controller_interface.h"

#include "ApplMain.h"
#include "app.h"
#include "temperature_sensor.h"
#include "board.h"
#include "FXOS8700CQ.h"
#include "tsi_sensor.h"
#include "buzzer_driver.h"
#include "potentiometer_driver.h"
#include "e-compass.h"
#include "ir_controller.h"

#if gAllowUartDebug
#include "SerialManager.h"
#endif

/************************************************************************************
*************************************************************************************
* Private macros
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Private type definitions
*************************************************************************************
************************************************************************************/
typedef struct advState_tag{
    bool_t      advOn;
}advState_t;
/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/

/* Host Stack Data*/
static bleDeviceAddress_t   maBleDeviceAddress;

/* Adv Parmeters */
static advState_t  mAdvState;
static tmrTimerID_t appTimerId;

/* Application Timers */
static tmrTimerID_t tmsTimerId;         //Chip temperature sensor timer
static tmrTimerID_t tsiTimerId;         //Touch Sensing sensor timer
static tmrTimerID_t potTimerId;         //Potentiometer timer
static tmrTimerID_t acsTimerId;         //Accelerometer sensor timer
static tmrTimerID_t cpsTimerId;         //E-Compass calculation timer

static deviceId_t  mPeerDeviceId = gInvalidDeviceId_c ;
#if gBondingSupported_d
static bleDeviceAddress_t   maPeerDeviceAddress;
static uint8_t mcBondedDevices = 0;
static bool_t   mSendDataAfterEncStart = FALSE;
static bleAddressType_t     mPeerDeviceAddressType;
#endif

/* Service Data*/
static disConfig_t disServiceConfig = {service_device_info};
static lcsConfig_t lcsServiceConfig = {service_led_control, 0};
static irsConfig_t irsServiceConfig = {service_input_report, 0};
static bcsConfig_t bcsServiceConfig = {service_buzzer_control, FALSE};
static psConfig_t psServiceConfig = {service_potentiometer, 0};
static ctsConfig_t ctsServiceConfig = {service_chip_temperature, 2500};
static acsConfig_t acsServiceConfig = {service_accelerometer, 0, 0, 0, 0};
static cpsConfig_t cpsServiceConfig = {service_compass, 0};
static rcsConfig_t rcsServiceConfig = {service_remote_controller, 0, 0};


/************************************************************************************
*************************************************************************************
* Private functions prototypes
*************************************************************************************
************************************************************************************/

/* Gatt and Att callbacks */
static void BleApp_AdvertisingCallback (gapAdvertisingEvent_t* pAdvertisingEvent);
static void BleApp_ConnectionCallback (deviceId_t peerDeviceId, gapConnectionEvent_t* pConnectionEvent);
static void BleApp_GattServerCallback (deviceId_t deviceId, gattServerEvent_t* pServerEvent);
static void BleApp_Config();
static void BleApp_SendAttWriteResponse (deviceId_t* pDeviceId, gattServerEvent_t* pGattServerEvent, bleResult_t* pResult);

/* Timer Callbacks */
#if cPWR_UsePowerDownMode
static void AdvertisingTimerCallback (void *);
static void DisconnectTimerCallback(void* );
#endif
static void BleApp_TsiSensorTimer(void* pParam);
static void BleApp_TemperatureSensorTimer(void* pParam);
static void BleApp_AccelerometerSensorTimer(void* pParam);
static void BleApp_CompassHeadingTimer (void* pParam);

/* Application callbacks */
static void BleApp_Advertise(void);
static void BleApp_HandleElectrodes(tsi_sensor_electrode_flags_t* pElectrodeFlags);
static void BleApp_PotentiometerTimer (void* pParam);
void BleApp_AccelerometerUpdateMeasurements(void);
void BleApp_PotentiometerUpdateMeasurements(void);
void BleApp_CompassHeadingUpdateMeasurements(void);

/************************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
************************************************************************************/
#if gAllowUartDebug
extern uint8_t debugUartId;
#endif

/************************************************************************************
*************************************************************************************
* Public functions prototypes
*************************************************************************************
************************************************************************************/
extern void App_AccelSignalAppThread(bool_t isSensitivityUpdateEvent, uint8_t newSensitivityValue);
extern void App_PotentiometerSignalAppThread (void);
extern void App_CompassSignalAppThread(void);

/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/


/*! *********************************************************************************
* \brief    Initializes application specific functionality before the BLE stack init.
*
********************************************************************************** */
void BleApp_Init(void)
{
    FXOS8700CQ_status_t fxos8700cqStatus;
    FXOS8700CQ_config_t configurationStructure = {
      .outputDataRate = kDataRate0,
      .enabledSensors = kHybridMode,
    };
    FXOS8700CQ_magnetometer_config_t magnetometerStructure = {
      .oversamplingRatio = kMagnetometerOSR0,
      .autoSensorResetFreq = kMagneticAutoReset1,
      .autoCalibrationEnabled = TRUE,
    };

    /* Initialize temperature sensor */
    temperature_sensor_init();
    /* Install timer for temperature measurements */
    tmsTimerId = TMR_AllocateTimer();

    /* Initialize TSI sensor */
    configure_tsi_pins(0);        //Configure TSI pins
    tsi_sensor_init(BleApp_HandleElectrodes);
    /* Install timer for continuous TSI measurements */
    tsiTimerId = TMR_AllocateTimer();

    /* Initialize buzzer sensor */
    configure_tpm_pins(1);      //Configure buzzer TPM pin.
    buzzer_driver_init();       //Initialize buzzer driver.
    
    /* Initialize potentiometer sensor */
    potentiometer_init();
    /* Install timer for potentiometer measurements */
    potTimerId = TMR_AllocateTimer();

    /* Initialize FXOS8700CQ */
    configure_i2c_pins(1);
    fxos8700cqStatus = FXOS8700CQ_init(&configurationStructure);
    assert(fxos8700cqStatus == kStatusSuccess);         //Error initializing FXOS8700CQ
    
    //Initialize magnetometer
    fxos8700cqStatus = FXOS8700CQ_magnetometer_configuration(&magnetometerStructure);
    assert(fxos8700cqStatus == kStatusSuccess); //Error configuring magnetometer
    
    //Initialize IR Controller
    assert(irControllerStatusOk == ir_controller_init()); //Error initializing IR controller
    
    configure_cmt_pin(0);       //Configure CMT_IRO pin
    
    //Install timer for continuous accelerometer measurements
    acsTimerId = TMR_AllocateTimer();
    
    //Install timer for e-compass measurements
    cpsTimerId = TMR_AllocateTimer();
}

/*! *********************************************************************************
* \brief    Starts the BLE application.
*
********************************************************************************** */
void BleApp_Start(void)
{
    Led1On();
    
    if (mPeerDeviceId == gInvalidDeviceId_c)
    {
        /* Device is not connected and not advertising*/
        if (!mAdvState.advOn)
        {
            BleApp_Advertise();
        }
        
#if (cPWR_UsePowerDownMode)    
        PWR_ChangeDeepSleepMode(1);
        PWR_AllowDeviceToSleep();    
#endif           
    }
    else
    {

    }
}

/*! *********************************************************************************
* \brief        Handles keyboard events.
*
* \param[in]    events    Key event structure.
********************************************************************************** */
void BleApp_HandleKeys(key_event_t events)
{  
  switch (events){
  case gKBD_EventPB1_c:
    if (mPeerDeviceId != gInvalidDeviceId_c){
      /* Device is connected, disconnect */
      Gap_Disconnect(mPeerDeviceId);
    }
    else if(mAdvState.advOn){
      /* Device is disconnected but advertising. Stop advertising */
      Gap_StopAdvertising();
    }
    else{
      /* Device is disconnected and not advertising. Start advertising */
      BleApp_Start();
    }
    break;
    
  case gKBD_EventPB2_c:
    Irs_RecordInputReportMeasurement(service_input_report, (1<<0));
    break;
    
  default:
    break;
  }
    
}

/*! *********************************************************************************
* \brief        Handles TSI events.
*
* \param[in]    pElectrodeFlags    Electrode flags.
********************************************************************************** */
void BleApp_HandleElectrodes(tsi_sensor_electrode_flags_t* pElectrodeFlags)
{
  if(pElectrodeFlags->overallFlagStatus){
    /* TSI touched, send report */
    Irs_RecordInputReportMeasurement (service_input_report, (uint8_t)(pElectrodeFlags->overallFlagStatus<<1));
    /* Clear TSI flags */
    pElectrodeFlags->overallFlagStatus = 0x0000u;
  }
}

/*! *********************************************************************************
* \brief        Handles BLE generic callback.
*
* \param[in]    pGenericEvent    Pointer to gapGenericEvent_t.
********************************************************************************** */
void BleApp_GenericCallback (gapGenericEvent_t* pGenericEvent)
{
    switch (pGenericEvent->eventType)
    {
        case gInitializationComplete_c:    
        {
            BleApp_Config();
        }
        break;    
            
        case gPublicAddressRead_c:
        {
            /* Use address read from the controller */
            FLib_MemCpyReverseOrder(maBleDeviceAddress, pGenericEvent->eventData.aAddress, sizeof(bleDeviceAddress_t));
        }
        break;            
            
        case gAdvertisingDataSetupComplete_c:
        {            
            
        }
        break;  
        
        case gAdvertisingParametersSetupComplete_c:
        {
            App_StartAdvertising(BleApp_AdvertisingCallback, BleApp_ConnectionCallback);
        }
        break;         

        case gInternalError_c:
        {
            Led2On();
            panic(0,0,0,0);
        }
        break;

        default: 
            break;
    }
}

/************************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
************************************************************************************/

/*! *********************************************************************************
* \brief        Configures BLE Stack after initialization. Usually used for
*               configuring advertising, scanning, white list, services, et al.
*
********************************************************************************** */
static void BleApp_Config()
{  
    /* Read public address from controller */
    Gap_ReadPublicDeviceAddress();

    /* Register for callbacks*/
    App_RegisterGattServerCallback(BleApp_GattServerCallback);
    
    /* Register security requirements */
#if gUseServiceSecurity_d
    Gap_RegisterDeviceSecurityRequirements(&deviceSecurityRequirements);
#endif

    /* Set local passkey */
#if gBondingSupported_d
    Gap_SetLocalPasskey(gPasskeyValue_c);
#endif

    /* Setup Advertising and scanning data */
    Gap_SetAdvertisingData(&gAppAdvertisingData, &gAppScanRspData);

    /* Populate White List if bonding is supported */
#if gBondingSupported_d
    bleDeviceAddress_t aBondedDevAdds[gcGapMaximumBondedDevices_d];
    bleResult_t result = Gap_GetBondedStaticAddresses(aBondedDevAdds, gcGapMaximumBondedDevices_d, &mcBondedDevices);

    if (gBleSuccess_c == result && mcBondedDevices > 0)
    {
        for (uint8_t i = 0; i < mcBondedDevices; i++)
        {
            Gap_AddDeviceToWhiteList(gBleAddrTypePublic_c, aBondedDevAdds[i]);
        }
    }
#endif

    mAdvState.advOn = FALSE;
    /* Start services */
    Lcs_Start(&lcsServiceConfig);
    Dis_Start(&disServiceConfig);
    Irs_Start(&irsServiceConfig);
    Bcs_Start(&bcsServiceConfig);
    Ps_Start(&psServiceConfig);
    Cts_Start(&ctsServiceConfig);
    Acs_Start(&acsServiceConfig);
    Cps_Start(&cpsServiceConfig);
    Rcs_Start(&rcsServiceConfig);
    
    /* Configure writtable attributes that require a callback action */
    uint16_t notifiableHandleArray[] = {value_led_control, value_buzzer, value_accelerometer_scale, value_controller_command, value_controller_configuration};
    uint8_t notifiableHandleCount = sizeof(notifiableHandleArray)/2;
    bleResult_t initializationResult = GattServer_RegisterHandlesForWriteNotifications(notifiableHandleCount, (uint16_t*)&notifiableHandleArray[0]);
    
    /* Allocate aplication timer */
    appTimerId = TMR_AllocateTimer();
    
#if (cPWR_UsePowerDownMode)    
    PWR_ChangeDeepSleepMode(3);
    PWR_AllowDeviceToSleep();    
#endif    
}

/*! *********************************************************************************
* \brief        Configures GAP Advertise parameters. Advertise will satrt after
*               the parameters are set.
*
********************************************************************************** */
static void BleApp_Advertise(void)
{
    /* Set advertising parameters*/
    Gap_SetAdvertisingParameters(&gAdvParameters);
}

/*! *********************************************************************************
* \brief        Handles BLE Advertising callback from host stack.
*
* \param[in]    pAdvertisingEvent    Pointer to gapAdvertisingEvent_t.
********************************************************************************** */
static void BleApp_AdvertisingCallback (gapAdvertisingEvent_t* pAdvertisingEvent)
{
    switch (pAdvertisingEvent->eventType)
    {
        case gAdvertisingStateChanged_c:
        {
            mAdvState.advOn = !mAdvState.advOn;

#if (cPWR_UsePowerDownMode)
            if(!mAdvState.advOn)
            {
                LED_StopFlashingAllLeds();
                Led1Off();
                PWR_ChangeDeepSleepMode(3);
                PWR_SetDeepSleepTimeInMs(cPWR_DeepSleepDurationMs);
                PWR_AllowDeviceToSleep();    
            }  
            else
            {
                /* Start advertising timer */
                TMR_StartLowPowerTimer(appTimerId, 
                           gTmrLowPowerSecondTimer_c,
                           TmrSeconds(gAdvTime_c),
                           AdvertisingTimerCallback, NULL);
                
                LED_StopFlashingAllLeds();
                Led1Flashing();
            }
#else    
            LED_StopFlashingAllLeds();
            if(mAdvState.advOn){
              /* Device is advertising */
              Led1Flashing();
            }
            else{
              /* Device is not advertising. Turn off LED */
              Led1Off();
            }
#endif  
        }
        break;

        case gAdvertisingCommandFailed_c:
        {
            Led2On();
            panic(0,0,0,0);
        }
        break;

        default:
            break;
    }
}

/*! *********************************************************************************
* \brief        Handles BLE Connection callback from host stack.
*
* \param[in]    peerDeviceId        Peer device ID.
* \param[in]    pConnectionEvent    Pointer to gapConnectionEvent_t.
********************************************************************************** */
static void BleApp_ConnectionCallback (deviceId_t peerDeviceId, gapConnectionEvent_t* pConnectionEvent)
{
    switch (pConnectionEvent->eventType)
    {
        case gConnEvtConnected_c:
        {
            bool_t isBonded = FALSE ;
            
#if gBondingSupported_d    
            /* Copy peer device address information */
            mPeerDeviceAddressType = pConnectionEvent->eventData.connectedEvent.peerAddressType;
            FLib_MemCpy(maPeerDeviceAddress, pConnectionEvent->eventData.connectedEvent.peerAddress, sizeof(bleDeviceAddress_t));
            
            mSendDataAfterEncStart = FALSE;
            if (gBleSuccess_c == Gap_CheckIfBonded(peerDeviceId, &isBonded) &&
                TRUE == isBonded) 
            {
                /* Send temperature data after encryption is started */
                mSendDataAfterEncStart = TRUE;
            }   
#endif  
            
            /* Advertising stops when connected */
            mAdvState.advOn = FALSE;
            TMR_StopTimer(appTimerId);         
            
            /* Subscribe client*/
            mPeerDeviceId = peerDeviceId; 
            Lcs_Subscribe(peerDeviceId);
            Irs_Subscribe(peerDeviceId);
            Bcs_Subscribe(peerDeviceId);
            Cts_Subscribe(peerDeviceId);
            Ps_Subscribe(peerDeviceId);
            Acs_Subscribe(peerDeviceId);
            Cps_Subscribe(peerDeviceId);
            Rcs_Subscribe(peerDeviceId);
            
            LED_StopFlashingAllLeds();
            Led1On();

#if (!cPWR_UsePowerDownMode)             
            /* UI */
            LED_StopFlashingAllLeds();
            Led1On();
#endif 
        }
        break;
        
        case gConnEvtDisconnected_c:
        {
        /* UI */
          Led1Off();
          
          /* Unsubscribe client */
          mPeerDeviceId = gInvalidDeviceId_c;
          Lcs_Unsubscribe();
          Irs_Unsubscribe();
          Bcs_Unsubscribe();
          Cts_Unsubscribe();
          Ps_Unsubscribe();
          Acs_Unsubscribe();
          Cps_Unsubscribe();
          Rcs_Unsubscribe();
          
#if (cPWR_UsePowerDownMode)
          
          /* Go to sleep */
          PWR_ChangeDeepSleepMode(3);
          PWR_SetDeepSleepTimeInMs(cPWR_DeepSleepDurationMs);
          PWR_AllowDeviceToSleep();
#else
            
#endif             
        }
        break;

#if gBondingSupported_d
        case gConnEvtKeysReceived_c:
        {
            /* Copy peer device address information when IRK is used */
            if (pConnectionEvent->eventData.keysReceivedEvent.pKeys->aIrk != NULL)
            {
                mPeerDeviceAddressType = pConnectionEvent->eventData.keysReceivedEvent.pKeys->addressType;
                FLib_MemCpy(maPeerDeviceAddress, pConnectionEvent->eventData.keysReceivedEvent.pKeys->aAddress, sizeof(bleDeviceAddress_t));
            }
        }
        break;

        case gConnEvtPairingComplete_c:
        {
            if (pConnectionEvent->eventData.pairingCompleteEvent.pairingSuccessful &&
                pConnectionEvent->eventData.pairingCompleteEvent.pairingCompleteData.withBonding)
            {
                /* If a bond is created, write device address in controllers White List */
                Gap_AddDeviceToWhiteList(mPeerDeviceAddressType, maPeerDeviceAddress);
            }
        }
        break;

        case gConnEvtPairingRequest_c:
        {
            gPairingParameters.centralKeys = pConnectionEvent->eventData.pairingEvent.centralKeys;
            gPairingParameters.peripheralKeys = pConnectionEvent->eventData.pairingEvent.peripheralKeys;
            Gap_AcceptPairingRequest(peerDeviceId, &gPairingParameters);
        }    
        break;
        
        case gConnEvtKeyExchangeRequest_c:
        {
            gapSmpKeys_t sentSmpKeys = gSmpKeys;
            
            if (!(pConnectionEvent->eventData.keyExchangeRequestEvent.requestedKeys & gLtk_c))
            {
                sentSmpKeys.aLtk = NULL;
                /* When the LTK is NULL EDIV and Rand are not sent and will be ignored. */
            }
            
            if (!(pConnectionEvent->eventData.keyExchangeRequestEvent.requestedKeys & gIrk_c))
            {
                sentSmpKeys.aIrk = NULL;
                /* When the IRK is NULL the Address and Address Type are not sent and will be ignored. */
            }
            
            if (!(pConnectionEvent->eventData.keyExchangeRequestEvent.requestedKeys & gCsrk_c))
            {
                sentSmpKeys.aCsrk = NULL;
            }
            
            Gap_SendSmpKeys(peerDeviceId, &sentSmpKeys);
            break;
        }
            
        case gConnEvtLongTermKeyRequest_c:
        {
            if (pConnectionEvent->eventData.longTermKeyRequestEvent.ediv == gSmpKeys.ediv &&
                pConnectionEvent->eventData.longTermKeyRequestEvent.randSize == gSmpKeys.cRandSize)
            {
                /* EDIV and RAND both matched */
                Gap_ProvideLongTermKey(peerDeviceId, gSmpKeys.aLtk, gSmpKeys.cLtkSize);
            }
            else
            /* EDIV or RAND size did not match */
            {
                Gap_DenyLongTermKey(peerDeviceId);
            }
        }
        break;
            
        case gConnEvtEncryptionChanged_c:
        {
            if (mSendDataAfterEncStart)
            {

            }
        }
        break;
        
#endif
        
    default:
        break;
    }
}

/*! *********************************************************************************
* \brief        Handles GATT server callback from host stack.
*
* \param[in]    deviceId        Peer device ID.
* \param[in]    pServerEvent    Pointer to gattServerEvent_t.
********************************************************************************** */
static void BleApp_GattServerCallback (deviceId_t deviceId, gattServerEvent_t* pServerEvent)
{
    switch (pServerEvent->eventType)
    {
      case gEvtCharacteristicCccdWritten_c:
        {
            /* 
            Attribute CCCD write handler: Create a case for your registered attribute and
            execute callback action accordingly
            */
            switch(pServerEvent->eventData.charCccdWrittenEvent.handle)
            {
            case cccd_input_report:{
              //Determine if the timer must be started or stopped
              if (pServerEvent->eventData.charCccdWrittenEvent.newCccd){
                // CCCD set, start timer
                TMR_StartTimer(tsiTimerId, gTmrIntervalTimer_c, gTsiUpdateTime_c ,BleApp_TsiSensorTimer, NULL);
#if gAllowUartDebug
                Serial_Print(debugUartId, "Input Report notifications enabled \n\r", gNoBlock_d);
#endif
              }
              else{
                // CCCD cleared, stop timer
                TMR_StopTimer(tsiTimerId);
#if gAllowUartDebug
                Serial_Print(debugUartId, "Input Report notifications disabled \n\r", gNoBlock_d);
#endif
              }
            }
              break;
              
            case cccd_potentiometer:{
              //Determine if the timer must be started or stopped
              if (pServerEvent->eventData.charCccdWrittenEvent.newCccd){
                // CCCD set, start timer
                TMR_StartTimer(potTimerId, gTmrIntervalTimer_c, gPotentiometerUpdateTime_c ,BleApp_PotentiometerTimer, NULL);
#if gAllowUartDebug
                Serial_Print(debugUartId, "Potentiometer notifications enabled \n\r", gNoBlock_d);
#endif
              }
              else{
                // CCCD cleared, stop timer
                TMR_StopTimer(potTimerId);
#if gAllowUartDebug
                Serial_Print(debugUartId, "Potentiometer notifications disabled \n\r", gNoBlock_d);
#endif
              }
            }
              break;
            
            case cccd_temperature:{
              //Determine if the timer must be started or stopped
              if (pServerEvent->eventData.charCccdWrittenEvent.newCccd){
                uint16_t chipTemperature;
                // CCCD set, start timer
                TMR_StartTimer(tmsTimerId, gTmrIntervalTimer_c, gChipTemperatureUpdateTime_c, BleApp_TemperatureSensorTimer, NULL);
                // Send a first temperature measurement
                chipTemperature = temperature_sensor_get_chip_temperature();
                Cts_RecordChipTemperatureMeasurement(service_chip_temperature, chipTemperature);
#if gAllowUartDebug
                Serial_Print(debugUartId, "Chip Temperature notifications enabled \n\r", gNoBlock_d);
#endif
              }
              else{
                // CCCD cleared, stop timer
                TMR_StopTimer(tmsTimerId);
#if gAllowUartDebug
                Serial_Print(debugUartId, "Chip Temperature notifications disabled \n\r", gNoBlock_d);
#endif
              }
            }
              break;
            
            case cccd_accelerometer_readings:{
              //Determine if the accelerometer timer must be started or stopped
              if (pServerEvent->eventData.charCccdWrittenEvent.newCccd){
                // CCCD set, start timer
                TMR_StartTimer(acsTimerId, gTmrIntervalTimer_c, gAccelerometerUpdateTime_c,BleApp_AccelerometerSensorTimer, NULL);
#if gAllowUartDebug
                Serial_Print(debugUartId, "Accelerometer notifications enabled \n\r", gNoBlock_d);
#endif
              }
              else{
                // CCCD cleared, stop timer
                TMR_StopTimer(acsTimerId);
#if gAllowUartDebug
                Serial_Print(debugUartId, "Accelerometer notifications disabled \n\r", gNoBlock_d);
#endif
              }
            }
              break;
              
            case cccd_compass_heading:{
              //Determine if the accelerometer timer must be started or stopped
              if (pServerEvent->eventData.charCccdWrittenEvent.newCccd){
                // CCCD set, start timer
                TMR_StartTimer(cpsTimerId, gTmrIntervalTimer_c, gCompassHeadingUpdateTime_c, BleApp_CompassHeadingTimer, NULL);
#if gAllowUartDebug
                Serial_Print(debugUartId, "E-Compass notifications enabled \n\r", gNoBlock_d);
#endif
              }
              else{
                // CCCD cleared, stop timer
                TMR_StopTimer(cpsTimerId);
#if gAllowUartDebug
                Serial_Print(debugUartId, "E-Compass notifications disabled \n\r", gNoBlock_d);
#endif
              }
            }
              break;
            
            default:
              break;
            }
        }
        break;
        
      case gEvtAttributeWritten_c:
        {
            /* 
            Attribute write handler: Create a case for your registered attribute and
            execute callback action accordingly
            */
            switch(pServerEvent->eventData.attributeWrittenEvent.handle){
              case value_led_control:{
                bleResult_t result;
                
                //Get written value
                uint8_t* pAttWrittenValue = pServerEvent->eventData.attributeWrittenEvent.aValue;
                
                //Create a new instance of the LED configurator structure
                lcsConfig_t lcs_LedConfigurator = {
                  .serviceHandle = service_led_control,
                  .ledControl.ledNumber = (uint8_t)*pAttWrittenValue,
                  .ledControl.ledCommand = (uint8_t)*(pAttWrittenValue + sizeof(uint8_t)),
                };
                
                //Call LED update function
                result = Lcs_SetNewLedValue(&lcs_LedConfigurator);
                
                //Send response to client
                BleApp_SendAttWriteResponse(&deviceId, pServerEvent, &result);
                
              }
              break;
              
            case value_buzzer:{
              bleResult_t result;
                
                //Get written value
                uint8_t* pAttWrittenValue = pServerEvent->eventData.attributeWrittenEvent.aValue;
                
                //Create a new instance of the Buzzer configurator structure
                bcsConfig_t bcs_BuzzerConfigurator = {
                  .serviceHandle = service_buzzer_control,
                  .buzzerStatus = (bool_t)*pAttWrittenValue
                };
                
                //Call Buzzer update function
                result = Bcs_SetNewBuzzerValue(&bcs_BuzzerConfigurator);
                
                //Send response to client
                BleApp_SendAttWriteResponse(&deviceId, pServerEvent, &result);
            }
              break;
              
            case value_accelerometer_scale:{
                bleResult_t result;
                
                //Get written value
                uint8_t* pAttWrittenValue = pServerEvent->eventData.attributeWrittenEvent.aValue;
               
                //Call Scale update function
                result = Acs_RecordAccelerometerScale(service_accelerometer, *pAttWrittenValue);
                
                //Send response to client
                BleApp_SendAttWriteResponse(&deviceId, pServerEvent, &result);
                
              }
              break;
              
            case value_controller_command:{
                bleResult_t result;
                
                //Get written value
                uint8_t* pAttWrittenValue = pServerEvent->eventData.attributeWrittenEvent.aValue;
               
                //Call Command update function
                result = Rcs_SetCommand(service_remote_controller, *pAttWrittenValue);
                
                //Send response to client
                BleApp_SendAttWriteResponse(&deviceId, pServerEvent, &result);
                
              }
              break;
              
            case value_controller_configuration:{
                bleResult_t result;
                
                //Get written value
                uint8_t* pAttWrittenValue = pServerEvent->eventData.attributeWrittenEvent.aValue;
               
                //Call Configuration update function
                result = Rcs_SetConfiguration(service_remote_controller, *pAttWrittenValue);
                
                //Send response to client
                BleApp_SendAttWriteResponse(&deviceId, pServerEvent, &result);
                
              }
              break;
              
              default:
                break;
            }
        }
        break;
      
    default:
        break;
    }
}

#if cPWR_UsePowerDownMode
/*! *********************************************************************************
* \brief        Handles advertising timer callback.
*
* \param[in]    pParam        Calback parameters.
********************************************************************************** */
static void AdvertisingTimerCallback(void* pParam)
{
    /* Stop advertising */
    if (mAdvState.advOn)
    {
        Gap_StopAdvertising();
    }
}
#endif

#if cPWR_UsePowerDownMode
/*! *********************************************************************************
* \brief        Handles disconnect timer callback.
*
* \param[in]    pParam        Calback parameters.
********************************************************************************** */
static void DisconnectTimerCallback(void* pParam)
{
    if (mPeerDeviceId != gInvalidDeviceId_c)
    {
        Gap_Disconnect(mPeerDeviceId);
    }
}
#endif

/*! *********************************************************************************
* \brief        Sends GATT response to the client
* 
* \param[in]    pDeviceId               Pointer to device ID structure (see @ref deviceId_t)
* \param[in]    pGattServerEvent        Pointer to GATT server event structure (see @ref gattServerEvent_t)
* \param[in]    pResult                 Pointer to BLE error result status structure (see @ref bleResult_t)
*
* \return       void
********************************************************************************** */
static void BleApp_SendAttWriteResponse (deviceId_t* pDeviceId, gattServerEvent_t* pGattServerEvent, bleResult_t* pResult){
  attErrorCode_t attErrorCode;
  
  // Determine response to send (OK or Error)
  if(*pResult == gBleSuccess_c)
    attErrorCode = gAttErrCodeNoError_c;
  else{
    attErrorCode = (attErrorCode_t)(*pResult & 0x00FF);
  }
  // Send response to client  
  GattServer_SendAttributeWrittenStatus(*pDeviceId, pGattServerEvent->eventData.attributeWrittenEvent.handle, attErrorCode);
}

/*! *********************************************************************************
* \brief        Handles TSI sensor timer callback
*
* \param[in]    pParam        Callback parameters.
********************************************************************************** */
static void BleApp_TsiSensorTimer(void* pParam){
  /* Start a new TSI single measurement */
  tsi_sensor_start_single_measurement();
}

/*! *********************************************************************************
* \brief        Handles Temperature sensor timer callback
*
* \param[in]    pParam        Callback parameters.
********************************************************************************** */
static void BleApp_TemperatureSensorTimer(void* pParam){
  int16_t chipTemperature;
  
  chipTemperature = temperature_sensor_get_chip_temperature();
  Cts_RecordChipTemperatureMeasurement(service_chip_temperature, chipTemperature);
}

/*! *********************************************************************************
* \brief        Handles Potentiometer timer callback
*
* \param[in]    pParam        Callback parameters.
********************************************************************************** */
static void BleApp_PotentiometerTimer (void* pParam){
  // Signal the main application that a new reading can be performed
  App_PotentiometerSignalAppThread();
}

/*! *********************************************************************************
* \brief        Updates potentiometer readings
*
* \param[in]    none
********************************************************************************** */
extern void BleApp_PotentiometerUpdateMeasurements(void){
  uint8_t potentiometerPosition;
  
  potentiometerPosition = potentiometer_get_position();
  
  Ps_RecordPotentiometerMeasurement (service_potentiometer, potentiometerPosition);
}

/*! *********************************************************************************
* \brief        Handles Accelerometer sensor timer callback
*
* \param[in]    pParam        Callback parameters.
********************************************************************************** */
static void BleApp_AccelerometerSensorTimer(void* pParam){
  // Signal the main application that a new reading can be performed
  App_AccelSignalAppThread(FALSE, 0);
}

/*! *********************************************************************************
* \brief        Handles Compass heading timer callback
*
* \param[in]    pParam        Callback parameters.
********************************************************************************** */
static void BleApp_CompassHeadingTimer (void* pParam){
  // Signal the main application that a new compass heading calculation can be performed
  App_CompassSignalAppThread();
}

/*! *********************************************************************************
* \brief        Updates accelerometer readings
*
* \param[in]    none
********************************************************************************** */
extern void BleApp_AccelerometerUpdateMeasurements(void){
  //Error handling flag
  FXOS8700CQ_status_t accelerometerReadingStatus;
  //Create a new accelerometer data instance
  FXOS8700CQ_output_data_t accelerometerReadings;
  

  
  //Read accelerometer data
  accelerometerReadingStatus = FXOS8700CQ_get_accelerometer_readings(&accelerometerReadings);
  assert(accelerometerReadingStatus == kStatusSuccess); //Error reading accelerometer data
  
  //Record new accelerometer measurements
  Acs_RecordAccelerometerMeasurement(service_accelerometer, accelerometerReadings.xAxisData,
                                     accelerometerReadings.yAxisData, accelerometerReadings.zAxisData);
}

/*! *********************************************************************************
* \brief        Updates compass heading
*
* \param[in]    none
********************************************************************************** */
extern void BleApp_CompassHeadingUpdateMeasurements (void){
  //Error handling flag
  FXOS8700CQ_status_t accelerometerReadingStatus;
  //Create a new accelerometer and magnetometer data instance
  FXOS8700CQ_output_data_t accelerometerReadings;
  FXOS8700CQ_output_data_t magnetometerReadings;
  //NED system definitions
  int16_t magX, magY, magZ, accX, accY, accZ;
  //Compass Heading result
  int32_t compassHeading;
  
  //Read Accelerometer + Magnetometer data
  accelerometerReadingStatus = FXOS8700CQ_get_hybrid_sensor_readings(&accelerometerReadings, &magnetometerReadings);
  assert(accelerometerReadingStatus == kStatusSuccess); //Error trying to read FXOS8700CQ
  
  //Adjust FXOS8700CQ measurements to satsify the NED system (see AN4284)
  magX = magnetometerReadings.yAxisData;
  magY = magnetometerReadings.xAxisData;
  magZ = magnetometerReadings.zAxisData*(-1);
  accX = accelerometerReadings.yAxisData*(-1);
  accY = accelerometerReadings.xAxisData*(-1);
  accZ = accelerometerReadings.zAxisData;
  
  //Calculate compass heading (-180 -> 180)
  compassHeading = ecompass_calculate_heading(magX, magY, magZ, accX, accY, accZ);
  
  //Obtain 360 equivalent (0-359)
  if (compassHeading >= 0){
    compassHeading = (compassHeading/100);
  }
  else{
    compassHeading = 359 + (compassHeading/100);
  }
  
  Cps_RecordCompassHeadingMeasurement (service_compass, (uint16_t)(compassHeading));
}


/*! *********************************************************************************
* @}
********************************************************************************** */
