/*!
* @defgroup e_compass e-Compass
*
* e-Compass module combines accelerometer and magnetometer readings to determine the magnetic north position relative to the board orientation.
* This module uses APIs from the @ref fxos8700cq module to obtain accelerometer and magnetometer readings. These measurements are passed to the
* @ref e_compass_driver module to calculate the current magnetic north position.
* @{
*******************************************************************************/
/*!
* @file e-compass.h
*
* @author  
*
* @version 1.0
*
* @date Mar-14-2016
*
* @brief Driver for E-Compass
*
********************************************************************************
*
* Copyright (c) 2016, Freescale Semiconductor.
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*
* o Redistributions of source code must retain the above copyright notice, this list
*   of conditions and the following disclaimer.
*
* o Redistributions in binary form must reproduce the above copyright notice, this
*   list of conditions and the following disclaimer in the documentation and/or
*   other materials provided with the distribution.
*
* o Neither the name of Freescale Semiconductor nor the names of its
*   contributors may be used to endorse or promote products derived from this
*   software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
* ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*!
* @defgroup e_compass_driver e-Compass Driver
* @{
* @brief e-Compass Driver implements the functions to calculate the compass heading based on the accelerometer and
*                  magnetometer data. All functions are taken from and explained in the NXP
*                  application note AN4248 "Implementing a Tilt-Compansated eCompass using Accelerometer
*                  and Magnetometer Sensors"
* @warning  The magnetometer sensor must be calibrated before calculating the compass heading by moving the board
*           in all directions while the magnetometer is enabled.
*
* Version 1.0
*
* This driver version does not include these features:
*
* 1. Calibration function
*/

#ifndef _E_COMPASS_H_
#define _E_COMPASS_H_

/******************************************************************************
* Headers
******************************************************************************/
#include "EmbeddedTypes.h"

/******************************************************************************
* Public function prototypes
******************************************************************************/

/*!
* @brief Calculates the current compass heading (in degrees) based on the current
*        accelerometer and magnetometer data.
* @note  The accelerometer and magnetometer input parameters must be passed using the
*        polarity and directions in the NED coordinate system. If your sensors have
*        a different position on the board, adjust the measurements obtained to fit
*        the NED system.
*
* @param [in] iBpx      Magnetometer X-axis reading.
* @param [in] iBpy      Magnetometer Y-Axis reading.
* @param [in] iBpz      Magnetometer Z-Axis reading.
* @param [in] iGpx      Accelerometer X-axis reading.
* @param [in] iGpy      Accelerometer Y-axis reading.
* @param [in] iGpz      Accelerometer Z-axis reading.
*
* @return Compass heading position (in degrees, from -180 to 180).
*
* @warning The magnetometer must be calibrated before using this function for accurate results.
*/
int16_t ecompass_calculate_heading(int16_t iBpx, int16_t iBpy, int16_t iBpz,
                                   int16_t iGpx, int16_t iGpy, int16_t iGpz);

/********************************************************************************/
/*!
* @brief Calibrates the magnetometer by determining the current hard iron offset.
* @note  This function must be periodically called during the program execution to constantly
*        calibrate the magnetometer sensor. It is not neccesary to call this function if the
*        magnetometer sensor has the autocalibration feature.
*
* @param [in] xAxisOffset      Magnetometer X-axis reading.
* @param [in] yAxisOffset      Magnetometer Y-Axis reading.
* @param [in] zAxisOffset      Magnetometer Z-Axis reading.
*
* @return void
*/
void ecompass_calibrate_hard_iron (int16_t xAxisOffset, int16_t yAxisOffset, int16_t zAxisOffset);

/*!
* @} End of e_compass_driver
*/

#endif /* End of _E_COMPASS_H_ */