/*!
* @file ir_controller.c
*
* @author  
*
* @version 1.0
*
* @date Mar-14-2016
*
* @brief 
*
********************************************************************************
*
* Copyright (c) 2016, Freescale Semiconductor.
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*
* o Redistributions of source code must retain the above copyright notice, this list
*   of conditions and the following disclaimer.
*
* o Redistributions in binary form must reproduce the above copyright notice, this
*   list of conditions and the following disclaimer in the documentation and/or
*   other materials provided with the distribution.
*
* o Neither the name of Freescale Semiconductor nor the names of its
*   contributors may be used to endorse or promote products derived from this
*   software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
* ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/******************************************************************************
* Headers
******************************************************************************/
#include "ir_controller.h"
#include "fsl_cmt_driver.h"
#include "fsl_osa_ext.h"

/******************************************************************************
* Definitions
******************************************************************************/
#define SWAP32(integer) ((integer>>24)&0xff) | ((integer<<8)&0xff0000) | ((integer>>8)&0xff00) | ((integer<<24)&0xff000000)   

/******************************************************************************
* Declarations
******************************************************************************/
extern void App_IrControllerSignalAppThread (void);

/******************************************************************************
* Type definitions
******************************************************************************/
/*! State machine function pointer definition */
typedef void (*pIrControllerStateMachineFuncPtr) (void);

/*! IR Controller state machine states */
typedef enum IrControllerState{
  irControllerStateIdle,                /*!<Module is idle */
  irControllerStateSendingHeader,       /*!<Module is sending the header */
  irControllerStateSendingData,         /*!<Module is sending data */
  irControllerStateSendingStopGap       /*!<Module is sending the stop and gap signals */
}ir_controller_state_t;

/*! IR Controller metadata structure */
typedef struct IrControllerMetadata{
  ir_controller_state_t irControllerState;              /*!<Current state machine state */
  ir_controller_parameters_t irControllerParameters;    /*!<Modulation parameters for the current controller */
  uint8_t irControllerBitCount;                         /*!<Variable containing the number of bits to send in the command */
  uint8_t irControllerRemainingCommandRepeat;           /*!<Variable containing the number of times left the command must be repeated. */
  uint32_t irControllerCommand;                         /*!<Variable containing the controller command to send */
  uint32_t irControllerCommandBackup;                   /*!<Backup of the last command sent, so it can be retransmitted if repeat is required */
}ir_controller_metadata_t;

/******************************************************************************
* Local variables
******************************************************************************/
uint8_t irControllerHeaderStopGapBit;
ir_controller_metadata_t irControllerMetadata;
osaSemaphoreId_t irControllerSemaphore;
cmt_modulate_data_t irControllerCmtData;
cmt_modulate_config_t irControllerCmtConfig = {
  .mode = kCMTBasebandMode,
  .highCount1 = 0xFF,
  .lowCount1 = 0xFF,
  .highCount2 = 0xFF,
  .lowCount2 = 0xFF,
};

/******************************************************************************
* Private function prototypes
******************************************************************************/
void irControllerIdleState (void);
void irControllerSendHeaderState (void);
void irControllerSendDataState (void);
void irControllerSendStopGapState (void);

pIrControllerStateMachineFuncPtr irControllerStateMachine[] = {
  irControllerIdleState,
  irControllerSendHeaderState,
  irControllerSendDataState,
  irControllerSendStopGapState
};

/******************************************************************************
* Public functions
******************************************************************************/
ir_controller_status_t ir_controller_init (void){
  cmt_status_t cmtStatus;
  const cmt_base_config_t cmtInit = {
    .isInterruptEnabled = true,
    .isIroEnabled = true,
    .iroPolarity = kCMTIROActiveHigh,
    .cmtDivider = kCMTSecClkDiv1
  };

  cmtStatus = CMT_DRV_Init(0, &cmtInit);

  if(cmtStatus != kStatus_CMT_Success){
    return irControllerStatusInitError;
  }
  
  /* Create application semaphore */
  irControllerSemaphore = OSA_EXT_SemaphoreCreate(0);
  
  return irControllerStatusOk;
}

ir_controller_status_t ir_controller_send_command (ir_controller_parameters_t* controllerParamPtr, uint32_t* ptrCommandToSend, uint8_t numberOfBits, uint8_t commandRepeatTimes){
  /* Check if module is not currently in use */
  if(irControllerMetadata.irControllerState != irControllerStateIdle){
    return irControllerStatusModuleBusy;
  }
  
  /* Store metadata */
  irControllerMetadata.irControllerParameters = *controllerParamPtr;
  irControllerMetadata.irControllerBitCount = numberOfBits;
  irControllerMetadata.irControllerRemainingCommandRepeat = commandRepeatTimes;
  irControllerMetadata.irControllerCommand = SWAP32(*ptrCommandToSend);
  irControllerMetadata.irControllerCommandBackup = SWAP32(*ptrCommandToSend);
  
  /* Indicate state machine to start sending a new command */
  irControllerMetadata.irControllerState = irControllerStateSendingHeader;
  OSA_EXT_SemaphorePost(irControllerSemaphore);
  /* Signal state machine execution in application thread */
  App_IrControllerSignalAppThread();
  
  return irControllerStatusOk;
}

void ir_controller_task (void){
  /* Check if semaphore has been set */
  osaStatus_t irControllerSemaphoreStatus = OSA_EXT_SemaphoreWait(irControllerSemaphore, 0);
  
  if(irControllerSemaphoreStatus != osaStatus_Success){
    return;
  }
  
  /* Execute action according with current state */
  irControllerStateMachine[irControllerMetadata.irControllerState]();
}

ir_controller_status_t ir_controller_get_current_status (void){
  if(irControllerMetadata.irControllerState != irControllerStateIdle){
    return irControllerStatusModuleBusy;
  }
  
  return irControllerStatusOk;
}
/******************************************************************************
* Private functions
******************************************************************************/
void irControllerIdleState (void){
  /* No actions to execute */
}

void irControllerSendHeaderState (void){
  /* Copy carrier configuration */
  if(irControllerMetadata.irControllerParameters.carrierOnTime == 0 ||
     irControllerMetadata.irControllerParameters.carrierOffTime == 0){
       irControllerCmtConfig.mode = kCMTBasebandMode;
     }
  else{
    irControllerCmtConfig.mode = kCMTTimeMode;
    irControllerCmtConfig.highCount1 = irControllerMetadata.irControllerParameters.carrierOnTime;
    irControllerCmtConfig.lowCount1 = irControllerMetadata.irControllerParameters.carrierOffTime;
  }
  
  /* Configure header settings */
  irControllerHeaderStopGapBit = 0x80;
  irControllerCmtData.markOne = irControllerMetadata.irControllerParameters.headerOnTime;       //Configure only mark and space for logic 1 with header times
  irControllerCmtData.spaceOne = irControllerMetadata.irControllerParameters.headerOffTime;
  irControllerCmtData.data = &irControllerHeaderStopGapBit;      //Data is only a logic 1 (header)
  irControllerCmtData.len = 0x01; //Send just one bit (header)
  irControllerCmtData.lenDone = 0x00; //Reset data sent counter
  
  /* Start header modulation */
  CMT_DRV_StartCarrierModulate (0, &irControllerCmtConfig, &irControllerCmtData, true);
}

void irControllerSendDataState (void){
  cmt_status_t cmtDriverStatus;
    
  /* Configure data settings */
  irControllerCmtData.markOne = irControllerMetadata.irControllerParameters.logic1OnTime;     
  irControllerCmtData.spaceOne = irControllerMetadata.irControllerParameters.logic1OffTime;
  irControllerCmtData.markZero = irControllerMetadata.irControllerParameters.logic0OnTime;
  irControllerCmtData.spaceZero = irControllerMetadata.irControllerParameters.logic0OffTime;
  irControllerCmtData.data = (uint8_t*)&irControllerMetadata.irControllerCommand;
  irControllerCmtData.len = irControllerMetadata.irControllerBitCount;
  irControllerCmtData.lenDone = 0x00; //Reset data sent counter
  
  /* Start data modulation */
  cmtDriverStatus = CMT_DRV_ModulateDataByBit(0, &irControllerCmtData, true);
  
  assert(cmtDriverStatus != kStatus_CMT_Fail);  //Error attempting CMT modulation
}

void irControllerSendStopGapState (void){
  cmt_status_t cmtDriverStatus;
  
/* Configure Stop and Gap settings */
  irControllerHeaderStopGapBit = 0x80;
  irControllerCmtData.markOne = irControllerMetadata.irControllerParameters.stopTime;     
  irControllerCmtData.spaceOne = irControllerMetadata.irControllerParameters.gapTime;
  irControllerCmtData.data = &irControllerHeaderStopGapBit;
  irControllerCmtData.len = 0x01;
  irControllerCmtData.lenDone = 0x00; //Reset data sent counter
  
  /* Start data modulation */
  cmtDriverStatus = CMT_DRV_ModulateDataByBit(0, &irControllerCmtData, true);
  
   assert(cmtDriverStatus != kStatus_CMT_Fail);  //Error attempting CMT modulation
}

extern void CMT_IRQHandler(void)
{
  cmt_status_t cmtDriverStatus;
  uint32_t dummyMarkCount = 0x1234;
  
  /* Clear interrupt flag */
  CMT_DRV_ClearEndofCycleFlag(0, dummyMarkCount);
  
  /* Continue sending next byte */
  cmtDriverStatus = CMT_DRV_ModulateDataByBit (0, &irControllerCmtData, true);
  
  /* Check for errors */
   assert(cmtDriverStatus != kStatus_CMT_Fail);  //Error attempting CMT modulation
  
  /* Check for transfer completion */
  if (cmtDriverStatus == kStatus_CMT_Done){

    /* Check if current state is not the last one */
    if(irControllerMetadata.irControllerState == irControllerStateSendingStopGap){
      /* Stop CMT */
      CMT_MSC &= ~(CMT_MSC_MCGEN_MASK); //@todo: Report to SDK developers problem stoping CMT using function CMT_DRV_StopCarrierModulate
      
      /* Check if command needs to be repeated */
      if(--irControllerMetadata.irControllerRemainingCommandRepeat > 0){
        /* Repeat required, send command again */
        irControllerMetadata.irControllerCommand = irControllerMetadata.irControllerCommandBackup; //Reload command data
        irControllerMetadata.irControllerState = irControllerStateSendingHeader;        //Signal Sending Header state
        OSA_EXT_SemaphorePost(irControllerSemaphore);   //Post semaphore
        /* Signal state machine execution in application thread */
        App_IrControllerSignalAppThread();
      }
      else{
        /* No repeat is needed, go to idle */
        irControllerMetadata.irControllerState = irControllerStateIdle;
      }
    }
    else{
      /* Go to the next state */
      irControllerMetadata.irControllerState++;
    }
    /* Unlock semaphore */
    OSA_EXT_SemaphorePost(irControllerSemaphore);
    /* Signal state machine execution in application thread */
    App_IrControllerSignalAppThread();
  }
}