/*!
* @defgroup ir_controller IR Controller
*
* @details IR controller module implements functions to command different TVs using an infrared transmitter. 
* @ref ir_controller_driver uses CMT to modulate the IR signal. Timing parameters and controller command codes 
* are defined using the structures on @ref ir_controller_parameters.
*
* @{
*******************************************************************************/
/*!
* @file ir_controller.h
*
* @author  
*
* @version 1.0
*
* @date Mar-14-2016
*
* @brief 
*
********************************************************************************
*
* Copyright (c) 2016, Freescale Semiconductor.
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*
* o Redistributions of source code must retain the above copyright notice, this list
*   of conditions and the following disclaimer.
*
* o Redistributions in binary form must reproduce the above copyright notice, this
*   list of conditions and the following disclaimer in the documentation and/or
*   other materials provided with the distribution.
*
* o Neither the name of Freescale Semiconductor nor the names of its
*   contributors may be used to endorse or promote products derived from this
*   software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
* ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
* ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*!
* @defgroup ir_controller_driver IR Controller Driver
* @{
*
* This module implements the APIs to initialize the IR controller driver and send
* IR commands
*/

#ifndef _IR_CONTROLLER_H_
#define _IR_CONTROLLER_H_

/******************************************************************************
* Headers
******************************************************************************/
#include "EmbeddedTypes.h"

/******************************************************************************
* Type definitions
******************************************************************************/
/*! IR Controller Status */
typedef enum IrControllerStatus{
  irControllerStatusOk,                 /*!<No error occurred */
  irControllerStatusInitError,          /*!<An error occurred during the initialization phase */
  irControllerStatusModuleBusy,         /*!<The IR Controller is busy and can't start a new command at this time */
}ir_controller_status_t;

/*! Modulation parameters for each remote controller */
typedef struct IrControllerParameters{
  uint16_t headerOnTime;                /*!<On time for the header in us */
  uint16_t headerOffTime;               /*!<Off time for the header in us */
  uint16_t logic0OnTime;                /*!<On time for the logic 0 signal in us */
  uint16_t logic0OffTime;               /*!<Off time for the logic 0 signal in us */
  uint16_t logic1OnTime;                /*!<On time for the logic 1 signal in us */
  uint16_t logic1OffTime;               /*!<Off time for the logic 1 signal in us */
  uint16_t stopTime;                    /*!<On time for the stop signaling in us */
  uint16_t gapTime;                     /*!<GAP time (the minimum time between one command and the next command) in us */
  uint8_t carrierOnTime;                /*!<Signal carrier On time in the multiples of 0.125 us. Must be 0 if the carrier is not needed. */
  uint8_t carrierOffTime;               /*!<Signal carrier Off time in the multiples of 0.125 us. Must be 0 if the carrier is not needed */
}ir_controller_parameters_t;

/******************************************************************************
* Public function prototypes
******************************************************************************/

/*!**********************************************************************************
* \brief        Initializes the IR Controller required modules. 
*
* \note         The pin mux must be initialized separately.
*
* \param[in]    None
*
* \return       @ref ir_controller_status_t IR Controller module status
************************************************************************************/
ir_controller_status_t ir_controller_init (void);

/*!**********************************************************************************
* \brief        Sends a new IR controller command.
*
* \param[in]    controllerParamPtr Controller modulation parameters (See @ref ir_controller_parameters_t)
* \param[in]    ptrCommandToSend Pointer to the 32-bit array containing the command to send. @warning Data bits must be LEFT-ALIGNED, 
*               Example: If the following 11 bits will be sent (0b0...10110001001 = 0x00000589) the array must be [0xB1, 0x20, 0x00, 0x00 = 0b1011000100100000...)
* \param[in]    numberOfBits Number of BITS to modulate.
* \param[in]    commandRepeatTimes Number of times the command must be repeatedly transmitted.
*
* \return       @ref ir_controller_status_t IR Controller module status
************************************************************************************/
ir_controller_status_t ir_controller_send_command (ir_controller_parameters_t* controllerParamPtr, uint32_t* ptrCommandToSend, uint8_t numberOfBits, uint8_t commandRepeatTimes);

/*!**********************************************************************************
* \brief        This task must be executed periodically for a proper module functionality.
*
* \param[in]    None
*
* \return       None
************************************************************************************/
void ir_controller_task (void);

/*!**********************************************************************************
* \brief        Gets the current status of the IR Controller module
*
* \param[in]    None
*
* \return       @ref ir_controller_status_t. @ref irControllerStatusOk if free, @ref irControllerStatusModuleBusy if busy
************************************************************************************/
ir_controller_status_t ir_controller_get_current_status (void);

/*!
* @} End of ir_controller_driver
*/

/*!
* @} End of ir_controller
*/


#endif /* End of _IR_CONTROLLER_H_ */