/****************************************************************************
Motor Control
****************************************************************************/

/************************************************************************************************/
/*                                      Includes Section                                        */
/************************************************************************************************/

#include <hidef.h> /* for EnableInterrupts macro */
#include "pub_def.h"
#include "APP_SMAC_API.h"
#include "freescale_radio_hardware.h"
#include "drivers.h"
#include "simple_phy.h"
#include "MotorControl.h"

/************************************************************************************************/
/*                                     Constants Section                                        */
/************************************************************************************************/

#define NoCoil                (0xFF87)
#define GPIO7                 (1<<6)
#define GPIO6                 (1<<5)
#define GPIO5                 (1<<4)
#define GPIO4                 (1<<3)
#define GPIO3                 (1<<2)
const UINT16 u16CoilsArray[] = {GPIO4,GPIO5,GPIO6,GPIO7};

typedef enum
{
     SMACMotorStopState,
     SMACMotorLowSpeedState,
     SMACMotorMedSpeedState,
     SMACMotorHiSpeedState,
     SMACMotorCCWDirectionState,
     SMACMotorCWDirectionState,
     SMACMotorControlState,
     SMACEnableReceptionState,
     SMACIDLEState
}_StateMachine;

/************************************************************************************************/
/*                                Global Variables Section                                      */
/************************************************************************************************/

#pragma DATA_SEG _DATA_ZEROPAGE

UINT8 gu8RTxMode;                            /* needed for s-mac, application can read this variable */
                                             /* gu8RTxMode is the software state variable of the radio */
                                             /* This variable reflects the current state of MC13192 */
INT8 gi8AppStatus = 0;                       /*State machine variable for the application    */
UINT8 gu8LedStatus = 0;                      /*led status reflects which power setting should be set.  */
UINT8 link_quality;                          /*Holds the link quality of the last received ZigBee Packet.*/

UINT8 gau8TxDataBuffer[20];
UINT8 gau8RxDataBuffer[20];
tTxPacket gsTxPacket;
tRxPacket gsRxPacket;

UINT16 u16DelayRotacion = 0;
UINT16 u16NewFreq = 7000;
UINT8 u8CoilIndex = 0;

UINT8 u8ActualState;
UINT8 u8NextState;

UINT8 u8CCWFlag;

#pragma DATA_SEG DEFAULT

/************************************************************************************************/
/*                                      Functions Section                                       */
/************************************************************************************************/

void vfnSMACMotorStopState(void)
{
     u16GPIO &= NoCoil;
     SPIDrvWrite(0x0C,u16GPIO);
     u8ActualState = SMACIDLEState;
}

void vfnSMACMotorLowSpeedState(void)
{
     u16NewFreq = 65500;
     u8ActualState = SMACMotorControlState;
}

void vfnSMACMotorMedSpeedState(void)
{
     u16NewFreq = 28000;
     u8ActualState = SMACMotorControlState;
}

void vfnSMACMotorHiSpeedState(void)
{
     u16NewFreq = 7000;
     u8ActualState = SMACMotorControlState;
}

void vfnSMACMotorCCWDirectionState(void)
{
     u8CCWFlag = 1;
     u8ActualState = SMACMotorControlState;
}

void vfnSMACMotorCWDirectionState(void)
{
     u8CCWFlag = 0;
     u8ActualState = SMACMotorControlState;
}

void vfnSMACMotorControlState(void)
{
     if(!(u16DelayRotacion--))
     {
          u16DelayRotacion = u16NewFreq;
          
          u16GPIO &= NoCoil;
          u16GPIO |= u16CoilsArray[u8CoilIndex];
          SPIDrvWrite(0x0C,u16GPIO);
          
          if(u8CCWFlag)
          {
               u8CoilIndex++;
          }
          else
          {
               u8CoilIndex--;
          }
          u8CoilIndex %= 4;
     }
}

void vfnSMACEnableReceptionState(void)
{
     (void)MLMERXEnableRequest(&gsRxPacket, 0);
     u8ActualState = u8NextState;
}

void vfnSMACIDLEState(void)
{
     
}

const void (* const paSMACFunctions[])(void) =
{
     vfnSMACMotorStopState,
     vfnSMACMotorLowSpeedState,
     vfnSMACMotorMedSpeedState,
     vfnSMACMotorHiSpeedState,
     vfnSMACMotorCCWDirectionState,
     vfnSMACMotorCWDirectionState,
     vfnSMACMotorControlState,
     vfnSMACEnableReceptionState,
     vfnSMACIDLEState
};

void vfnSMACDriver(void)
{
	void (* lpMotorFuncPtr)(void);
	lpMotorFuncPtr = (void *)paSMACFunctions[u8ActualState];
	lpMotorFuncPtr();
}


void main(void) 
{

    /*Initialize the packet.*/
    gsTxPacket.u8DataLength = 0;                  /*Set TX default length to 0*/
    gsTxPacket.pu8Data = &gau8TxDataBuffer[0];    /*Load the address of our txbuffer into the tx structure*/
    gsRxPacket.u8DataLength = 0;                  /*Set RX default length to 0*/
    gsRxPacket.pu8Data = &gau8RxDataBuffer[0];    /*Load the address of our rxbuffer into the rx structure*/
    gsRxPacket.u8MaxDataLength = 20;              /*Define the max buffer we are interested in.*/
    gsRxPacket.u8Status = 0;                      /*initial status variable for rx to 0.*/
    

    MCUInit();                                    /*Initialize the mcu, setup GPIOs, SPI, clock scheme, etc.*/
    RadioInit();

    (void)MLMESetMC13192ClockRate(2);                            /* Set initial Clk speed */
    UseExternalClock();                                          /* switch clock sources */

     EnableInterrupts;                                           /* Turn on system interrupts */
     (void)MLMESetChannelRequest(CHANNEL_NUMBER);            /*Set channel to 0 (2.405GHz)  */
     
     /******************************************************************
     To adjust output power call the MLME_MC13192_PA_output_adjust() with:
     MAX_POWER     (+3 to +5dBm)
     NOMINAL_POWER (0 dBm)
     MIN_POWER     ~(-16dBm)
     or somewhere custom ? (0-15, 11 (NOMINAL_POWER) being Default power)
     ******************************************************************/
     (void)MLMEMC13192PAOutputAdjust(OUTPUT_POWER);              /*Set MAX power setting*/
     

    /* include your start up code here */    
    
      gau8TxDataBuffer[0] = '\n';
      gau8TxDataBuffer[1] = '\r';
      gau8TxDataBuffer[2] = 'M';
      gau8TxDataBuffer[3] = 'O';
      gau8TxDataBuffer[4] = 'T';
      gau8TxDataBuffer[5] = 'O';
      gau8TxDataBuffer[6] = 'R';
      gau8TxDataBuffer[7] = '1';
      gau8TxDataBuffer[8] = '\0';                           /* Sending String */
      gsTxPacket.u8DataLength = 9;
      
      MCPSDataRequest(&gsTxPacket);

    
    u8ActualState = SMACEnableReceptionState;                /* Initial State */
    u8NextState = SMACMotorStopState;
    
    for(;;)
    {
          vfnSMACDriver();                                  /* Main Machine */
    }
}

void MCPSDataIndication(tRxPacket *gsRxPacket) 
{
     u8NextState = u8ActualState;
     u8ActualState = SMACEnableReceptionState;
     
    if (gsRxPacket->u8Status == SUCCESS) 
    {
          /* Execute Command */
          if(((gsRxPacket->pu8Data[0]) >= 'A') && ((gsRxPacket->pu8Data[0]) <= 'F'))
          {
               u8NextState = (gsRxPacket->pu8Data[0]) - 'A';
          }
    }    
}

void MLMEMC13192ResetIndication(void) 
{

}
