;/**************************************************************************
;* FILENAME:      ECC.s                                                    *         
;*                                                                         *  
;* VERSION:       V1.0                                                     *  
;*                                                                         *  
;* DESCRIPTION:   Error Correction Code for NXP LPC2200 interface to NAND  *                       
;*                flash reference design. For more detail, Pls. refer to   * 
;*                corresponding Hamming Code based ECC code articles       *  
;*                                                                         *  
;* TOOLS:         ARM Developer Suite v1.2                                 *
;*                Flash Magic V3.33.157                                    *
;*                                                                         *  
;*                                                                         *  
;* REVISION HISTORY                                                        *  
;* Version  Author           Date         Remarks                          *
;* 1.0      Xiaodong Xie     12/12/2006   Created                          *         
;***************************************************************************/

			AREA	|ECC|, CODE, READONLY

;//***************************************************************************
;//* Function Name: ecc512
;//* Input(s) : 	unsigned char 		*Buffer, 
;//*				unsigned char		*EccByte.
;//* Returns :  	void.
;//* Description : generate ECC code for 512 data bytes
;//* Notes:        EccByte
;//*               D0 - D17   ECC row information from LP0 to LP17
;//*               D18 - D23  ECC column information from CP0 to CP5
;//***************************************************************************
			EXPORT	ecc512
			
ecc512

			;register used in function ecc512
			;R0  the first argument of function ecc512
			;R1  the second argument of function ecc512
			;R2  register to load the first 4 bytes of 512 data bytes 
			;R3  register to load the second 4 bytes of 512 data bytes 
			;R4  register to load the third 4 bytes of 512 data bytes 
			;R5  register to load the fourth 4 bytes of 512 data bytes 
			;R6  byte index of 512 data bytes
			;R7-R12  temp value

	        STMFD   SP!, {R0-R12, LR}	;store R0-R12 & LR in stack
			
			MOV		R6,#0				;init the byte index
			LDR		R11,[R1]			;load EccByte[0] to R11

_loopBuffer
			;load 4 words of 512 data byte at one time
			;single instruction multiple data (SIMD)
			LDMIA	R0!,{R2-R5}		

;Calculating ECC row information

			MOV		R10,R2
			EOR		R10,R10,R3
			EOR		R10,R10,R4
			EOR		R10,R10,R5			;R10 is the EOR result of R2,R3,R4,R5
			MOV		R12,R10				;store R10 in R12

			MOV		R7,R10
			EOR		R7,R7,R10,LSR #1
			EOR		R7,R7,R10,LSR #2
			EOR		R7,R7,R10,LSR #3
			EOR		R7,R7,R10,LSR #4
			EOR		R7,R7,R10,LSR #5
			EOR		R7,R7,R10,LSR #6
			EOR		R7,R7,R10,LSR #7

			MOV		R8,R7
			EOR		R8,R8,R7,LSR #8
			EOR		R8,R8,R7,LSR #16
			EOR		R8,R8,R7,LSR #24
			AND		R8,R8,#1			;1st bit of R8 is the EOR result of each 
										;sigle byte stored in R2,R3,R4,R5

			;calculate ECC LP8-LP17
			MACRO
			LP8_LP17	$NthLP, $NextLP, $IndBitPos
				MOV		R9,#0
				ANDS	R9,R6,#$IndBitPos
				EOREQ	R11,R11,R8,LSL #$NthLP
				EORNE	R11,R11,R8,LSL #$NextLP
			MEND

			LP8_LP17	8, 9, 0x10			;ECC LP8, LP9
			
			LP8_LP17	10, 11, 0x20		;ECC LP10, LP11

			LP8_LP17	12, 13, 0x40		;ECC LP12, LP13

			LP8_LP17	14, 15, 0x80		;ECC LP14, LP15

			LP8_LP17	16, 17, 0x100		;ECC LP16, LP17
			

			;calculate ECC LP0-LP3
			MACRO
			LP0_LP3	$NthLP, $Shifter1, $Shifter2
				IF	$Shifter1 = 0
					MOV		R8,R7
				ELSE
					MOV		R8,R7,LSR #$Shifter1
				ENDIF
				EOR		R8,R8,R7,LSR #$Shifter2
				AND		R8,R8,#1
				IF	$NthLP = 0
					EOR		R11,R11,R8
				ELSE
					EOR		R11,R11,R8,LSL #$NthLP
				ENDIF
			MEND
			
			LP0_LP3	1, 8, 24		;ECC LP1

			LP0_LP3	0, 0, 16		;ECC LP0

			LP0_LP3	3, 16, 24		;ECC LP3
			
			LP0_LP3	2, 0, 8			;ECC LP2


			;calculate LP4-LP7
			MACRO
			LP4_LP7	$NthLP, $Reg1, $Reg2
				MOV		R10,$Reg1
				EOR		R10,R10,$Reg2
				
				MOV		R7,R10
				EOR		R7,R7,R10,LSR #1
				EOR		R7,R7,R10,LSR #2
				EOR		R7,R7,R10,LSR #3
				EOR		R7,R7,R10,LSR #4
				EOR		R7,R7,R10,LSR #5
				EOR		R7,R7,R10,LSR #6
				EOR		R7,R7,R10,LSR #7

				MOV		R8,R7
				EOR		R8,R8,R7,LSR #8
				EOR		R8,R8,R7,LSR #16
				EOR		R8,R8,R7,LSR #24
				AND		R8,R8,#1

				EOR		R11,R11,R8,LSL #$NthLP
			MEND

			LP4_LP7	5, R3, R5		;ECC LP5
			
			LP4_LP7	4, R2, R4		;ECC LP4
			
			LP4_LP7	7, R4, R5		;ECC LP7
		
			LP4_LP7	6, R2, R3		;ECC LP6

			
;Calculating ECC column information
			;calculate CP0-CP7
			MACRO
			CP0_CP7	$NthCP, $Shifter1, $Shifter2, $Shifter3, $Shifter4
				IF	$Shifter1 = 0				
					MOV		R7,R12
				ELSE
					MOV		R7,R12,LSR #$Shifter1
				ENDIF
				EOR		R7,R7,R12,LSR #$Shifter2
				EOR		R7,R7,R12,LSR #$Shifter3
				EOR		R7,R7,R12,LSR #$Shifter4	
				
				MOV		R8,R7
				EOR		R8,R8,R7,LSR #8
				EOR		R8,R8,R7,LSR #16
				EOR		R8,R8,R7,LSR #24
				AND		R8,R8,#1
				
				EOR		R11,R11,R8,LSL #$NthCP
			MEND

			CP0_CP7	18, 0, 2, 4, 6		;ECC CP0
			
			CP0_CP7	19, 1, 3, 5, 7		;ECC CP1
			
			CP0_CP7	20, 0, 1, 4, 5		;ECC CP2
			
			CP0_CP7	21, 2, 3, 6, 7		;ECC CP3
			
			CP0_CP7	22, 0, 1, 2, 3		;ECC CP4

			CP0_CP7	23, 4, 5, 6, 7		;ECC CP5


			ADD		R6,R6,#16			;Buffer[Index + 16]
			CMP		R6,#512				;if (Index + 16) != 512
			BNE		_loopBuffer			;then jump to _loopBuffer
			
			STR		R11,[R1]			;EccByte[0] = R11
			
	        LDMFD   SP!, {R0-R12, LR}
			MOV		PC,LR	

		
;//***************************************************************************
;//* Function Name: eccCheck512
;//* Input(s) : 	unsigned char 		*Buffer, 
;//*				unsigned char	 	*EccByte.
;//*				struct EccAddr 		*ErrAddr
;//* Returns :  	unsigned char	  	EccStatus
;//* Description : ECC based data check and correction for 512 bytes data
;//* Notes:        EccStatus:
;//*               	0		No Error found
;//*               	1		One bit Error found and corrected
;//*				2		Not Correctable Error found
;//***************************************************************************
			EXPORT	eccCheck512

eccCheck512

			STMFD	SP!,{R1-R7,LR}
			BL		ecc512			;jump to ecc512 function

			MVN		R4,#0xFF
			LDR		R3,[R1]
			AND		R3,R3,R4,LSR #8
			
			MOV		R4,#0			;How many "1" in ECC information bytes?
			MOV		R5,#0
_loopCount_1
			CMP		R3,#0
			BEQ		_endLoopCount_1
			ADD		R4,R4,#1;		;R4 stores the number of "1"
			SUB		R5,R3,#1
			AND		R3,R3,R5
			B		_loopCount_1			
_endLoopCount_1			
			
			CMP		R4,#0			;if R4 == 0, no error, R0 = 0
			MOVEQ	R0,#0
			BEQ		_endEccCheck512
			
			CMP		R4,#12			;if R4 != 14, not correctable err, R0 = 2
			MOVNE	R0,#2			
			BNE		_endEccCheck512

			;Correctable error
			;Firstly, calculate the error bit''s bit address			
			MOV		R4,#0
			MOV		R5,#0
			MOV		R6,#0
			LDR		R3,[R1]
						
			MOV		R4,R3,LSR #23

			ANDS	R4,R4,#1
			ADDNE	R5,R5,R4,LSL #2	;If CP5 != 0, R5 += 1<<2

			MOV		R4,R3,LSR #21
			ANDS	R4,R4,#1
			ADDNE	R5,R5,R4,LSL #1	;If CP3 != 0, R5 += 1<<1	
 		
			MOV		R4,R3,LSR #19
			ANDS	R4,R4,#1
			ADDNE	R5,R5,#2		;If CP1 != 0, R5 += #2	
			ADDEQ	R5,R5,#1		;If CP1 == 0, R5 += #1
		
			;R5 now contain the error bit address
			;store error bit''s bit address in ErrAddr.bit_addr
			STRB	R5,[R2,#4]
			
			;Then calcuate the error bit''s byte address
			MOV		R4,#0
			MOV		R5,#0
			MOV		R6,#0
			
			MOV		R4,R3,LSR #17
			ANDS	R4,R4,#1
			ADDNE	R5,R5,R4,LSL #8	;If LP17 is not zero, R5 += 1<<8
			
			MOV		R4,R3,LSR #15
			ANDS	R4,R4,#1
			ADDNE	R5,R5,R4,LSL #7			

			MOV		R4,R3,LSR #13
			ANDS	R4,R4,#1
			ADDNE	R5,R5,R4,LSL #6			

			MOV		R4,R3,LSR #11
			ANDS	R4,R4,#1
			ADDNE	R5,R5,R4,LSL #5			

			MOV		R4,R3,LSR #9
			ANDS	R4,R4,#1
			ADDNE	R5,R5,R4,LSL #4			

			MOV		R4,R3,LSR #7
			ANDS	R4,R4,#1
			ADDNE	R5,R5,R4,LSL #3	

			MOV		R4,R3,LSR #5
			ANDS	R4,R4,#1
			ADDNE	R5,R5,R4,LSL #2

			MOV		R4,R3,LSR #3
			ANDS	R4,R4,#1
			ADDNE	R5,R5,R4,LSL #1	

			MOV		R4,R3,LSR #1
			ANDS	R4,R4,#1
			ADDNE	R5,R5,#2		
			ADDEQ	R5,R5,#1
			;R5 now contain the error byte address
			SUB		R5,R5,#1
			;store erroe bit''s byte address in ErrAddr.byte_addr
			STR		R5,[R2]
			
			MOV		R0,#1	;one error bit detected and corrected, R0 = 1

_endEccCheck512			

			LDMFD	SP!,{R1-R7,LR}
			MOV		PC,LR
			
			END
