/***********************************************************************
 * $Id: DTMF.c 3473 2010-05-17 23:59:27Z nxp27266 $
 *
 * Project: M3 DSP Library Examples
 *
 * Description: Decodes DTMF tones from the result of an FFT and uses
 *              this information to determine the corresponding key.
 *
 * Copyright(C) 2010, NXP Semiconductor
 * All rights reserved.
 *
 ***********************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
 **********************************************************************/
#include "DTMF.h"

/* Define maximum keypad size */
#define MAX_ROWS					4
#define MAX_COLS					3

/* If the power level in DTMF bins as percentage of total power is above
   this threshold then a tone is assumed to be present */
#define DTMF_PERCENT_TOTAL_POWER	25

/* Lookup table for a standard keypad */
static uint8_t au8Keypad[MAX_ROWS][MAX_COLS] =
{
	{ '1' , '2', '3' },
	{ '4' , '5', '6' },
	{ '7' , '8', '9' },
	{ '*' , '0', '#' },
};

/* Lookup tables used to determine which bin in FFT output corresponds
   to which row or column on the keypad */
static uint16_t au16RowBins[MAX_ROWS] = { 22, 25, 27, 30 };
static uint16_t au16ColBins[MAX_COLS] = { 39, 43, 47 };

/* Local function prototype */
static uint8_t u8DTMF_TonePresent(uint32_t *pu32Data, uint16_t u16NbrBins);
static uint8_t u8DTMF_FindMaxPos(uint32_t *pu32Data, uint16_t *pu16SearchTable, uint8_t u8TableLen);

/******************************************************************************
** Function name : u8DTMF_Decode
**
** Description	 : Decode a DTMF tone and determine the corresponding key.
**
** Parameters	 : pu32BinData - Pointer to an array containing the result of
**                 an FFT (magnitude only - no phase information).
**
**                 u16NbrBins - Number of bins contained in the bin data array.
**
** Returned value: The key corresponding to the DTMF tone.
**
******************************************************************************/
uint8_t u8DTMF_Decode(uint32_t *pu32BinData, uint16_t u16NbrBins)
{
	uint8_t u8Col, u8Row;
	uint8_t u8Key = 0;

	/* Only attempt to decode if results contain a DTMF tone */
	if (u8DTMF_TonePresent(pu32BinData, u16NbrBins) > 0)
	{
		/* Find row and column */
		u8Row = u8DTMF_FindMaxPos(pu32BinData, &au16RowBins[0], MAX_ROWS);
		u8Col = u8DTMF_FindMaxPos(pu32BinData, &au16ColBins[0], MAX_COLS);

		/* Determine which key corresponds to the row and column found */
		u8Key = au8Keypad[u8Row][u8Col];
	}
	return (u8Key);
}

/******************************************************************************
** Function name : u16DTMF_FindMaxPos
**
** Description	 : Finds the location of the maximum value in an array. The
**                 locations in the array that are to be searched are contained
**                 in an array pointed to by one of the function parameters.
**
** Parameters	 : pu32Data - Pointer to array which is to be search to find
**                 the location of the maximum value;
**
**                 puSearchTable - Pointer to an array containing the locations
**                 that are to be searched.
**
**                 u8SearchLen - Number of entries to be searched.
**
** Returned value: The location of the entry in the search table that points
**                 to the maximum value found in the data table.
**
******************************************************************************/
static uint8_t u8DTMF_FindMaxPos(uint32_t *pu32Data, uint16_t *pu16SearchTable, uint8_t u8TableLen)
{
	uint8_t  u8Index;
	uint32_t u32MaxValue = 0;
	uint8_t  u8MaxLocation = 0;

	for (u8Index = 0; u8Index < u8TableLen; u8Index++)
	{
		/* Only search subset of data define by contents of puSearchTable */
		if(pu32Data[pu16SearchTable[u8Index]] > u32MaxValue)
		{
			/* This is the maximum value found so far */
			u32MaxValue = pu32Data[pu16SearchTable[u8Index]];
			u8MaxLocation = u8Index;
		}
	}
	return (u8MaxLocation);
}

/******************************************************************************
** Function name : u8DTMF_TonePresent
**
** Description	 : Determines if a DTMF tone is present in FFT results by
**                 checking that the amount of power in the DTMF bins is greater
**                 than the total power across all bins.
**
** Parameters	 : pu32BinData - Pointer to an array containing the result of
**                 an FFT (magnitude only - no phase information).
**
**                 u16NbrBins - Number of bins contained in the bin data array.
**
** Returned value: If no DTMF tone present in FFT result returns 0, otherwise 1.
**
******************************************************************************/
static uint8_t u8DTMF_TonePresent(uint32_t *pu32Data, uint16_t u16NbrBins)
{
	uint16_t i;
	uint8_t u8Result = 0;
	uint32_t u32TotalPower = 0;
	uint32_t u32DTMFPower = 0;

	/* Calculate total power present in all bins */
	for (i = 0; i < (u16NbrBins / 2); i++)
	{
		u32TotalPower += pu32Data[i];
	}

	/* Calculate power contained in DTMF bins */
	for (i = 0; i < MAX_ROWS; i++)
	{
		u32DTMFPower += pu32Data[au16RowBins[i]];
	}
	for (i = 0; i < MAX_COLS; i++)
	{
		u32DTMFPower += pu32Data[au16ColBins[i]];
	}

	/* A tone is considered to be present if the power in the DTMF bins is a
	 * certain percentage greater than the total power present across all bins. */
	if (((u32DTMFPower * 100UL) / u32TotalPower) > DTMF_PERCENT_TOTAL_POWER)
	{
		/* Tone present */
		u8Result = 1;
	}
	return u8Result;
}

/*****************************************************************************
**                            End Of File
*****************************************************************************/
