/**********************************************************************************************************
*      This LPC1200 sample project allows the user to enter and exit the low power saving modes:
*      1) Sleep Mode
*      2) Deep-Sleep Mode (self timed via RTC)
*      3) Deep Power-down Mode (self timed via RTC)
*
*      It also allows the user to measure the wake up times for each low power modes.
*      This project displays a menu via the UART interface.
*      See the "Using LPC111x/x01 and LPC11C1x/x01 power modes on the LPCXpresso" Application Note for further details.
*
*      For Sleep and Deep Sleep modes, the wake-up times can be measured by using the following pins:
*		Trigger pin ( PIO0_1 RXD )
*		Used to get the device out of the sleep mode or deep-sleep mode.
*		The pin is triggered externally (falling edge) to wake the device up using serial character reception.
*		
*		Strobe pin (PIO2_8)
*		After wake-up, the device returns into run mode, and this pin is set high within
*		the WAKEUP_IRQHANDLER() subroutine.
*
*	    The wakeup time is the difference between the falling edge of the trigger pin
*	    and the rising edge of the Strobe pin.
* 
**********************************************************************************************************/
#include "LPC122x.h"
#include "uart.h"
#include "util.h"
#include "rtc.h"

#define UART_BAUD	9600
#define WAKEUP_PIN_MASK (1<<1)
#define WAKEUP_IRQ_SRC (WAKEUP1_IRQn)

extern volatile uint8_t  UARTBuffer0[BUFSIZE];
extern volatile uint32_t UARTCount0;

static void SetIOPinsInactive(void);
static void SetupStartLogicPin(void);
static void SetupStartLogicRTC(void);
static void setupSelfTimedDeepSleep(uint32_t sleepTime);
static void DisplayMenu(void);
static void DeInitUART(void);
static void SetupClock(uint32_t sel);
static void PMU_Sleep( void );
static void PMU_DeepSleep( uint32_t SleepCtrl );
static void PMU_PowerDown( void );
static void WDT_CLK_Setup ( void );

typedef enum{
  PD_SLEEP_IRC='a',
  PD_SLEEP_IRC_PLL = 'b',
  PD_SLEEP_OSC='c',
  PD_SLEEP_OSC_PLL='d',
  PD_DEEP_SLEEP='e',
  PD_DEEP_PD='f',
  PD_DEEP_SLEEP_RTC='g',
  PD_DEEP_PD_RTC='h',
}WAKEUP_COMMAND;

int main (void) 
{	
  const uint8_t strError[] = "\r\nIncorrect Key\r\n";
  uint8_t cmd=0;
  
  UARTInit(0, UART_BAUD);
  
  if (0!=RTC_Init(RTC_CLK_1HZ,0))
  {
    RTC_Load(0);
    print_string(0,"Real Time Clock Initialization Complete.\r\n");
  }
  
  if (LPC_PMU->PCON & (1<<11))
  {
    print_string(0,"Woke from Deep Power Down\r\n");
    LPC_PMU->PCON |= 1<<11;
  }
  DisplayMenu();
  
  /* Loop forever */  
  while (1)
  {	
    __WFI();
    cmd = UARTBuffer0[UARTCount0-1];		
    print_string(0,">");
    UARTSend(0,&cmd,1);
    
    switch(cmd)
    {
      /* Sleep Mode Options */
    case PD_SLEEP_IRC:
    case PD_SLEEP_IRC_PLL:
    case PD_SLEEP_OSC:
    case PD_SLEEP_OSC_PLL:
      print_string(0,"\r\nEntering ");
      switch (cmd)
      {
      case PD_SLEEP_IRC:
        print_string(0,"IRC");
        break;
      case PD_SLEEP_IRC_PLL:
        print_string(0,"IRC+PLL");
        break;
      case PD_SLEEP_OSC:
        print_string(0,"SYSOSC");
        break;
      case PD_SLEEP_OSC_PLL:
        print_string(0,"SYSOSC+PLL");
        break;
      }				
      print_string(0," Sleep...\r\n");
      /* Disable UART */
      DeInitUART();
      SetupClock(cmd-PD_SLEEP_IRC);
      SetupStartLogicPin();											
      SetIOPinsInactive();
      
      /* Specify Sleep mode before entering mode */
      PMU_Sleep();
      NVIC_SystemReset();
      break;
      
      /* Deep Sleep Mode Options */
    case PD_DEEP_SLEEP:
      print_string(0,"\r\nEnterting Deep-Sleep...\r\n");
      /* Disable UART */
      DeInitUART();
      
      SetupClock(0);//Select IRC to generate clean edge.		
      SetIOPinsInactive();
      SetupStartLogicPin();
      
      /* Specify Deep-Sleep mode before entering mode */
      PMU_DeepSleep(0x0000FFFF);
      NVIC_SystemReset();
      break;
      
    case PD_DEEP_SLEEP_RTC:
      print_string(0,"\r\nRTC Timed Deep-Sleep[10s]...\r\n");
      /* Disable UART */
      DeInitUART();
      SetIOPinsInactive();
      setupSelfTimedDeepSleep(10);
      NVIC_SystemReset();
      break;
      
      /* Deep Power-down Mode Options */
    case PD_DEEP_PD:
      print_string(0,"\r\nEnterting Deep Power-Down...\r\n");
      /* Disable UART */
      DeInitUART();
      
      PMU_PowerDown();
      break;
      
      /* Deep Power-down Mode Options */
    case PD_DEEP_PD_RTC:
      print_string(0,"\r\nRTC Timed Deep Power-Down[10s]...\r\n");
      /* Disable UART */
      DeInitUART();
      RTC_Match(10);
      PMU_PowerDown();
      break;
      
      /* Default case whenever an incorrect cmd has been entered */
    default:
      UARTSend(0, (uint8_t*)strError, sizeof(strError));
      DisplayMenu();
      break;
      
    }											  
  }
}

/* To get accurate wake up time measurements, this is called in startup ASM */
void startupTogglePin(void)
{
  if (LPC_PMU->PCON & (1<<11))
  {
    /*Toggle IO! - Used to measure time to wake from Deep Power Down*/
    LPC_SYSCON->SYSAHBCLKCTRL |= 0xE0010000UL;
    LPC_GPIO2->DIR = (1<<8);
    LPC_GPIO2->CLR = (1<<8);
    LPC_GPIO2->NOT = (1<<8);
    
    /* Do NOT clear flag, cleared later in main... */
  }
}

/* To get accurate wake up time measurements, P2_8 is toggled in the ISR */
void WAKEUP_IRQHandler(void)
{
  /* Toggle IO */
  LPC_SYSCON->SYSAHBCLKCTRL |= 0xE0010000UL;
  LPC_GPIO2->NOT = (1<<8);
  LPC_GPIO2->NOT = (1<<8);
  
  LPC_SYSCON->STARTRSRP0CLR |= WAKEUP_PIN_MASK;
  return;
}

/* Configure WD Oscilator for lowest frequency */
static void WDT_CLK_Setup ( void )
{
  /* Watchdog configuration. */
  /* Freq = 0.5MHz, div_sel is 0x1F, divided by 64. WDT_OSC should be 7.8125khz */
  LPC_SYSCON->WDTOSCCTRL = (0x1<<5)|0x1F;
  LPC_SYSCON->PDRUNCFG &= ~(0x1<<6);    /* Let WDT clock run */

  /* Enables clock for WDT */
  LPC_SYSCON->SYSAHBCLKCTRL |= (1<<15);
  LPC_WWDT->CLKSEL = 0x1;
  return;
}

static void PMU_Sleep( void )
{
  SCB->SCR &= ~(1<<2);
  LPC_PMU->PCON &= ~(1<<1);
  
  /* Sleep until Interrupt, BOD or Reset */
  __WFI();						
  return;
}

static void PMU_DeepSleep( uint32_t SleepCtrl )
{
  /* Restore existing Power-Down settings after wakeup */
  LPC_SYSCON->PDAWAKECFG = LPC_SYSCON->PDRUNCFG;
  
  /* Only 4 possible values can be used */
  switch(SleepCtrl)
  {
  case 0x0000FFB7:	/* BOD and WD osc. enabled */
  case 0x0000FFBF:	/* BOD disabled, WD osc. enabled */
  case 0x0000FFF7:	/* BOD enabled, WD osc. disabled */
    LPC_SYSCON->PDSLEEPCFG = SleepCtrl;
    break;
  default:			/* BOD and WD osc. disabled */
    LPC_SYSCON->PDSLEEPCFG = 0x0000FFFF;
    break;
  }
  
  /* NVIC_LP_SLEEPDEEP defined as (0x04) */
  SCB->SCR |= (1<<2);
  LPC_PMU->PCON &= ~(1<<1);
  
  __WFI();
  return;
}

static void PMU_PowerDown( void )
{
  /* Specify DPDEN to power control register */
  LPC_PMU->PCON	= (1<<1)|(1<<11);	
  /* Specify Deep Power-down mode before entering mode */
  SCB->SCR	|=	(1<<2);
  
  __WFI();
  return;
}

/*  Configure system for self timed wake up from deep sleep.
 *  This requires use of Start Logic 1, RTC, WDO and PMU.
 */
static void setupSelfTimedDeepSleep(uint32_t sleepTime)
{
  SetupStartLogicRTC();  
  RTC_Match(sleepTime);
  
  /* See ERRATA - Deep Sleep wake from RTC requires use of WDOsc*/  
  SetupClock(4);
  PMU_DeepSleep(0x0000FFBF);
}

/* Falling edge of RXD will wake device from Deep-Sleep */
static void SetupStartLogicPin(void)
{
  NVIC_DisableIRQ(WAKEUP_IRQ_SRC);
  
  /* Falling Edge */
  LPC_SYSCON->STARTAPRP0 &= ~(WAKEUP_PIN_MASK);
  LPC_SYSCON->STARTERP0 |= (WAKEUP_PIN_MASK);	
  LPC_SYSCON->STARTRSRP0CLR |= (WAKEUP_PIN_MASK);
  
  NVIC_ClearPendingIRQ(WAKEUP_IRQ_SRC);
  NVIC_EnableIRQ(WAKEUP_IRQ_SRC);
}

/* RTC Match Interrupt will wake device from Deep-Sleep */
static void SetupStartLogicRTC(void)
{
  /* Enable Start Logic 1 to wake from RTC */
  LPC_SYSCON->STARTAPRP1 = (1<<18);
  LPC_SYSCON->STARTRSRP1CLR = (1<<18);
  LPC_SYSCON->STARTERP1 = (1<<18);
}

/* Function prints a menu of options onto the UART */
static void DisplayMenu(void)
{
  print_string(0,    "\r\nSELECT WHICH MODE YOU WANT TO TEST");
  print_string(0,    " - LPC1220\r\n");
  print_string(0,    "Device ID:");
  print_hex32(0,LPC_SYSCON->DEVICE_ID);
  print_string(0,    "\r\nSLEEP mode\r\n");
  print_string(0,    "    'a' - IRC (12MHz)\r\n");
  print_string(0,    "    'b' - IRC with PLL (24MHz)\r\n");
  print_string(0,    "    'c' - System Osc. (12MHz)\r\n");
  print_string(0,    "    'd' - System Osc. with PLL (24MHz)\r\n");
  print_string(0,    "\r\n");
  print_string(0,    "DEEP SLEEP mode\r\n");
  print_string(0,    "    'e' - All clocks off\r\n");
  print_string(0,    "\r\n");
  print_string(0,    "DEEP POWER DOWN mode\r\n");
  print_string(0,    "    'f' - Enter DEEP POWER DOWN\r\n");
  print_string(0,    "\r\n");  
  print_string(0,    "RTC Timed DEEP SLEEP mode [10s]\r\n");
  print_string(0,    "    'g' - All clocks off\r\n");
  print_string(0,    "\r\n");
  print_string(0,    "RTC Timed  DEEP POWER DOWN mode [10s]\r\n");
  print_string(0,    "    'h' - Enter DEEP POWER DOWN\r\n");
  print_string(0,    "\r\n");  
  print_string(0,    "Option:");
}

/* Function turns off the UART peripheral for the current test */
static void DeInitUART(void)
{
  /* Disable NVIC UART interrupts */
  NVIC_DisableIRQ(UART0_IRQn);
  NVIC_ClearPendingIRQ(UART0_IRQn);
  LPC_UART0->IER = 0;
  
  /* Set the UART clock divider to 0, disabling the clock */
  LPC_SYSCON->UART0CLKDIV = 0;		   
  
  /* Power down the UART */
  LPC_SYSCON->SYSAHBCLKCTRL &= ~(1<<12);
  return;
}

/*  Used to select from one of four clock configurations:
    IRC
    SYSOSC
    IRC+PLL
    SYSOSC+PLL
    WDOsc (Self Timed Wakeup from RTC)
*/
static void SetupClock(uint32_t sel)
{
  uint32_t enPLL = 0;
  uint32_t enPLL_IN = 0;
  uint32_t sourcePLL = 0;
  uint32_t sourceMain = 0;
  uint32_t powerBits = 0;
  volatile uint32_t i;
  
  switch(sel)
  {
  case 0x00:	/* IRC Only */
    enPLL = 0;
    enPLL_IN = 0;
    sourcePLL = 0;
    sourceMain = 0x0;
    /* IRC - Turn Turn off SYSOSC,WDOSC and PLL */
    powerBits = ((1<<7)|(1<<6)|(1<<5));
    break;
    
  case 0x01:	/* PLL w IRC */
    enPLL = 1;
    enPLL_IN = 1;
    sourcePLL = 0;
    sourceMain = 0x3;
    /* IRC and PLL - Turn off SYSOSC, WDOSC */
    powerBits = ((1<<6)|(1<<5));
    break;		
    
  case 0x02:	/* SYSOSC Only */
    enPLL = 0;
    enPLL_IN = 1;
    sourcePLL = 1;
    sourceMain = 0x1;
    /* SYSOSC - Turn off PLL,WDOSC and IRC */
    powerBits = ((1<<7)|(1<<6)|(1<<1)|(1<<0));
    break;
    
  case 0x03:	/* PLL w SYSOSC */
    enPLL = 1;
    enPLL_IN = 1;
    sourcePLL = 1;
    sourceMain = 0x3;
    /* SYSOSC and PLL - Turn off IRC and WDOSC */
    powerBits = ((1<<6)|(1<<1)|(1<<0));
    break;
    
  case 0x4:
    enPLL = 0;
    enPLL_IN = 0;
    sourcePLL = 0;
    sourceMain = 0x2;
    /* WDOSC- Turn Turn off SYSOSC,IRC and PLL */
    powerBits = ((1<<7)|(1<<5)|(1<<1)|(1<<0));
    /* Power Up WD Osc */
    WDT_CLK_Setup();
    break;
  }
  
  /* Momentarily enable all bits */ 
  LPC_SYSCON->PDRUNCFG &= ~((1<<7)|(1<<6)|(1<<5)|(1<<1)|(1<<0));
  
  /* 	Wait 200us for OSC to be stablized, no status 
  indication, dummy wait. */
  for ( i = 0; i < 0x100; i++ );

  /* Switch to IRC initially */
  LPC_SYSCON->MAINCLKSEL = 0x00;
  LPC_SYSCON->MAINCLKUEN = 0x01;		
  LPC_SYSCON->MAINCLKUEN = 0x00;		
  LPC_SYSCON->MAINCLKUEN = 0x01;
  
  if (enPLL_IN)
  {
    LPC_SYSCON->SYSPLLCLKSEL = sourcePLL;
    LPC_SYSCON->SYSPLLCLKUEN = 0x01;
    LPC_SYSCON->SYSPLLCLKUEN = 0x00;
    LPC_SYSCON->SYSPLLCLKUEN = 0x01;
    while ( !(LPC_SYSCON->SYSPLLCLKUEN & 0x01) );
  }
  
  /* Fixed PLL Settings (2x) */
  if (enPLL)
  {
    LPC_SYSCON->SYSPLLCTRL = (1<<5) | 1;
    while ( !(LPC_SYSCON->SYSPLLSTAT & 0x01) );	/* Wait until it's locked */
  }	
  
  LPC_SYSCON->MAINCLKSEL = sourceMain;
  LPC_SYSCON->MAINCLKUEN = 0x01;		
  LPC_SYSCON->MAINCLKUEN = 0x00;		
  LPC_SYSCON->MAINCLKUEN = 0x01;
  while ( !(LPC_SYSCON->MAINCLKUEN & 0x01) );	/* Wait until updated */

  /* Shut power down to the peripherals which weren't neeed */  
  LPC_SYSCON->PDRUNCFG |= powerBits;
  return;
}

/* SetIOPinsInactive - Will turn all unused GPIO pins
 *  to a state of output low, and disable internal Pull-ups.
 *
 * Note - The code here is intentionally verbose to allow easier
 *  reconfiguration/retargeting.
 */
static void SetIOPinsInactive(void)
{
  LPC_SYSCON->SYSAHBCLKCTRL |= 0xE0010000UL;
  /* Configure all IOs as GPIO w/o pull-up resistors */
  LPC_IOCON->PIO0_0			&= ~(0x1F);
  /* 
  Most PCBs will wake on falling edge of RXD.
  If another pin is used to define WAKEUP_PIN_MASK,
  comment the line below which disables internal pull ups
  */
  //LPC_IOCON->PIO0_1			&= ~(0x1F);	/*Don't contend with RS-232 Driver*/
  //LPC_IOCON->PIO0_2			&= ~(0x1F);	/*Don't Transmit null characters */
  LPC_IOCON->PIO0_3			&= ~(0x1F);
  LPC_IOCON->PIO0_4			&= ~(0x1F);
  LPC_IOCON->PIO0_5			&= ~(0x1F);
  LPC_IOCON->PIO0_6			&= ~(0x1F);
  LPC_IOCON->PIO0_7			&= ~(0x1F);
  LPC_IOCON->PIO0_8			&= ~(0x1F);
  LPC_IOCON->PIO0_9			&= ~(0x1F);
  LPC_IOCON->PIO0_10			&= ~(0x1F);
  LPC_IOCON->PIO0_11			&= ~(0x1F);
  LPC_IOCON->PIO0_12			&= ~(0x1F);
  LPC_IOCON->RESET_PIO0_13		&= ~(0x1F);
  LPC_IOCON->PIO0_14			&= ~(0x1F);
  LPC_IOCON->PIO0_15			&= ~(0x1F);
  LPC_IOCON->PIO0_16			&= ~(0x1F);
  LPC_IOCON->PIO0_17			&= ~(0x1F);
  LPC_IOCON->PIO0_18			&= ~(0x1F);
  LPC_IOCON->PIO0_19			&= ~(0x1F);
  LPC_IOCON->PIO0_20			&= ~(0x1F);
  LPC_IOCON->PIO0_21			&= ~(0x1F);
  LPC_IOCON->PIO0_22			&= ~(0x1F);
  LPC_IOCON->PIO0_23			&= ~(0x1F);
  LPC_IOCON->PIO0_24			&= ~(0x1F);
  LPC_IOCON->SWDIO_PIO0_25	  	&= ~(0x1F);
  LPC_IOCON->SWCLK_PIO0_26     	&= ~(0x1F);
  LPC_IOCON->PIO0_27			&= ~(0x1F);
  LPC_IOCON->PIO0_28			&= ~(0x1F);
  LPC_IOCON->PIO0_29			&= ~(0x1F);
  LPC_IOCON->R_PIO0_30			&= ~(0x1F);
  LPC_IOCON->R_PIO0_31			&= ~(0x1F);
  
  LPC_IOCON->R_PIO1_0			&= ~(0x1F);
  LPC_IOCON->R_PIO1_1			&= ~(0x1F);
  LPC_IOCON->PIO1_2			&= ~(0x1F);
  LPC_IOCON->PIO1_3			&= ~(0x1F);
  LPC_IOCON->PIO1_4			&= ~(0x1F);
  LPC_IOCON->PIO1_5			&= ~(0x1F);
  LPC_IOCON->PIO1_6			&= ~(0x1F);
  
  LPC_IOCON->PIO2_0			&= ~(0x1F);
  LPC_IOCON->PIO2_1			&= ~(0x1F);
  LPC_IOCON->PIO2_2			&= ~(0x1F);
  LPC_IOCON->PIO2_3			&= ~(0x1F);
  LPC_IOCON->PIO2_4			&= ~(0x1F);
  LPC_IOCON->PIO2_5			&= ~(0x1F);
  LPC_IOCON->PIO2_6			&= ~(0x1F);
  LPC_IOCON->PIO2_7			&= ~(0x1F);
  LPC_IOCON->PIO2_8			&= ~(0x1F);
  LPC_IOCON->PIO2_9			&= ~(0x1F);
  LPC_IOCON->PIO2_10			&= ~(0x1F);
  LPC_IOCON->PIO2_11			&= ~(0x1F);
  LPC_IOCON->PIO2_12			&= ~(0x1F);
  LPC_IOCON->PIO2_13			&= ~(0x1F);
  LPC_IOCON->PIO2_14			&= ~(0x1F);
  LPC_IOCON->PIO2_15			&= ~(0x1F);
  
  /* GPIOs at outputs */
  LPC_GPIO0->DIR = ~((WAKEUP_PIN_MASK)|(1<<2)|(1<<1));
  LPC_GPIO1->DIR = 0xFFFFFFFF;
  LPC_GPIO2->DIR = 0xFFFFFFFF;
  
  /* GPIO outputs to LOW */
  LPC_GPIO0->CLR = ~((WAKEUP_PIN_MASK)|(1<<2)|(1<<1));
  LPC_GPIO1->CLR = 0xFFFFFFFF;
  LPC_GPIO2->CLR = 0xFFFFFFFF;
  
  LPC_SYSCON->SYSAHBCLKCTRL = ~0xE0000000UL;
  return;
}
