/****************************************************************************
 *   $Id:: util.c 3904 2010-07-21 17:13:39Z usb00423                        $
 *   Project: NXP LPC122x utility APIs
 *
 *   Description:
 *     This file contains some utility APIs mostly for debugging.
 *
 ****************************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
****************************************************************************/
#include "LPC122x.h"			/* LPC122x Peripheral Registers */
#include "uart.h"
#include "util.h"

/*****************************************************************************
** Function name:		ConvertDigital
**
** Descriptions:		Convert single digit to ASCII
**
** parameters:		single digit
** Returned value:	ascii value
** 
*****************************************************************************/
uint8_t ConvertDigital ( uint8_t digital )
{
  uint8_t ascii_char;

  if ( digital <= 9 )
  {
	ascii_char = digital + 0x30;	/* 0~9 */
  }
  else
  {
	ascii_char = digital - 0x0A;
	ascii_char += 0x41;				/* A~F */
  }
  return ( ascii_char );
}

/*****************************************************************************
** Function name:		print_string
**
** Descriptions:		Dump ascii string to the serial port
**
** parameters:		serial port number, 32-bit hex value
** Returned value:	None.
** 
*****************************************************************************/
void print_string( uint32_t port_num, uint8_t *string_ptr  )
{
  uint8_t UARTBuffer0[40];
  uint32_t str_len = 0;

  while ( (*string_ptr != '\0') && (str_len <= 40) )
  {
    UARTBuffer0[str_len] = *string_ptr++;
    str_len++;
  }
  UARTSend( port_num, (uint8_t *)UARTBuffer0, str_len );
  return;
}

/*****************************************************************************
** Function name:		print_hex32
**
** Descriptions:		Dump 32-bit hex to the serial port
**
** parameters:		serial port number, 32-bit hex value
** Returned value:	None.
** 
*****************************************************************************/
void print_hex32( uint32_t port_num, uint32_t hexvalue )
{
  uint8_t UARTBuffer0[10];

  UARTBuffer0[0] = ConvertDigital( (uint8_t)(hexvalue>>28));
  UARTBuffer0[1] = ConvertDigital( (uint8_t)((hexvalue>>24)&0xF));
  UARTBuffer0[2] = ConvertDigital( (uint8_t)((hexvalue>>20)&0xF));
  UARTBuffer0[3] = ConvertDigital( (uint8_t)((hexvalue>>16)&0xF));
  UARTBuffer0[4] = ConvertDigital( (uint8_t)((hexvalue>>12)&0xF));
  UARTBuffer0[5] = ConvertDigital( (uint8_t)((hexvalue>>8)&0xF));
  UARTBuffer0[6] = ConvertDigital( (uint8_t)((hexvalue>>4)&0xF));
  UARTBuffer0[7] = ConvertDigital( (uint8_t)(hexvalue&0xF));
  UARTBuffer0[8] = '\r';
  UARTBuffer0[9] = '\n';
  UARTSend( port_num, (uint8_t *)UARTBuffer0, 10 );
  return;
}

/*****************************************************************************
** Function name:		print_hex16
**
** Descriptions:		Dump 16-bit hex to the serial port
**
** parameters:		serial port number, 16-bit hex value
** Returned value:	None.
** 
*****************************************************************************/
void print_hex16( uint32_t port_num, uint16_t hexvalue )
{
  uint8_t UARTBuffer0[6];

  UARTBuffer0[0] = ConvertDigital( (uint8_t)((hexvalue>>12)&0xF));
  UARTBuffer0[1] = ConvertDigital( (uint8_t)((hexvalue>>8)&0xF));
  UARTBuffer0[2] = ConvertDigital( (uint8_t)((hexvalue>>4)&0xF));
  UARTBuffer0[3] = ConvertDigital( (uint8_t)(hexvalue&0xF));
  UARTBuffer0[4] = '\r';
  UARTBuffer0[5] = '\n';
  UARTSend( port_num, (uint8_t *)UARTBuffer0, 6 );
  return;
}

/*****************************************************************************
** Function name:		print_hex8
**
** Descriptions:		Dump 8-bit hex to the serial port
**
** parameters:		serial port number, 8-bit hex value
** Returned value:	None.
** 
*****************************************************************************/
void print_hex8( uint32_t port_num, uint8_t hexvalue )
{
  uint8_t UARTBuffer0[4];

  UARTBuffer0[0] = ConvertDigital( (uint8_t)((hexvalue>>4)&0xF));
  UARTBuffer0[1] = ConvertDigital( (uint8_t)(hexvalue&0xF));
  UARTBuffer0[2] = '\r';
  UARTBuffer0[3] = '\n';
  UARTSend( port_num, (uint8_t *)UARTBuffer0, 4 );
  return;
}

/******************************************************************************
**                            End Of File
******************************************************************************/
