/****************************************************************************
 *   $Id:: rtc.c 7002 2011-03-28 19:39:10Z nxp28548                         $
 *   Project: NXP LPC122x real-time clock(RTC) example
 *
 *   Description:
 *     This file contains RTC code example which include RTC 
 *     initialization, RTC interrupt handler, and related APIs for 
 *     RTC setup.
 *
 ****************************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.
****************************************************************************/
#include "LPC122x.h"
#include "rtc.h"

/*****************************************************************************
** Function name:		RTC_IRQHandler
**
** Descriptions:		Real Time Clock Interrupt Handler
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void RTC_IRQHandler(void)
{
  uint32_t i;
  LPC_RTC->ICR = 0x01; /* Clear RTC interrupt */
  
  LPC_RTC->DR;
  i = LPC_RTC->DR;
  LPC_RTC->MR = i + RTC_MATCH;	/* Set match value */
  
  LPC_GPIO1->NOT |= ((1<<4)|(1<<5));
  return;
}																		  

/* The act of loading the RTC doesn't require disabling interrupts, however
 * the match register is presumed to be invalid after a load, so this code
 * will also mask interrupts and clear the interrupt flag*/
void RTC_Load(uint32_t timerValue)
{
	LPC_RTC->ICR = 0x01;	/* Clear any interrupts */
	LPC_RTC->ICSC = 0x00;	/* Mask the RTC Interrupt */
	NVIC_DisableIRQ(RTC_IRQn);

	/* Set Load value */
	LPC_RTC->LR = timerValue;
}

/* The act of setting the match register doesn't require interrupts to be
 * disabled, however in the general case it is safer to do so. */
void RTC_Match(uint32_t matchValue)
{
	/* Critical Region */
	NVIC_DisableIRQ(RTC_IRQn);
	LPC_RTC->ICSC = 0x00;
	/* Clear any interrupts */
	LPC_RTC->ICR = 0x01;

	LPC_RTC->DR;
	LPC_RTC->MR = LPC_RTC->DR + matchValue;

	/* Enable the RTC Interrupt */
	LPC_RTC->ICSC = 0x01;
	NVIC_EnableIRQ(RTC_IRQn);
}

/*****************************************************************************
 ** Function name:		RTC_Init
 **
 ** Descriptions:		Initialize Real-time clock. timer.
 **
 ** parameters:			None
 ** Returned value:		0 : None POR reset occurred
 ** 					non-0 : POR reset occurred
 **
 *****************************************************************************/
uint32_t RTC_Init(RTC_CLK_SEL mode, uint32_t pclkDiv)
{
	/* Check for POR */
	if((LPC_SYSCON->SYSRESSTAT & 0x1) && ( (LPC_PMU->PCON&(1<<11))==0))
	{
		/* Make sure clock is off prior to changing the RTC_CLK source */
		LPC_SYSCON->SYSAHBCLKCTRL &= ~(1 << 19);

		LPC_PMU->SYSCFG &= ~(0xF << 11);
		switch (mode)
		{
		case RTC_CLK_PCLK:
			LPC_SYSCON->RTCCLKDIV = pclkDiv;
		case RTC_CLK_DELAY_1HZ:
		case RTC_CLK_1KHZ:
			LPC_PMU->SYSCFG |= ((mode & 0xF) << 11);
			break;
		default:
			//Bits are already cleared, so default to 1Hz
			break;
		}
		LPC_SYSCON->SYSAHBCLKCTRL |= (1 << 19);
		/* Pending interrupts after POR shouldn't be possible, but just in case... */
		LPC_RTC->ICR = 0x01;

		LPC_RTC->CR=1;					/* Enable RTC Start */
		LPC_SYSCON->SYSRESSTAT = 0x1;	/* Clear POR flag */

		/* !! Handle ERRATA - see ERRATA Sheet !! */
		while(LPC_RTC->DR <3 )
		{
		  __NOP();
		}
		return 1;
	}
	else
	{
		/* Enable access to RTC registers */
		LPC_SYSCON->SYSAHBCLKCTRL |= (1 << 19);
                LPC_RTC->ICR = 0x01;
                NVIC_ClearPendingIRQ(RTC_IRQn);
		return 0;
	}
}


/******************************************************************************
**                            End Of File
******************************************************************************/
