/*****************************************************************************
 *   uart.c:  UART API file for NXP LPC18xx Family Microprocessors
 *
 *   Copyright(C) 2010, NXP Semiconductor
 *   All rights reserved.
 *
 *   History
 *   2010.04.20  ver 1.00    Preliminary version, first Release
 *
******************************************************************************/
#include "lpc18xx.h"
#include "type.h"
#include "uart.h"
#include "scu.h"

volatile uint32_t UART0Status, UART1Status, UART2Status, UART3Status;
volatile uint8_t UART0TxEmpty = 1, UART1TxEmpty = 1, UART2TxEmpty = 1, UART3TxEmpty = 1;
volatile uint8_t UART0Buffer[BUFSIZE], UART1Buffer[BUFSIZE], UART2Buffer[BUFSIZE], UART3Buffer[BUFSIZE];
volatile uint32_t UART0Count = 0, UART1Count = 0, UART2Count = 0, UART3Count = 0;

/*****************************************************************************
** Function name:		UART0_IRQHandler
**
** Descriptions:		UART0 interrupt handler
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void UART0_IRQHandler (void) 
{
  uint8_t IIRValue, LSRValue;
  uint8_t Dummy = Dummy;
	
  IIRValue = LPC_USART0->IIR;
    
  IIRValue >>= 1;			/* skip pending bit in IIR */
  IIRValue &= 0x07;			/* check bit 1~3, interrupt identification */
  if ( IIRValue == IIR_RLS )		/* Receive Line Status */
  {
	LSRValue = LPC_USART0->LSR;
	/* Receive Line Status */
	if ( LSRValue & (LSR_OE|LSR_PE|LSR_FE|LSR_RXFE|LSR_BI) )
	{
	  /* There are errors or break interrupt */
	  /* Read LSR will clear the interrupt */
	  UART0Status = LSRValue;
	  Dummy = LPC_USART0->RBR;		/* Dummy read on RX to clear 
							interrupt, then bail out */
	  return;
	}
	if ( LSRValue & LSR_RDR )	/* Receive Data Ready */			
	{
	  /* If no error on RLS, normal ready, save into the data buffer. */
	  /* Note: read RBR will clear the interrupt */
	  UART0Buffer[UART0Count] = LPC_USART0->RBR;
	  UART0Count++;
	  if ( UART0Count == BUFSIZE )
	  {
		UART0Count = 0;		/* buffer overflow */
	  }	
	}
  }
  else if ( IIRValue == IIR_RDA )	/* Receive Data Available */
  {
	/* Receive Data Available */
	UART0Buffer[UART0Count] = LPC_USART0->RBR;
	UART0Count++;
	if ( UART0Count == BUFSIZE )
	{
	  UART0Count = 0;		/* buffer overflow */
	}
  }
  else if ( IIRValue == IIR_CTI )	/* Character timeout indicator */
  {
	/* Character Time-out indicator */
	UART0Status |= 0x100;		/* Bit 9 as the CTI error */
  }
  else if ( IIRValue == IIR_THRE )	/* THRE, transmit holding register empty */
  {
	/* THRE interrupt */
	LSRValue = LPC_USART0->LSR;		/* Check status in the LSR to see if
									valid data in U0THR or not */
	if ( LSRValue & LSR_THRE )
	{
	  UART0TxEmpty = 1;
	}
	else
	{
	  UART0TxEmpty = 0;
	}
  }
  return;    
}

/*****************************************************************************
** Function name:		UART1_IRQHandler
**
** Descriptions:		UART1 interrupt handler
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void UART1_IRQHandler (void) 
{
  uint8_t IIRValue, LSRValue;
  uint8_t Dummy = Dummy;
	
  IIRValue = LPC_UART1->IIR;
    
  IIRValue >>= 1;			/* skip pending bit in IIR */
  IIRValue &= 0x07;			/* check bit 1~3, interrupt identification */
  if ( IIRValue == IIR_RLS )		/* Receive Line Status */
  {
	LSRValue = LPC_UART1->LSR;
	/* Receive Line Status */
	if ( LSRValue & (LSR_OE|LSR_PE|LSR_FE|LSR_RXFE|LSR_BI) )
	{
	  /* There are errors or break interrupt */
	  /* Read LSR will clear the interrupt */
	  UART1Status = LSRValue;
	  Dummy = LPC_UART1->RBR;		/* Dummy read on RX to clear 
								interrupt, then bail out */
	  return;
	}
	if ( LSRValue & LSR_RDR )	/* Receive Data Ready */			
	{
	  /* If no error on RLS, normal ready, save into the data buffer. */
	  /* Note: read RBR will clear the interrupt */
	  UART1Buffer[UART1Count] = LPC_UART1->RBR;
	  UART1Count++;
	  if ( UART1Count == BUFSIZE )
	  {
		UART1Count = 0;		/* buffer overflow */
	  }	
	}
  }
  else if ( IIRValue == IIR_RDA )	/* Receive Data Available */
  {
	/* Receive Data Available */
	UART1Buffer[UART1Count] = LPC_UART1->RBR;
	UART1Count++;
	if ( UART1Count == BUFSIZE )
	{
	  UART1Count = 0;		/* buffer overflow */
	}
  }
  else if ( IIRValue == IIR_CTI )	/* Character timeout indicator */
  {
	/* Character Time-out indicator */
	UART1Status |= 0x100;		/* Bit 9 as the CTI error */
  }
  else if ( IIRValue == IIR_THRE )	/* THRE, transmit holding register empty */
  {
	/* THRE interrupt */
	LSRValue = LPC_UART1->LSR;		/* Check status in the LSR to see if
								valid data in U0THR or not */
	if ( LSRValue & LSR_THRE )
	{
	  UART1TxEmpty = 1;
	}
	else
	{
	  UART1TxEmpty = 0;
	}
  }
  return;
}

/*****************************************************************************
** Function name:		UART2_IRQHandler
**
** Descriptions:		UART2 interrupt handler
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void UART2_IRQHandler (void) 
{
  uint8_t IIRValue, LSRValue;
  uint8_t Dummy = Dummy;
	
  IIRValue = LPC_USART2->IIR;
    
  IIRValue >>= 1;			/* skip pending bit in IIR */
  IIRValue &= 0x07;			/* check bit 1~3, interrupt identification */
  if ( IIRValue == IIR_RLS )		/* Receive Line Status */
  {
	LSRValue = LPC_USART2->LSR;
	/* Receive Line Status */
	if ( LSRValue & (LSR_OE|LSR_PE|LSR_FE|LSR_RXFE|LSR_BI) )
	{
	  /* There are errors or break interrupt */
	  /* Read LSR will clear the interrupt */
	  UART2Status = LSRValue;
	  Dummy = LPC_USART2->RBR;		/* Dummy read on RX to clear 
							interrupt, then bail out */
	  return;
	}
	if ( LSRValue & LSR_RDR )	/* Receive Data Ready */			
	{
	  /* If no error on RLS, normal ready, save into the data buffer. */
	  /* Note: read RBR will clear the interrupt */
	  UART2Buffer[UART2Count] = LPC_USART2->RBR;
	  UART2Count++;
	  if ( UART2Count == BUFSIZE )
	  {
		UART2Count = 0;		/* buffer overflow */
	  }	
	}
  }
  else if ( IIRValue == IIR_RDA )	/* Receive Data Available */
  {
	/* Receive Data Available */
	UART2Buffer[UART2Count] = LPC_USART2->RBR;
	UART2Count++;
	if ( UART2Count == BUFSIZE )
	{
	  UART2Count = 0;		/* buffer overflow */
	}
  }
  else if ( IIRValue == IIR_CTI )	/* Character timeout indicator */
  {
	/* Character Time-out indicator */
	UART2Status |= 0x100;		/* Bit 9 as the CTI error */
  }
  else if ( IIRValue == IIR_THRE )	/* THRE, transmit holding register empty */
  {
	/* THRE interrupt */
	LSRValue = LPC_USART2->LSR;		/* Check status in the LSR to see if
									valid data in U0THR or not */
	if ( LSRValue & LSR_THRE )
	{
	  UART2TxEmpty = 1;
	}
	else
	{
	  UART2TxEmpty = 0;
	}
  }
  return;    
}

/*****************************************************************************
** Function name:		UART3_IRQHandler
**
** Descriptions:		UART3 interrupt handler
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void UART3_IRQHandler (void) 
{
  uint8_t IIRValue, LSRValue;
  uint8_t Dummy = Dummy;
	
  IIRValue = LPC_USART3->IIR;
    
  IIRValue >>= 1;			/* skip pending bit in IIR */
  IIRValue &= 0x07;			/* check bit 1~3, interrupt identification */
  if ( IIRValue == IIR_RLS )		/* Receive Line Status */
  {
	LSRValue = LPC_USART3->LSR;
	/* Receive Line Status */
	if ( LSRValue & (LSR_OE|LSR_PE|LSR_FE|LSR_RXFE|LSR_BI) )
	{
	  /* There are errors or break interrupt */
	  /* Read LSR will clear the interrupt */
	  UART3Status = LSRValue;
	  Dummy = LPC_USART3->RBR;		/* Dummy read on RX to clear 
							interrupt, then bail out */
	  return;
	}
	if ( LSRValue & LSR_RDR )	/* Receive Data Ready */			
	{
	  /* If no error on RLS, normal ready, save into the data buffer. */
	  /* Note: read RBR will clear the interrupt */
	  UART3Buffer[UART3Count] = LPC_USART3->RBR;
	  UART3Count++;
	  if ( UART3Count == BUFSIZE )
	  {
		UART3Count = 0;		/* buffer overflow */
	  }	
	}
  }
  else if ( IIRValue == IIR_RDA )	/* Receive Data Available */
  {
	/* Receive Data Available */
	UART3Buffer[UART3Count] = LPC_USART3->RBR;
	UART3Count++;
	if ( UART3Count == BUFSIZE )
	{
	  UART3Count = 0;		/* buffer overflow */
	}
  }
  else if ( IIRValue == IIR_CTI )	/* Character timeout indicator */
  {
	/* Character Time-out indicator */
	UART3Status |= 0x100;		/* Bit 9 as the CTI error */
  }
  else if ( IIRValue == IIR_THRE )	/* THRE, transmit holding register empty */
  {
	/* THRE interrupt */
	LSRValue = LPC_USART3->LSR;		/* Check status in the LSR to see if
									valid data in U0THR or not */
	if ( LSRValue & LSR_THRE )
	{
	  UART3TxEmpty = 1;
	}
	else
	{
	  UART3TxEmpty = 0;
	}
  }
  return;    
}

/*****************************************************************************
** Function name:		UARTInit
**
** Descriptions:		Initialize UART port, setup pin select,
**						clock, parity, stop bits, FIFO, etc.
**
** parameters:			portNum(0 or 1) and UART baudrate
** Returned value:		true or false, return false only if the 
**						interrupt handler can't be installed to the 
**						VIC table
** 
*****************************************************************************/
uint32_t UARTInit( uint32_t PortNum, uint32_t baudrate )
{
  uint32_t Fdiv;

  if ( PortNum == 0 )
  {
	LPC_SCU->SFSPF_10 = (0x3<<2) | 0x1;	/*	PF_10 U0_TXD */
	LPC_SCU->SFSPF_11 = (0x3<<2) | 0x1;	/*	PF_11 U0_RXD */
/*	Added from default code */	
//Port F
	LPC_SCU->SFSPF_9  = (0x3<<2) | 0x1;	/*	PF_9 U0_DIR */
	LPC_SCU->SFSPF_8  = (0x3<<2) | 0x1;	/*	PF_8 U0_UCLK */
//Port 6

/*	TEST */
	LPC_SCU->SFSP6_5 = (0x3<<2) | 0x2;	/*	P6_5 U0_RXD */
	LPC_SCU->SFSP6_4 = (0x3<<2) | 0x2;	/*	P6_4 U0_TXD */
	LPC_SCU->SFSP6_2 = (0x3<<2) | 0x2;	/*	P6_2 U0_DIR */
	LPC_SCU->SFSP6_1 = (0x3<<2) | 0x2;	/*	P6_1 U0_UCLK */
//Port 2
	LPC_SCU->SFSP2_1 = (0x3<<2) | 0x1;	/*	P2_1 U0_RXD */
	LPC_SCU->SFSP2_0 = (0x3<<2) | 0x1;	/*	P2_0 U0_TXD */
	LPC_SCU->SFSP2_6 = (0x3<<2) | 0x1;	/*	P2_6 U0_DIR */
	LPC_SCU->SFSP2_2 = (0x3<<2) | 0x1;	/*	P2_2 U0_UCLK */
/*	End added code */			

    LPC_USART0->LCR = 0x83;		/* 8 bits, no Parity, 1 Stop bit */
	Fdiv = ( UART0Frequency / 16 ) / baudrate ;	/*baud rate */
    LPC_USART0->DLM = Fdiv / 256;							
    LPC_USART0->DLL = Fdiv % 256;
	LPC_USART0->LCR = 0x03;		/* DLAB = 0 */
    LPC_USART0->FCR = 0x07;		/* Enable and reset TX and RX FIFO. */

   	NVIC_EnableIRQ(UART0_IRQn);

    LPC_USART0->IER = IER_RBR | IER_THRE | IER_RLS;	/* Enable UART0 interrupt */
    return (TRUE);
  }
  else if ( PortNum == 1 )
  {
	LPC_SCU->SFSPC_13 = (0x3<<2) | 0x2;	/*	PC_13 U1_TXD */
	LPC_SCU->SFSPC_14 = (0x3<<2) | 0x2;	/*	PC_14 U1_RXD */
	#if 0
/*	Added from default code */
//	Port C
	LPC_SCU->SFSPC_2  = (0x3<<2) | 0x2;	/*	PC_2 U1_CTS */
	LPC_SCU->SFSPC_11 = (0x3<<2) | 0x2;	/*	PC_11 U1_DCD */
	LPC_SCU->SFSPC_10 = (0x3<<2) | 0x2;	/*	PC_2 U1_DSR */
	LPC_SCU->SFSPC_12 = (0x3<<2) | 0x2;	/*	PC_2 U1_DTR */
	LPC_SCU->SFSPC_1  = (0x3<<2) | 0x2;	/*	PC_2 U1_RI */
	LPC_SCU->SFSPC_3  = (0x3<<2) | 0x2;	/*	PC_2 U1_RTS */
//	Port E
	LPC_SCU->SFSPE_12 = (0x3<<2) | 0x2;	/*	PE_12 U1_RXD */
	LPC_SCU->SFSPE_11 = (0x3<<2) | 0x2;	/*	PE_11 U1_TXD */
	LPC_SCU->SFSPE_7  = (0x3<<2) | 0x2;	/*	PE_7 U1_CTS */
	LPC_SCU->SFSPE_9  = (0x3<<2) | 0x2;	/*	PE_9 U1_DCD */
	LPC_SCU->SFSPE_8  = (0x3<<2) | 0x2;	/*	PE_8 U1_DSR */
	LPC_SCU->SFSPE_10 = (0x3<<2) | 0x2;	/*	PE_10 U1_DTR */
	LPC_SCU->SFSPE_6  = (0x3<<2) | 0x2;	/*	PE_6 U1_RI */
	LPC_SCU->SFSPE_5  = (0x3<<2) | 0x2;	/*	PE_5 U1_RTS */
//	Port 1
	LPC_SCU->SFSP1_14 = (0x3<<2) | 0x1;	/*	P1_14 U1_RXD */
	LPC_SCU->SFSP1_13 = (0x3<<2) | 0x1;	/*	P1_13 U1_TXD */
	LPC_SCU->SFSP1_11 = (0x3<<2) | 0x1;	/*	P1_11 U1_CTS */
	LPC_SCU->SFSP1_12 = (0x3<<2) | 0x1;	/*	P1_12 U1_DCD */
	LPC_SCU->SFSP1_7  = (0x3<<2) | 0x1;	/*	P1_7 U1_DSR */
	LPC_SCU->SFSP1_8  = (0x3<<2) | 0x1;	/*	P1_8 U1_DTR */
	LPC_SCU->SFSP1_10 = (0x3<<2) | 0x1;	/*	P1_10 U1_RI */
	LPC_SCU->SFSP1_9  = (0x3<<2) | 0x1;	/*	P1_9 U1_RTS */
/*	End added code */
#endif

    LPC_UART1->LCR = 0x83;		/* 8 bits, no Parity, 1 Stop bit */
	Fdiv = ( UART1Frequency / 16 ) / baudrate ;	/*baud rate */
    LPC_UART1->DLM = Fdiv / 256;							
    LPC_UART1->DLL = Fdiv % 256;
	LPC_UART1->LCR = 0x03;		/* DLAB = 0 */
    LPC_UART1->FCR = 0x07;		/* Enable and reset TX and RX FIFO. */

   	NVIC_EnableIRQ(UART1_IRQn);

    LPC_UART1->IER = IER_RBR | IER_THRE | IER_RLS;	/* Enable UART1 interrupt */
    return (TRUE);
  }
  else if ( PortNum == 2 )
  {
	LPC_SCU->SFSP1_15 = (0x3<<2) | 0x1;	/*	P1_15 U2_TXD */
	LPC_SCU->SFSP1_16 = (0x3<<2) | 0x1;	/*	P1_16 U2_RXD */
/*	Added from default code */	
//Port 1
	LPC_SCU->SFSP1_18 = (0x3<<2) | 0x1;	/*	P1_18 U2_DIR */
	LPC_SCU->SFSP1_17 = (0x3<<2) | 0x1;	/*	P1_17 U2_UCLK */
//Port 2
	LPC_SCU->SFSP2_11 = (0x3<<2) | 0x2;	/*	P2_11 U2_RXD */
	LPC_SCU->SFSP2_10 = (0x3<<2) | 0x2;	/*	P2_10 U2_TXD */
/*	End added code */
	
    LPC_USART2->LCR = 0x83;		/* 8 bits, no Parity, 1 Stop bit */
	Fdiv = ( UART2Frequency / 16 ) / baudrate ;	/*baud rate */
    LPC_USART2->DLM = Fdiv / 256;							
    LPC_USART2->DLL = Fdiv % 256;
	LPC_USART2->LCR = 0x03;		/* DLAB = 0 */
    LPC_USART2->FCR = 0x07;		/* Enable and reset TX and RX FIFO. */

   	NVIC_EnableIRQ(UART2_IRQn);

    LPC_USART2->IER = IER_RBR | IER_THRE | IER_RLS;	/* Enable UART2 interrupt */
    return (TRUE);
  }
  else if ( PortNum == 3 )
  {
	LPC_SCU->SFSPF_2 = (0x3<<2) | 0x1;	/*	PF_2 U3_TXD */
	LPC_SCU->SFSPF_3 = (0x3<<2) | 0x1;	/*	PF_3 U3_RXD */
/*	Added from default code */	
//Port F
	LPC_SCU->SFSPF_6 = (0x3<<2) | 0x1;	/*	PF_6 U3_DIR */
	LPC_SCU->SFSPF_5 = (0x3<<2) | 0x1;	/*	PF_5 U3_UCLK */
//Port 2
	LPC_SCU->SFSP2_4 = (0x3<<2) | 0x2;	/*	P2_4 U3_RXD */
	LPC_SCU->SFSP2_3 = (0x3<<2) | 0x2;	/*	P2_3 U3_TXD */
	LPC_SCU->SFSP2_8 = (0x3<<2) | 0x2;	/*	P2_8 U3_DIR */
	LPC_SCU->SFSP2_7 = (0x3<<2) | 0x2;	/*	P2_7 U3_UCLK */
/*	End added code */
	
    LPC_USART3->LCR = 0x83;		/* 8 bits, no Parity, 1 Stop bit */
	Fdiv = ( UART3Frequency / 16 ) / baudrate ;	/*baud rate */
    LPC_USART3->DLM = Fdiv / 256;							
    LPC_USART3->DLL = Fdiv % 256;
	LPC_USART3->LCR = 0x03;		/* DLAB = 0 */
    LPC_USART3->FCR = 0x07;		/* Enable and reset TX and RX FIFO. */

   	NVIC_EnableIRQ(UART3_IRQn);

    LPC_USART3->IER = IER_RBR | IER_THRE | IER_RLS;	/* Enable UART3 interrupt */
    return (TRUE);
  }
  return( FALSE ); 
}

/*****************************************************************************
** Function name:		UARTSend
**
** Descriptions:		Send a block of data to the UART 0 port based
**						on the data length
**
** parameters:			portNum, buffer pointer, and data length
** Returned value:		None
** 
*****************************************************************************/
void UARTSend( uint32_t portNum, uint8_t *BufferPtr, uint32_t Length )
{
  if ( portNum == 0 )
  {
    while ( Length != 0 )
    {
	  /* THRE status, contain valid data */
	  while ( !(UART0TxEmpty & 0x01) );	
	  LPC_USART0->THR = *BufferPtr;
	  UART0TxEmpty = 0;	/* not empty in the THR until it shifts out */
	  BufferPtr++;
	  Length--;
	}
  }
  else if ( portNum == 1 )
  {
	while ( Length != 0 )
    {
	  /* THRE status, contain valid data */
	  while ( !(UART1TxEmpty & 0x01) );	
	  LPC_UART1->THR = *BufferPtr;
	  UART1TxEmpty = 0;	/* not empty in the THR until it shifts out */
	  BufferPtr++;
	  Length--;
    }
  }
  else if ( portNum == 2 )
  {
	while ( Length != 0 )
    {
	  /* THRE status, contain valid data */
	  while ( !(UART2TxEmpty & 0x01) );	
	  LPC_USART2->THR = *BufferPtr;
	  UART2TxEmpty = 0;	/* not empty in the THR until it shifts out */
	  BufferPtr++;
	  Length--;
    }
  }
  else if ( portNum == 3 )
  {
	while ( Length != 0 )
    {
	  /* THRE status, contain valid data */
	  while ( !(UART3TxEmpty & 0x01) );	
	  LPC_USART3->THR = *BufferPtr;
	  UART3TxEmpty = 0;	/* not empty in the THR until it shifts out */
	  BufferPtr++;
	  Length--;
    }
  }
  return;
}

/******************************************************************************
**                            End Of File
******************************************************************************/
