/***********************************************************************
 * $Id:: hw_i2cd_rom_api.h 174 2011-04-21 00:31:04Z usb10131                   $
 *
 * Project: I2c device ROM Stack
 *
 * Description:
 *     ROM API Module definitions.
 *
 ***********************************************************************
 *   Copyright(C) 2011, NXP Semiconductor
 *   All rights reserved.
 *
 ****************************************************************************
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * products. This software is supplied "AS IS" without any warranties.
 * NXP Semiconductors assumes no responsibility or liability for the
 * use of the software, conveys no license or title under any patent,
 * copyright, or mask work right to the product. NXP Semiconductors
 * reserves the right to make changes in the software without
 * notification. NXP Semiconductors also make no representation or
 * warranty that such application will be suitable for the specified
 * use without further testing or modification.

 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors�
 * relevant copyright in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.

 ****************************************************************************/

#ifndef __HW_I2CD_ROM_API_H
#define __HW_I2CD_ROM_API_H


#include "stdint.h"    // added to define uint32_t, etc.
#include "error.h"

typedef  void*   I2C_HANDLE_T ;    // define TYPE for i2c handle pointer

typedef void  (*I2C_CALLBK_T) (  uint32_t  err_code, uint32_t  n) ; //define TYPE

typedef struct i2c_A {		 //parms passed to ROM function
	uint32_t num_bytes_send ;
	uint32_t num_bytes_rec ;
	uint8_t *buffer_ptr_send ;
	uint8_t *buffer_ptr_rec ;
	I2C_CALLBK_T func_pt; 		// callback function
	uint8_t stop_flag;
	uint8_t dummy[3] ;         // to make word boundry
} I2C_PARAM ;

typedef struct i2c_R {	     // RESULTs struct--results are here when returned
	uint32_t n_bytes_sent ;
	uint32_t n_bytes_recd ;
} I2C_RESULT ;

typedef enum I2C_mode {   	 
	IDLE,
	MASTER_SEND,
	MASTER_RECEIVE,
	SLAVE_SEND,
	SLAVE_RECEIVE  } I2C_MODE_T;

	/* slave state */
	typedef enum I2C_state {
		SLAVE_STATE_IDLE = 0x00,
		SLAVE_STATE_ID_READ,
		SLAVE_STATE_ID_WRITTEN,
		SLAVE_STATE_CONFIG,
		SLAVE_STATE_CONFIGURED,
		SLAVE_STATE_TEMP_READ,
		SLAVE_STATE_TEMP_WRITTEN } I2C_STATE_T;

		//typedef  void *I2C_HANDLE_T;    // define TYPE for i2c handle pointer

		//************************************************
		// *** Below is the structure providing the function types for all the
		//     I2C functions.  One structure is used to pass params to most of the functions.
		//     The parms that are required to be set before calling the function are
		//     listed in comments below each function.	***

		typedef struct  I2CD_API {	   // index of all the i2c driver functions
			void  (*i2c_isr_handler) ( I2C_HANDLE_T*  h_i2c) ;  // ISR  interrupt service request
			// MASTER functions ***

			ErrorCode_t (*i2c_master_transmit_poll)(  I2C_HANDLE_T* h_i2c, I2C_PARAM* ptp,
					I2C_RESULT* ptr ) ;

			ErrorCode_t (*i2c_master_receive_poll)(I2C_HANDLE_T* h_i2c, I2C_PARAM* ptp,
					I2C_RESULT* ptr ) ;

			ErrorCode_t (*i2c_master_tx_rx_poll)(I2C_HANDLE_T* h_i2c,	I2C_PARAM* ptp,
					I2C_RESULT* ptr ) ;

			ErrorCode_t (*i2c_master_transmit_intr)(I2C_HANDLE_T*  h_i2c, I2C_PARAM*  ptp, I2C_RESULT* ptr ) ;


			ErrorCode_t (*i2c_master_receive_intr)(I2C_HANDLE_T*  h_i2c, I2C_PARAM*  ptp, I2C_RESULT* ptr ) ;


			ErrorCode_t (*i2c_master_tx_rx_intr)( I2C_HANDLE_T*  h_i2c, I2C_PARAM*  ptp, I2C_RESULT* ptr ) ;


			// SLAVE functions ***
			ErrorCode_t (*i2c_slave_receive_poll)( I2C_HANDLE_T*  h_i2c, I2C_PARAM*  ptp, I2C_RESULT* ptr ) ;


			ErrorCode_t (*i2c_slave_transmit_poll)( I2C_HANDLE_T*  h_i2c, I2C_PARAM*  ptp, I2C_RESULT* ptr ) ;


			ErrorCode_t (*i2c_slave_receive_intr)(  I2C_HANDLE_T*  h_i2c, I2C_PARAM*  ptp, I2C_RESULT* ptr ) ;

			ErrorCode_t (*i2c_slave_transmit_intr)(  I2C_HANDLE_T*  h_i2c, I2C_PARAM*  ptp, I2C_RESULT* ptr ) ;

			ErrorCode_t  (*i2c_set_slave_addr)(  I2C_HANDLE_T*  h_i2c,
					uint32_t slave_addr_0_3, uint32_t  slave_mask_0_3)	;

			// OTHER functions

			uint32_t  (*i2c_get_mem_size)( void) ;  //"ramsize_in_bytes" memory needed by I2C drivers

			I2C_HANDLE_T*   (*i2c_setup)( uint32_t  i2c_base_addr, uint32_t *start_of_ram ) ;

			ErrorCode_t   (*i2c_set_bitrate)(I2C_HANDLE_T*  h_i2c,  uint32_t  P_clk_in_hz,
					uint32_t  bitrate_in_bps)  ;

			uint32_t  (*i2c_get_firmware_version)( ) ;

			I2C_MODE_T  (*i2c_get_status)( I2C_HANDLE_T*  h_i2c ) ;

		} I2CD_API_T  ;	// end of structure ************************************



		extern const I2CD_API_T  i2c_api ;  //so application program can access	pointer to
		// function table

#endif /* __HW_I2CD_ROM_API_H */

		/* *********************************************************
   List of functions and the parameters needed to be initilized in
   the structure before called

 i2c_isr_handler  -- none    // ISR  interrupt service request

		 *** MASTER functions ***

 i2c_master_transmit_poll --  num_bytes_send, *buffer_ptr_send, stop_flag 

 i2c_master_receive_poll -- num_bytes_rec,   *buffer_ptr_rec,  stop_flag  

 i2c_master_tx_rx_poll -- num_bytes_send,   num_bytes_rec,
		 *buffer_ptr_send,	  *buffer_ptr_rec ,   stop_flag

 i2c_master_transmit_intr -- num_bytes_send, *buffer_ptr_send,  stop_flag, *callback_fp   

 i2c_master_receive_intr -- num_bytes_rec, *buffer_ptr_rec, stop_flag, *callback_fp     

 i2c_master_tx_rx_intr --   num_bytes_send,   num_bytes_rec,  *buffer_ptr_send,
		 *buffer_ptr_rec,  *callback_fp,  stop_flag

		 *** SLAVE functions ***

 i2c_slave_receive_poll --  num_bytes_rec,   *buffer_ptr_rec  

 i2c_slave_transmit_poll -- num_bytes_send,   *buffer_ptr_send  

 i2c_slave_receive_intr  -- num_bytes_rec,   *buffer_ptr_rec, *callback_fp   

 i2c_slave_transmit_intr --  num_bytes_send, *buffer_ptr_send, *callback_fp  

 i2c_set_slave_addr --  i2c_handle,  slave_addr_0_3,    slave_mask_0_3 

// OTHER functions

 i2c_get_mem_size -- none   //"ramsize_in_bytes" memory needed by I2C drivers

 i2c_setup --    i2c_base_addr,   *start_of_ram  

 i2c_set_bitrate -- i2c_handle,  P_clk_in_hz, bitrate_in_bps 

 i2c_get_firmware_version) -- none

 i2c_get_status -- i2c_handle   

		 */
