/***********************************************************************
 * Code Red Technologies Flash driver                                  *
 *                                                                     */
/***********************************************************************/
/*                                                                     */
/*  FlashDev.C:  Flash Programming Functions adapted                   */
/*               for SST39x3201B (16-bit Bus) on the LPC18xx and       */
/*               LPC18xx board version 2 from Hitex					   */
/*                                                                     */
/***********************************************************************/


//#include "crt_flash_if.h"


#include "LPC18xx.h"                    /* LPC18xx definitions                */
#include "config.h"
#include "crt_flash_if.h"
#include <stdio.h>

// Defines
#define M8(adr)  (*((volatile unsigned char  *) (adr)))
#define M16(adr) (*((volatile unsigned short *) (adr)))
#define M32(adr) (*((volatile unsigned long  *) (adr)))

#define STACK_SIZE   64        // Stack Size
void reset(void);

// Parameter definitions

union fsreg 	// Flash Status Register
{                  
  struct b  
  {
	    unsigned int q0:1;
	    unsigned int q1:1;
	    unsigned int q2:1;
	    unsigned int q3:1;
	    unsigned int q4:1;
	    unsigned int q5:1;
	    unsigned int q6:1;
	    unsigned int q7:1;
		unsigned int rl:8;

	    unsigned int q0h:1;
	    unsigned int q1h:1;
	    unsigned int q2h:1;
	    unsigned int q3h:1;
	    unsigned int q4h:1;
	    unsigned int q5h:1;
	    unsigned int q6h:1;
	    unsigned int q7h:1;
	    unsigned int rh:8;
	  } b;
	  unsigned int v;
} fsr;

unsigned long base_adr;



/************************************************************************
 * Check if Program/Erase completed
 *    Parameter:      adr:  Block Start Address
 *    Return Value:   0 - OK,  1 - Failed
 ************************************************************************/

int Polling (unsigned long adr) 
{

  unsigned int q6, q6h;

  fsr.v = M32(adr);
  q6 = fsr.b.q6;
  q6h = fsr.b.q6h;
  do {
	fsr.v = M32(adr);
	if ((fsr.b.q6 == q6) && (fsr.b.q6h == q6h)) {
		return (0);  // Done
	}
	q6 = fsr.b.q6;
	q6h = fsr.b.q6h;
	// if (fsr.b.q3 == 1) break;        // VPP not high enough
  } while ((fsr.b.q5 == 0) || (fsr.b.q5h == 0));           // Check for Timeout
  fsr.v = M32(adr);
  q6 = fsr.b.q6;
  q6h = fsr.b.q6h;
  fsr.v = M32(adr);
  if ((fsr.b.q6 == q6) && (fsr.b.q6h == q6h)) {
	return (0);    // Done
  }
  M32(adr) = 0x00F000F0;                   // Reset Device
  return (1);                        // Failed

}


/************************************************************************
 *  Initialize Flash Programming Functions
 *    Parameter:      adr:  Device Base Address
 *                    clk:  Unused
 *                    fnc:  Unused
 *    Return Value:   0 - OK,  1 - Failed
 ************************************************************************/

int Init (unsigned long adr, unsigned long clk, unsigned long fnc) 
{	
	base_adr = adr;

	// Configure clocks and stuff- use 12 meg irc
	SystemInit();
	/* Set PL160M @ 6*12=72MHz */
	SetPL160M(SRC_IRC, 6);							
	/* Run base M3 clock from PL160M, div by 1 = no division */
	SetClock(BASE_M3_CLK, SRC_PL160M_0, DIV1);

	// Set all EMI related pins to the correct function
	// Data bus (32 bits wide)
	scu_pinmux(1,7,MD_PLN_FAST,FUNC3); // D0
	scu_pinmux(1,8,MD_PLN_FAST,FUNC3); // D1
	scu_pinmux(1,9,MD_PLN_FAST,FUNC3); // D2
	scu_pinmux(1,10,MD_PLN_FAST,FUNC3);// D3
	scu_pinmux(1,11,MD_PLN_FAST,FUNC3);// D4
	scu_pinmux(1,12,MD_PLN_FAST,FUNC3);// D5
	scu_pinmux(1,13,MD_PLN_FAST,FUNC3);// D6
	scu_pinmux(1,14,MD_PLN_FAST,FUNC3);// D7
	scu_pinmux(5,4,MD_PLN_FAST,FUNC2); // D8
	scu_pinmux(5,5,MD_PLN_FAST,FUNC2); // D9
	scu_pinmux(5,6,MD_PLN_FAST,FUNC2); // D10
	scu_pinmux(5,7,MD_PLN_FAST,FUNC2); // D11
	scu_pinmux(5,0,MD_PLN_FAST,FUNC2); // D12
	scu_pinmux(5,1,MD_PLN_FAST,FUNC2); // D13
	scu_pinmux(5,2,MD_PLN_FAST,FUNC2); // D14
	scu_pinmux(5,3,MD_PLN_FAST,FUNC2); // D15

	scu_pinmux(0xD,2,MD_PLN_FAST,FUNC2); // D16
	scu_pinmux(0xD,3,MD_PLN_FAST,FUNC2); // D17
	scu_pinmux(0xD,4,MD_PLN_FAST,FUNC2); // D18
	scu_pinmux(0xD,5,MD_PLN_FAST,FUNC2); // D19
	scu_pinmux(0xD,6,MD_PLN_FAST,FUNC2); // D20
	scu_pinmux(0xD,7,MD_PLN_FAST,FUNC2); // D21
	scu_pinmux(0xD,8,MD_PLN_FAST,FUNC2); // D22
	scu_pinmux(0xD,9,MD_PLN_FAST,FUNC2); // D23
	scu_pinmux(0xE,5,MD_PLN_FAST,FUNC3); // D24
	scu_pinmux(0xE,6,MD_PLN_FAST,FUNC3); // D25
	scu_pinmux(0xE,7,MD_PLN_FAST,FUNC3); // D26
	scu_pinmux(0xE,8,MD_PLN_FAST,FUNC3); // D27
	scu_pinmux(0xE,9,MD_PLN_FAST,FUNC3); // D28
	scu_pinmux(0xE,10,MD_PLN_FAST,FUNC3);// D29
	scu_pinmux(0xE,11,MD_PLN_FAST,FUNC3);// D30
	scu_pinmux(0xE,12,MD_PLN_FAST,FUNC3);// D31

	// Address bus
	scu_pinmux(2,9,MD_PLN_FAST,FUNC3);  // A0 (not used)
	scu_pinmux(2,10,MD_PLN_FAST,FUNC3); // A1 on LPC18xx connected to A0 on flash
	scu_pinmux(2,11,MD_PLN_FAST,FUNC3); // A2
	scu_pinmux(2,12,MD_PLN_FAST,FUNC3); // A3
	scu_pinmux(2,13,MD_PLN_FAST,FUNC3); // A4
	scu_pinmux(1,0,MD_PLN_FAST,FUNC2);  // A5
	scu_pinmux(1,1,MD_PLN_FAST,FUNC2);  // A6
	scu_pinmux(1,2,MD_PLN_FAST,FUNC2);  // A7
	scu_pinmux(2,8,MD_PLN_FAST,FUNC3);  // A8
	scu_pinmux(2,7,MD_PLN_FAST,FUNC3);  // A9
	scu_pinmux(2,6,MD_PLN_FAST,FUNC2);  // A10
	scu_pinmux(2,2,MD_PLN_FAST,FUNC2);  // A11
	scu_pinmux(2,1,MD_PLN_FAST,FUNC2);  // A12
	scu_pinmux(2,0,MD_PLN_FAST,FUNC2);  // A13
	scu_pinmux(6,8,MD_PLN_FAST,FUNC1);  // A14
	scu_pinmux(6,7,MD_PLN_FAST,FUNC1);  // A15
	scu_pinmux(0xD,16,MD_PLN_FAST,FUNC2);  // A16
	scu_pinmux(0xD,15,MD_PLN_FAST,FUNC2);  // A17
	scu_pinmux(0xE,0,MD_PLN_FAST,FUNC3);  // A18
	scu_pinmux(0xE,1,MD_PLN_FAST,FUNC3);  // A19
	scu_pinmux(0xE,2,MD_PLN_FAST,FUNC3);  // A20

	//new once
	scu_pinmux(0xE,3,MD_PLN_FAST,FUNC3); // A21
	scu_pinmux(0xE,4,MD_PLN_FAST,FUNC3); // A22
	scu_pinmux(0xA,4,MD_PLN_FAST,FUNC3); // A23

	// misc signals
	scu_pinmux(1,6,MD_PLN_FAST,FUNC3); //EMC_WE
	scu_pinmux(1,3,MD_PLN_FAST,FUNC3); //EMC_OE
	scu_pinmux(1,5,MD_PLN_FAST,FUNC3); //EMC_CS0 //Flash
	scu_pinmux(1,4,MD_PLN_FAST,FUNC3); //EMC_BLS0
	scu_pinmux(6,3,MD_PLN_FAST,FUNC3); //EMC_CS1
	scu_pinmux(6,6,MD_PLN_FAST,FUNC1); //EMC_BLS1
	scu_pinmux(0xD,12,MD_PLN_FAST,FUNC2); //EMC_CS2  //Static SRAM
	scu_pinmux(0xD,13,MD_PLN_FAST,FUNC2); //EMC_BLS2
	scu_pinmux(0xD,11,MD_PLN_FAST,FUNC2); //EMC_CS3 //No cnxn
	scu_pinmux(0xD,10,MD_PLN_FAST,FUNC2); //EMC_BLS3

	// initialise the control pin signal registers
	*(uint32_t *)0x40005000 = 0x00000001;      // Enable
	*(uint32_t *)0x40005200 = 0x00000082;      // CS0: 32 bit, WE + buffer
	*(uint32_t *)0x40005208 = 0x00000002;      // CS0: WAITOEN = 1
	*(uint32_t *)0x4000520C = 0x0000000B;      // CS0: WAITRD = 6

  	return (0);
}



/************************************************************************
 *  De-Initialize Flash Programming Functions
 *    Parameter:      fnc:  Unused
 *    Return Value:   0 - OK,  1 - Failed
 ************************************************************************/

int UnInit (unsigned long fnc) 
{
	reset();
  	return (0);
}



/************************************************************************
 *  Erase complete Flash Memory
 *    Return Value:   0 - OK,  1 - Failed
 ************************************************************************/

int EraseChip (void) 
{
	  // Start Chip Erase Command
	  M32(base_adr + (0x555 << 2)) = 0x00AA00AA;
	  M32(base_adr + (0x2AA << 2)) = 0x00550055;
	  M32(base_adr + (0x555 << 2)) = 0x00800080;
	  M32(base_adr + (0x555 << 2)) = 0x00AA00AA;
	  M32(base_adr + (0x2AA << 2)) = 0x00550055;
	  M32(base_adr + (0x555 << 2)) = 0x00100010;

	  return (Polling(base_adr));  // Wait until Erase completed	 //Mod
}


void reset() {
  M32(base_adr + (0x555 << 2)) = 0x00AA00AA;
  M32(base_adr + (0x2AA << 2)) = 0x00550055;
  M32(base_adr) = 0x00F000F0;
  Polling(base_adr);
}

/************************************************************************
 *  Erase Sector in Flash Memory
 *    Parameter:      adr:  Sector Address
 *    Return Value:   0 - OK,  1 - Failed
 ************************************************************************/

int EraseSector (unsigned long adr) 
{

    // Start Erase Sector Command
  M32(base_adr + (0x555 << 2)) = 0x00AA00AA;
  M32(base_adr + (0x2AA << 2)) = 0x00550055;
  M32(base_adr + (0x555 << 2)) = 0x00800080;
  M32(base_adr + (0x555 << 2)) = 0x00AA00AA;
  M32(base_adr + (0x2AA << 2)) = 0x00550055;
  M32(adr) = 0x00300030;

  do {					           // Wait for Sector Erase Timeout
    fsr.v = M32(adr);
  } while ((fsr.b.q3 == 0) || (fsr.b.q3h == 0));

  return (Polling(adr));       // Wait until Erase completed    //Mod
}


/************************************************************************
 *  Program Page in Flash Memory
 *    Parameter:      adr:  Page Start Address
 *                    sz:   Page Size
 *                    buf:  Page Data
 *    Return Value:   0 - OK,  1 - Failed
 ************************************************************************/

int ProgramPage (unsigned long adr, unsigned long sz, unsigned char *buf) 
{
	  int i;

	  for (i = 0; i < ((sz+3)/4); i++)  {
	    // Start Program Command
	    M32(base_adr + (0x0555 << 2)) = 0x00AA00AA;
	    M32(base_adr + (0x02AA << 2)) = 0x00550055;
	    M32(base_adr + (0x0555 << 2)) = 0x00A000A0;
	    M32(adr) = *((unsigned long *) buf);
	    if (Polling(adr) != 0) return (1);
	    buf += 4;
	    adr += 4;
	  }

	  return (0);                                  // Finished without Errors
}

/************************************************************************
 *  Verify the Flash Memory
 *    Parameter:      adr:  Sector Address
 *                    sz:   Page Size
 *                    buf:  Page Data
 *    Return Value:   0 - OK,  1 - Failed
 ************************************************************************/
unsigned long Verify	(unsigned long adr,   // Verify Function
                         unsigned long sz,
                         unsigned char *buf) {
	return (0);
}
