/*********************************************************************
*                SEGGER Microcontroller GmbH & Co. KG                *
*        Solutions for real time microcontroller applications        *
**********************************************************************
*                                                                    *
*        (c) 1996 - 2012  SEGGER Microcontroller GmbH & Co. KG       *
*                                                                    *
*        Internet: www.segger.com    Support:  support@segger.com    *
*                                                                    *
**********************************************************************

** emWin V5.18 - Graphical user interface for embedded applications **
All  Intellectual Property rights  in the Software belongs to  SEGGER.
emWin is protected by  international copyright laws.  Knowledge of the
source code may not be used to write a similar product.  This file may
only be used in accordance with a license and should not be re-
distributed in any way. We appreciate your understanding and fairness.
----------------------------------------------------------------------
File        : LCDConf.c
Purpose     : Display controller configuration (single layer)
---------------------------END-OF-HEADER------------------------------
*/

#include "GUI.h"
#ifndef _WINDOWS
  #include "LCDConf.h"
	#ifdef LPC4088
	  #include "LPC407x_8x_177x_8x.h"
	  #include "system_LPC407x_8x_177x_8x.h"
	#else
    #include "LPC177x_8x.h"
    #include "system_LPC177x_8x.h"
	#endif
#else
	#include "WinConf.h"
#endif

/*********************************************************************
*
*       Layer configuration (to be modified)
*
**********************************************************************
*/


//
// Physical display size
//
#ifndef _WINDOWS
  #define XSIZE_PHYS  _LcdParams.PPL
  #define YSIZE_PHYS  _LcdParams.LPP
#else
  #define XSIZE_PHYS  FB_XSIZE
  #define YSIZE_PHYS  FB_YSIZE
#endif

#define VYSIZE_PHYS VY_IMAGE_SIZE
#define VXSIZE_PHYS	VX_IMAGE_SIZE

//
// Color conversion
//
#define COLOR_CONVERSION  GUICC_M565

//
// Display driver
//
#define DISPLAY_DRIVER_TRULY  &GUIDRV_Lin_OSX_16_API
#define DISPLAY_DRIVER_OTHER  &GUIDRV_Lin_16_API


//
// Video RAM
//
#ifdef __ICCARM__
  #pragma data_alignment=8  // 8 byte align frame buffer
  #pragma location="VRAM"
  static __no_init U32 _aVRAM[(VXSIZE_PHYS * VYSIZE_PHYS*PIXEL_WIDTH)/4];
#endif
#ifdef __CC_ARM
  __align(8) U32 _aVRAM[(VXSIZE_PHYS * VYSIZE_PHYS * PIXEL_WIDTH)/4] __attribute__ ((section ("VRAM"), zero_init));
#endif
#ifdef __GNUC__
  U32 _aVRAM[(VXSIZE_PHYS * VYSIZE_PHYS*PIXEL_WIDTH)/4]  __attribute__ ((section(".VRAM"))) = { 0 };
#endif

//
// Video RAM address
//
#ifndef _WINDOWS
#define VRAM_ADDR_PHYS  (U32)&_aVRAM[0]

//
// LCD
//
LCD_PARAM          _LcdParams;
U8                 _Display;

//
// Touch screen results
//
int _TouchX;
int _TouchY;
	
//
// Touch screen
//
#define TOUCH_CS_BIT                   20
#define TS_CS_SET()                    LPC_GPIO0->CLR = (1 << TOUCH_CS_BIT)
#define TS_CS_CLR()                    LPC_GPIO0->SET = (1 << TOUCH_CS_BIT)

#if GUI_SUPPORT_TOUCH  // Used when touch screen support is enabled

//
// Touch screen results
//
static int _TouchX;
static int _TouchY;


#endif
#endif

/*********************************************************************
*
*       Configuration checking
*
**********************************************************************
*/
#ifndef   VXSIZE_PHYS
  #define VXSIZE_PHYS XSIZE_PHYS
#endif
#ifndef   VYSIZE_PHYS
  #define VYSIZE_PHYS YSIZE_PHYS
#endif
#ifndef   VRAM_ADDR_PHYS
  #define VRAM_ADDR_PHYS  0
#endif
#ifndef   VRAM_ADDR_VIRT
  #define VRAM_ADDR_VIRT  0
#endif

#ifndef   XSIZE_PHYS
  #error Physical X size of display is not defined!
#endif
#ifndef   YSIZE_PHYS
  #error Physical Y size of display is not defined!
#endif
#ifndef   COLOR_CONVERSION
  #error Color conversion not defined!
#endif

//
// I2C
//
#define I2C0_BASE_ADDR  0x4001C000

//
// PCA9532 (I2C)
//
#define PCA9532_I2C_ADDR         0x64  // Address of the PCA9532 on the display PCB
#define PCA9532_DISP_3V3_EN_BIT  0
#define PCA9532_DISP_5V_EN_BIT   1
#define PCA9532_DISP_EN_BIT      4
#define PCA9532_DISP_BL_BIT      8

//
// EEPROM (I2C)
//
#define EEPROM_I2C_ADDR         0x56  // Address of the EEPROM that holds the display configurations
#define EEPROM_MAGIC_NUMBER     0xEA01CDAE
#define EEPROM_LCD_CONFIG_OFFS  64
#define EEPROM_MAX_SEQ_VER      1

/*********************************************************************
*
*       Local functions
*
**********************************************************************
*/

/*********************************************************************
*
*       _SetDisplayOrigin()
*/
#ifndef _WINDOWS
static void _SetDisplayOrigin(int x, int y) {
  (void)x;
  //
  // Set start address for display data and enable LCD controller
  //
  LPC_LCD->UPBASE = VRAM_ADDR_PHYS + (y * XSIZE_PHYS * PIXEL_WIDTH);  // Needs to be set, before LCDC is enabled
}
#endif

/*********************************************************************
*
*       Global functions
*
**********************************************************************
*/

/*********************************************************************
*
*       _InitController
*
* Function description
*   Initializes the LCD controller and touch screen
*/
#ifndef _WINDOWS
void _InitController(unsigned LayerIndex) {
  //_InitLCD();
  //
  // Set display size and video-RAM address
  //
  LCD_SetSizeEx (XSIZE_PHYS, YSIZE_PHYS, LayerIndex);
  LCD_SetVSizeEx(VXSIZE_PHYS, VYSIZE_PHYS, LayerIndex);
  LCD_SetVRAMAddrEx(LayerIndex, (void*)VRAM_ADDR_PHYS);
  //
  // Init touch
  //
#if GUI_SUPPORT_TOUCH  // Used when touch screen support is enabled
  InitTouch();
#endif
}
#endif


/*********************************************************************
*
*       Global functions for display init
*
**********************************************************************
*/

/*********************************************************************
*
*       LCD_X_Config
*
* Purpose:
*   Called during the initialization process in order to set up the
*   display driver configuration.
*/
void LCD_X_Config(void) {
  const GUI_DEVICE_API * pDriver;

   //
  // Set display driver and color conversion for 1st layer
  //
  #ifndef _WINDOWS
  if (_Display == DISPLAY_TRULY_240_320) {
    pDriver = DISPLAY_DRIVER_TRULY;
  } else {
    pDriver = DISPLAY_DRIVER_OTHER;
  }
  #else
  pDriver = GUIDRV_WIN32;
  #endif
  GUI_DEVICE_CreateAndLink(pDriver, COLOR_CONVERSION, 0, 0);
  //
  // Display driver configuration, required for Lin-driver
  //
  LCD_SetPosEx(0, 0, 0);
  if (LCD_GetSwapXYEx(0)) {
    LCD_SetSizeEx  (0, YSIZE_PHYS , XSIZE_PHYS);
    LCD_SetVSizeEx (0, VYSIZE_PHYS, VXSIZE_PHYS);
  } else {
    LCD_SetSizeEx  (0, XSIZE_PHYS , YSIZE_PHYS);
    LCD_SetVSizeEx (0, VXSIZE_PHYS, VYSIZE_PHYS);
  }
  LCD_SetVRAMAddrEx(0, (void*)VRAM_ADDR_VIRT);
  //
  // Set user palette data (only required if no fixed palette is used)
  //
  #if defined(PALETTE)
    LCD_SetLUTEx(0, PALETTE);
  #endif
}

/*********************************************************************
*
*       LCD_X_DisplayDriver
*
* Purpose:
*   This function is called by the display driver for several purposes.
*   To support the according task the routine needs to be adapted to
*   the display controller. Please note that the commands marked with
*   'optional' are not cogently required and should only be adapted if
*   the display controller supports these features.
*
* Parameter:
*   LayerIndex - Index of layer to be configured
*   Cmd        - Please refer to the details in the switch statement below
*   pData      - Pointer to a LCD_X_DATA structure
*
* Return Value:
*   < -1 - Error
*     -1 - Command not handled
*      0 - Ok
*/
int LCD_X_DisplayDriver(unsigned LayerIndex, unsigned Cmd, void * pData) {
  #ifndef _WINDOWS
  LCD_X_SETORG_INFO * pSetOrg;
  #endif
  int r;

  (void) LayerIndex;

  switch (Cmd) {
  //
  // Required
  //
  case LCD_X_INITCONTROLLER:
    //
    // Called during the initialization process in order to set up the
    // display controller and put it into operation. If the display
    // controller is not initialized by any external routine this needs
    // to be adapted by the customer...
    //
    // ...
    #ifndef _WINDOWS
      _InitController(0);
    #endif
    return 0;
  case LCD_X_SETORG:
    //
    // Required for setting the display origin which is passed in the 'xPos' and 'yPos' element of p
    //
    #ifndef _WINDOWS
      pSetOrg = (LCD_X_SETORG_INFO *)pData;
      _SetDisplayOrigin(pSetOrg->xPos, pSetOrg->yPos);
    #endif
    return 0;
  default:
    r = -1;
  }
  return r;
}

#ifndef _WINDOWS
/*********************************************************************
*
*       Global functions for GUI touch
*
**********************************************************************
*/

#if GUI_SUPPORT_TOUCH  // Used when touch screen support is enabled
#if 0
/*********************************************************************
*
*       _InitTouch
*
* Function description
*   Initializes the touch screen.
*/
static void _InitTouch(void) {
  U32 TouchOrientation;
  U8  Config[3];
  U8  TouchHasSwapXY;
  int TouchADLeft;
  int TouchADRight;
  int TouchADTop;
  int TouchADBottom;

  //
  // Init ports and SSP interface as needed for touch
  //
  LPC_GPIO0->DIR |= (1 << TOUCH_CS_BIT);  // P0.20 as output (touch controller CS)
  InitSSP(SSP_MODE_TS);
  Config[0] = REF_ON;
  Config[1] = (READ_12BIT_SER(ADS_A2A1A0_vaux) | ADS_PD10_ALL_ON);
  Config[2] = PWRDOWN;
  TS_CS_SET();
  SSP_Send(Config, 3);
  TS_CS_CLR();
  //
  // Set touch orientation and calibrate touch
  //
  if (_Display == DISPLAY_TRULY_240_320) {
    //
    // 3.2" Truly display
    // Swap XY for touch as touch does not have the same orientation as the display
    //
    if (LCD_GetSwapXYEx(0)) {
      TouchHasSwapXY = 0;
    } else {
      TouchHasSwapXY = 1;
    }
    //
    // Calibrate touch
    //
    TouchOrientation = (GUI_MIRROR_X * LCD_GetMirrorYEx(0)) |  // XY swapped
                       (GUI_MIRROR_Y * LCD_GetMirrorXEx(0)) |  // XY swapped
                       (GUI_SWAP_XY  * TouchHasSwapXY) ;
    GUI_TOUCH_SetOrientation(TouchOrientation);
    if (TouchHasSwapXY) {
      GUI_TOUCH_Calibrate(GUI_COORD_X, 0, YSIZE_PHYS, TOUCH_TRULY_240_320_AD_LEFT, TOUCH_TRULY_240_320_AD_RIGHT);
      GUI_TOUCH_Calibrate(GUI_COORD_Y, 0, XSIZE_PHYS, TOUCH_TRULY_240_320_AD_TOP , TOUCH_TRULY_240_320_AD_BOTTOM);
    } else {
      GUI_TOUCH_Calibrate(GUI_COORD_Y, 0, XSIZE_PHYS, TOUCH_TRULY_240_320_AD_LEFT, TOUCH_TRULY_240_320_AD_RIGHT);
      GUI_TOUCH_Calibrate(GUI_COORD_X, 0, YSIZE_PHYS, TOUCH_TRULY_240_320_AD_TOP , TOUCH_TRULY_240_320_AD_BOTTOM);
    }
  } else {
    //
    // LCD board other than truly
    //
    if        (_Display == DISPLAY_BOARD_480_272) {
      TouchADLeft   = TOUCH_BOARD_480_272_AD_LEFT;
      TouchADRight  = TOUCH_BOARD_480_272_AD_RIGHT;
      TouchADTop    = TOUCH_BOARD_480_272_AD_TOP;
      TouchADBottom = TOUCH_BOARD_480_272_AD_BOTTOM;
    } else if (_Display == DISPLAY_BOARD_800_480) {
      TouchADLeft   = TOUCH_BOARD_800_480_AD_LEFT;
      TouchADRight  = TOUCH_BOARD_800_480_AD_RIGHT;
      TouchADTop    = TOUCH_BOARD_800_480_AD_TOP;
      TouchADBottom = TOUCH_BOARD_800_480_AD_BOTTOM;
    } else {
      while (1);  // Touch not supported
    }
    //
    // Calibrate touch
    //
    TouchOrientation = (GUI_MIRROR_X * LCD_GetMirrorXEx(0)) |
                       (GUI_MIRROR_Y * LCD_GetMirrorYEx(0)) |
                       (GUI_SWAP_XY  * LCD_GetSwapXYEx (0)) ;
    GUI_TOUCH_SetOrientation(TouchOrientation);
    GUI_TOUCH_Calibrate(GUI_COORD_X, 0, XSIZE_PHYS, TouchADLeft, TouchADRight);
    GUI_TOUCH_Calibrate(GUI_COORD_Y, 0, YSIZE_PHYS, TouchADTop , TouchADBottom);
  }
  _IsInited = 1;
}
#endif
/*********************************************************************
*
*       GUI_TOUCH_X_ActivateX()
*
* Function decription:
*   Called from GUI, if touch support is enabled.
*   Switches on voltage on X-axis,
*   prepares measurement for Y-axis.
*   Voltage on Y-axis is switched off.
*/
void GUI_TOUCH_X_ActivateX(void) {
}

/*********************************************************************
*
*       GUI_TOUCH_X_ActivateY()
*
* Function decription:
*   Called from GUI, if touch support is enabled.
*   Switches on voltage on Y-axis,
*   prepares measurement for X-axis.
*   Voltage on X-axis is switched off.
*/
void GUI_TOUCH_X_ActivateY(void) {
}

/*********************************************************************
*
*       GUI_TOUCH_X_MeasureX()
*
* Function decription:
*   Called from GUI, if touch support is enabled.
*   Measures voltage of X-axis.
*/
int  GUI_TOUCH_X_MeasureX(void) {
  return _TouchX;
}

/*********************************************************************
*
*       GUI_TOUCH_X_MeasureY()
*
* Function decription:
*   Called from GUI, if touch support is enabled.
*   Measures voltage of Y-axis.
*/
int  GUI_TOUCH_X_MeasureY(void) {
  return _TouchY;
}

#endif  // GUI_SUPPORT_TOUCH

#endif

/*************************** End of file ****************************/
