// ===================================================================
// sdcard_MCI_Init.c - Code Red Technologies
//
// Extracted from NXP's LPC407x_8x_177x_8x peripheral driver library
// file 'lpc_mci.c' - in order to split the board specific
// initialisation code from the generic generic mci driver code.
//
// Version : 121029
//
// ===================================================================

/**********************************************************************
* $Id$         lpc177x_8x_mci.c            2011-06-02
*//**
* @file        lpc177x_8x_mci.c
* @brief       Contains all functions support for MCI firmware library
*              on LPC177x_8x
* @version     2.0
* @date        29. June. 2011
* @author      NXP MCU SW Application Team
* 
* Copyright(C) 2011, NXP Semiconductor
* All rights reserved.
*
***********************************************************************
* Software that is described herein is for illustrative purposes only
* which provides customers with programming information regarding the
* products. This software is supplied "AS IS" without any warranties.
* NXP Semiconductors assumes no responsibility or liability for the
* use of the software, conveys no license or title under any patent,
* copyright, or mask work right to the product. NXP Semiconductors
* reserves the right to make changes in the software without
* notification. NXP Semiconductors also make no representation or
* warranty that such application will be suitable for the specified
* use without further testing or modification.
* Permission to use, copy, modify, and distribute this software and its
* documentation is hereby granted, under NXP Semiconductors'
* relevant copyright in the software, without fee, provided that it
* is used in conjunction with NXP Semiconductors microcontrollers.  This
* copyright, permission, and disclaimer notice must appear in all copies of
* this code.
**********************************************************************/

#include "LPC407x_8x_177x_8x.h"
#include "lpc_types.h"
#include "lpc_mci.h"
#include "lpc_gpdma.h"
#include "lpc_clkpwr.h"
#include "lpc_pinsel.h"

#include "bsp.h"

extern volatile en_Mci_CardType MCI_CardType;

/************************************************************************//**
 * @brief         Do initialization the MCI block as set its clock, registers,
 *                setup NVIC for interrupts, configure the pins used for MCI 
 *                function, do initialize the card in slot...
 *
 * @param[in]    powerActiveLevel the power level to activate the card in slot
 *
 * @return         MCI_FUNC_OK in case of success
 ***************************************************************************/
int32_t MCI_Init(uint8_t powerActiveLevel )
{
     volatile uint32_t i;

    MCI_CardType = MCI_CARD_UNKNOWN;

    // Following block of code added to ensure card VCC drops to zero
    // before card is initialized

#if !(_CURR_USING_BRD == _RDB4078_BOARD)
   // Force all MCI control pins to basic I/O mode
   LPC_IOCON->P1_2  &= ~0x1F; /* SD_CLK @ P1.2 */
   LPC_IOCON->P1_3  &= ~0x1F; /* SD_CMD @ P1.3 */
   LPC_IOCON->P1_5  &= ~0x1F; /* SD_PWR @ P1.5 */
   LPC_IOCON->P1_6  &= ~0x1F; /* SD_DAT_0 @ P1.6 */
   LPC_IOCON->P1_7  &= ~0x1F; /* SD_DAT_1 @ P1.7 */
   LPC_IOCON->P1_11 &= ~0x1F; /* SD_DAT_2 @ P1.11 */
   LPC_IOCON->P1_12 &= 0x1F; /* SD_DAT_3 @ P1.12 */

   // Set all MCI pins to outputs
   LPC_GPIO1->DIR |= 0x18EC;
  
   // Force all pins low (except power control pin)
   LPC_GPIO1->CLR = 0x1000;
   LPC_GPIO1->CLR = 0x0800;
   LPC_GPIO1->CLR = 0x0080;
   LPC_GPIO1->CLR = 0x0040;
  
   LPC_GPIO1->SET = 0x0020;
  
   LPC_GPIO1->CLR = 0x0008;
   LPC_GPIO1->CLR = 0x0004;
#else
   // Force all MCI control pins to basic I/O mode
   LPC_IOCON->P0_19  &= ~0x1F; /* SD_CLK @ P0.19 */
   LPC_IOCON->P0_20  &= ~0x1F; /* SD_CMD @ P0.20 */
   LPC_IOCON->P0_21  &= ~0x1F; /* SD_PWR @ P0.21 */
   LPC_IOCON->P0_22  &= ~0x1F; /* SD_DAT_0 @ P0.22 */
   LPC_IOCON->P2_11  &= ~0x1F; /* SD_DAT_1 @ P2.11 */
   LPC_IOCON->P2_12 &= ~0x1F; /* SD_DAT_2 @ P2.12 */
   LPC_IOCON->P2_13 &= 0x1F; /* SD_DAT_3 @ P2.13 */
   // Set all MCI pins to outputs
   LPC_GPIO0->DIR |= (1 << 19) | (1 << 20) | (1 << 21) | (1 << 22);
   LPC_GPIO2->DIR |= (1 << 11) | (1 << 12) | (1 << 13);
   // Force all pins low (except power control pin)
   LPC_GPIO0->CLR = (1 << 19) | (1 << 20) | (1 << 22);
   LPC_GPIO0->SET = (1 << 21);
   LPC_GPIO2->CLR = (1 << 11) | (1 << 12) | (1 << 13);

#endif


   // Crude delay of 50ms at 120MHz
   for ( i = 0; i < 0x100000; i++ );

    LPC_SC->PCONP |= ( 1 << 28 );            /* Enable clock to the MCI block */

    if ( LPC_MCI->CLOCK & (1 << 8) )
    {
        LPC_MCI->CLOCK &= ~(1 << 8);
        for ( i = 0; i < 0x10; i++ );    /* delay 3MCLK + 2PCLK  */
    }

    if ( LPC_MCI->POWER & 0x02 )
    {
        LPC_MCI->POWER = 0x00;
        for ( i = 0; i < 0x10; i++ );    /* delay 3MCLK + 2PCLK  */
    }

    for ( i = 0; i < 0x1000; i++ );

    /* Disable all interrupts for now */
    LPC_MCI->MASK0 = 0;

#if !(_CURR_USING_BRD == _RDB4078_BOARD)
    //SD_CLK
    PINSEL_ConfigPin(1, 2, 2);

    //SD_CMD
    PINSEL_ConfigPin(1, 3, 2);

    //SD_PWR
    PINSEL_ConfigPin(1, 5, 2);

    //SD_DAT_0
    PINSEL_ConfigPin(1, 6, 2);

    //SD_DAT_1
    PINSEL_ConfigPin(1, 7, 2);

    //SD_DAT_2
    PINSEL_ConfigPin(1, 11, 2);

    //SD_DAT_3
    PINSEL_ConfigPin(1, 12, 2);

#else

	//SD_CLK
	PINSEL_ConfigPin(0, 19, 2);

	//SD_CMD
	PINSEL_ConfigPin(0, 20, 2);

	//SD_PWR
	PINSEL_ConfigPin(0, 21, 2);

	//SD_DAT_0
	PINSEL_ConfigPin(0, 22, 2);

	//SD_DAT_1
	PINSEL_ConfigPin(2, 11, 2);

	//SD_DAT_2
	PINSEL_ConfigPin(2, 12, 2);

	//SD_DAT_3
	PINSEL_ConfigPin(2, 13, 2);

#endif

    // SD_PWR is active high (follows the output of the SD Card interface block).
    if(powerActiveLevel == LOW_LVL)
    {
        LPC_SC->SCS &= ~ 0x08;//Becase on EA board SD_PWR is active low
    }
    else
    {
        LPC_SC->SCS |= 0x08;
    }

    //Setting for timeout problem
    LPC_MCI->DATATMR = 0x1FFFFFFF;

    /*set up clocking default mode, clear any registers as needed */
    LPC_MCI->COMMAND = 0;
    for ( i = 0; i < 0x10; i++ );    /* delay 3MCLK + 2PCLK  */
    LPC_MCI->DATACTRL = 0;
    for ( i = 0; i < 0x10; i++ );    /* delay 3MCLK + 2PCLK  */
    LPC_MCI->CLEAR = 0x7FF;        /* clear all pending interrupts */

    LPC_MCI->POWER = 0x02;        /* power up */
    for ( i = 0; i < 0x10; i++ );    /* delay 3MCLK + 2PCLK  */

    
    /* delays for the supply output is stable*/
    for ( i = 0; i < 0x80000; i++ );

    /* During identification phase, the clock should be less than
    400Khz. Once we pass this phase, the normal clock can be set up
    to 25Mhz on SD card and 20Mhz on MMC card. */
    MCI_Set_MCIClock(MCI_SLOW_RATE );

    LPC_MCI->POWER |= 0x01;        /* bit 1 is set already, from power up to power on */
    for ( i = 0; i < 0x10; i++ );    /* delay 3MCLK + 2PCLK  */

    NVIC_EnableIRQ(MCI_IRQn);

    MCI_CardInit();

    /* During the initialization phase, to simplify the process, the CMD related
    interrupts are disabled. The DATA related interrupts are enabled when
    the FIFOs are used and just before WRITE_BLOCK READ_BLOCK cmds are issues, and
    disabled after the data block has been written and read. Please also note,
    before WRITE_BLOCK only TX related data interrupts are enabled, and before
    READ_BLOCK only RX related data interrupts are enabled. */
    return MCI_FUNC_OK;
}

