/* 
* TITLE: NM1LIB.C
* Written and Developed by: 
* Orchid Technologies Engineering and Consulting, Inc. 
* 147 Main Street 
* Maynard, Ma.  01754 
* TEL: 978-461-2000  
* 
* This Source Code is the property of Orchid Technologies 
* Engineering and Consulting, Inc.  Copyright 2003, 2004, 2005, 
* 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 
* and NXP Semiconductors 2013.

* THIS SOFTWARE IS PROVIDED BY ORCHID TECHNOLOGIES ENGINEERING AND CONSULTING INC. 
* AND NXP SEMICONDUCTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, 
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
* PARTICULAR PURPOSE, AND NON-INFRINGEMENT OF INTELLECTUAL PROPERTY RIGHTS ARE 
* DISCLAIMED. IN NO EVENT SHALL ORCHID TECHNOLOGIES ENGINEERING AND CONSULTING, 
* INC. OR NXP SEMICONDUCTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
* SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
* PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
* BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
* IN CONTRACT, STRICT LIABILITY, OR TORT(INCLUDING NEGLIGENCE OR OTHERWISE) 
* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
* POSSIBILITY OF SUCH DAMAGE. ORCHID TECHNOLOGIES ENGINEERING AND CONSULTING INC. 
* AND NXP SEMICONDUCTORS RESERVE THE RIGHT TO MAKE CHANGES IN THE SOFTWARE 
* WITHOUT NOTIFICATION. 
*
* @par
* Permission to use, copy, modify, and distribute this software and its
* documentation is hereby granted, under NXP Semiconductors' and its
* licensor's relevant copyrights in the software, without fee, provided that it
* is used in conjunction with NXP Semiconductors microcontrollers.  This
* copyright, permission, and disclaimer notice must appear in all copies of
* this code.
*/

//Include Files
#include "arm_comm.h"
#include "iolpc15xx.h"
#include "nm1def.h"            /* Control Defines    */
#include "nm1lib.h"            /* Library Prototypes */
#include "nm1com.h"            /* Communications     */
#include "nm1app.h"            /* Application        */
#include "nm1mem.h"            /* Global Memory      */
#include <stdio.h>
#include <math.h>

//Initialize NXP Clock
//12MHz Input Clock, CPU Clock at 96MHz, PCLK at 24MHz
void InitClock(void)
{
   unsigned int  *rom_api_ptr;
   unsigned int  *api_ptr1;
   unsigned int  *api_ptr2;
   unsigned int  command[10];
   unsigned int  result[10];

   //Setup LPC15XX IAP's
   rom_api_ptr = (unsigned int *)  rom_power_api;
   api_ptr1    = (unsigned int *) *rom_api_ptr;
   api_ptr1    = api_ptr1 + 0x00000003;
   api_ptr2    = (unsigned int *) *api_ptr1;
   set_pll     = (rom_set_pll) *api_ptr2;
   set_power   = (rom_set_power) *(api_ptr2+1);

   //Set PLL Frequency
   command[0] = 12000;
   command[1] = 72000;
   command[2] = 0;   //Frequency EQU
   command[3] = 0;   //Wait to Lock
   set_pll(command, result);

   //Set System Power
   command[0] = 72;  //Clock Rate
   command[1] = 1;   //Performance
   command[2] = 72;  //System Clock
   set_power(command, result);
}


//initialize_cpu
//This routine sets up the CPU Operating conditions
void initialize_cpu(void)
{
   unsigned int temp;

   //Turn on Peripherals
   SYSAHBCLKDIV   = 0x00000001;     //72MHz
   temp = SYSAHBCLKCTRL0;           //Reset Values
   temp = temp | 0x7801FA9B;        //Turn most everything on
   SYSAHBCLKCTRL0 = temp;           //Done

   temp = SYSAHBCLKCTRL1;           //Reset Values
   temp = temp | 0x0022007C;        //Turn most everything on
   SYSAHBCLKCTRL1 = temp;           //Done

   //Power On Peripherals
   temp = PDRUNCFG;
   temp = temp & 0xFFFFE3FF;        //ADC0 On, ADC1 On, DAC On
   PDRUNCFG = temp;

   //Setup Peripheral Clocks
   //running at 72MHz
   SYSTICKCLKDIV = 72;              //1 MHz Systick Clock
   IOCONCLKDIV   =  0;              //Forget about it
   UARTCLKDIV    =  0x27;           //UART Clock for 115200

   FRGCTRL       =  0xFF;           //UART Clock
   TRACECLKDIV   =  0;              //Forget about it
   USBCLKDIV     =  0;              //Forget about it
   ADCASYNCCLKDIV = 2;              //36 MHz to ADC's
   CLKOUTDIV     = 72;              //1MHz Clock Out
   CLKOUTSELA    =  3;              //Select System Clock
   CLKOUTSELB    =  0;              //Select System Clock

   //Setup IOCON Here
   //No IOCON Changes at this time, all setup for Pull-Down IO

   //Setup SWM Here
   //Setup for EA Motor Control Board and LPC15XX
   PINASSIGN0  = 0xFFFF0D12;   //UART0
   PINASSIGN1  = 0xFFFFFFFF;   //UART1
   PINASSIGN2  = 0xFFFFFFFF;   //UART2
   PINASSIGN3  = 0xFFFFFFFF;   //SSP0
   PINASSIGN4  = 0xFFFFFFFF;   //SSP0
   PINASSIGN5  = 0xFFFFFFFF;   //SSP1
   PINASSIGN6  = 0xFFFFFFFF;   //SSP1, CAN, USB
   PINASSIGN7  = 0x1A1D06FF;   //SCT0 OUT0, 1, 2, USB VBUS
   PINASSIGN8  = 0xFFFFFFFF;   //SCT1 OUT0, 1, 2, SCT2 OUT 0
   PINASSIGN9  = 0xFFFFFFFF;   //SCT2 OUT0, 1, 2, SCT3 OUT 0
   PINASSIGN10 = 0xFFFFFFFF;   //SCT2, ABORT, ADC Trig
   PINASSIGN11 = 0xFFFFFFFF;   //ADC Trig, DAC Trig
   PINASSIGN12 = 0xFFFFFFFF;   //ACMP Out
   PINASSIGN13 = 0xFFFF0EFF;   //ACMP Out, CLKOUT-PIN 30, PIO0-14
   PINASSIGN14 = 0x111E02FF;   //USB, QEI - Rev 2 Board
   PINASSIGN15 = 0xFFFFFFFF;   //SWO

   PINENABLE0  = 0xFEFD9FE4;   //Enables, Enable = 0, ADC0, ADC1, ADC3, ADC4, ADC1_1, ADC1_2, ADC1_5, DAC
   PINENABLE1  = 0x001FFE9F;   //CAREFUL! Debugger Here!  Don't disable.
                               //Enable SCT0_3, SCT0_4, SCT0_6

   //Configure INPUT_MUX here for SCT and other functions
   //Not Implemented at this time.

   //Clocks to SWM and INPUT_MUX can be turned off now.
   //This would lock configuration
   //Not Implemented at this time.

   //CPU CONFIGURATION COMPLETE

   //Start GPIO Setup
   GPIO_B3   = 1;            //Green LED
   GPIO_B25  = 1;            //Red   LED
   GPIO_B27  = 1;            //Current Trip = 1 DISABLED for NOW!!!  Motor Off

   GPIO_DIR0 = 0x0A400008;   //Current Trip,  LED, PIO0-22 Output for Test
#if  TRIG_TEST 
   GPIO_DIR1 = 0x00000004;   //Regular Setup, LED
   GPIO_B34 =1;
#else
  GPIO_DIR1 = 0x00000000;   //Regular Setup, LED
#endif   
   GPIO_DIR2 = 0x00000000;
   //End BPIO Setup

// Console Channel
// Initialize USART0 (UART) - Channel 0
// 8 Bit Word, 19200 Baud, 1 Start, 1 Stop, No Parity
// 24MHz Clock
   USART0_CFG  = 0x00000007;      //Enable 8Bit, No Parity, 1 Stop
   USART0_CTRL = 0x00000000;
   USART0_BRG  = 0x00000000;      //115200 Baud w/72MHz Sysclk Clock approx
}


//initialize_systick
//This routine initializes the system tick exception.
//This is a high priority exception which provides 10mSec
//exceptions for the system timer.  This routine is setup
//for Sysclk of 72MHz, SYSTICKDIV of 72, 1MHz Base Time.
void initialize_systick(void)
{
   SYSTICKRVR = 10000;       //10mSec Ticks - 100 Times/Second
   SYSTICKCSR = 0x00000003;  //Enable, Interrupt, 1MHz Source
}

//initialize_memory
//This routine sets up the global memory conditions
void initialize_memory(void)
{
   //Setup SCI (UART) Ring Buffers
   txring_inptr0  = sci_txbuffer0;
   txring_outptr0 = sci_txbuffer0;
   rxring_inptr0  = sci_rxbuffer0;
   rxring_outptr0 = sci_rxbuffer0;
   txring_count0  = 0x00;
   rxring_count0  = 0x00;

   //Setup SCI0 Line Input
   comm_lineflag0 = 0;
   comm_linecount0 = 0;

   //Setup RTC Variables
   rtc_delay = 0;                //Delay services are cleared
   rtc_running = 500;            //Fractional Seconds Counter Setup - wait to start for Show Morse

   //Application Globals
   service_adc_offset_state = STATE0;
   service_adc_offset_flag  = 0;
   service_adc_offset_count = 0;
   service_adc_offset_suma = 0;
   service_adc_offset_sumb = 0;
   service_adc_offset_sumc = 0;
   service_adc_offset_sumall = 0;
#if GUI_Interface  
    service_show_state = STATE0;
#else   
   service_show_state = STATE_IDLE;
#endif   
   service_joy_state = STATE0;
   service_joy_timer = 0;
   service_foc_cal_state = STATE_IDLE;
   service_foc_cal_timer = 0;
   service_rpm_pi_state = STATE0;
   estimator_flag = 0;
   estimator_state = STATE0;

   mtr1.adc_state = STATE0;
   mtr1.pi_flag   = 0;
   mtr1.piq_setpt = PIQ_SETPT;
   mtr1.piq_pgain = PIQ_PGAIN;
   mtr1.piq_igain = PIQ_IGAIN;
   mtr1.piq_tgain = PIQ_TGAIN;
   mtr1.piq_sum   = 0.0;
   mtr1.pid_setpt = PID_SETPT;
   mtr1.pid_pgain = PID_PGAIN;
   mtr1.pid_igain = PID_IGAIN;
   mtr1.pid_tgain = PID_TGAIN;
   mtr1.pid_sum   = 0.0;

   mtr1.rpm_flag  = 0;
   mtr1.rpm_setpt =  RPM_SETPT;
   mtr1.rpm_pgain  = RPM_PGAIN;
   mtr1.rpm_igain  = RPM_IGAIN;
   mtr1.rpm_tgain  = RPM_TGAIN;
   mtr1.rpm_sum    = 0.0;

   //Setup User Constants
   user_torque = 0.0;
   user_flux_est0 = 0.0;
   user_angle_est0 = -54;
   user_angle_est1 = 100;

   //Constant for now
   mtr1.vsdref = 0.0;      //Torque
   mtr1.vsqref = 0.0;      //Flux
   user_motor_r = MOTOR_R;
   user_motor_l = MOTOR_L;

   //Estimator Startup
   est1.start_count = 0;
   est1.start_theta = 0.0;
   est1.start_incr  = 0;
 }


//initialize_uarts_isr
//This routine sets up and starts the UARTs
void initialize_uarts_isr(void)
{
   NVIC_IntEnable(NVIC_UART0);                  //Enable Interrupt
   USART0_INTENSET = 0x00000001;                //Enable Receive Interrupt
}


//initialize_sct
//This routine initializes SCT0 for motor control operation
//We assume clocking is On, and SCT Outputs are already mapped.
void initialize_sct(void)
{
   SCT0_CONFIG = 0x00020001;  //32 Bit Operation
   SCT0_CTRL   = 0x0000001C;  //Bidir Count, Prescale = 0 and Halt
   SCT0_STATE  = 0x00000000;  //State Zero

   SCT0_MATCH0    = 0x400;    //17.5KHz
   SCT0_MATCHREL0 = 0x400;    //17.5KHz
   SCT0_MATCH1    = 0x000;    //Phase A
   SCT0_MATCHREL1 = 0x000;    //Phase A
   SCT0_MATCH2    = 0x000;    //Phase B
   SCT0_MATCHREL2 = 0x000;    //Phase B
   SCT0_MATCH3    = 0x000;    //Phase C
   SCT0_MATCHREL3 = 0x000;    //Phase C
   SCT0_MATCH4    = 0x000;    //ADC Trigger Clear
   SCT0_MATCHREL4 = 0x000;    //ADC Trigger Clear

   //Overall PWM Period
   SCT0_EV0_STATE = 0xFFFFFFFF;  //Operate in all states
   SCT0_EV0_CTRL  = 0x00001000;  //Match

   //Phase A
   SCT0_EV1_STATE = 0xFFFFFFFF;  //Operate in all states
   SCT0_EV1_CTRL  = 0x00201001;  //Match Up
   SCT0_EV2_STATE = 0xFFFFFFFF;  //Operate in all states
   SCT0_EV2_CTRL  = 0x00401001;  //Match Down

   //Phase B
   SCT0_EV3_STATE = 0xFFFFFFFF;  //Operate in all states
   SCT0_EV3_CTRL  = 0x00201002;  //Match Up
   SCT0_EV4_STATE = 0xFFFFFFFF;  //Operate in all states
   SCT0_EV4_CTRL  = 0x00401002;  //Match Down

   //Phase C
   SCT0_EV5_STATE = 0xFFFFFFFF;  //Operate in all states
   SCT0_EV5_CTRL  = 0x00201003;  //Match Up
   SCT0_EV6_STATE = 0xFFFFFFFF;  //Operate in all states
   SCT0_EV6_CTRL  = 0x00401003;  //Match Down

   //Phase A
   SCT0_OUT0_SET = 0x00000002;   //Set   on Event 1 - Low
   SCT0_OUT0_CLR = 0x00000004;   //Clear on Event 2 - Low
   SCT0_OUT1_SET = 0x00000004;   //Set   on Event 2 - High
   SCT0_OUT1_CLR = 0x00000002;   //Clear on Event 1 - High

   //Phase B
   SCT0_OUT2_SET = 0x00000008;   //Set   on Event 3 - Low
   SCT0_OUT2_CLR = 0x00000010;   //Clear on Event 4 - Low
   SCT0_OUT3_SET = 0x00000010;   //Set   on Event 4 - High
   SCT0_OUT3_CLR = 0x00000008;   //Clear on Event 3 - High

   //Phase C
   SCT0_OUT4_SET = 0x00000020;   //Set   on Event 5 - Low
   SCT0_OUT4_CLR = 0x00000040;   //Clear on Event 6 - Low
   SCT0_OUT6_SET = 0x00000040;   //Set   on Event 6 - High
   SCT0_OUT6_CLR = 0x00000020;   //Clear on Event 5 - High

   //ADC Trigger
   SCT0_EV7_STATE = 0xFFFFFFFF;  //Operate in all states
   SCT0_EV7_CTRL  = 0x00001004;  //Match All
   SCT0_OUT7_SET = 0x00000001;   //Set   on Event 0 - Set
   SCT0_OUT7_CLR = 0x00000080;   //Clear on Event 7 - Clear

   //Start SCT0
   SCT0_CTRL   = 0x00000038;  //Bidir Count, Prescale = 1 and Run, 17KHz
}

//initialize_adc0
//This routine initializes ADC0.  ADC0 is used for phase current measurements
//This routine initializes us to trigger on SCT0-7 internal.
//ADC0_4: PHA_CURR
//ADC0_3: PHB_CURR
//ADC0_1: PHC_CURR
//ADC0_0: TOTAL_CURR
//Here we setup to read this once per trigger.
void initialize_adc0(void)
{
   int temp;

   ADC0_CTRL      = 0x40000002;   //36MHz ADC Clock
   ADC0_INSEL     = 0x00000000;   
   ADC0_SEQA_CTRL = 0x4000201B;   //Enable Chan 0, 1, 3, 4, SCT0-7 Trigger
   ADC0_SEQB_CTRL = 0x00000000;
   ADC0_INTEN     = 0x00000001;   //End of Sequence Interrupt

   ADC0_SEQA_CTRL = 0xC000201B;   //Enable Chan 0, 1, 3, 4, SCT0-7 Trigger
   for(temp = 0; temp < 0xffff; temp++);  //delay
   NVIC_IntEnable(NVIC_ADC0_SEQA); //Enable Interrupt
}

//initialize_adc1
//This routine initializes ADC1.  ADC1 is used for phase BEMF measurements
//This routine initializes us to trigger on SCT0-7 internal.
//ADC1_1: PHA_BEMF
//ADC1_2: PHB_BEMF
//ADC1_5: PHC_BEMF
//Here we setup to read this once per trigger.
//We biggy back onto the ADC0 interrupt and service this at the same time!
void initialize_adc1(void)
{
   ADC1_CTRL      = 0x40000002;   //36MHz ADC Clock
   ADC1_INSEL     = 0x00000000;   
   ADC1_SEQA_CTRL = 0x40002026;   //Enable Chan 1, 2, 5 SCT0-7 Trigger
   ADC1_SEQB_CTRL = 0x00000000;
   ADC1_INTEN     = 0x00000000;   //No Interrupt

   ADC1_SEQA_CTRL = 0xC0002026;   //Enable Chan 1, 2, 5 SCT0-7 Trigger
}

//initialize_qei
//This routine initializes the QEI system.  Quadrature 4x counts
//with an index pulse reset and velocity timer.
void initialize_qei(void)
{
   QEIMAXPSOS = MOTOR_COUNTSPP;
   QEICONF    = 0x000F0014;
}

//byte2ascii
//This routine converts a single byte value into a decimal ASCII four
//byte string.  The first byte is alway 0.
void byte2ascii(unsigned char data, unsigned char *result)
{
   unsigned char temp1;

   //First ascii character
   *result++ = 0x30;

   //Second ascii character
   temp1 = (unsigned char) (data / 100);
   *result++ = 0x30 + temp1;
   data = data - (temp1 * 100);

   //Third ascii character
   temp1 = (unsigned char) (data / 10);
   *result++ = 0x30 + temp1;
   data = data - (temp1 * 10);

   //Fourth ascii character
   *result++ = 0x30 + data;
}

//word2bin
//This routine converts a single sixteen bit word to an ASCII Binary
//string.  A space is included between bit 7 and 8.  String is shown
//MSB to LSB.
void word2bin(unsigned int data, unsigned char *result)
{
   unsigned char i, j;

   for(i = 15, j = 0; j < 16; i--, j++)
   {
      if(j == 8)
      *result++ = 0x20;

      if(_btst(data, i))
      *result++ = 0x31;
      else
      *result++ = 0x30;
   }
   *result = 0;
}

///MISC Services
///MISC Services
///MISC Services


//led_run
//This routine turns on/off the Blue RUN LED
void led_run(unsigned char state)
{
   if(state)
   GPIO_B3 = 0x00;
   else
   GPIO_B3 = 0x01;
}


// SysTick_Handler
// This routine handles SysTick, Every 10mSec
void SysTick_Handler(void)
{
   volatile unsigned int temp1;

   //Keep track of Elapsed Time
   rtc_running--;
   if(rtc_running == 0x00)
   {
   rtc_running = 100;      //100 Interrupts per second! Flash Slow

   //Blink Running Status
   GPIO_NOT0 = 0x00000008; //Cool LPC15xx Toggle Function!!
   }

   //General Purpose Delay
   if(rtc_delay > 0)
   rtc_delay--;

   if(service_show_timer > 0)
   service_show_timer--;

   if(service_joy_timer > 0)
   service_joy_timer--;

   if(service_foc_cal_timer > 0)
   service_foc_cal_timer--;

   temp1 = SYSTICKCSR;
}

//ADC0_SEQ0_IRQHandler
//This routine handles ADC0 Sequence A Interrupts
//We get here once per PWM Cycle - Every 17.5KHz.  Here we read and
//store the motor phase currents as integers.
void ADC0_SEQA_IRQHandler(void)
{
   //Calibrate ADC Only
   if(service_adc_offset_state != STATE_IDLE)
   {
      //read adc data - Current
      mtr1.crnt_all = (int) ((ADC0_DAT0 & 0x0000FFF0) >> 4);   //Link Current
      mtr1.crnt_c   = (int) ((ADC0_DAT1 & 0x0000FFF0) >> 4);   //Phase C Current
      mtr1.crnt_b   = (int) ((ADC0_DAT3 & 0x0000FFF0) >> 4);   //Phase B Current
      mtr1.crnt_a   = (int) ((ADC0_DAT4 & 0x0000FFF0) >> 4);   //Phase A Current

      //Adjust Zero Offset - Current
      mtr1.crnt_all = mtr1.crnt_all + service_adc_offset_all;
      mtr1.crnt_a   = mtr1.crnt_a + service_adc_offset_a;
      mtr1.crnt_b   = mtr1.crnt_b + service_adc_offset_b;
      mtr1.crnt_c   = mtr1.crnt_c + service_adc_offset_c;

      //Set Flag
      service_adc_offset_flag = 1;
   }
#if TRIG_TEST   
    GPIO_B34 =0;
#endif   
   //Normal ADC Operation
   if(service_adc_offset_state == STATE_IDLE)
   {
      //Toggle PIO0-22 Test
      GPIO_NOT0 = 0x00400000; //Cool LPC15xx Toggle Function!! - PIO0-22 Test Only

      //Theta provided by startup routine.
      if(mtr1.su_flag == 0)
      {
         //Find Theta
         mtr1.theta = QEIPOS;

         //Estimate Theta
         if(estimator_flag != 0)
         mtr1.theta = mtr1.est_theta;

         //Adjust Theta
         if(estimator_flag == 0)
         {
         mtr1.theta = mtr1.theta + user_angle_est0;
         if(mtr1.theta < 0)
         mtr1.theta = MOTOR_COUNTSPP + mtr1.theta;
         if(mtr1.theta > MOTOR_COUNTSPP)
         mtr1.theta = mtr1.theta - MOTOR_COUNTSPP;

         mtr1.piq_setpt = user_flux_est0;
         est1.start_count = est1.start_time + 1;   //Already started
         }

         //Adjust Theta
         if(estimator_flag == 1)
         {
         mtr1.theta = mtr1.theta + user_angle_est1;
         if(mtr1.theta < 0)
         mtr1.theta = MOTOR_COUNTSPP + mtr1.theta;
         if(mtr1.theta > MOTOR_COUNTSPP)
         mtr1.theta = mtr1.theta - MOTOR_COUNTSPP;

         mtr1.piq_setpt = user_flux_est1;
         }

         //Calculate RPM
         get_rpm(&mtr1);
      }

     //DAC Diagnostic
     DAC_VAL = mtr1.theta << 6;

     //Divide Up the FOC Calculations
     switch(mtr1.adc_state)
     {
     case STATE0:
                    //Get Analog I and V Readings
                    get_analog(&mtr1);
                  
                    //Run Proportional-Integral Control
                    run_pi(&mtr1);

                    mtr1.adc_state = STATE1;
                    break;

     case STATE1: 
                    //Perform Sin and Cos Calculations for transforms below
                    sincos(&mtr1);

                    //Perform Clarke Transform
                    clarke(&mtr1);

                    //Perform Parke Transform
                    parke(&mtr1);

                    //Perform Inverse Park Transform
                    inverse_park(&mtr1);

                    //Perform SV_PWM 
                    sv_pwm(&mtr1);

                    mtr1.adc_state = STATE0;
                    break;
   }

   //Toggle PIO0-22 Test
   GPIO_NOT0 = 0x00400000; //Cool LPC15xx Toggle Function!! - PIO0-22 Test Only
   }

   //Clear Interrupts
   ADC0_FLAGS = 0x10000000;         //Clear Sequence A
   NVIC_ClrPend(NVIC_ADC0_SEQA);
}


//Math Helper Functions
//Math Helper Functions
//Math Helper Functions

//These functions perform sin, cos, and atan functions.
//The angle is expressed from 0 to 1023 giving a twelve bit
//value.  Thus 0 degress is 0.  360 degrees is 1024.  The 
//native math functions are expressed in radian.  Thus our
//angle counts are converted to radians and calculated.
//We get rougly an 0.25 degree angle accuracy simply with lookup
//tables, thus do not bother to interpolate and simply round down
//when doing table look up.
//
//sin and cos use the same table with cos offset by 90 
//degrees (256 counts)
//


//initialize_sincos
//This routine initializes the sin, cos lookup table.
//Angle is express in counts from 0 to MOTOR_COUNTSPP which correspond to degrees
//from 0 to 360.  We make the table (MOTOR_COUNTSPP/4) + 10 counts longer for cos and
//overflow management.
void initialize_sincos(void)
{
   int   angle;
   float fraction;

   fraction = (2.0 * 3.141592635) / MOTOR_COUNTSPP;
   for(angle = 0; angle < (MOTOR_COUNTSPP + MOTOR_COUNTSPP/4 + 10); angle++)
   sincostab[angle] = sin(fraction * (float) angle);
}

//initialize_tan
//This routine initializes the tan lookup table.
//Angle is express in counts from 0 to MOTOR_COUNTSPP which correspond to degrees
//from 0 to 360.  We make the table MOTOR_COUNTSPP/2 counts long.
//The tan lookup table is used for calculating 
//the get_atan function.  The tan lookup table runs from 0 to 180+ degrees.
//A negative number is the numerator indicates 180 to 360 degrees section.
void initialize_tan(void)
{
   int   angle;
   float fraction;

   fraction = (2.0 * 3.141592635) / MOTOR_COUNTSPP;
   for(angle = 0; angle < (MOTOR_COUNTSPP/2); angle++)
   tantab[angle] = tan(fraction * (float) angle);
}


//get_sin
//This routine performs a sin function on a floating point angle
//using table lookup and interpolation.  Angle = 0 to MOTOR_COUNTSPP which
//corresponds to 0 to 360 degrees
float get_sin(int angle)
{
   int   look_up_angle;
   float answer;

   look_up_angle = (int) angle;
   answer        = sincostab[look_up_angle];

   return(answer);
}

//get_cos
//This routine performs a cos function on a floating point angle
//using table lookup and interpolation.  Angle = 0 to 360
float get_cos(int angle)
{
   int   look_up_angle;
   float answer;

   look_up_angle = (int) angle + (MOTOR_COUNTSPP/4);
   answer        = sincostab[look_up_angle];

   return(answer);
}

//get_atan
//This routine performs an arc tangent on the input value.
//This routine returns an integer from 0 to MOTOR_COUNTSPP/2 which represents
//an angle from 0 to 180 degrees.  This routine performs a binary
//search to determine the arc tan using values in a table.
int get_atan(float value)
{
   int result, count;
   unsigned int index;
   unsigned int neg_flag;

   result = 0;
   if(value < 0.0)
   {
      result = result | 0x100;
      neg_flag = 1;
   }

   //Now perform binary search
   if(neg_flag == 0)
   {
      for(count = 7; count >= 0; count--)
      {
         index = result | (0x001 << count);
         if(tantab[index] >= value)
         result = result | (0x001 << count);
      }
   }
   else
   {
      for(count = 7; count >= 0; count--)
      {
         index = result | (0x001 << count);
         if(tantab[index] <= value)
         result = result | (0x001 << count);
      }
   }

   return(result);
}


//ADC Helper Functions
//service_adc_offset
//This routine corrects for Current Amplifier Zero Offset Errors
void service_adc_offset(void)
{
   if(service_adc_offset_state == STATE_IDLE)
   return;

   switch(service_adc_offset_state)
   {
      case STATE0:   //Clear
                     service_adc_offset_suma = 0;
                     service_adc_offset_sumb = 0;
                     service_adc_offset_sumc = 0;
                     service_adc_offset_sumall = 0;
                     service_adc_offset_count = 0;
                     service_adc_offset_state = STATE1;
                     break;

      case STATE1:   //Do we have a value?
                     if(service_adc_offset_flag)
                     service_adc_offset_state = STATE2;
                     break;

      case STATE2:   //Sum the values
                     service_adc_offset_flag = 0;
                     service_adc_offset_suma = service_adc_offset_suma + mtr1.crnt_a;
                     service_adc_offset_sumb = service_adc_offset_sumb + mtr1.crnt_b;
                     service_adc_offset_sumc = service_adc_offset_sumc + mtr1.crnt_c;
                     service_adc_offset_sumall = service_adc_offset_sumall + mtr1.crnt_all;
                     service_adc_offset_count++;
                     service_adc_offset_state = STATE3;
                     break;

      case STATE3:   //Are we done
                     if(service_adc_offset_count == 100)
                     service_adc_offset_state = STATE4;
                     else
                     service_adc_offset_state = STATE1;
                     break;

      case STATE4:   //Calculate Offset and Done
                     service_adc_offset_suma = service_adc_offset_suma / 100;
                     service_adc_offset_sumb = service_adc_offset_sumb / 100;
                     service_adc_offset_sumc = service_adc_offset_sumc / 100;
                     service_adc_offset_sumall = service_adc_offset_sumall / 100;
                     service_adc_offset_a = -service_adc_offset_suma;
                     service_adc_offset_b = -service_adc_offset_sumb;
                     service_adc_offset_c = -service_adc_offset_sumc;
                     service_adc_offset_all = -service_adc_offset_sumall;
                     service_adc_offset_state = STATE_IDLE;
                     break;
   }
}

//get_analog
//This routine reads the ADC0 and ADC1 values.  This routine
//applies calibration.  This routine applies scaling for current and voltage.
void get_analog(PMSMotor *ip)
{
   //read adc data - Current
   ip->crnt_c   = (int) ((ADC0_DAT1 & 0x0000FFF0) >> 4);   //Phase C Current
   ip->crnt_b   = (int) ((ADC0_DAT3 & 0x0000FFF0) >> 4);   //Phase B Current
   ip->crnt_a   = (int) ((ADC0_DAT4 & 0x0000FFF0) >> 4);   //Phase A Current

   //Adjust Zero Offset - Current
   ip->crnt_a   = ip->crnt_a + service_adc_offset_a;
   ip->crnt_b   = ip->crnt_b + service_adc_offset_b;
   ip->crnt_c   = ip->crnt_c + service_adc_offset_c;

   //Calibrate Current in Amps
   ip->ia = (float) ip->crnt_a / 62.0;
   ip->ib = (float) ip->crnt_b / 62.0;
// ip->ic = (float) ip->crnt_c / 62.0;
}

//get_rpm
//This routine calculates RPM.
void get_rpm(PMSMotor *ip)
{
   int   temp;

   temp = ip->theta - ip->last_theta;
   if(temp < 0)
   temp = -temp;

   if(temp > 500)
   temp = 0;

   ip->last_theta = ip->theta;

   ip->rpm_cnts = ip->rpm_cnts + temp;
   ip->rpm_time++;

   if(ip->rpm_time >= 1750)
   {
      ip->rpm_time = 0;
      if(QEISTAT & 0x00000001)
      ip->rpm = -(600 * ip->rpm_cnts) / 4000;
      else
      ip->rpm =  (600 * ip->rpm_cnts) / 4000;

      if((ip->rpm < -10000) || (ip->rpm > 10000))
      ip->rpm = 0;

      ip->rpm_cnts = 0;
      ip->rpm_flag = 1;
   }
}


//run_pi
//This routine performs the dual Proportional Integral control
//calculations which feed back into the motor control system.
//The inputs are the setpt's and the outputs are the vs'ref's.
void run_pi(PMSMotor *ip)
{
   float pterm;

   if(ip->pi_flag)
   {
     //First do Q Control
     pterm = ip->piq_setpt - (ip->iqout);
     ip->piq_sum  = ip->piq_sum + (pterm / ip->piq_tgain);
     ip->piq_out  = (ip->piq_pgain * pterm) + (ip->piq_igain * ip->piq_sum);

     if(ip->piq_out > MAX_QPID)
     {
       ip->piq_out = MAX_QPID;
       ip->piq_sum = ip->piq_sum - (pterm / ip->piq_tgain);
     }

     if(ip->piq_out < -MAX_QPID)
     {
       ip->piq_out = -MAX_QPID;
       ip->piq_sum = ip->piq_sum + (pterm / ip->piq_tgain);
     }

     //Next do D Control
     pterm = ip->pid_setpt - (ip->idout);
     ip->pid_sum  = ip->pid_sum + (pterm / ip->pid_tgain);
     ip->pid_out  = (ip->pid_pgain * pterm) + (ip->pid_igain * ip->pid_sum);

     if(ip->pid_out > MAX_PID)
     {
       ip->pid_out = MAX_PID;
       ip->pid_sum = ip->pid_sum - (pterm / ip->pid_tgain);
     }

     if(ip->pid_out < -MAX_PID)
     {
     ip->pid_out = -MAX_PID;
     ip->pid_sum = ip->pid_sum + (pterm / ip->pid_tgain);
     }

     if(mtr1.su_flag == 0)
     {
         ip->vsqref = ip->piq_out;
         ip->vsdref = ip->pid_out;
     }
   }
   else
   {
        ip->vsqref =  0.0;
        ip->vsdref =  0.0;
        ip->piq_sum = 0.0;
        ip->pid_sum = 0.0;
   }
}


//CORTEX-M3 Interrupt Enable
void NVIC_IntEnable(unsigned long IntNumber)
{
   volatile unsigned long * pNVIC_SetEn = &SETENA0;

// assert((NVIC_WDT_IRQ <= IntNumber) && (NVIC_PLL1 >= IntNumber));
   IntNumber -= NVIC_WDT_IRQ;
   pNVIC_SetEn += IntNumber/32;
   *pNVIC_SetEn = (1UL<<(IntNumber%32));
}

//CORTEX-M3 Interrupt Disable
void NVIC_IntDisable(unsigned long IntNumber)
{
   volatile unsigned long * pNVIC_ClrEn = &CLRENA0;

// assert((NVIC_WDT_IRQ <= IntNumber) && (NVIC_PLL1 >= IntNumber));
   IntNumber -= NVIC_WDT_IRQ;
   pNVIC_ClrEn += IntNumber/32;
   *pNVIC_ClrEn = (1UL<<(IntNumber%32));
}

//CORTEX-M3 Clear Pending Interrupt
void NVIC_ClrPend(unsigned long IntNumber)
{
   volatile unsigned long * pNVIC_ClrPend = &CLRPEND0;

//  assert((NVIC_WDT_IRQ <= IntNumber) && (NVIC_PLL1 >= IntNumber));
    IntNumber -= NVIC_WDT_IRQ;
    pNVIC_ClrPend += IntNumber/32;
   *pNVIC_ClrPend = (1UL<<(IntNumber%32));
}


//CORTEX-M3 Interrupt Priority Setting
void NVIC_IntPri(unsigned long IntNumber, unsigned char Priority)
{
   volatile unsigned char * pNVIC_IntPri = (unsigned char *)&IP0;

// assert((NVIC_WDT_IRQ <= IntNumber) && (NVIC_PLL1 >= IntNumber));
   IntNumber -= NVIC_WDT_IRQ;
   pNVIC_IntPri += IntNumber;
   *pNVIC_IntPri = Priority;
}


