/*
 * @brief State Configurable Timer (SCT) PWM example
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2014
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include "board.h"

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

#define SCT_PWM            LPC_SCT
#define SCT_PRE_SCALE      180
#define SCT_PWM1_PIN    	 10        /* COUT10 [index 1] Generate PWM1 */
#define SCT_PWM2_PIN    	 14        /* COUT14 [index 2] Generate PWM2 */


#define SCT_PWM1_OUT        1        /* Index of PWM1 */
#define SCT_PWM2_OUT        2        /* Index of PWM2 */
#define SCT_PWM_RATE   40000*(SCT_PRE_SCALE)    /* PWM frequency 40 KHz */

/* Systick timer tick rate, to change duty cycle */
#define TICKRATE_HZ     1000        /* 1 ms Tick rate */


/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Private functions
 ****************************************************************************/

/* Setup board specific pin muxing */
static void app_setup_pin(void)
{
	/* SCT_OUT14 on PD.11 mapped to FUNC6: LED2 */
	Chip_SCU_PinMuxSet(0xD, 11, (SCU_MODE_INACT | SCU_MODE_FUNC6));
	/* SCT_OUT10 on PD.12 mapped to FUNC6: LED3 & Oscilloscope input */
	Chip_SCU_PinMuxSet(0xD, 12, (SCU_MODE_INACT | SCU_MODE_FUNC6));
}

/*****************************************************************************
 * Public functions
 ****************************************************************************/

/**
 * @brief	Handle interrupt from SysTick timer
 * @return	Nothing
 */
void SysTick_Handler(void)
{
	/* You can used it to wakeup up the device from sleep */
	/*Not used in this application*/
}

/* Example entry point */
int main(void)
{
	
	/* Generic Initialization */
	SystemCoreClockUpdate();
	Board_Init();

	/* Initialize the SCT as PWM and set frequency */
	Chip_SCTPWM_Init(SCT_PWM);
	
		
	Chip_SCT_SetControl(SCT_PWM,SCT_CTRL_CLRCTR_L|SCT_CTRL_HALT_L|SCT_CTRL_PRE_L(SCT_PRE_SCALE-1));
				
	Chip_SCTPWM_SetRate(SCT_PWM, SCT_PWM_RATE);

	/* Setup Board specific output pin */
	app_setup_pin();

	/* COUT10 on PD.12 pin Duty cycle Match1*/
	Chip_SCTPWM_SetOutPin(SCT_PWM, SCT_PWM1_OUT, SCT_PWM1_PIN);
	/*COUT14  on PD11 Duty cycle Match2*/
	Chip_SCTPWM_SetOutPin(SCT_PWM, SCT_PWM2_OUT, SCT_PWM2_PIN);

	LPC_SCT->DITHER_L = 0;
	// Choose dither pattern here.port pin D.12(PWM1) uses dither pattern below
	//Set_Fractional_Mat_Register
	LPC_SCT->FRAMAT[1].U  &= ~0xF;
	LPC_SCT->FRAMAT[1].U  |= 0x8;
	LPC_SCT->RELFRAMAT[1].U &=~0xF;
	LPC_SCT->RELFRAMAT[1].U |=0x8;
	
	/* With 50% duty cycle */
	Chip_SCTPWM_SetDutyCycle(SCT_PWM, SCT_PWM1_OUT, Chip_SCTPWM_GetTicksPerCycle(SCT_PWM)/2);
	Chip_SCTPWM_SetDutyCycle(SCT_PWM, SCT_PWM2_OUT, Chip_SCTPWM_GetTicksPerCycle(SCT_PWM)/2);
	Chip_SCTPWM_Start(SCT_PWM);


	/* Enable SysTick Timer */
	SysTick_Config(SystemCoreClock / TICKRATE_HZ);

	while (1) {
		__WFI();
	}
	return 0;
}
