/*
 * @brief boot user application with communication to PC througn COM
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2012
 * Copyright(C) Dean Camera, 2011, 2012
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include "fsusb_cfg.h"
#include "ff.h"
#include "iap_17xx_40xx.h"

#define KEYBD_BACKSPACE					(0x7F)
#define	KEYBD_ENTER							(0x0D)

#define COM_OUT(...) printf(__VA_ARGS__)

extern USB_ClassInfo_MS_Host_t FlashDisk_MS_Interface;
extern volatile uint32_t udisk_change;

#define ApplicationAddress    (0x00005000)
#define ApplicationStartSec		(5)
#define	COPY_SIZE							(512)

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/
uint32_t sysclock;
static uint8_t binary_buff[COPY_SIZE+1];

STATIC FATFS fatFS;	/* File system object */
STATIC FIL fileObj;	/* File object */

const char title_str[] = {"\r\n	<Secondary USB Massstorage Host Bootloader>\r\n"};
const char version[] = {"		---Version 1.00---\r\n"};
const char cmd_menu[] = {"\r\n-------- Command List -------\r\n \
													\r\n ls - list root files in U-disk    \
													\r\n ld - load user application to flash like 'ld xxx.bin' \
													\r\n go - run user application "};
const char cmd_symbol[] = {"\r\n> "};
/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/
 /* Function to spin forever when there is an error */

/** Configures the board hardware and chip peripherals for the demo's functionality. */
static void SetupHardware(void)
{
	SystemCoreClockUpdate();
	Board_Init();
	Chip_USB_Init();
	/* Hardware Initialization */
	Board_Debug_Init();
}
static void die(FRESULT rc)
{
#if 0
	COM_OUT("*******DIE %d*******\r\n", rc);
	while (1) {}/* Spin for ever */
#endif
}
 
void DelayCnt(uint32_t cnt)
{
	uint32_t i = 1000;
	
	while(cnt--) {
		while(i--);
		i = 1000;
	}
}
uint32_t wait_enter(uint32_t max_cnt)
{
	uint32_t basecnt = SystemCoreClock/45, i, cnt;
	uint8_t Uartdata;
	
	i = basecnt;
	cnt = 1;
	COM_OUT("Press [Enter] to input command \r\nDelay... ");
	while(max_cnt--) {
		while(i--) {
			Uartdata = Board_UARTGetChar();
			if(Uartdata == KEYBD_ENTER) {//enter
				COM_OUT(cmd_menu);
				COM_OUT(cmd_symbol);
				return 1;
			}
		}

		COM_OUT("%d ", cnt);
		i = basecnt;
		cnt++;
	}
	COM_OUT("\r\nBooting user application...\r\n");
	return 0;
}

/*********************************************************************//**
 * @brief        Get Sector Number
 *
 * @param[in]    adr	 Sector Address
 *
 * @return       Sector Number.
 *
 **********************************************************************/
 uint32_t GetSecNum (uint32_t adr)
{
    uint32_t n;

    n = adr >> 12;                               //  4kB Sector
    if (n >= 0x10) {
      n = 0x0E + (n >> 3);                       // 32kB Sector
    } 

    return (n);                                  // Sector Number
}

/** Open a file in usb disk. */
FRESULT usb_bin_open(uint8_t * filename)
{
	FRESULT rc;		/* Result code */
	
	f_mount(0, &fatFS);		/* Register volume work area (never fails) */

	rc = f_open(&fileObj, (const TCHAR*)filename, FA_READ);
	if (rc) {
		COM_OUT("\r\nUnable to open %s! Check the file name!\r\n", filename);
	}	else {
		COM_OUT("\r\nFile %s Opened\r\n", filename);
	}

	return rc;
}
/** List files in usb disk. */
void list_usb_disk(void) 
{
	DIR fs_dir;
	TCHAR path[50]={""};  
	FILINFO fileinfo;
	uint32_t filefound;
	
	f_mount(0, &fatFS);		/* Register volume work area (never fails) */

	filefound = 0;
	if( f_opendir (&fs_dir, path) == FR_OK ) {
		while(f_readdir (&fs_dir, &fileinfo) == FR_OK ) {
			if(!fileinfo.fname[0]) {
				break;
			}
			if (fileinfo.fattrib & AM_DIR) {
			}
			else {
				COM_OUT("%s\r\n", fileinfo.fname); 
				filefound = 1;
			}
		}
	}
	if(!filefound) COM_OUT("\r\n No file found!\r\n");
	COM_OUT(cmd_symbol);				
}
/** load binary files from usb disk. */
void usb_bin_load(void)
{
	uint32_t filesize,copysize;
	uint32_t start_sec, end_sec;
	uint32_t copytimes = 0;
	uint32_t i;
	FRESULT rc;		/* Result code */
	UINT br;
	
	
	filesize = fileObj.fsize;
	COM_OUT("File size: %d\r\n", filesize);
// 	__disable_irq();
	
	start_sec = GetSecNum(ApplicationAddress);
	end_sec = GetSecNum(ApplicationAddress+filesize-1);
	
	Chip_IAP_PreSectorForReadWrite(start_sec, end_sec);
	Chip_IAP_EraseSector(start_sec, end_sec);
	
	copytimes = (filesize-1)/COPY_SIZE + 1;
	
	COM_OUT("Loading to sectors: %d - %d\r\n", start_sec, end_sec);	
		
	for(i=0; i<copytimes; i++) {
		if(i==(copytimes-1))
		{
			memset(binary_buff, 0xFF, COPY_SIZE);
			copysize = filesize%COPY_SIZE;
			if(copysize)
			{
				rc = f_read(&fileObj, binary_buff, copysize, &br);
			}
			else
			{
				rc = f_read(&fileObj, binary_buff, COPY_SIZE, &br);
			}
		}
		else
		{
			rc = f_read(&fileObj, binary_buff, COPY_SIZE, &br);	
		}
		if (rc || !br) {
			break;					/* Error or end of file */
		}
		Chip_IAP_PreSectorForReadWrite(start_sec, end_sec);	
		Chip_IAP_CopyRamToFlash(ApplicationAddress+(i*COPY_SIZE), (uint32_t *)binary_buff, COPY_SIZE);
	}
	if (rc) {
		die(rc);
	}
	
// 	__enable_irq();
	COM_OUT("Done!\r\n");
	
	rc = f_close(&fileObj);
	if (rc) {
		die(rc);
	}
	COM_OUT(cmd_menu);
	COM_OUT(cmd_symbol);
}

typedef  void (*pFunction)(void);
pFunction Jump_To_Application;
static uint32_t JumpAddress;


void run_to_app(void)
{
		JumpAddress = *(__IO uint32_t*) (ApplicationAddress + 4);

		/* Jump to user application */
		Jump_To_Application = (pFunction) JumpAddress;
		/* Initialize user application's S4tack Pointer */
		__set_MSP(*(__IO uint32_t*) ApplicationAddress);
		SCB->VTOR  = ApplicationAddress & 0x3FFFFF80;
		Jump_To_Application();
}

/*****************************************************************************
 * Public functions
 ****************************************************************************/

/** Main program entry point. This routine configures the hardware required by the application, then
 *  calls the filesystem function to read files from USB Disk
 */

#define 	RECV_BUF_SIZE		80
uint8_t RecvBuf[RECV_BUF_SIZE];
uint32_t RecvBufCount = 0;
int main(void)
{
	uint8_t bUartData;;
	char *result, *result1;
	uint32_t i, cmd;
	static uint32_t k;
	FRESULT rlt;
	uint8_t ap_status;
	
	SetupHardware();

	COM_OUT(title_str);
	COM_OUT(version);
	
	USB_Init(FlashDisk_MS_Interface.Config.PortNumber, USB_MODE_Host);
		
	k = 0;
	RecvBufCount = 0;
	for(i=0;i<RECV_BUF_SIZE;i++) {
		RecvBuf[i] = 0x0;
	}
	ap_status = Chip_IAP_BlankCheckSector(ApplicationStartSec, ApplicationStartSec);
	if(ap_status!=IAP_SECTOR_NOT_BLANK) {//if no user application in flash
		COM_OUT("\r\nNo user application! Please update with U-disk\r\n");
		COM_OUT(cmd_menu);
		while(!udisk_change);
		COM_OUT(cmd_symbol);
		cmd = 1;	
	}
	else {
		DelayCnt(100);
		cmd = wait_enter(5);
	}
	while (cmd) {
		bUartData = Board_UARTGetChar();
		if((bUartData != 0x00) && (bUartData != 0xFF)) {
			switch (bUartData) {
			case KEYBD_BACKSPACE:			//Backspace
				if(RecvBufCount) {
					RecvBuf[--RecvBufCount] = 0x0;		
				}
				break;
				
			case KEYBD_ENTER:					//enter
				//command: ls
				result = NULL;
				result = strstr((const char *)RecvBuf, "ls");
				if(result != NULL) {
					COM_OUT("\r\n");
					DelayCnt(100);
					list_usb_disk();
					while(RecvBufCount) {
						RecvBuf[--RecvBufCount] = 0x0;
					}
					break;
				}
				//command: go
				result = NULL;
				result = strstr((const char *)RecvBuf, "go");
				if(result != NULL) {
					COM_OUT("\r\n");
					cmd = 0;
					break;
				}	
				//command: ld
				result = NULL;
				result = strstr((const char *)RecvBuf, "ld");
				result1 = NULL;
				result1 = strstr((const char *)&RecvBuf[k], "bin");
				if(result1 == NULL) {
					result1 = strstr((const char *)&RecvBuf[k], "BIN");
				}
				if((result != NULL)&&(result1 != NULL)) {
					COM_OUT("\r\n");
					rlt = usb_bin_open(&RecvBuf[3]);
					if(rlt == FR_OK) {
						usb_bin_load();
					} else {
						COM_OUT(cmd_symbol);
					}
					k = 0;
					while(RecvBufCount) {
						RecvBuf[--RecvBufCount] = 0x0;
					}
					break;
				} 
				//error handling
				if(result==NULL) {
					COM_OUT("\r\nInvalid command!");
					COM_OUT(cmd_symbol);
					while(RecvBufCount) {
						RecvBuf[--RecvBufCount] = 0x0;
					}
				} else if(result1==NULL) {
						COM_OUT("\r\nThe file extension name must be .bin or .BIN");
						COM_OUT(cmd_symbol);
						COM_OUT("%s", RecvBuf);
				}
				
				break;
			
			default:
				RecvBuf[RecvBufCount] = bUartData; 
				RecvBufCount++;
				if(bUartData == '.') {
					k = RecvBufCount;
				}
				break;
			}
		}
	}
	DelayCnt(100);
	run_to_app();
}
