/*
===============================================================================
 Name        : comm_task.c
 Author      : nlv11817
 Description : serial communication task
===============================================================================
*/
#include "comm_task.h"
#include "touch_task.h"
#include "lcd.h"

uint8_t  service_comm_timer = 0;
uint8_t  send_pos_data      = 0;                            // flag to send position data via comm port
uint8_t  ts_loaded          = 0;                            // flag to indicate training set is loaded

static uint8_t     timeout = 0;
static COM_STATE_T com_state;

void COM_Init(void)
{
    com_state = WAIT_CMD;
    USART0_Init(115200, TXD, RXD);
}

void COM_Task(void)
{
  uint16_t crc16 = 0;
  uint16_t i;
  uint8_t  packet_nr;
  uint8_t  txbuf[32];

    if (service_comm_timer > 0)
        return;

    if (timeout++ > 49)                                                         // comm timeout 50 msec 
        com_state = WAIT_CMD;

    switch (com_state)
    {
      case WAIT_CMD :       timeout = 0;                                        // reset timeout
                            if (uart_ptr > 0)                                   // data / command received?
                            {
                                switch (uart_buf[0])
                                {
                                  case 'T' :    com_state = START_TRAINING;     // command start training
                                                break;
                                  case 'W' :    com_state = SET_PARAMETER;      // command write parameters
                                                LCD_PrintFullString("PARAM ");
                                                break;
                                  case 'R' :    com_state = READ_PARAMETER;     // command read parameters
                                                break;
//                                case 0x44 :   com_state = DEBUGGING;
//                                              break;
                                  case 'M'  :   com_state = READ_MODE;          // command read mode
                                                break;
                                  case 'U'  :   com_state = UPLOAD;             // command upload training set
                                                ts_loaded = 0;                  // receive 40 packets of 64 bytes
                                                break;
                                  default :     com_state = WAIT_CMD;
                                                uart_ptr  = 0;                  // reset uart read pointer
                                                txbuf[0]  = 0xFF;               // send not acknowledge
                                                UARTSend(&txbuf[0], 1);
                                                break;
                                }
                            }
                            break;
      case START_TRAINING : send_pos_data = 1;
                            uart_ptr      = 0;                                  // reset uart read pointer
                            txbuf[0]      = 'A';                                // send acknowledge
                            UARTSend(&txbuf[0], 1);
                            com_state = WAIT_CMD;
                            break;
      case SET_PARAMETER :  if (uart_ptr > PARAM_BUF_LEN)                       // wait until all params received
                            {
                                par.agc_mode    =  uart_buf[1];
                                par.agc_min     = (uart_buf[2] << 8) | uart_buf[3];
                                par.agc_max     = (uart_buf[4] << 8) | uart_buf[5];
                                par.system_gain =  uart_buf[6];
                                par.dt_mode     =  uart_buf[7];
                                par.dt_min      = (uart_buf[8]  << 8) | uart_buf[9];
                                par.dt_max      = (uart_buf[10] << 8) | uart_buf[11];
                                par.fd_pos      =  uart_buf[12];
                                par.fd_ref      =  uart_buf[13];
                                par.fd_raw      =  uart_buf[14];
                                par.touch_mode  =  uart_buf[15];
                                StoreTouchParameters();                         // store parameters in flash/eeprom
                                uart_ptr = 0;                                   // reset uart read pointer
                                txbuf[0] = 'A';                                 // send acknowledge
                                UARTSend(&txbuf[0], 1);
                                com_state = WAIT_CMD;
                            }
                            break;
      case READ_PARAMETER : uart_ptr  = 0;                                      // reset uart read pointer
                            txbuf[0]  = 'A';                                    // send ACK + parameters
                            txbuf[1]  = par.agc_mode;
                            txbuf[2]  = par.agc_min >> 8;
                            txbuf[3]  = par.agc_min;
                            txbuf[4]  = par.agc_max >> 8;
                            txbuf[5]  = par.agc_max;
                            txbuf[6]  = par.system_gain;
                            txbuf[7]  = par.dt_mode;
                            txbuf[8]  = par.dt_min >> 8;
                            txbuf[9]  = par.dt_min;
                            txbuf[10] = par.dt_max >> 8;
                            txbuf[11] = par.dt_max;
                            txbuf[12] = par.fd_pos;
                            txbuf[13] = par.fd_ref;
                            txbuf[14] = par.fd_raw;
                            txbuf[15] = par.touch_mode;
                            UARTSend(&txbuf[0], 16);
                            com_state = WAIT_CMD;
                            break;
      case READ_MODE :      txbuf[0] = 'A';                                     // send acknowledge
                            txbuf[1] = par.touch_mode;                          // + in which mode we are
                            uart_ptr = 0;                                       // reset uart read pointer
                            UARTSend(&txbuf[0], 2);
                            com_state = WAIT_CMD;
                            break;
      case UPLOAD :         if (uart_ptr > 67)                                  // wait until all data received
                            {
                                packet_nr = uart_buf[1];                        // packet nr (lenght 64 bytes)
                                for (i = 0; i < 64; i++)
                                {
                                    crc16 += uart_buf[i+4];                     // calculate checksum
                                }
                                i = (uart_buf[2] << 8) | uart_buf[3];           // received checksum
                                if (crc16 == i)                                 // compare checksums
                                {
                                    txbuf[0] = 'A';                             // crc okay send Ack
                                    StoreTrainingSet(&uart_buf[4], packet_nr);  // store packet in flash
                                }
                                else
                                {
                                    txbuf[0] = 0xFF;                            // CRC error
                                }
                                uart_ptr  = 0;                                  // reset uart read pointer
                                com_state = WAIT_CMD;
                                UARTSend(&txbuf[0], 1);
                                sys.service_timer = 100;                        // stop touch scanner for a while 

                                if (packet_nr == 39)                            // last packet?
                                {
                                    ts_loaded = 1;
                                    gr.nr_of_gestures = 10;
                                    gr.trs = (uint8_t *)0x00007400;
                                    Gesture_Init();
                                }
                            }
                            break;
      default :             break;
    }
    service_comm_timer = 1;
}
