/*
 * @brief Image programming code
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2014
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#ifndef SL_FLASH_H
#define SL_FLASH_H

#include "chip.h"
#ifdef __cplusplus
extern "C" {
#endif
	
#include "sl_protocol.h"
	
/* Writes user image blocks to flash. Erases flash sectors as needed. src address
    should be on a 4 byte boundary. */
extern int32_t flashWriteBlock(uint32_t *src, uint32_t block_nr);

extern int32_t flashReadBlock(uint32_t *dest, uint32_t block_nr);

extern int32_t flashReadPage(uint32_t *dest, uint32_t page_nr);

/* Erase one or more pages of flash memory. Addresses should be on a page boundary.
    Pages occupied by the secondary loader are not erased. Erase page takes same amount
    of time as sector erase hence only suitable for writing small amount of data in
    flash memory. */
extern int32_t flashErasePage(uint32_t start_addr, uint32_t end_addr);

/* Writes a page to flash memory. Destination address should be on a page boundary. Source
    address should be on a 4 byte boundary. Page should be erased before writing. Pages
    occupied by the secondary loader are not written. */
extern int32_t flashWritePage(uint32_t dst, uint32_t *src);

/* Erase one or more sectors of flash memory.
    Sector(s) occupied by the secondary loader are not erased. */
extern int32_t flashEraseSector(uint32_t start_sect, uint32_t end_sect);

/* Write one or more sectors of flash memory. */
extern int32_t flashWriteSector(uint32_t *src, uint32_t start_sect, uint32_t end_sect);

// define for LPC11U6X
#include "flash.h"
#define Chip_IAP_PreSectorForReadWrite Chip_FLASH_PreSectorForReadWrite
#define Chip_IAP_CopyRamToFlash        Chip_FLASH_CopyRamToFlash
#define Chip_IAP_ErasePage             Chip_FLASH_ErasePage
#define Chip_IAP_EraseSector           Chip_FLASH_EraseSector
#define Chip_IAP_ReinvokeISP           Chip_FLASH_ReinvokeISP
#define Chip_IAP_ReadUID               Chip_FLASH_ReadUID


#ifdef __cplusplus
}
#endif

#endif /* SL_FLASH_H */
