/*
===============================================================================
 Name        : sbl_dfu_lpc11u60.c
 Author      : $(NXP LPC MCU)
 Version     : 1.0
 Copyright   : $(NXP)
 Description : main definition
===============================================================================
*/

#include "chip.h"
#include "board.h"

#include <cr_section_macros.h>
#include "stdio.h"
#include "stdlib.h"

#include "sl_common.h"
#include "sl_flash.h"
#include "app_usbd_cfg.h"
#include "sl_protocol.h"
#include "sbl_lpc.h"

static USBD_HANDLE_T g_hUsb;

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/
const USBD_API_T *g_pUsbApi;		/* USB ROM API Handler */
volatile uint32_t dwSysTicks = 0;	/* System Ticks Counter */

/*****************************************************************************
 * Private functions
 ****************************************************************************/
volatile  uint8_t *pushAppFlag = (uint8_t *) SL_ADDRESS_APPCALLEDFL;

/*****************************************************************************
 * Public functions
 ****************************************************************************/

/* Handle interrupt from SystemTick */
void Tick_Handler(void)
{
  /* system ticks */
  dwSysTicks++;
}

/* Interrupt service routine of USB */
void USB_Handler(void)
{
  uint32_t *addr = (uint32_t *) LPC_USB->EPLISTSTART;

  /* WORKAROUND for artf32289 ROM driver BUG:
     As part of USB specification the device should respond
     with STALL condition for any unsupported setup packet. The host will send
     new setup packet/request on seeing STALL condition for EP0 instead of sending
     a clear STALL request. Current driver in ROM doesn't clear the STALL
     condition on new setup packet which should be fixed.
  */
  /* if setup packet is received */
  if ( LPC_USB->DEVCMDSTAT & _BIT(8) )
  {
    addr[0] &= ~(_BIT(29));	/* clear EP0_OUT stall */
    addr[2] &= ~(_BIT(29));	/* clear EP0_IN stall */
  }

  /* USB Device Status -- connect change */
  if( LPC_USB->DEVCMDSTAT & _BIT(24) )
  {
    *pushAppFlag = 0x11;
  }
  USBD_API->hw->ISR(g_hUsb);
}

static ErrorCode_t update_device_status_patch(USBD_HANDLE_T hUsb)
{
  USB_CORE_CTRL_T *pCtrl;
  pCtrl = (USB_CORE_CTRL_T *) hUsb;      /* convert the handle to control structure */
  if ( (((USB_CONFIGURATION_DESCRIPTOR *) (pCtrl->full_speed_desc))->bmAttributes & USB_CONFIG_POWERED_MASK) != 0 )
  {
    /* This is SELF_POWERED. */
    pCtrl->device_status |= (0x01 << 0);
  }
  else {
    /* This is BUS_POWERED. */
    pCtrl->device_status &= ~(0x01 << 0);
  }
  if ( (((USB_CONFIGURATION_DESCRIPTOR *) (pCtrl->full_speed_desc))->bmAttributes & USB_CONFIG_REMOTE_WAKEUP) != 0 )
  {
    /* This is REMOTE_WAKEUP enabled. */
    pCtrl->device_status |= (0x01 << 1);
  }
  else
  {
    /* This is REMOTE_WAKEUP disabled. */
    pCtrl->device_status &= ~(0x01 << 1);
  }
  return LPC_OK;
}

/* Handler for WCID USB device requests. */
static ErrorCode_t WCID_hdlr(USBD_HANDLE_T hUsb, void *data, uint32_t event)
{
  USB_CORE_CTRL_T *pCtrl = (USB_CORE_CTRL_T *) hUsb;
  ErrorCode_t ret = ERR_USBD_UNHANDLED;

  /* Handle Microsoft's WCID request for install less WinUSB operation.
     Check https://github.com/pbatard/libwdi/wiki/WCID-Devices for more details.
  */
  if (event == USB_EVT_SETUP)
  {
    switch (pCtrl->SetupPacket.bmRequestType.BM.Type)
    {
      case REQUEST_STANDARD:
        if ( (pCtrl->SetupPacket.bmRequestType.BM.Recipient == REQUEST_TO_DEVICE) &&
             (pCtrl->SetupPacket.bRequest == USB_REQUEST_GET_DESCRIPTOR) &&
             (pCtrl->SetupPacket.wValue.WB.H == USB_STRING_DESCRIPTOR_TYPE) &&
             (pCtrl->SetupPacket.wValue.WB.L == 0x00EE))
        {
          pCtrl->EP0Data.pData = (uint8_t *) WCID_String_Descriptor;
          pCtrl->EP0Data.Count = pCtrl->SetupPacket.wLength;
          USBD_API->core->DataInStage(pCtrl);
          ret = LPC_OK;
        }
        break;

      case REQUEST_VENDOR:
        if (pCtrl->SetupPacket.bRequest != WCID_String_Descriptor[16])
        {
          break;
        }
        switch (pCtrl->SetupPacket.bmRequestType.BM.Recipient)
        {
          case REQUEST_TO_DEVICE:
            if (pCtrl->SetupPacket.wIndex.W == 0x0004)
            {
              pCtrl->EP0Data.pData = (uint8_t *) WCID_CompatID_Descriptor;
              pCtrl->EP0Data.Count = pCtrl->SetupPacket.wLength;
              USBD_API->core->DataInStage(pCtrl);
              ret = LPC_OK;
            }
           /* Fall-through. Check note1 of
              https://github.com/pbatard/libwdi/wiki/WCID-Devices#wiki-Defining_a_Device_Interface_GUID_or_other_device_specific_properties
              break;
           */

          case REQUEST_TO_INTERFACE:
            if (pCtrl->SetupPacket.wIndex.W == 0x0005)
            {
              pCtrl->EP0Data.pData = (uint8_t *) WCID_ExtProp_Descriptor;
              pCtrl->EP0Data.Count = pCtrl->SetupPacket.wLength;
              USBD_API->core->DataInStage(pCtrl);
              ret = LPC_OK;
            }
            break;
        }
        break;
    }
  }
  return ret;
}

/**
 * @brief	Disable USB
 * @param   Nothing
 * @return	Nothing
 */
void Disable_USB(void)
{
  /* Signal disconnect */
  USBD_API->hw->Connect(g_hUsb, 0);
  /* Disable USB controller & Clear flags */
  /* First disconnect the USB pins & powerdown PHY */
		
  /* LPC11U6X */
  /* PIO0_6 used for GPIO not USB_CONNECT */
  Chip_IOCON_PinMuxSet( LPC_IOCON, 0,  6,  (IOCON_FUNC0 | IOCON_MODE_INACT));
  /* PIO0_3 used for GPIO not USB_VBUS */
  Chip_IOCON_PinMuxSet( LPC_IOCON, 0,  3,  (IOCON_FUNC0 | IOCON_MODE_INACT));
  /* Disable USB PAD power */
  Chip_SYSCTL_PowerDown(SYSCTL_POWERDOWN_USBPAD_PD);

  /* Toggle connect bit to reset usb address. This should also clear the pending setup packet if any */
  USBD_API->hw->Connect(g_hUsb, 1);
  USBD_API->hw->Connect(g_hUsb, 0);

  /* Final Reset the Controller registers to reset value */
  LPC_USB->DEVCMDSTAT = 0x07000800;
  LPC_USB->EPLISTSTART = 0x0;
  LPC_USB->DATABUFSTART = 0x0;
  LPC_USB->LPM = 0x0;
  LPC_USB->EPSKIP = 0x0;
  LPC_USB->EPINUSE = 0x0;
  LPC_USB->EPBUFCFG = 0x0;
  LPC_USB->INTSTAT = 0xFFFFFFFF;
  LPC_USB->INTEN = 0x0;

  LPC_USB->INTROUTING	= 0x0;
  /* Switching Main clock resource to IRC */
  Chip_Clock_SetMainClockSource(SYSCTL_MAINCLKSRC_IRC);
  /* Disable Peripheral powers */
  Chip_SYSCTL_PowerDown(SYSCTL_POWERDOWN_USBPLL_PD|SYSCTL_POWERDOWN_SYSOSC_PD|SYSCTL_POWERDOWN_SYSPLL_PD);
}

/**
 * @brief	ReinvokeISP
 * @param   Nothing
 * @return	Nothing
 */
void ReinvokeISP(void)
{
  Disable_USB();
  /* Enable core interrupts */
  __enable_irq();

  /* Set stack pointer to ROM value (reset default) This must be the last
     piece of code executed before calling ISP, because most C expressions
     and function returns will fail after the stack pointer is changed. */
#if SBL_LPC11U68
  __set_MSP(0x10001800); // Hardcoded to top of minimum ram available in 11U2X
#endif

#if SBL_LPC5411X
  __set_MSP(0x20001000); // Hardcoded to top of minimum ram available in LPC54110
#endif

  /* Enter ISP */
  Chip_IAP_ReinvokeISP();
  /* Not supposed to come back! */
}


/* Main application entry point */
int main(void)
{
  USBD_API_INIT_PARAM_T usb_param;
  USB_CORE_DESCS_T desc;
  ErrorCode_t ret = LPC_OK;
  IRQn_Type irqNum;

  uint32_t timerFreq;

  pushAppFlag = (uint8_t *) SL_ADDRESS_APPCALLEDFL;

  /* Comment Board_Init(), due to no need initial uart or other pins */
  //Board_Init();
  /* Initialize GPIO */
  Chip_GPIO_Init(LPC_GPIO);
  /* Update SystemCoreClock */
  SystemCoreClockUpdate();

  /* Calculate flash size */
  LPC_SBL_CalFlashSize();

  /* Disable sysTick */
  SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;
  // LPC11U6X
  /* Disable all NVIC interrupts in case this loader is called from the user application */
  /* there're total 32 interrupts in LPC11U6x */
  for (irqNum = PIN_INT0_IRQn; irqNum <= 31; irqNum++)
  {
    NVIC_DisableIRQ(irqNum);
    NVIC_ClearPendingIRQ(irqNum);
  }

  /* Booted from Reset or Power On */
  if(*pushAppFlag == 0x44)
  {
	  /* If One of Images is valid, then jump to the right application image */
	  LPC_SBL_BootImageCheck();
	  /* If no Images are valid, enter DFU process */
  }

  /* enable clocks and pinmux */
  Chip_USB_Init();
  /* initialize USBD ROM API pointer. */
  g_pUsbApi = (const USBD_API_T *) LPC_ROM_API->usbdApiBase;

  /* initialize call back structures */
  memset((void *) &usb_param, 0, sizeof(USBD_API_INIT_PARAM_T));

  usb_param.usb_reg_base = LPC_USB0_BASE;

  /* WORKAROUND for artf44835 ROM driver BUG:
     Code clearing STALL bits in endpoint reset routine corrupts memory area
     next to the endpoint control data. For example When EP0, EP1_IN, EP1_OUT,
     EP2_IN are used we need to specify 3 here. But as a workaround for this
     issue specify 4. So that extra EPs control structure acts as padding buffer
     to avoid data corruption. Corruption of padding memory doesn?t affect the
     stack/program behavior.
  */
  usb_param.max_num_ep = 1 + 1;
  usb_param.mem_base = USB_STACK_MEM_BASE;
  usb_param.mem_size = USB_STACK_MEM_SIZE;
  usb_param.USB_Reset_Event = update_device_status_patch;
  /* Set the USB descriptors */
  desc.device_desc = (uint8_t *) &USB_DeviceDescriptor;
  desc.string_desc = (uint8_t *) USB_StringDescriptor;

  /* Report SBL version via device descriptor */
  /* Bit 6 of the major version number is set when DFU SL function is active in rom
     Bit 5 of major version number indicates fw update mode */
  USB_DeviceDescriptor.bcdDevice = ( (CFG_MAJOR_VERSION) << 8 ) | CFG_MINOR_VERSION;

  /* Note, to pass USBCV test full-speed only devices should have both
  * descriptor arrays point to same location and device_qualifier set
  * to 0.
  */
  desc.high_speed_desc = (uint8_t *) USB_FsConfigDescriptor;
  desc.full_speed_desc = (uint8_t *) USB_FsConfigDescriptor;
  desc.device_qualifier = 0;
  /* USB Initialization */
  ret = USBD_API->hw->Init(&g_hUsb, &desc, &usb_param);

  if (ret == LPC_OK)
  {
    /* WORKAROUND for artf32219 ROM driver BUG:
       The mem_base parameter part of USB_param structure returned
       by Init() routine is not accurate causing memory allocation issues for
       further components.
    */
    usb_param.mem_base = USB_STACK_MEM_BASE + (USB_STACK_MEM_SIZE - usb_param.mem_size);

    ret = usb_dfu_init( g_hUsb,
                       (USB_INTERFACE_DESCRIPTOR *) &USB_FsConfigDescriptor[sizeof(USB_CONFIGURATION_DESCRIPTOR)],
                       &usb_param.mem_base, &usb_param.mem_size);

    /* register WCID handler */
    ret = USBD_API->core->RegisterClassHandler(g_hUsb, WCID_hdlr, 0);

    if (ret == LPC_OK)
    {
      /* now connect */
      USBD_API->hw->Connect(g_hUsb, 1);
    }
  }

  // Start timer
  Chip_TIMER_Init(LPC_TIMER32_0);
  timerFreq = Chip_Clock_GetSystemClockRate();
  Chip_TIMER_MatchEnableInt(LPC_TIMER32_0, 1);
  Chip_TIMER_SetMatch(LPC_TIMER32_0, 1, (timerFreq));
  Chip_TIMER_ResetOnMatchEnable(LPC_TIMER32_0, 1);
  Chip_TIMER_Enable(LPC_TIMER32_0);
  NVIC_ClearPendingIRQ(TIMER_32_0_IRQn);

#if   defined ( __GNUC__ )
  __ASM volatile ("cpsid i" : : : "memory");
#elif defined ( __CC_ARM )
  __ASM ("cpsid i");
#else
  XXXX    // error, reminder different compiler
#endif
  /* Reset System Counter here */
  dwSysTicks = 0;

  /* Reset any variables here */
  gLPCflashOffset = 0;

  /* Poll for USB events. All interrupts in SBL flash are redirected to user app */
  while(1)
  {
    /* Polling process USB interrupt status based on flags */
    if(NVIC_GetPendingIRQ(USB0_IRQn))
    {
      /* USB Operation */
      USB_Handler();
      /* Clear USB interrupt status */
      NVIC_ClearPendingIRQ(USB0_IRQn);
    }
    else
    {
      /* DFU Operation */
      DFU_ProcessData();
    }
    /* Polling process Timer interrupt status based on flags */
    if(NVIC_GetPendingIRQ(TIMER_32_0_IRQn))
    {
      /* Clear Match event flag */
      Chip_TIMER_ClearMatch(LPC_TIMER32_0, 1);
      /* Increase Tick Counter Value */
      Tick_Handler();
      /* Clear Match interrupt status */
      NVIC_ClearPendingIRQ(TIMER_32_0_IRQn);
    }

    /* USB detached & goto application */
    if(*pushAppFlag == 0x11) 
    {
      /* Clean dwSysTicks */
      dwSysTicks = 0;
	  /* Reset any variables here */
      gLPCflashOffset = 0;
      /* De-initialize USB */
      Disable_USB();
      /* Only booted from app */
      /* Check Image -> Clean system(if image valid) -> Reboot System */
      LPC_SBL_BootImageCheck();
    }
    else
    {
      /* Ignor --Was loader booted from app (0x01) or from FLASH on reset (0x44)? */
      /* Only booted from app */
      /* Time out, beyond 5 seconds */
      if ( dwSysTicks > 5 )
      {
    	/* Clean dwSysTicks */
        dwSysTicks = 0;
        /* De-initialize USB */
        Disable_USB();
        /* Check image in flash and boot */
        LPC_SBL_BootImageCheck();
      }
    }
  }
//  return 0;
}

// end file

