/*
 * @brief Dual Image code
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2017
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

/*
 * Author:
 * Date:    19, Dec, 2016
 * Purpose: 
 * Ver:     v0.1
 */
#include <string.h>
#include "board.h"

#include "sl_common.h"
#include "sl_protocol.h"
#include "sl_flash.h"
#include "sbl_lpc.h"

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/
volatile uint32_t gLPCFlashSize = 0;
volatile uint32_t gLPCdualImageBoundary = 0;
volatile uint32_t gLPCflashOffset = 0;

/*****************************************************************************
 * Private functions
 ****************************************************************************/

/*****************************************************************************
 * Public functions
 ****************************************************************************/

/**
 * @brief	This function is get chip's flash size
 * @param   Nothing
 * @return	Nothing
 */
uint32_t LPC_SBL_GetFlashSize(void)
{
    return gLPCFlashSize;
}

/**
 * @brief	This function is get dual image boundary information
 * @param   Nothing
 * @return	Nothing
 */
uint32_t LPC_SBL_GetDualImgBoundary(void)
{
    return gLPCdualImageBoundary;
}

/**
 * @brief	This function is check vector data is valid or not
 * @param   Nothing
 * @return	Nothing
 */
uint32_t LPC_SBL_VectCheckSum(uint32_t *vector)
{
    volatile uint32_t sum = 0;
    uint8_t i;
    for(i = 0; i < 8; i++) {
        sum += *(vector + i);
    }
    sum = sum;
    return sum;
}

/**
 * @brief	This function is check boot image is valid or not
 * @param   timeoutflag - 0, not enter app asap / 1, enter app asap.
 * @return	Nothing
 */
extern void Disable_USB(void);
void LPC_SBL_BootImageCheck(void)
{
    volatile uint32_t flashSize, appSize, currVersion = 0, bootImg = 0, imgAddr = SL_APP_START_ADDR;
    IMG_HEADER_T *imgHdr;
    
    flashSize = LPC_SBL_GetFlashSize();

    /* LPC11U68's sector is not unify */
    if(flashSize >= (256 *1024))
    {
    	appSize = 120*1024;
    }
    else
    {
    	appSize = (flashSize > 0) ?(flashSize - 8*1024) >> 1 : 0;
    }

    while(imgAddr < flashSize)
    {
        imgHdr = (IMG_HEADER_T *)(imgAddr + SL_BOOTAPP_IMGHDR_OFS);
        if( (LPC_SBL_VectCheckSum((uint32_t *)imgAddr) == 0) &&
            (imgHdr->header_marker == IMG_HEADER_MARKER) &&
            ( ((imgHdr->img_type == IMG_NORMAL) && (checkAppCRC(imgAddr) == 0)) || (imgHdr->img_type == IMG_NO_CRC)))
        {
                
            if(imgHdr->userFmwrVersion > currVersion)
            {
                currVersion = imgHdr->userFmwrVersion;
                bootImg = imgAddr;
            }
        }
        imgAddr += appSize;
    }

    if (bootImg >=  SL_APP_START_ADDR )
    {
		doCleanBoot(bootImg);
    }
}

static uint32_t LPC_IAP_ReadPID(void)
{
	return Chip_FLASH_ReadPID();
    /* failed */
	return 0;
}

/**
 * @brief	This function is calculate the flash size based on ChipID
 * @param   Nothing
 * @return	Nothing
 */
void LPC_SBL_CalFlashSize(void)
{
    switch(LPC_IAP_ReadPID())
    {
/* LPC11U6x */
        case 0x0000DCC8:	// LPC11U66JBD48 64kB
        	gLPCFlashSize = 64 * 1024;
        	gLPCdualImageBoundary = (8 + (64 - 8)/2) * 1024;
            break;
        case 0x0000BC88:	// LPC11U67JBD48/LPC11U67JBD64
        	gLPCFlashSize = 128 * 1024;
        	gLPCdualImageBoundary = (8 + (128 - 8)/2) * 1024;
            break;
        case 0x0000BC80:	// LPC11U67JBD100
        	gLPCFlashSize = 128 * 1024;
        	gLPCdualImageBoundary = (8 + (128 - 8)/2) * 1024;
            break;
        case 0x00007C08:	// LPC11U68JBD48/LPC11U68JBD64
        	gLPCFlashSize = 256 * 1024;
        	/* LPC11U68 Flash Sector is not unified as the same, the 24-28 is 32KB, the 0-23 is 4KB */
        	/* Here the Image Boundary is Sector 25,  */
        	/* To prevent waste of flash, we define the 2nd Image boundary start @0x20000 not 0x21000 */
        	/* So the 1st Image will start@0x2000  end@0x20000, total size is 120KB */
        	/* So the 2nd Image will start@0x20000 end@0x40000, total size is 128KB */
        	//gLPCdualImageBoundary = (8 + (256 - 8)/2) * 1024; // 0x21000
        	gLPCdualImageBoundary = 0x20000;
            break;
        case 0x00007C00:	// LPC11U68JBD100
        	gLPCFlashSize = 256 * 1024;
        	/* LPC11U68 Flash Sector is not unified as the same, the 24-28 is 32KB, the 0-23 is 4KB */
        	/* Here the Image Boundary is Sector 25,  */
        	/* To prevent waste of flash, we define the 2nd Image boundary start @0x20000 not 0x21000 */
        	/* So the 1st Image will start@0x2000  end@0x20000, total size is 120KB */
        	/* So the 2nd Image will start@0x20000 end@0x40000, total size is 128KB */
        	//gLPCdualImageBoundary = (8 + (256 - 8)/2) * 1024; // 0x21000
        	gLPCdualImageBoundary = 0x20000;
            break;

/* None of LPC11U6x series */
        default:
        	gLPCFlashSize = 0;
        	gLPCdualImageBoundary = 0;
            break;
        
    }
}

/**
 * @brief	This function is cleanup system before jump to application
 * @param   Nothing
 * @return	Nothing
 */
void LPC_SBL_CleanUpSystem(void)
{
	/* Switch Main clock resource to IRC */
	Chip_Clock_SetMainClockSource(SYSCTL_MAINCLKSRC_IRC);
	/* No need to update the systemclock as user app is going to takeover anyway*/
	/* Setup FLASH access to 2 clock */
	Chip_FMC_SetFLASHAccess(FLASHTIM_2CLK_CPU);
	Chip_SYSCTL_PowerDown(SYSCTL_POWERDOWN_SYSPLL_PD);
}

/**
 * @brief	doCleanBoot
 * @param   Nothing
 * @return	Nothing
 */
typedef void (*iapfun)(void);
iapfun jump2app;
void doCleanBoot(uint32_t imgAddr)
{
	/* Cleanup before going to app */
	LPC_SBL_CleanUpSystem();

	Chip_PMU_WriteGPREG(LPC_PMU, 3, imgAddr);

	/* Boot Valid Application */
	/* Set Stack */
	__set_MSP(*(volatile uint32_t*)  (imgAddr));
	/* Set app entry point */
	jump2app =(iapfun)*(uint32_t*)(imgAddr+4);

	/* enable interrupt */
#if   defined ( __GNUC__ )
	__ASM volatile ("cpsie i" : : : "memory");
#elif defined ( __CC_ARM )
	__ASM ("cpsie i");
#else
	XXXX  // Error reminder for different compiler
#endif

	/* Jump_To_Application = (pFunction) JumpAddress; */
	jump2app();
	/***** Control should never come here *****/
}

// end file

