/*
 * @brief Image programming code
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2014
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

/*
 * Author:
 * Date:    19, Dec, 2016
 * Purpose: compatable with LPC11u6x and lpc5411x
 * Ver:     v0.1
 */
#include <string.h>
#include "board.h"

#include "sl_common.h"
#include "sl_protocol.h"
#include "sl_flash.h"

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Private functions
 ****************************************************************************/

/*****************************************************************************
 * Public functions
 ****************************************************************************/

/* Calculate the flash sector number based on address input */
uint32_t SBL_GetSectNdx(uint32_t addr, uint32_t *pSectSz)
{
	uint32_t ndx, sectSz;
// Only for LPC11U6x
	if (addr < SL_FLASH_LRGSECT_ADDR) {
		ndx = (addr / SL_FLASH_SECT_SZ);
		sectSz = SL_FLASH_SECT_SZ;
	}
	else {
		ndx = ((addr - SL_FLASH_LRGSECT_ADDR) / SL_FLASH_SECTLRG_SZ) + SL_FLASH_SECT_CNT;
		sectSz = SL_FLASH_SECTLRG_SZ;
	}
	if (pSectSz)
		pSectSz[0] = sectSz;

	return ndx;
}

/* Flash write blocks */
volatile uint32_t flashwrflag = 0, era_start_sect=0, era_end_sect=0;
int32_t flashWriteBlock(uint32_t *src, uint32_t block_nr)
{
	volatile uint32_t dst, start_sect, end_sect, rc;
	volatile uint32_t pmask = __get_PRIMASK();
	flashwrflag = 1;
	volatile uint32_t i;
	for(i=0; i<1000; i++);

	volatile uint32_t sectSz = SL_FLASH_SECT_SZ;
	dst = (SL_FLASH_BLOCK_SZ * block_nr);

	start_sect = SBL_GetSectNdx(dst, (uint32_t *)&sectSz);
	end_sect = start_sect;	

	/* Sectors size is 4KB , for LPC11U6x*/
	if(start_sect <= 23)
	{
		if ( (dst % sectSz) == 0x0)
		{
			__disable_irq();
			rc = Chip_IAP_PreSectorForReadWrite((uint32_t)start_sect, end_sect);
			if (rc == LPC_OK)
			{
				rc = Chip_IAP_EraseSector(start_sect, end_sect);
			}
			__set_PRIMASK(pmask);
			if (rc != LPC_OK)
			{
				return rc;
			}
		}
	}
	/* Sectors size is 32KB */
	else
	{
		if ( (dst % SL_FLASH_SECTLRG_SZ) == 0x0)
		{
			__disable_irq();
			rc = Chip_IAP_PreSectorForReadWrite((uint32_t)start_sect, end_sect);
			if (rc == LPC_OK)
			{
				rc = Chip_IAP_EraseSector(start_sect, end_sect);
			}
			__set_PRIMASK(pmask);
			if (rc != LPC_OK)
			{
				return rc;
			}
		}
	}

	/* Disable Interrupt, prevent data corrupt */
	__disable_irq();
	/* Prepare Sector first of all */
	rc = Chip_IAP_PreSectorForReadWrite(start_sect, end_sect);
	if (rc == LPC_OK)
	{
		/* Copy data from RAM buffer to Flash */
		Chip_IAP_CopyRamToFlash(dst, src, SL_FLASH_BLOCK_SZ);
	}
    __set_PRIMASK(pmask);
	/* Enable Interrupt */
	__disable_irq();

	flashwrflag = 0;

	return rc;
}

/* Flash erase page */
int32_t flashErasePage(uint32_t start_addr, uint32_t end_addr)
{
	uint32_t rc, start_sect, end_sect, start_page, end_page;
	uint32_t pmask = __get_PRIMASK();

	start_sect = SBL_GetSectNdx(start_addr, 0);
	end_sect = SBL_GetSectNdx(end_addr, 0);
	start_page = start_addr / SL_FLASH_PAGE_SZ;
	end_page = end_addr / SL_FLASH_PAGE_SZ;
	__disable_irq();
	rc = Chip_IAP_PreSectorForReadWrite(start_sect, end_sect);
	if ( rc == LPC_OK )
	{
		rc = Chip_IAP_ErasePage(start_page, end_page);
	}
    __set_PRIMASK(pmask);
	return rc;
}

/* Flash write page */
int32_t flashWritePage(uint32_t dst, uint32_t *src)
{
	uint32_t rc, start_sect, end_sect;
	uint32_t pmask = __get_PRIMASK();

	if ( (dst  < SL_APP_START_ADDR) )
	{
		/* Protect secondary loader */
		return ERR_ISP_INVALID_SECTOR;
	}

	if ( (dst % SL_FLASH_PAGE_SZ) != 0 )
	{
		/* Protect secondary loader */
		return ERR_ISP_DST_ADDR_ERROR;
	}

	start_sect = SBL_GetSectNdx(dst, 0);
	end_sect = start_sect;
	__disable_irq();
	rc = Chip_IAP_PreSectorForReadWrite(start_sect, end_sect);
	if (rc == LPC_OK)
	{
		Chip_IAP_CopyRamToFlash(dst, src, SL_FLASH_PAGE_SZ);
	}
    __set_PRIMASK(pmask);
	return rc;
}

/* Flash erase sectors */
int32_t flashEraseSector(uint32_t start_sect, uint32_t end_sect)
{
	uint32_t rc;
	uint32_t pmask = __get_PRIMASK();
	if ( (start_sect < 2) || (end_sect < 2)) {
		/* Protect secondary loader */
		return ERR_ISP_INVALID_SECTOR;
	}
	__disable_irq();
	rc = Chip_IAP_PreSectorForReadWrite(start_sect, end_sect);
	if ( rc == LPC_OK ) {
		rc = Chip_IAP_EraseSector(start_sect, end_sect);
	}
    __set_PRIMASK(pmask);
	return rc;
}

/* Flash write sector */
int32_t flashWriteSector(uint32_t *src, uint32_t start_sect, uint32_t end_sect)
{
	uint32_t rc;
	flashwrflag = 1;
	uint32_t pmask = __get_PRIMASK();

	__disable_irq();
	rc = Chip_IAP_PreSectorForReadWrite(start_sect, end_sect);
	if (rc == LPC_OK)
	{
		rc = Chip_IAP_EraseSector(start_sect, end_sect);
	}
	__set_PRIMASK(pmask);
	if (rc != LPC_OK)
	{
		return rc;
	}

	__disable_irq();
	rc = Chip_IAP_PreSectorForReadWrite(start_sect, end_sect);
	if (rc == LPC_OK)
	{
		Chip_IAP_CopyRamToFlash(start_sect*SL_FLASH_SECT_SZ, src, SL_FLASH_SECT_SZ);
	}
    __set_PRIMASK(pmask);
	flashwrflag = 0;
	return rc;
}

/* Flash read blocks */
int32_t flashReadBlock(uint32_t *dest, uint32_t block_nr)
{
	uint32_t src = (SL_FLASH_BLOCK_SZ * block_nr);

	if ( (src + SL_FLASH_BLOCK_SZ) > SL_FLASH_END )
	{
		return ERR_API_INVALID_PARAMS;
	}
	memcpy(dest, (void *) src, SL_FLASH_BLOCK_SZ);

	return LPC_OK;
}

/* Flash read page */
int32_t flashReadPage(uint32_t *dest, uint32_t page_nr)
{
	uint32_t src = (SL_FLASH_PAGE_SZ * page_nr);

	if ( (src + SL_FLASH_PAGE_SZ) > SL_FLASH_END )
	{
		return ERR_API_INVALID_PARAMS;
	}
	memcpy(dest, (void *) src, SL_FLASH_PAGE_SZ);

	return LPC_OK;
}

// end file

