/*--------------------------------------------------------------------------	
*
* File:  MPC82xx.H
*
*
* Description:  
*
* Main Internal Memory Maps for the PowerQUICC 2 family
*
* Note that different structures can be overlaid at the same offsets for the 
* different modes of operation.
*
*
* History:
*
* 13 JUL 98    ggh    initial version derived from MSIL 8260 drivers of 6/98
* 12 DEC 98    jes    changes for ATM app. software
* 31 Jul 03    pdw    Includes Internal TC layer registers
* 14 Nov 03    pdw 	  Added trap regs 5-8 fro Hip7 silicon 
*
*----------------------------------------------------------------------------*/

#define _Packed 
#define _PackedType 

#pragma pack(push,1)

/******************************************************************************
*
*  PARAMETER RAM (PRAM) FOR EACH PERIPHERAL
*  
*  Each subsection contains protocol-specific PRAM for each peripheral,
*  followed by the PRAM common to all protocols for that peripheral.  These 
*  structs are used as needed in the main MPC8260 memory map structure.  Note 
*  that different modes of operation will require the use of different PRAM 
*  structs, and that certain structs may overlay and conflict with the use of 
*  other PRAM areas.  Consult the MPC8260 User Manual for details as to what
*  is unavailable when certain protocols are run on certain peripherals.
*
******************************************************************************/
								  


/*---------------------------------------------------------------------------*/
/*                   SERIAL COMMUNICATION CONTROLLER (SCC)                 */
/*---------------------------------------------------------------------------*/

/*----------*/
/* SCC HDLC */
/*----------*/

typedef  _Packed struct 
{
    UBYTE    reserved1[4]; /* Reserved area */
    UWORD    c_mask;       /* CRC constant */
    UWORD    c_pres;       /* CRC preset */
    UHWORD   disfc;        /* discarded frame counter */
    UHWORD   crcec;        /* CRC error counter */
    UHWORD   abtsc;        /* abort sequence counter */
    UHWORD   nmarc;        /* nonmatching address rx cnt */
    UHWORD   retrc;        /* frame transmission counter. */
                           /* For FCC this area is reserved.*/
    UHWORD   mflr;         /* maximum frame length reg */
    UHWORD   max_cnt;      /* maximum length counter */
    UHWORD   rfthr;        /* received frames threshold */
    UHWORD   rfcnt;        /* received frames count */
    UHWORD   hmask;        /* user defined frm addr mask */
    UHWORD   haddr1;       /* user defined frm address 1 */
    UHWORD   haddr2;       /* user defined frm address 2 */
    UHWORD   haddr3;       /* user defined frm address 3 */
    UHWORD   haddr4;       /* user defined frm address 4 */
    UHWORD   tmp;          /* temp */
    UHWORD   tmp_mb;       /* temp */
}_PackedType t_HdlcScc_Pram;

 
/*--------------*/
/* SCC Ethernet */
/*--------------*/

typedef  _Packed struct 
{
    UWORD    c_pres;      /* CRC preset */
    UWORD    c_mask;      /* CRC constant mask*/
    UWORD    crcec;       /* CRC error counter */
    UWORD    alec;        /* alignment error counter */
    UWORD    disfc;       /* discarded frame counter */
    UHWORD   pads;        /* Short frame pad character. */
    UHWORD   ret_lim;     /* Retry limit threshold. */
    UHWORD   ret_cnt;     /* Retry limit counter. */
    UHWORD   mflr;        /* maximum frame length reg */
    UHWORD   minflr;      /* minimum frame length reg */
    UHWORD   maxd1;       /* max DMA1 length register. */
    UHWORD   maxd2;       /* max DMA2 length register. */
    UHWORD   maxd;        /* Rx max DMA. */
    UHWORD   dma_cnt;     /* Rx DMA counter. */
    UHWORD   max_b;       /* max buffer descriptor byte count. */
    UHWORD   gaddr1;      /* group address filter */
    UHWORD   gaddr2;      /* group address filter */
    UHWORD   gaddr3;      /* group address filter */
    UHWORD   gaddr4;      /* group address filter */
    UWORD    tbuf0_data0; /* Saved area 0, current frame. */
    UWORD    tbuf0_data1; /* Saved area 1, current frame. */
    UWORD    tbuf0_rba0;
    UWORD    tbuf0_crc;
    UHWORD   tbuf0_bcnt;
    UHWORD   paddr1_h;    /* physical address (MSB) */
    UHWORD   paddr1_m;    /* physical address */
    UHWORD   paddr1_l;    /* physical address (LSB) */
    UHWORD   p_per;       /* persistence */
    UHWORD   rfbd_ptr;    /* Rx first BD pointer. */
    UHWORD   tfbd_ptr;    /* Tx first BD pointer. */
    UHWORD   tlbd_ptr;    /* Tx last BD pointer. */
    UWORD    tbuf1_data0; /* Saved area 0, next frame. */
    UWORD    tbuf1_data1; /* Saved area 1, next frame. */
    UWORD    tbuf1_rba0;
    UWORD    tbuf1_crc;
    UHWORD   tbuf1_bcnt;
    UHWORD   tx_len;      /* tx frame length counter */
    UHWORD   iaddr1;      /* individual address filter. */
    UHWORD   iaddr2;      /* individual address filter.  */
    UHWORD   iaddr3;      /* individual address filter. */
    UHWORD   iaddr4;      /* individual address filter.  */
    UHWORD   boff_cnt;    /* back-off counter */
    UHWORD   taddr_h;     /* temp address (MSB) */
    UHWORD   taddr_m;     /* temp address */
    UHWORD   taddr_l;     /* temp address (LSB) */
} _PackedType t_EnetScc_Pram;

/*----------*/
/* SCC UART */
/*----------*/

typedef  _Packed struct 
{
    UBYTE       reserved1[8];   /* Reserved area */
    UHWORD      max_idl;        /* maximum idle characters */
    UHWORD      idlc;           /* rx idle counter (internal) */
    UHWORD      brkcr;          /* break count register */
    UHWORD      parec;          /* Rx parity error counter */
    UHWORD      frmec;          /* Rx framing error counter */
    UHWORD      nosec;          /* Rx noise counter */
    UHWORD      brkec;          /* Rx break character counter */
    UHWORD      brkln;          /* Receive break length */
    UHWORD      uaddr1;         /* address character 1 */
    UHWORD      uaddr2;         /* address character 2 */
    UHWORD      rtemp;          /* temp storage */
    UHWORD      toseq;          /* Tx out of sequence char */
    UHWORD      cc[8];          /* Rx control characters */
    UHWORD      rccm;           /* Rx control char mask */
    UHWORD      rccr;           /* Rx control char register */
    UHWORD      rlbc;           /* Receive last break char */
} _PackedType t_UartScc_Pram;


/*-----------------*/
/* SCC Transparent */
/*-----------------*/

typedef  _Packed struct  
{
    UWORD    c_mask;     /* CRC constant */
    UWORD    c_pres;     /* CRC preset */
} _PackedType t_TransScc_Pram;


/*------------*/
/* SCC Bisync */
/*------------*/

typedef  _Packed struct  
{
   UBYTE 	reserved1[4];		/* Reserved area */
   UWORD	crcc;			/* CRC Constant Temp Value */
   UHWORD	prcrc;			/* Preset Receiver CRC-16/LRC */
   UHWORD	ptcrc;			/* Preset Transmitter CRC-16/LRC */
   UHWORD	parec;			/* Receive Parity Error Counter */
   UHWORD	bsync;			/* BISYNC SYNC Character */
   UHWORD	bdle;			/* BISYNC DLE Character */
   UHWORD	cc[8];			/* Rx control characters */
   UHWORD	rccm;			/* Receive Control Character Mask */
} _PackedType t_BisyncScc_Pram;


/*-----------------*/
/* SCC Common PRAM */
/*-----------------*/

typedef  _Packed struct 
{
    UHWORD   rbase;      /* RX BD base address */
    UHWORD   tbase;      /* TX BD base address */
    UBYTE    rfcr;       /* Rx function code */
    UBYTE    tfcr;       /* Tx function code */
    UHWORD   mrblr;      /* Rx buffer length */
    UWORD    rstate;     /* Rx internal state */
    UWORD    rptr;       /* Rx internal data pointer */
    UHWORD   rbptr;      /* rb BD Pointer */
    UHWORD   rcount;     /* Rx internal byte count */
    UWORD    rtemp;      /* Rx temp */
    UWORD    tstate;     /* Tx internal state */
    UWORD    tptr;       /* Tx internal data pointer */
    UHWORD   tbptr;      /* Tx BD pointer */
    UHWORD   tcount;     /* Tx byte count */
    UWORD    ttemp;      /* Tx temp */
    UWORD    rcrc;       /* temp receive CRC */
    UWORD    tcrc;       /* temp transmit CRC */
    union 
    {
	    t_HdlcScc_Pram    h;
	    t_EnetScc_Pram    e;
	    t_UartScc_Pram    u;
	    t_TransScc_Pram   t;
        t_BisyncScc_Pram  b;
    } SpecificProtocol;
    VUBYTE COMPLETE_SIZE_OF_DPRAM_PAGE[0x5c];
} _PackedType t_Scc_Pram;



/*---------------------------------------------------------------------------*/
/*	                    FAST COMMUNICATION CONTROLLER (FCC)					 */
/*---------------------------------------------------------------------------*/

/*----------*/
/* FCC HDLC */
/*----------*/

typedef  _Packed struct 
{
    UBYTE    reserved1[8]; /* Reserved area */
    UWORD    c_mask;       /* CRC constant */
    UWORD    c_pres;       /* CRC preset */
    UHWORD   disfc;        /* discarded frame counter */
    UHWORD   crcec;        /* CRC error counter */
    UHWORD   abtsc;        /* abort sequence counter */
    UHWORD   nmarc;        /* nonmatching address rx cnt */
    UWORD    max_cnt;      /* maximum length counter */
    UHWORD   mflr;         /* maximum frame length reg */
    UHWORD   rfthr;        /* received frames threshold */
    UHWORD   rfcnt;        /* received frames count */
    UHWORD   hmask;        /* user defined frm addr mask */
    UHWORD   haddr1;       /* user defined frm address 1 */
    UHWORD   haddr2;       /* user defined frm address 2 */
    UHWORD   haddr3;       /* user defined frm address 3 */
    UHWORD   haddr4;       /* user defined frm address 4 */
    UHWORD   tmp;          /* temp */
    UHWORD   tmp_mb;       /* temp */
} _PackedType t_HdlcFcc_Pram;


/*--------------*/
/* FCC Ethernet */
/*--------------*/

typedef  _Packed struct 
{
    UWORD    stat_bus;       /* Internal use buffer. */
    UWORD    cam_ptr;        /* CAM address. */
    UWORD    c_mask;         /* CRC constant mask*/
    UWORD    c_pres;	     /* CRC preset */
    UWORD    crcec;          /* CRC error counter */
    UWORD    alec;           /* alignment error counter */
    UWORD    disfc;          /* discarded frame counter */
    UHWORD   ret_lim;        /* Retry limit threshold. */
    UHWORD   ret_cnt;        /* Retry limit counter. */
    UHWORD   p_per;          /* persistence */
    UHWORD   boff_cnt;       /* back-off counter */
    UWORD    gaddr_h;        /* group address filter, high */
    UWORD    gaddr_l;        /* group address filter, low */
    UHWORD   tfcstat;        /* out of sequece Tx BD staus. */
    UHWORD   tfclen;         /* out of sequece Tx BD length. */
    UWORD    tfcptr;         /* out of sequece Tx BD data pointer. */
    UHWORD   mflr;           /* maximum frame length reg */
    UHWORD   paddr1_h;       /* physical address (MSB) */
    UHWORD   paddr1_m;       /* physical address */
    UHWORD   paddr1_l;       /* physical address (LSB) */
    UHWORD   ibd_cnt;        /* internal BD counter. */
    UHWORD   ibd_start;      /* internal BD start pointer. */
    UHWORD   ibd_end;        /* internal BD end pointer. */
    UHWORD   tx_len;         /* tx frame length counter */
    UBYTE    ibd_base[0x20]; /* internal micro code usage. */
    UWORD    iaddr_h;        /* individual address filter, high */
    UWORD    iaddr_l;        /* individual address filter, low */
    UHWORD   minflr;         /* minimum frame length reg */
    UHWORD   taddr_h;        /* temp address (MSB) */
    UHWORD   taddr_m;        /* temp address */
    UHWORD   taddr_l;        /* temp address (LSB) */
    UHWORD   pad_ptr;        /* pad_ptr. */
    UHWORD   cf_type;        /* flow control frame type coding. */
    UHWORD   cf_range;       /* flow control frame range. */
    UHWORD   max_b;          /* max buffer descriptor byte count. */
    UHWORD   maxd1;          /* max DMA1 length register. */
    UHWORD   maxd2;          /* max DMA2 length register. */
    UHWORD   maxd;           /* Rx max DMA. */
    UHWORD   dma_cnt;        /* Rx DMA counter. */
    
    /* counter: */
    UWORD    octc;           /* received octets counter. */
    UWORD    colc;           /* estimated number of collisions */
    UWORD    broc;           /* received good packets of broadcast address */
    UWORD    mulc;           /* received good packets of multicast address */
    UWORD    uspc;           /* received packets shorter then 64 octets. */
    UWORD    frgc;           /* as uspc + bad packets */
    UWORD    ospc;           /* received packets longer then 1518 octets. */
    UWORD    jbrc;           /* as ospc + bad packets  */
    UWORD    p64c;           /* received packets of 64 octets.. */
    UWORD    p65c;           /* received packets of 65-128 octets.. */
    UWORD    p128c;          /* received packets of 128-255 octets.. */
    UWORD    p256c;          /* received packets of 256-511 octets.. */
    UWORD    p512c;          /* received packets of 512-1023 octets.. */
    UWORD    p1024c;         /* received packets of 1024-1518 octets.. */
    UWORD    cam_buf;        /* cam respond internal buffer. */
    UHWORD   rfthr;          /* received frames threshold */
    UHWORD   rfcnt;          /* received frames count */
} _PackedType t_EnetFcc_Pram;


/*-----------------*/
/* FCC Common PRAM */
/*-----------------*/

typedef  _Packed struct 
{
    UHWORD   riptr;      /* Rx internal temporary data pointer. */
    UHWORD   tiptr;      /* Tx internal temporary data pointer. */
    UHWORD   reserved0;  /* Reserved */
    UHWORD   mrblr;      /* Rx buffer length */
    UWORD    rstate;     /* Rx internal state */
    UWORD    rbase;      /* RX BD base address */
    UHWORD   rbdstat;    /* Rx BD status and control */
    UHWORD   rbdlen;     /* Rx BD data length */
    UWORD    rdptr;      /* rx BD data pointer */
    UWORD    tstate;     /* Tx internal state */
    UWORD    tbase;      /* TX BD base address */
    UHWORD   tbdstat;    /* Tx BD status and control */
    UHWORD   tbdlen;     /* Tx BD data length */
    UWORD    tdptr;      /* Tx  data pointer */
    UWORD    rbptr;      /* rx BD pointer */
    UWORD    tbptr;      /* Tx BD pointer */
    UWORD    rcrc;       /* Temp receive CRC */
    UWORD    reserved_1[0x1];
    UWORD    tcrc;       /* Temp transmit CRC */
    union                /* Protocol-Specific parameter ram */
	{
        t_HdlcFcc_Pram    h;
	    t_EnetFcc_Pram    e;
    } SpecificProtocol;      
} _PackedType t_Fcc_Pram;



/*---------------------------------------------------------------------------*/
/*	                MULTICHANNEL COMMUNICATION CONTROLLER (MCC)              */
/*---------------------------------------------------------------------------*/

/******************************************************************************
* Note that each MCC uses multiple logical channels.  We first define the     *
* PRAM for a logical channel (which can be used in either HDLC or Transparent *
* mode;  wherever there are differences, it is specified), followed by the    *
* PRAM for an MCC itself.                                                     *
******************************************************************************/

/*---------------------*/
/* MCC Logical Channel */
/*---------------------*/

typedef  _Packed struct
{
    UWORD    tstate;     /* Tx internal state. */ 
    UWORD    zistate;    /* Zero insertion machine state. */
    UWORD    zidata0;    /* Zero insertion high UHWORD. */
    UWORD    zidata1;    /* Zero insertion low UHWORD. */
    UHWORD   tbdflags;   /* Tx internal BD flags. */
    UHWORD   tbdcnt;     /* Tx internal byte count . */
    UWORD    tbdptr;     /* Tx internal data pointer. */
    UHWORD   intmask;    /* Interrupt mask flags. */
    UHWORD   chamr;      /* channel mode register. */
    UWORD    tcrc;       /* Transparent: reserved. */
                         /* Hdlc: Temp receive CRC.*/
    UWORD    rstate;     /* Rx internal state. */ 
    UWORD    zdstate;    /* Zero deletion machine state. */
    UWORD    zddata0;    /* Zero deletion  high UHWORD. */
    UWORD    zddata1;    /* Zero deletion  low UHWORD. */
    UHWORD   rbdflags;   /* Rx internal BD flags. */
    UHWORD   rbdcnt;     /* Rx internal byte count . */
    UWORD    rbdptr;     /* Rx internal data pointer. */
    UHWORD   maxrlen;    /* Transparent: Max receive buffer length. */
                         /* Hdlc: Max receive frame length. */
    UHWORD   sync_maxcnt;/* Transparent: Receive synchronization pattern*/
                         /* Hdlc: Max length counter. */
    UWORD    rcrc;       /* Transparent: reserved. */
                         /* Hdlc: Temp receive CRC.*/
} _PackedType t_Mch_Pram;


/*----------*/
/* MCC PRAM */
/*----------*/

typedef  _Packed struct
{
    UWORD    mccbase;    /* A pointer to starting address of BD rings. */
    UHWORD   mccstate;   /* Controller state. */
    UHWORD   mrblr;      /* Maximum receive buffer length. */
    UHWORD   grfthr;     /* Global receive frame threshold. */
    UHWORD   grfcnt;     /* Global receive frame counter. */
    UWORD    rinttmp;    /* Temp location for receive interrupt table entry. */                            
    UWORD    data0;      /* Temporary location for holding data. */
    UWORD    data1;      /* Temporary location for holding data. */
    UWORD    tintbase;   /* Transmit interrupt table base address. */
    UWORD    tintptr;    /* Transmit interrupt table pointer. */ 
    UWORD    tinttmp;    /* Temp location for receive interrupt table entry. */
    UHWORD   sctpbase;   /* A pointer to the super channel transmit table*/
    UBYTE    res0[0x2];  /* Reserved area */
    UWORD    c_mask32;   /* CRC constant. */
    UHWORD   xtrabase;   /* A pointer to the beginning of extra parameters */                            
    UHWORD   c_mask16;   /* CRC constant. */
    UWORD    rinttmp0;   /* Temp location for receive interrupt table entry. */                            
    UWORD    rinttmp1;   /* Temp location for receive interrupt table entry. */
    UWORD    rinttmp2;   /* Temp location for receive interrupt table entry. */
    UWORD    rinttmp3;   /* Temp location for receive interrupt table entry. */
    UWORD    rintbase0;  /* Receive interrupt table base address. */
    UWORD    rintptr0;   /* Receive interrupt table pointer. */
    UWORD    rintbase1;  /* Receive interrupt table base address. */
    UWORD    rintptr1;   /* Receive interrupt table pointer. */
    UWORD    rintbase2;  /* Receive interrupt table base address. */
    UWORD    rintptr2;   /* Receive interrupt table pointer. */
    UWORD    rintbase3;  /* Receive interrupt table base address. */
    UWORD    rintptr3;   /* Receive interrupt table pointer. */
    UBYTE    pad[0xa0];
} _PackedType t_Mcc_Pram;



/*---------------------------------------------------------------------------*/
/*                              ATM PARAMETER RAM                            */
/*---------------------------------------------------------------------------*/


/*--------------------------------------*/
/* Address Compression parameters table */
/*--------------------------------------*/

 _Packed struct  AddressCompressionPram 
{
    VUWORD  VptBase;  /* VP-level addressing table base address */
    VUWORD  VctBase;  /* VC-level addressing table base address */
    VUWORD  Vpt1Base; /* VP1-level addressing table base address */	
    VUWORD  Vct1Base; /* VC1-level addressing table base address */
    VUHWORD VpMask;	  /* VP mask for address compression look-up */
} _PackedType;


/*-------------------------------*/
/* External CAM parameters table */
/*-------------------------------*/

 _Packed struct ExtCamPram 
{
    VUWORD  ExtCamBase;		/* Base address of the external CAM */
    VUBYTE  reserved00[4];  /* Reserved */
    VUWORD  ExtCam1Base;	/* Base address of the external CAM1 */
    VUBYTE  reserved01[6];  /* Reserved */
} _PackedType;


/*---------------------------*/
/* ATM mode parameters table */
/*---------------------------*/

typedef  _Packed struct AtmPram 
{
    VUBYTE  reserved0[64];    /* Reserved */
    VUHWORD RxCellTmpBase;    /* Rx cell temporary base address */
    VUHWORD TxCellTmpBase;    /* Tx cell temporary base address */
    VUHWORD UdcTmpBase;       /* UDC temp base address (in UDC mode only) */
    VUHWORD IntRctBase;       /* Internal RTC base address */
    VUHWORD IntTctBase;       /* Internal TCT base address */
    VUHWORD IntTcteBase;      /* Internal ACT base address */
    VUBYTE  reserved1[4];     /* reserved four bytes */
    VUWORD  ExtRctBase;       /* Extrnal RTC base address */
    VUWORD  ExtTctBase;       /* Extrnal TCT base address */
    VUWORD  ExtTcteBase;      /* Extrnal ACT base address */
    VUHWORD UeadOffset;       /* The offset in half-wordunits of the UEAD
                                 entry in the UDC extra header. Should be
                                 even address. If little-endian format is
                                 used, the UeadOffset is of the little-endian
                                 format. */
    VUBYTE  reserved2[2];     /* Reserved */
    VUHWORD PmtBase;		  /* Performance monitoring table base address */
    VUHWORD ApcParamBase;	  /* APC Parameters table base address */
    VUHWORD FbpParamBase;	  /* Free buffer pool parameters base address */
    VUHWORD IntQParamBase;	  /* Interrupt queue parameters table base */
    VUBYTE  reserved3[2];
    VUHWORD UniStatTableBase; /* UNI statistics table base */
    VUWORD  BdBaseExt;        /* BD ring base address extension */
    union 
    {
	    struct AddressCompressionPram   AddrCompression;
	    struct ExtCamPram               ExtCam;
    } AddrMapping;            /* Address look-up mechanism */
    VUHWORD VciFiltering;     /* VCI filtering enable bits. If bit i is set,
                                 the cell with VCI=i will be sent to the
                                 raw cell queue. The bits 0-2 and 5 should
                                 be zero. */
    VUHWORD Gmode;            /* Global mode */
    VUHWORD CommInfo1;        /* The information fields associated with the */
    VUHWORD Channel_num;      /* the last host command (cpcr). Channel num  */
    VUHWORD Burst_tol;        /* and burst tolerance (VBR channels).        */
    VUBYTE  reserved4[4];     /* Reserved */
    VUWORD  CRC32Preset;      /* Preset for CRC32 */
    VUWORD  CRC32Mask;        /* Constant mask for CRC32 */
    VUHWORD AAL1SnpTableBase; /* AAl1 SNP protection look-up table base */
    VUHWORD reserved5;        /* Reserved */
    VUWORD  SrtsBase;         /* External SRTS logic base address. For AAL1
                                 only. Should be 16 bytes aligned */
    VUHWORD IdleBase;         /* Idle cell base address */
    VUHWORD IdleSize;         /* Idle cell size: 52, 56, 60, 64 */
    VUWORD  EmptyCellPayload; /* Empty cell payload (little-indian) */
    
    /* ABR specific only */
    VUWORD  Trm; /* Upper bound on time between F-RM cells for active source */                                 
    VUHWORD Nrm; /* Controls the maximum data cells sent for each F-RM cell. */                           
    VUHWORD Mrm; /* Controls bandwidth between F-RM, B-RM and user data cell */
    VUHWORD Tcr;              /* Tag cell rate */
    VUHWORD AbrRxTcte;        /* ABR reserved area address (2-UHWORD aligned)*/
    VUBYTE  reserved7[58];    /* Reserved */
    VUHWORD IMARoot;        /* IMA root table */
    VUBYTE  reserved8[16];    /* Reserved */
} _PackedType t_Atm_Pram;



/*---------------------------------------------------------------------------*/
/*	                    SERIAL MANAGEMENT CHANNEL  (SMC)	                 */
/*---------------------------------------------------------------------------*/

typedef  _Packed struct  
{
   UHWORD	rbase;		/* Rx BD Base Address */
   UHWORD	tbase;		/* Tx BD Base Address */
   UBYTE	rfcr;		/* Rx function code */
   UBYTE	tfcr;		/* Tx function code */
   UHWORD	mrblr;		/* Rx buffer length */
   UWORD	rstate;		/* Rx internal state */
   UWORD	rptr;		/* Rx internal data pointer */
   UHWORD	rbptr;		/* rb BD Pointer */
   UHWORD	rcount;		/* Rx internal byte count */
   UWORD	rtemp;		/* Rx temp */
   UWORD    tstate;		/* Tx internal state */
   UWORD	tptr;		/* Tx internal data pointer */
   UHWORD	tbptr;		/* Tx BD pointer */
   UHWORD	tcount;		/* Tx byte count */
   UWORD	ttemp;	    /* Tx temp */
  
   /* SMC UART-specific PRAM */
   UHWORD	max_idl;	/* Maximum IDLE Characters */
   UHWORD	idlc;		/* Temporary IDLE Counter */
   UHWORD	brkln;		/* Last Rx Break Length */
   UHWORD	brkec;		/* Rx Break Condition Counter */
   UHWORD	brkcr;		/* Break Count Register (Tx) */
   UHWORD	r_mask;		/* Temporary bit mask */
  
} _PackedType t_Smc_Pram;



/*---------------------------------------------------------------------------*/
/*						      IDMA PARAMETER RAM                             */
/*---------------------------------------------------------------------------*/

typedef  _Packed struct  
{
    UHWORD      ibase;       /* IDMA BD Base Address               */
    UHWORD      dcm;         /* DMA channel mode register          */
    UHWORD      ibdptr;      /* next bd ptr                        */
    UHWORD      DPR_buf;     /* ptr to internal 64 byte buffer     */
    UHWORD      BUF_inv;     /* The quantity of data in DPR_buf    */
    UHWORD      SS_max;      /* Steady State Max. transfer size    */
    UHWORD      DPR_in_ptr;  /* write ptr for the internal buffer  */
    UHWORD      sts;         /* Source Transfer Size               */
    UHWORD      DPR_out_ptr; /* read ptr for the internal buffer   */
    UHWORD      seob;        /* Source end of burst                */
    UHWORD      deob;        /* Destination end of burst           */
    UHWORD      dts;         /* Destination Transfer Size          */
    UHWORD      RetAdd;      /* return address when ERM==1         */
    UHWORD      Reserved;    /* reserved */
    UWORD       BD_cnt;      /* Internal byte count                */
    UWORD       S_ptr;       /* source internal data ptr           */
    UWORD       D_ptr;       /* destination internal data ptr      */
    UWORD       istate;      /* Internal state                     */

} _PackedType t_Idma_Pram;



/*-------------------------------------------------------------------*/
/*         INTER-INTEGRATED CIRCUIT  (I2C)	                     */
/*-------------------------------------------------------------------*/

typedef  _Packed struct 
{
   UHWORD	rbase;		/* RX BD base address */
   UHWORD	tbase;		/* TX BD base address */
   UBYTE	rfcr;		/* Rx function code */
   UBYTE	tfcr;		/* Tx function code */
   UHWORD	mrblr;		/* Rx buffer length */
   UWORD	rstate;		/* Rx internal state */
   UWORD	rptr;		/* Rx internal data pointer */
   UHWORD	rbptr;		/* rb BD Pointer */
   UHWORD	rcount;		/* Rx internal byte count */
   UWORD	rtemp;		/* Rx temp */
   UWORD	tstate;		/* Tx internal state */
   UWORD	tptr;		/* Tx internal data pointer */
   UHWORD	tbptr;		/* Tx BD pointer */
   UHWORD	tcount;		/* Tx byte count */
   UWORD	ttemp;		/* Tx temp */
} _PackedType t_I2c_Pram;



/*---------------------------------------------------------------------------*/
/*                       SERIAL PERIPHERAL INTERFACE  (SPI)                  */
/*---------------------------------------------------------------------------*/

typedef  _Packed struct  
{
   UHWORD	rbase;      /* Rx BD Base Address */
   UHWORD	tbase;      /* Tx BD Base Address */
   UBYTE    rfcr;       /* Rx function code */
   UBYTE    tfcr;       /* Tx function code */
   UHWORD	mrblr;      /* Rx buffer length */
   UWORD    rstate;     /* Rx internal state */
   UWORD    rptr;       /* Rx internal data pointer */
   UHWORD	rbptr;      /* Rx BD Pointer */
   UHWORD	rcount;     /* Rx internal byte count */
   UWORD    rtemp;      /* Rx temp */
   UWORD    tstate;     /* Tx internal state */
   UWORD    tptr;       /* Tx internal data pointer */
   UHWORD	tbptr;      /* Tx BD pointer */
   UHWORD	tcount;     /* Tx byte count */
   UWORD    ttemp;      /* Tx temp */
   UBYTE    reserved[8];
} _PackedType t_Spi_Pram;



/*---------------------------------------------------------------------------*/
/*   					RISC TIMER PARAMETER RAM                             */
/*---------------------------------------------------------------------------*/

typedef  _Packed struct  
{
 
   UHWORD	tm_base;    /* RISC timer table base adr */
   UHWORD	tm_ptr;     /* RISC timer table pointer */
   UHWORD	r_tmr;      /* RISC timer mode register */
   UHWORD	r_tmv;      /* RISC timer valid register */
   UWORD	tm_cmd;     /* RISC timer cmd register */
   UWORD	tm_cnt;     /* RISC timer internal cnt */
} _PackedType t_timer_pram;



/*--------------------------------------------------------------------------*/
/*                  ROM MICROCODE PARAMETER RAM AREA                        */
/*--------------------------------------------------------------------------*/

typedef  _Packed struct  
{
   UHWORD   rev_num;    /* Ucode Revision Number */
   UHWORD   d_ptr;      /* MISC Dump area pointer */
} _PackedType t_ucode_pram;








/*--------------------------------------------------------------------------*/
/*                MAIN DEFINITION OF MPC8260 INTERNAL MEMORY MAP            */
/*--------------------------------------------------------------------------*/

typedef  _Packed struct 
{

/* cpm_ram */
    t_Mch_Pram    mch_pram[256]; /* MCC logical channels parameter ram */
    VUBYTE reserved0[0x4000];    /* Reserved area */
    
/* DPR_BASE+0x8000*/
    union 
    {
    
    /*for access to the PRAM structs for SCCs, FCCs, and MCCs */ 
	    struct serials 
        {
	        t_Scc_Pram scc_pram[4];
	        t_Fcc_Pram fcc_pram[3];
	        t_Mcc_Pram mcc_pram[2];
	        VUBYTE reserved1[0x700];
	    } serials;
        
    /* for access to ATM PRAM structs */
	    struct atm
        {
	        VUBYTE reserved2[0x400];
	        t_Atm_Pram atm_pram[2];
	        VUBYTE reserved3[0xa00];
	    } atm;
        
    /* for access to the memory locations holding user-defined 
       base addresses of PRAM for SMCs, IDMA, SPI, and I2C. */     
       struct general
       {
            VUBYTE scc1[0x100];
            VUBYTE scc2[0x100];
            VUBYTE scc3[0x100];
            VUBYTE scc4[0x100];
            VUBYTE fcc1[0x100];
            VUBYTE fcc2[0x100];
            VUBYTE fcc3[0x100];
            VUBYTE mcc1[0x80];
            VUBYTE reserved_0[0x7c];
            VUBYTE smc1[0x2];
            VUBYTE idma1[0x2];
            VUBYTE mcc2[0x80];
            VUBYTE reserved_1[0x7c];
            VUBYTE smc2[0x2];
            VUBYTE idma2[0x2];
            VUBYTE reserved_2[0xfc];
            VUBYTE spi[0x2];
            VUBYTE idma3[0x2];
            VUBYTE reserved_3[0xe0];
            VUBYTE timers[0x10];
            VUBYTE Rev_num[0x2];
            VUBYTE D_ptr[0x2];
            VUBYTE reserved_4[0x4];
            VUBYTE rand[0x4];
            VUBYTE i2c[0x2];
            VUBYTE idma4[0x2];
            VUBYTE reserved_5[0x500];
	    } general;
        
    } pram;
    
    VUBYTE reserved11[0x2000];      /* Reserved area */
    VUBYTE cpm_ram_dpram_2[0x1000]; /* Internal RAM */
    VUBYTE reserved12[0x4000];      /* Reserved area */

/* siu */
    VUWORD siu_siumcr;        /* SIU Module Configuration Register */
    VUWORD siu_sypcr;         /* System Protection Control Register */
    VUBYTE reserved13[0x6];   /* Reserved area */
    VUHWORD siu_swsr;         /* Software Service Register */

/* buses */
    VUBYTE reserved14[0x14];  /* Reserved area */
    VUWORD bcr;               /* Bus Configuration Register */
    VUBYTE ppc_acr;           /* Arbiter Configuration Register */
    VUBYTE reserved15[0x3];   /* Reserved area */
    VUWORD ppc_alrh;          /* Arbitration level Register (First clients)*/
    VUWORD ppc_alrl;          /* Arbitration Level Register (Next clients) */
    VUBYTE lcl_acr;           /* LCL Arbiter Configuration Register */
    VUBYTE reserved16[0x3];   /* Reserved area */
    VUWORD lcl_alrh;        /* LCL Arbitration level Register (First clients)*/
    VUWORD lcl_alrl;        /* LCL Arbitration Level Register (Next clients) */
    VUWORD tescr1;       /* PPC bus transfer error status control register 1 */
    VUWORD tescr2;       /* PPC bus transfer error status control register 2 */
    VUWORD ltescr1;    /* Local bus transfer error status control register 1 */
    VUWORD ltescr2;    /* Local bus transfer error status control register 2 */
    VUWORD pdtea;             /* PPC bus DMA Transfer Error Address */
    VUBYTE pdtem;             /* PPC bus DMA Transfer Error MSNUM  */
    VUBYTE reserved17[0x3];   /* Reserved area */
    VUWORD ldtea;             /* PPC bus DMA Transfer Error Address */
    VUBYTE ldtem;             /* PPC bus DMA Transfer Error MSNUM  */
    VUBYTE reserved18[0xa3];  /* Reserved area */

/* memc */
    struct memc_regs 
    {
        VUWORD br;       /* Base Register */
        VUWORD or;       /* Option Register */
    } memc_regs[12];

    VUBYTE reserved19[0x8];   /* Reserved area */
    VUWORD memc_mar;          /* Memory Address Register */
    VUBYTE reserved20[0x4];   /* Reserved area */
    VUWORD memc_mamr;         /* Machine A Mode Register */
    VUWORD memc_mbmr;         /* Machine B Mode Register */
    VUWORD memc_mcmr;         /* Machine C Mode Register */
    VUWORD memc_mdmr;         /* Machine D Mode Register */
    VUBYTE reserved21[0x4];   /* Reserved area */
    VUHWORD memc_mptpr;       /* Memory Periodic Timer Prescaler */
    VUBYTE reserved22[0x2];   /* Reserved area */
    VUWORD memc_mdr;          /* Memory Data Register */
    VUBYTE reserved23[0x4];   /* Reserved area */
    VUWORD memc_psdmr;        /* PowerPC Bus SDRAM machine Mode Register */
    VUWORD memc_lsdmr;        /* Local Bus SDRAM machine Mode Registe */
    VUBYTE memc_purt;         /* PowerPC Bus assigned UPM Refresh Timer */
    VUBYTE reserved24[0x3];   /* Reserved area */
    VUBYTE memc_psrt;         /* PowerPC BusBus assigned SDRAM Refresh Timer */
    VUBYTE reserved25[0x3];   /* Reserved area */
    VUBYTE memc_lurt;         /* Local Bus assigned UPM Refresh Timer */
    VUBYTE reserved26[0x3];   /* Reserved area */
    VUBYTE memc_lsrt;         /* Local Bus assigned SDRAM Refresh Timer */
    VUBYTE reserved27[0x3];   /* Reserved area */
    VUWORD memc_immr;         /* Internal Memory Map Register */

/* pci */
    VUWORD pcibr0;            /* Base address+valid for PCI window 1 */
    VUWORD pcibr1;            /* Base address+valid for PCI window 2 */
    VUBYTE reserved28[0x10];  /* Reserved area */
    VUWORD pcimsk0;           /* Mask for PCI window 1 */
    VUWORD pcimsk1;           /* Mask for PCI window 2 */
    VUBYTE reserved29[0x54];  /* Reserved area */

/* si_timers */
    VUHWORD si_timers_tmcntsc; /* Time Counter Status and Control Register */
    VUBYTE reserved30[0x2];    /* Reserved area */
    VUWORD si_timers_tmcnt;    /* Time Counter Register */
    VUWORD si_timers_tmcntsec; /* Time Counter Seconds*/
    VUWORD si_timers_tmcntal;  /* Time Counter Alarm Register */
    VUBYTE reserved31[0x10];   /* Reserved area */
    VUHWORD si_timers_piscr;   /* Periodic Interrupt Status and Control Reg. */
    VUBYTE reserved32[0x2];    /* Reserved area */
    VUWORD si_timers_pitc;     /* Periodic Interrupt Count Register */
    VUWORD si_timers_pitr;     /* Periodic Interrupt Timer Register */
    VUBYTE reserved33[0x54];   /* Reserved area */

/* test module registers */
    VUWORD tstmhr;             
    VUWORD tstmlr;
    VUHWORD tster;
    VUBYTE reserved34[0x156]; /* Reserved area */
    
/* pci, part 2 */
    VUWORD pci_pci;           /* PCI Configuration space */
    VUBYTE reserved35[0x7fc]; /* Reserved area */
    
/* ic */
    VUHWORD ic_sicr;          /* Interrupt Configuration Register */
    VUBYTE reserved36[0x2];   /* Reserved area */
    VUWORD ic_sivec;          /* CP Interrupt Vector Register */
    VUWORD ic_sipnr_h;        /* Interrupt Pending Register (HIGH) */
    VUWORD ic_sipnr_l;        /* Interrupt Pending Register (LOW)    */
    VUWORD ic_siprr;          /* SIU Interrupt Priority Register     */
    VUWORD ic_scprr_h;        /* Interrupt Priority Register (HIGH) */
    VUWORD ic_scprr_l;        /* Interrupt Priority Register (LOW)   */
    VUWORD ic_simr_h;         /* Interrupt Mask Register (HIGH)      */
    VUWORD ic_simr_l;         /* Interrupt Mask Register (LOW)       */
    VUWORD ic_siexr;          /* External Interrupt Control Register */
    VUBYTE reserved37[0x58];  /* Reserved area */

/* clocks */
    VUWORD clocks_sccr;       /* System Clock Control Register */
    VUBYTE reserved38[0x4];   /* Reserved area */
    VUWORD clocks_scmr;       /* System Clock Mode Register  */
    VUBYTE reserved39[0x4];   /* Reserved area */
    VUWORD clocks_rsr;        /* Reset Status Register */
    VUWORD clocks_rmr;        /* Reset Moode Register  */
    VUBYTE reserved40[0x68];  /* Reserved area */

/* io_ports */
    struct io_regs 
    {
	    VUWORD pdir;    /* Port A-D Data Direction Register */
	    VUWORD ppar;    /* Port A-D Pin Assignment Register */
	    VUWORD psor;    /* Port A-D Special Operation Register */
	    VUWORD podr;    /* Port A-D Open Drain Register */
	    VUWORD pdat;    /* Port A-D Data Register */
	    VUBYTE reserved41[0xc]; /* Reserved area */
    } io_regs[4];

#define PORTA 0
#define PORTB 1
#define PORTC 2
#define PORTD 3

/* cpm_timers */
    VUBYTE cpm_timers_tgcr1;   /* Timer Global Configuration Register */
    VUBYTE reserved42[0x3];    /* Reserved area */
    VUBYTE cpm_timers_tgcr2;   /* Timer Global Configuration Register */
    VUBYTE reserved43[0xb];    /* Reserved area */
    VUHWORD cpm_timers_tmr1;   /* Timer Mode Register */
    VUHWORD cpm_timers_tmr2;   /* Timer Mode Register */
    VUHWORD cpm_timers_trr1;   /* Timer Reference Register */
    VUHWORD cpm_timers_trr2;   /* Timer Reference Register */
    VUHWORD cpm_timers_tcr1;   /* Timer Capture Register */
    VUHWORD cpm_timers_tcr2;   /* Timer Capture Register */
    VUHWORD cpm_timers_tcn1;   /* Timer Counter */
    VUHWORD cpm_timers_tcn2;   /* Timer Counter */
    VUHWORD cpm_timers_tmr3;   /* Timer Mode Register */
    VUHWORD cpm_timers_tmr4;   /* Timer Mode Register */
    VUHWORD cpm_timers_trr3;   /* Timer Reference Register */
    VUHWORD cpm_timers_trr4;   /* Timer Reference Register */
    VUHWORD cpm_timers_tcr3;   /* Timer Capture Register */
    VUHWORD cpm_timers_tcr4;   /* Timer Capture Register */
    VUHWORD cpm_timers_tcn3;   /* Timer Counter */
    VUHWORD cpm_timers_tcn4;   /* Timer Counter */
    VUHWORD cpm_timers_ter[4]; /* Timer Event Register */
    VUBYTE reserved44[0x260];  /* Reserved area */

/* sdma general */
    VUBYTE sdma_sdsr;         /* SDMA Status Register */
    VUBYTE reserved45[0x3];   /* Reserved area */
    VUBYTE sdma_sdmr;         /* SDMA Mask Register */
    VUBYTE reserved46[0x3];   /* Reserved area */

/* idma */
    VUBYTE idma_idsr1;        /* IDMA Status Register */
    VUBYTE reserved47[0x3];   /* Reserved area */
    VUBYTE idma_idmr1;        /* IDMA Mask Register */
    VUBYTE reserved48[0x3];   /* Reserved area */
    VUBYTE idma_idsr2;        /* IDMA Status Register */
    VUBYTE reserved49[0x3];   /* Reserved area */
    VUBYTE idma_idmr2;        /* IDMA Mask Register */
    VUBYTE reserved50[0x3];   /* Reserved area */
    VUBYTE idma_idsr3;        /* IDMA Status Register */
    VUBYTE reserved51[0x3];   /* Reserved area */
    VUBYTE idma_idmr3;        /* IDMA Mask Register */
    VUBYTE reserved52[0x3];   /* Reserved area */
    VUBYTE idma_idsr4;        /* IDMA Status Register */
    VUBYTE reserved53[0x3];   /* Reserved area */
    VUBYTE idma_idmr4;        /* IDMA Mask Register */
    VUBYTE reserved54[0x2c3]; /* Reserved area */
    
/* fcc */
     struct fcc_regs 
    {
        VUWORD gfmr;            /* FCC General Mode Register */
        VUWORD psmr;            /* FCC Protocol Specific Mode Register */
        VUHWORD todr;           /* FCC Transmit On Demand Register */
        VUBYTE reserved55[0x2]; /* Reserved area */
        VUHWORD dsr;            /* FCC Data Sync. Register */
        VUBYTE reserved56[0x2]; /* Reserved area */
        VUWORD fcce;            /* FCC Event Register */
        VUWORD fccm;            /* FCC Mask Register */
        VUBYTE fccs;            /* FCC Status Register */
        VUBYTE reserved57[0x3]; /* Reserved area */
        VUWORD ftirr;           /* FCC Transmit Internal Rate Register */
    } fcc_regs[3];
    VUBYTE reserved58[0xA0];   /* Reserved area */
 
 /*tc*/
    struct tc_regs
   {
   		VUHWORD tcmdr;			/* TC Mode Register */
   	  	VUHWORD cdsmr;			/* TC cell delineation state machine Register */
   	  	VUHWORD tcer;			/* TC event Register */
   	  	VUHWORD rcc;				/* TC received cells count register */
   	  	VUHWORD tcmr;			/* TC mask Register */
   	  	VUHWORD fcc;				/* TC filtered cells count register */
   	  	VUHWORD ccc;				/* TC Corrected cells count Register */
   	  	VUHWORD icc;				/* TC idle cells count Register */
   	  	VUHWORD tcc;				/* TC transmitted cells count Register */
   	  	VUHWORD ecc;				/* TC error cells count register */
   	  	VUBYTE reserved95[0xC];/* Reserved area */
   	} tc_regs[8];
   	VUHWORD tcgsr;				/* TC general status register */
   	VUHWORD tcger;				/* TC general event register */ 
   	VUBYTE reserved96[0xEC];  	/* Reserved area */
 
 
    
/* brgs */
    VUWORD brgs_brgc5;     /* BRG Configuration Register */
    VUWORD brgs_brgc6;     /* BRG Configuration Register */
    VUWORD brgs_brgc7;     /* BRG Configuration Register */
    VUWORD brgs_brgc8;     /* BRG Configuration Register */
    VUBYTE reserved59[0x260]; /* Reserved area */
    
/* i2c */
    VUBYTE i2c_i2mod;         /* IC Mode Register */
    VUBYTE reserved60[0x3];   /* Reserved area */
    VUBYTE i2c_i2add;         /* IC Address Register */
    VUBYTE reserved61[0x3];   /* Reserved area */
    VUBYTE i2c_i2brg;         /* IC BRG Register */
    VUBYTE reserved62[0x3];   /* Reserved area */
    VUBYTE i2c_i2com;         /* IC Command Register */
    VUBYTE reserved63[0x3];   /* Reserved area */
    VUBYTE i2c_i2cer;         /* IC Event Register */
    VUBYTE reserved64[0x3];   /* Reserved area */
    VUBYTE i2c_i2cmr;         /* IC Mask Register */
    VUBYTE reserved65[0x143]; /* Hip7 Reserved area */
    
/* cpm */
	VUHWORD cpm_rctr5;        /* Hip7 RISC Controller Trap Register */
    VUHWORD cpm_rctr6;        /* Hip7 RISC Controller Trap Register */
    VUHWORD cpm_rctr7;        /* Hip7 RISC Controller Trap Register */
    VUHWORD cpm_rctr8;        /* Hip7 RISC Controller Trap Register */
    VUWORD cpm_cpcr;          /* Communication Processor Command Register */
    VUWORD cpm_rccr;          /* RISC Configuration Register */
    VUWORD cpm_rmdr;          /* RISC Microcode Dev. Support Control Reg. */
    VUHWORD cpm_rctr1;        /* RISC Controller Trap Register */
    VUHWORD cpm_rctr2;        /* RISC Controller Trap Register */
    VUHWORD cpm_rctr3;        /* RISC Controller Trap Register */
    VUHWORD cpm_rctr4;        /* RISC Controller Trap Register */
    VUBYTE reserved66[0x2];   /* Reserved area */
    VUHWORD cpm_rter;         /* RISC Timers Event Register */
    VUBYTE reserved67[0x2];   /* Reserved area */
    VUHWORD cpm_rtmr;         /* RISC Timers Mask Register */
    VUHWORD cpm_rtscr;        /* RISC Time-Stamp Timer Control Register */
    VUHWORD cpm_rmds;         /* RISC Development Support Status Register */
    VUWORD cpm_rtsr;          /* RISC Time-Stamp Register */
    VUBYTE reserved68[0xc];   /* Reserved area */

/* brgs */
    VUWORD brgs_brgc1;     /* BRG Configuration Register */
    VUWORD brgs_brgc2;     /* BRG Configuration Register */
    VUWORD brgs_brgc3;     /* BRG Configuration Register */
    VUWORD brgs_brgc4;     /* BRG Configuration Register */
    
/* scc */
    struct scc_regs_8260
    {
        VUWORD scc_gsmr_l;      /* SCC General Mode Register */
        VUWORD scc_gsmr_h;      /* SCC General Mode Register */
        VUHWORD scc_psmr;       /* SCC Protocol Specific Mode Register */
        VUBYTE reserved69[0x2]; /* Reserved area */
        VUHWORD scc_todr;       /* SCC Transmit-On-Demand Register */
        VUHWORD scc_dsr;        /* SCC Data Synchronization Register */
        VUHWORD scc_scce;       /* SCC Event Register */
        VUBYTE reserved70[0x2]; /* Reserved area */
        VUHWORD scc_sccm;       /* SCC Mask Register */
        VUBYTE reserved71;      /* Reserved area */
        VUBYTE scc_sccs;        /* SCC Status Register */
        VUBYTE reserved72[0x8]; /* Reserved area */
    } scc_regs[4];
    
/* smc */
    struct smc_regs_8260
    {
        VUBYTE reserved73[0x2]; /* Reserved area */
        VUHWORD smc_smcmr;      /* SMC Mode Register */
        VUBYTE reserved74[0x2]; /* Reserved area */
        VUBYTE smc_smce;        /* SMC Event Register */
        VUBYTE reserved75[0x3]; /* Reserved area */
        VUBYTE smc_smcm;        /* SMC Mask Register */
        VUBYTE reserved76[0x5]; /* Reserved area */
    } smc_regs[2];
    
/* spi */
    VUHWORD spi_spmode;       /* SPI Mode Register */
    VUBYTE reserved77[0x4];   /* Reserved area */
    VUBYTE spi_spie;          /* SPI Event Register */
    VUBYTE reserved78[0x3];   /* Reserved area */
    VUBYTE spi_spim;          /* SPI Mask Register */
    VUBYTE reserved79[0x2];   /* Reserved area */
    VUBYTE spi_spcom;         /* SPI Command Register */
    VUBYTE reserved80[0x52];  /* Reserved area */
    
/* cpm_mux */
    VUBYTE cpm_mux_cmxsi1cr;  /* CPM MUX SI Clock Route Register */
    VUBYTE reserved81;        /* Reserved area */
    VUBYTE cpm_mux_cmxsi2cr;  /* CPM MUX SI Clock Route Register */
    VUBYTE reserved82;        /* Reserved area */
    VUWORD cpm_mux_cmxfcr;    /* CPM MUX FCC Clock Route Register */
    VUWORD cpm_mux_cmxscr;    /* CPM MUX SCC Clock Route Register */
    VUBYTE cpm_mux_cmxsmr;    /* CPM MUX SMC Clock Route Register */
    VUBYTE reserved83;        /* Reserved area */
    VUHWORD cpm_mux_cmxuar;   /* CPM MUX UTOPIA Address Register */
    VUBYTE reserved84[0x10];  /* Reserved area */
    
/* si */
    struct si_regs 
    {
        VUHWORD si_si1mr[4];    /* SI TDM Mode Registers */
        VUBYTE si_si1gmr;       /* SI Global Mode Register */
        VUBYTE reserved85;      /* Reserved area */
        VUBYTE si_si1cmdr;      /* SI Command Register */
        VUBYTE reserved86;      /* Reserved area */
        VUBYTE si_si1str;       /* SI Status Register */
        VUBYTE reserved87;      /* Reserved area */
        VUHWORD si_si1rsr;      /* SI RAM Shadow Address Register */
        VUHWORD mcc_mcce;       /* MCC Event Register */
        VUBYTE reserved88[0x2]; /* Reserved area */
        VUHWORD mcc_mccm;       /* MCC Mask Register */
        VUBYTE reserved89[0x2]; /* Reserved area */
        VUBYTE mcc_mccf;        /* MCC Configuration Register */
        VUBYTE reserved90[0x7]; /* Reserved area */
    } si_regs[2];
    VUBYTE reserved91[0x4a0];   /* Reserved area */
    
/* si_ram */
    struct si_ram 
    {
        UHWORD si1_ram_si1_tx_ram[0x100]; /* SI Transmit Routing RAM */
        VUBYTE reserved92[0x200];         /* Reserved area */
        UHWORD si1_ram_si1_rx_ram[0x100]; /* SI Receive Routing RAM */
        VUBYTE reserved93[0x200];         /* Reserved area */
    } si_ram[2];
    VUBYTE reserved94[0x1000];  /* Reserved area */

} _PackedType t_PQ2IMM;


#pragma pack(pop)



/***************************************************************************/
/*                   General Global Definitions                            */
/***************************************************************************/

#define PAGE1       0         /* SCC1 Index into SCC Param RAM Array */
#define PAGE2       1         /* SCC2 Index into SCC Param RAM Array */
#define PAGE3       2         /* SCC3 Index into SCC Param RAM Array */
#define PAGE4       3         /* SCC4 Index into SCC Param RAM Array */

#define SCC1_REG    0         /* SCC1 Index into SCC Regs Array  */				
#define SCC2_REG    1         /* SCC2 Index into SCC Regs Array  */				
#define SCC3_REG    2         /* SCC3 Index into SCC Regs Array  */				
#define SCC4_REG    3         /* SCC4 Index into SCC Regs Array  */				


/*--------------------------------*/
/* KEEP ALIVE POWER REGISTERS KEY */
/*--------------------------------*/

#define KEEP_ALIVE_KEY 0x55ccaa33


