/****************************************************************************/
/*!
 *\MODULE              Coordinator (Application Layer)
 *
 *\COMPONENT           $HeadURL: https://www.collabnet.nxp.com/svn/lprf_apps/Application_Notes/JN-AN-1069-IEEE-802.15.4-Serial-Cable-Replacement/Trunk/Coordinator/Source/crd_coordinator.c $
 *
 *\VERSION			   $Revision: 18318 $
 *
 *\REVISION            $Id: crd_coordinator.c 18318 2017-02-09 11:36:46Z nxp29759 $
 *
 *\DATED               $Date: 2017-02-09 11:36:46 +0000 (Thu, 09 Feb 2017) $
 *
 *\AUTHOR              $Author: nxp29759 $
 *
 *\DESCRIPTION         Coordinator (Application Layer) - implementation.
 */
/*****************************************************************************
 *
 * This software is owned by Jennic and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on Jennic products. You, and any third parties must reproduce
 * the copyright and warranty notice and any other legend of ownership on each
 * copy or partial copy of the software.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS". JENNIC MAKES NO WARRANTIES, WHETHER
 * EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED TO, IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE,
 * ACCURACY OR LACK OF NEGLIGENCE. JENNIC SHALL NOT, IN ANY CIRCUMSTANCES,
 * BE LIABLE FOR ANY DAMAGES, INCLUDING, BUT NOT LIMITED TO, SPECIAL,
 * INCIDENTAL OR CONSEQUENTIAL DAMAGES FOR ANY REASON WHATSOEVER.
 *
 * Copyright Jennic Ltd 2009. All rights reserved
 *
 ****************************************************************************/

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/
/* Jennic defines */
#include <jendefs.h>
#include <AppHardwareApi.h>
#include <AppQueueApi.h>
#include <LedControl.h>
#include <mac_sap.h>
#include <mac_pib.h>
/* Common defines */
#include "config.h"
#include "dbg.h"
#include "node.h"
#include "rnd.h"
#include "wuart.h"
#ifdef SWDEBUG
    #include <gdb.h>
#endif

/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/
#define CRD_TRACE FALSE

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/
/* Data type for storing data related to all end devices that have associated */
typedef struct
{
    uint16 u16ShortAddr;
    uint32 u32ExtAddrL;
    uint32 u32ExtAddrH;
} CRD_tsChild;

typedef struct
{
    /* Data related to associated end devices */
    uint16          u16Children;
    CRD_tsChild      asChild[MAX_CHILDREN];
} CRD_tsData;

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/
PRIVATE void CRD_vMain(void);
PRIVATE void CRD_vInit(void);
PRIVATE void CRD_vMlmeDcfmInd(MAC_MlmeDcfmInd_s *psMlmeInd);
PRIVATE void CRD_vMcpsDcfmInd(MAC_McpsDcfmInd_s *psMcpsInd);
PRIVATE void CRD_vHwInd(AppQApiHwInd_s *psAHI_Ind);
PRIVATE void CRD_vIndData(MAC_McpsDcfmInd_s *psMcpsInd);
PRIVATE void CRD_vDcfmData(MAC_McpsDcfmInd_s *psMcpsInd);
PRIVATE void CRD_vReqEnergyScan(void);
PRIVATE void CRD_vDcfmScanEnergy(MAC_MlmeDcfmInd_s *psMlmeInd);
PRIVATE void CRD_vReqStartCoordinator(void);
PRIVATE void CRD_vIndAssociate(MAC_MlmeDcfmInd_s *psMlmeInd);

/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/
PRIVATE CRD_tsData CRD_sData;

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

/****************************************************************************
 *
 * NAME: AppColdStart
 *
 * DESCRIPTION:
 * Entry point for application from boot loader. Initialises system and runs
 * main loop.
 *
 * RETURNS:
 * Never returns.
 *
 ****************************************************************************/
PUBLIC void AppColdStart(void)
{
	/* Disable watchdog if enabled by default */
	#ifdef WATCHDOG_ENABLED
		vAHI_WatchdogStop();
	#endif

	/* Perform main processing */
	CRD_vMain();
}

/****************************************************************************
 *
 * NAME: AppWarmStart
 *
 * DESCRIPTION:
 * Entry point for application from boot loader. Simply jumps to AppColdStart
 * as, in this instance, application will never warm start.
 *
 * RETURNS:
 * Never returns.
 *
 ****************************************************************************/
PUBLIC void AppWarmStart(void)
{
	/* Disable watchdog if enabled by default */
	#ifdef WATCHDOG_ENABLED
		vAHI_WatchdogStop();
	#endif

	/* Perform main processing */
	CRD_vMain();
}

/****************************************************************************/
/***        Local Functions                                               ***/
/****************************************************************************/

/****************************************************************************
 *
 * NAME: CRD_vMain
 *
 * DESCRIPTION:
 * Check each of the three event queues and process and items found.
 *
 * PARAMETERS:      Name            RW  Usage
 * None.
 *
 * RETURNS:
 * None.
 *
 * NOTES:
 * None.
 ****************************************************************************/
PRIVATE void CRD_vMain(void)
{
    MAC_MlmeDcfmInd_s *psMlmeInd;
	MAC_McpsDcfmInd_s *psMcpsInd;
    AppQApiHwInd_s    *psAHI_Ind;

	/* Initialise */
    CRD_vInit();

	/* Start state machine by requesting an energy scan */
    CRD_vReqEnergyScan();

	/* Loop processing events */
    while (1)
    {

		/* Check for anything on the MLME upward queue */
		do
		{
			psMlmeInd = psAppQApiReadMlmeInd();
			if (psMlmeInd != NULL)
			{
				CRD_vMlmeDcfmInd(psMlmeInd);
				vAppQApiReturnMlmeIndBuffer(psMlmeInd);
			}
		} while (psMlmeInd != NULL);

		/* Check for anything on the MCPS upward queue */
		do
		{
			psMcpsInd = psAppQApiReadMcpsInd();
			if (psMcpsInd != NULL)
			{
				CRD_vMcpsDcfmInd(psMcpsInd);
				vAppQApiReturnMcpsIndBuffer(psMcpsInd);
			}
		} while (psMcpsInd != NULL);

		/* Check for anything on the AHI upward queue */
		do
		{
			psAHI_Ind = psAppQApiReadHwInd();
			if (psAHI_Ind != NULL)
			{
				CRD_vHwInd(psAHI_Ind);
				vAppQApiReturnHwIndBuffer(psAHI_Ind);
			}
		} while (psAHI_Ind != NULL);
	}
}

/****************************************************************************
 *
 * NAME: CRD_vInit
 *
 * DESCRIPTION:
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void CRD_vInit(void)
{
    /* Setup interface to MAC */
    (void)u32AHI_Init();
    (void)u32AppQApiInit(NULL, NULL, NULL);

	/* Initialise debugging */
	DBG_vUartInit(DBG_UART, DBG_BAUD);
	/* Debug */
	DBG_vPrintf(CRD_TRACE, "\r\n%d CRD < vInitSystem()", NODE_sData.u32Timer);

	/* Initialise and turn on LEDs */
	vLedInitRfd();
	vLedControl(0, FALSE);
	vLedControl(1, FALSE);

    /* Initialise coordinator data */
    CRD_sData.u16Children = 0;

    /* Initialise common 15.4 node data */
    NODE_vInit();

    /* Initialise the random number generator */
	RND_vInit();

    /* We are the co-ordinator so choose our own short address */
	NODE_sData.u16Address = (uint16) (NODE_sData.u64Address & 0xffff);
	/* We are the co-ordinator so choose our own pan id which is fixed */
	NODE_sData.u16PanId = PAN_ID;
	/* Set short address in pib */
    MAC_vPibSetShortAddr(NODE_sData.pvMac, NODE_sData.u16Address);
	/* Set pan id in pib */
    MAC_vPibSetPanId(NODE_sData.pvMac, NODE_sData.u16PanId);
    /* Enable receiver to be on when idle */
    MAC_vPibSetRxOnWhenIdle(NODE_sData.pvMac, TRUE, FALSE);
    /* Allow nodes to associate */
    NODE_sData.psPib->bAssociationPermit = 1;

	/* Debug */
	DBG_vPrintf(CRD_TRACE, " Sht=%#x Pan=%#x", NODE_sData.u16Address, NODE_sData.u16PanId);

	/* Initialise wuart module */
	WUART_vInit();

	/* Start tick timer 10ms repeating */
	vAHI_TickTimerInterval(10 * TICK_TIMER_MS);
#ifdef JENNIC_CHIP_FAMILY_JN517x
	vAHI_TickTimerClear();
#else
    vAHI_TickTimerWrite(0);
#endif

	vAHI_TickTimerIntEnable(TRUE);
	vAHI_TickTimerConfigure(E_AHI_TICK_TIMER_RESTART);
}

/****************************************************************************
 *
 * NAME: CRD_vMlmeDcfmInd
 *
 * DESCRIPTION:
 * Process any incoming managment events from the stack.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  psMlmeInd
 *
 * RETURNS:
 * None.
 *
 * NOTES:
 * None.
 ****************************************************************************/
PRIVATE void CRD_vMlmeDcfmInd(MAC_MlmeDcfmInd_s *psMlmeInd)
{
	/* Which event type ? */
    switch (psMlmeInd->u8Type)
    {
		/* Incoming scan results ? */
		case MAC_MLME_DCFM_SCAN:
		{
			/* Waiting to create network ? */
			if (NODE_sData.eNwkState == NODE_NWKSTATE_CREATE)
			{
				/* Energy scan results ? */
				if (psMlmeInd->uParam.sDcfmScan.u8ScanType == MAC_MLME_SCAN_TYPE_ENERGY_DETECT)
				{
					/* Process energy scan results and start device as coordinator */
					CRD_vDcfmScanEnergy(psMlmeInd);
				}
			}
		}
		break;

		/* Incoming association request ? */
		case MAC_MLME_IND_ASSOCIATE:
		{
			/* Network is up ? */
			if (NODE_sData.eNwkState == NODE_NWKSTATE_UP)
			{
				/* Handle incoming association request */
				CRD_vIndAssociate(psMlmeInd);
			}
		}
		break;

		/* Others ? */
		default:
		{
			/* Do nothing */
			;
		}
		break;
    }
}

/****************************************************************************
 *
 * NAME: CRD_vMcpsDcfmInd
 *
 * DESCRIPTION:
 * Process incoming data events from the stack.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  psMcpsInd
 *
 * RETURNS:
 * None.
 *
 * NOTES:
 * None.
 ****************************************************************************/
PRIVATE void CRD_vMcpsDcfmInd(MAC_McpsDcfmInd_s *psMcpsInd)
{
	/* Which event type ? */
	switch(psMcpsInd->u8Type)
	{
		/* Incoming data frame ? */
        case MAC_MCPS_IND_DATA:
		{
    		/* Network is up ? */
    		if (NODE_sData.eNwkState == NODE_NWKSTATE_UP)
    		{
				/* Handle incoming data */
            	CRD_vIndData(psMcpsInd);
			}
		}
        break;

		/* Incoming acknowledgement or ack timeout ? */
        case MAC_MCPS_DCFM_DATA:
		{
    		/* Network is up ? */
    		if (NODE_sData.eNwkState == NODE_NWKSTATE_UP)
    		{
				/* Handle incoming delayed confirm data */
	            CRD_vDcfmData(psMcpsInd);
			}
		}
        break;

		/* Others ? */
        default:
		{
			/* Do nothing */
			;
		}
        break;
    }
}

/****************************************************************************
 *
 * NAME: CRD_vHwInd
 *
 * DESCRIPTION:
 * Process any hardware events.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  psAHI_Ind
 *
 * RETURNS:
 * None.
 *
 * NOTES:
 * None.
 ****************************************************************************/
PRIVATE void CRD_vHwInd(AppQApiHwInd_s *psAHI_Ind)
{
	/* Tick timer ? */
	if (psAHI_Ind->u32DeviceId == E_AHI_DEVICE_TICK_TIMER)
	{
		/* Call tick function for node */
		NODE_vTick();

		/* Call the wuart tick function */
		WUART_vTick();
	}
}

/****************************************************************************
 *
 * NAME: CRD_vIndData
 *
 * DESCRIPTION:
 *
 * PARAMETERS:      Name            RW  Usage
 *
 * RETURNS:
 *
 * NOTES:
 ****************************************************************************/
PRIVATE void CRD_vIndData(MAC_McpsDcfmInd_s *psMcpsInd)
{
    MAC_RxFrameData_s *psFrame;

    psFrame = &psMcpsInd->uParam.sIndData.sFrame;
    uint64 u64SrcAddr;

	/* Short addressing used ? */
	if (psFrame->sSrcAddr.u8AddrMode == 2)
	{
		/* Debug */
		DBG_vPrintf(CRD_TRACE, "\r\n%d CRD < CRD_vIndData() Mode=%d Short=%#x",
			NODE_sData.u32Timer, psFrame->sSrcAddr.u8AddrMode, psFrame->sSrcAddr.uAddr.u16Short);

		/* Pass data to wuart for handling */
		WUART_vRx((uint64) psFrame->sSrcAddr.uAddr.u16Short, psFrame->u8SduLength, psFrame->au8Sdu);
	}
	/* Extended addressing used ? */
	else if (psFrame->sSrcAddr.u8AddrMode == 3)
	{
		/* Debug */
		DBG_vPrintf(CRD_TRACE, "\r\n%d CRD < CRD_vIndData() Mode=%d Ext=%#x:%x",
			NODE_sData.u32Timer, psFrame->sSrcAddr.u8AddrMode,
			psFrame->sSrcAddr.uAddr.sExt.u32H, psFrame->sSrcAddr.uAddr.sExt.u32L);

		/* Construct extended address */
		u64SrcAddr = ((uint64) psFrame->sSrcAddr.uAddr.sExt.u32H << 32) | (uint64) psFrame->sSrcAddr.uAddr.sExt.u32L;
		/* Pass data to wuart for handling */
		WUART_vRx(u64SrcAddr, psFrame->u8SduLength, psFrame->au8Sdu);
	}
}

/****************************************************************************
 *
 * NAME: CRD_vDcfmData
 *
 * DESCRIPTION:
 *
 * PARAMETERS:      Name            RW  Usage
 *
 * RETURNS:
 *
 * NOTES:
 ****************************************************************************/
PRIVATE void CRD_vDcfmData(MAC_McpsDcfmInd_s *psMcpsInd)
{
	/* Debug */
	DBG_vPrintf(CRD_TRACE, "\r\n%d CRD < CRD_vDcfmData() Status=%d",
		NODE_sData.u32Timer, psMcpsInd->uParam.sDcfmData.u8Status);

    if (psMcpsInd->uParam.sDcfmData.u8Status == MAC_ENUM_SUCCESS)
    {
        /* Data frame transmission successful */
    }
    else
    {
        /* Data transmission falied after 3 retries at MAC layer. */
    }
}

/****************************************************************************
 *
 * NAME: CRD_vReqEnergyScan
 *
 * DESCRIPTION:
 * Starts an enery sacn on the channels specified.
 *
 * PARAMETERS:      Name            RW  Usage
 * None.
 *
 * RETURNS:
 * None.
 *
 * NOTES:
 * None.
 ****************************************************************************/
PRIVATE void CRD_vReqEnergyScan(void)
{
	/* Debug */
	DBG_vPrintf(CRD_TRACE, "\r\n%d CRD < CRD_vReqEnergyScan()",
		NODE_sData.u32Timer);

    /* Structures used to hold data for MLME request and response */
    MAC_MlmeReqRsp_s   sMlmeReqRsp;
    MAC_MlmeSyncCfm_s  sMlmeSyncCfm;

    NODE_sData.eNwkState = NODE_NWKSTATE_CREATE;

    /* Start energy detect scan */
    sMlmeReqRsp.u8Type = MAC_MLME_REQ_SCAN;
    sMlmeReqRsp.u8ParamLength = sizeof(MAC_MlmeReqStart_s);
    sMlmeReqRsp.uParam.sReqScan.u8ScanType = MAC_MLME_SCAN_TYPE_ENERGY_DETECT;
    sMlmeReqRsp.uParam.sReqScan.u32ScanChannels = SCAN_CHANNELS;
    sMlmeReqRsp.uParam.sReqScan.u8ScanDuration = ENERGY_SCAN_DURATION;

    vAppApiMlmeRequest(&sMlmeReqRsp, &sMlmeSyncCfm);
}

/****************************************************************************
 *
 * NAME: CRD_vDcfmScanEnergy
 *
 * DESCRIPTION:
 * Selects a channel with low enery content for use by the wireless UART.
 *
 * PARAMETERS:      Name            RW  Usage
 * None.
 *
 * RETURNS:
 * None.
 *
 * NOTES:
 * None.
 ****************************************************************************/
PRIVATE void CRD_vDcfmScanEnergy(MAC_MlmeDcfmInd_s *psMlmeInd)
{
    uint8 i = 0;
    uint8 u8MinEnergy;

	u8MinEnergy = (psMlmeInd->uParam.sDcfmScan.uList.au8EnergyDetect[0]);

    NODE_sData.u8Channel = CHANNEL_MIN;

	/* Debug */
	DBG_vPrintf(CRD_TRACE, "\r\n%d CRD < CRD_vDcfmScanEnergy()",
		NODE_sData.u32Timer);

	/* Search list to find quietest channel */
    while (i < psMlmeInd->uParam.sDcfmScan.u8ResultListSize)
    {
        if ((psMlmeInd->uParam.sDcfmScan.uList.au8EnergyDetect[i]) < u8MinEnergy)
        {
			u8MinEnergy = (psMlmeInd->uParam.sDcfmScan.uList.au8EnergyDetect[i]);
			NODE_sData.u8Channel = i + CHANNEL_MIN;
		}
		i++;
    }

	/* Debug */
	DBG_vPrintf(CRD_TRACE, "Chan=%d Energy=%d",	NODE_sData.u8Channel, u8MinEnergy);

    CRD_vReqStartCoordinator();
}

/****************************************************************************
 *
 * NAME: CRD_vReqStartCoordinator
 *
 * DESCRIPTION:
 * Starts the network by configuring the controller board to act as the PAN
 * coordinator.
 *
 * PARAMETERS:      Name            RW  Usage
 * None.
 *
 * RETURNS:
 * TRUE if network was started successfully otherwise FALSE
 *
 * NOTES:
 * None.
 ****************************************************************************/
PRIVATE void CRD_vReqStartCoordinator(void)
{
    /* Structures used to hold data for MLME request and response */
    MAC_MlmeReqRsp_s   sMlmeReqRsp;
    MAC_MlmeSyncCfm_s  sMlmeSyncCfm;

	/* Network is now up */
    NODE_sData.eNwkState = NODE_NWKSTATE_UP;

	/* Debug */
	DBG_vPrintf(CRD_TRACE, "\r\n%d CRD < CRD_vReqStartCoordinator()",
		NODE_sData.u32Timer);

    /* Start Pan */
    sMlmeReqRsp.u8Type = MAC_MLME_REQ_START;
    sMlmeReqRsp.u8ParamLength = sizeof(MAC_MlmeReqStart_s);
    sMlmeReqRsp.uParam.sReqStart.u16PanId = NODE_sData.u16PanId;
    sMlmeReqRsp.uParam.sReqStart.u8Channel = NODE_sData.u8Channel;
    sMlmeReqRsp.uParam.sReqStart.u8BeaconOrder = 0x0F;
    sMlmeReqRsp.uParam.sReqStart.u8SuperframeOrder = 0x0F;
    sMlmeReqRsp.uParam.sReqStart.u8PanCoordinator = TRUE;
    sMlmeReqRsp.uParam.sReqStart.u8BatteryLifeExt = FALSE;
    sMlmeReqRsp.uParam.sReqStart.u8Realignment = FALSE;
    sMlmeReqRsp.uParam.sReqStart.u8SecurityEnable = FALSE;

    vAppApiMlmeRequest(&sMlmeReqRsp, &sMlmeSyncCfm);

	/* Wuart not running ? */
	if (WUART_sData.u8State == WUART_STATE_NONE)
	{
		/* Put wuart into idle state */
		WUART_vState(WUART_STATE_IDLE);
	}
}

/****************************************************************************
 *
 * NAME: CRD_vIndAssociate
 *
 * DESCRIPTION:
 * Handle request by node to join the network.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  psMlmeInd
 *
 * RETURNS:
 * None.
 *
 * NOTES:
 * None.
 ****************************************************************************/
PRIVATE void CRD_vIndAssociate(MAC_MlmeDcfmInd_s *psMlmeInd)
{
    uint16 u16ShortAddr = 0xffff;
    uint16 u16EndDeviceIndex;

    MAC_MlmeReqRsp_s   sMlmeReqRsp;
    MAC_MlmeSyncCfm_s  sMlmeSyncCfm;

	/* Debug */
	DBG_vPrintf(CRD_TRACE, "\r\n%d CRD < CRD_vIndAssociate() Ext=%#x:%x",
		NODE_sData.u32Timer,
		psMlmeInd->uParam.sIndAssociate.sDeviceAddr.u32H,
		psMlmeInd->uParam.sIndAssociate.sDeviceAddr.u32L);

	/* Loop through current end devices */
	for (u16EndDeviceIndex = 0;
		 u16EndDeviceIndex < MAX_CHILDREN && u16ShortAddr == 0xffff;
		 u16EndDeviceIndex++)
	{
		/* Already got this end device ? */
		if (CRD_sData.asChild[u16EndDeviceIndex].u32ExtAddrL 	==
            psMlmeInd->uParam.sIndAssociate.sDeviceAddr.u32L				&&
		    CRD_sData.asChild[u16EndDeviceIndex].u32ExtAddrH 	==
            psMlmeInd->uParam.sIndAssociate.sDeviceAddr.u32H)
		{
			/* Note the short address of this entry */
			u16ShortAddr = CRD_sData.asChild[u16EndDeviceIndex].u16ShortAddr;

			sMlmeReqRsp.uParam.sRspAssociate.u8Status = 0; /* Access granted */

			/* Debug */
			DBG_vPrintf(CRD_TRACE, " Short=%#x REJOIN",
				NODE_sData.u32Timer, u16ShortAddr);
		}
	}

	/* Not an existing end device and have room for more ? */
    if (u16ShortAddr                       == 0xffff &&
        CRD_sData.u16Children <  MAX_CHILDREN)
    {
        /* Store end device address data */
        u16EndDeviceIndex = CRD_sData.u16Children;
        u16ShortAddr = (uint16)(psMlmeInd->uParam.sIndAssociate.sDeviceAddr.u32L & 0xffff);

        CRD_sData.asChild[u16EndDeviceIndex].u16ShortAddr = u16ShortAddr;

        CRD_sData.asChild[u16EndDeviceIndex].u32ExtAddrL  =
        psMlmeInd->uParam.sIndAssociate.sDeviceAddr.u32L;

        CRD_sData.asChild[u16EndDeviceIndex].u32ExtAddrH  =
        psMlmeInd->uParam.sIndAssociate.sDeviceAddr.u32H;

        CRD_sData.u16Children++;

        sMlmeReqRsp.uParam.sRspAssociate.u8Status = 0; /* Access granted */

		/* Debug */
		DBG_vPrintf(CRD_TRACE, " Short=%#x JOIN",
			NODE_sData.u32Timer, u16ShortAddr);
    }
    else if (u16ShortAddr == 0xffff)
    {
        sMlmeReqRsp.uParam.sRspAssociate.u8Status = 2; /* Denied */

		/* Debug */
		DBG_vPrintf(CRD_TRACE, " DENIED");
    }

    /* Create association response */
    sMlmeReqRsp.u8Type = MAC_MLME_RSP_ASSOCIATE;
    sMlmeReqRsp.u8ParamLength = sizeof(MAC_MlmeRspAssociate_s);
    sMlmeReqRsp.uParam.sRspAssociate.sDeviceAddr.u32H = psMlmeInd->uParam.sIndAssociate.sDeviceAddr.u32H;
    sMlmeReqRsp.uParam.sRspAssociate.sDeviceAddr.u32L = psMlmeInd->uParam.sIndAssociate.sDeviceAddr.u32L;
    sMlmeReqRsp.uParam.sRspAssociate.u16AssocShortAddr = u16ShortAddr;
    sMlmeReqRsp.uParam.sRspAssociate.u8SecurityEnable = FALSE;

    /* Send association response. There is no confirmation for an association
       response, hence no need to check */
    vAppApiMlmeRequest(&sMlmeReqRsp, &sMlmeSyncCfm);
}

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
