/*****************************************************************************
 *
 * MODULE:              PER Test Master
 */
/****************************************************************************
*
* This software is owned by NXP B.V. and/or its supplier and is protected
* under applicable copyright laws. All rights are reserved. We grant You,
* and any third parties, a license to use this software solely and
* exclusively on NXP products [NXP Microcontrollers such as JN5148, JN5142, JN5139].
* You, and any third parties must reproduce the copyright and warranty notice
* and any other legend of ownership on each copy or partial copy of the
* software.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
* ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
* INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
* POSSIBILITY OF SUCH DAMAGE.

* Copyright NXP B.V. 2012. All rights reserved
*/

/****************************************************************************/

/****************************************************************************/
/***        Include Files                                                 ***/
/****************************************************************************/

#include <jendefs.h>
#include <AppHardwareApi.h>
#include <JPT.h>
#include "Button.h"
#include "LedControl.h"
#include "UartBuffered.h"
#include "Printf.h"

#ifdef JENNIC_PCB_DEVKIT4
#include "GenericBoard.h"
#endif

/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/

#define UART_TO_PC          E_AHI_UART_0            /* Uart to PC           */
#define BAUD_RATE           E_AHI_UART_RATE_38400   /* Baud rate to use     */

#define KEY_DEBOUNCE    10
#define KEY_REPEAT      50
#define KEY_WAITREPEAT  200

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

typedef enum {
    RV_CHANGE_CHANNEL,
    RV_CHANGE_MODE,
    RV_STOPSTART,
} teReturnValue;

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/
PRIVATE void vChooseOptions(void);
PRIVATE void vUpdateDisplays(bool_t bUpdateLcd, bool_t bUpdateSerialConsole);
PRIVATE uint8 u8UpdateUI(void);

PRIVATE bool_t bReadButtons(uint8 *u8Key);
PRIVATE void vTickTimerISR(void);
PRIVATE void vPutC(uint8 u8Data);
PRIVATE char acGetC(void);


/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/

uint8  u8KeysDown = 0;
bool_t bEndpointOk = TRUE;
bool_t bLedsEnabled;
uint8 u8RunMode;
uint32 u32LastSeen = 0;

tsJPT_SSPT_MasterState sMasterData;             /* holds master state data */

tsUartFifo sFifo;                               /* TX & RX Buffers for UART */

/* scratchpad area and cursor info for building lcd contents */
volatile uint32 u32Debounce = KEY_DEBOUNCE;

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

PUBLIC void AppColdStart(void)
{

    /* Disable watchdog if enabled by default */
    #ifdef WATCHDOG_ENABLED
    vAHI_WatchdogStop();
    #endif

    /* Initialise the production test API */
    u32JPT_Init();

    /* Initialise the hardware API */
    u32AHI_Init();

    /* Set up UART 0: enable interrupts for RX and TX */
    vUartInit(UART_TO_PC, BAUD_RATE, &sFifo);   /* uart for console */
    vInitPrintf((void*)vPutC);

    /* set up the tick timer to run at 100Hz with interrupt */
    vAHI_TickTimerInit((void*)vTickTimerISR);
    vAHI_TickTimerWrite(0);
    vAHI_TickTimerInterval(16000000 / 100);
    vAHI_TickTimerConfigure(E_AHI_TICK_TIMER_RESTART);
    vAHI_TickTimerIntEnable(TRUE);

#ifdef  JENNIC_PCB_DEVKIT4
	vGenericLEDInit();
#else
	vLedInitFfd();
	vLedControl(0,FALSE);
	vLedControl(1,FALSE);
#endif

    /* Show startup options screen */
    vChooseOptions();

    /* Initialise site survey PER test master */
    vJPT_SSPT_MasterInit();

    /* Locate the endpoint */
    sMasterData.u8Mode = E_JPT_SSPT_MODE_LOCATE;
    u8RunMode = sMasterData.u8Mode;
    vUpdateDisplays(TRUE, FALSE);
    bEndpointOk = bJPT_SSPT_MasterSetState(&sMasterData);
    vUpdateDisplays(TRUE, FALSE);

    if(!bEndpointOk){
        vPrintf("\nFailed to contact the endpoint!");
    }

    /* Start the PER test running in Ack's mode */
    sMasterData.u8Mode = E_JPT_SSPT_MODE_RUNNING_ACKS;
    u8RunMode = sMasterData.u8Mode;
    bEndpointOk = bJPT_SSPT_MasterSetState(&sMasterData);

    while(1){

        switch(u8UpdateUI())
		{

        case RV_CHANGE_MODE:
            if(u8RunMode == E_JPT_SSPT_MODE_RUNNING_NO_ACKS){
                u8RunMode = E_JPT_SSPT_MODE_RUNNING_ACKS;
            } else {
                u8RunMode = E_JPT_SSPT_MODE_RUNNING_NO_ACKS;
            }
            if(sMasterData.u8Mode != E_JPT_SSPT_MODE_STOPPED){
                sMasterData.u8Mode = u8RunMode;
            }
            break;

        case RV_CHANGE_CHANNEL:
            sMasterData.u8Channel++;

/* Note: End device starts with high power enabled, JN5148 does not support high power
on Channel 26 (FCC compliance), therefore only channels 11 to 25 are supported */
/*ch 26 now supported #ifdef JENNIC_CHIP_JN5148

            if(sMasterData.u8Channel > 25){
#else */
            if(sMasterData.u8Channel > 26){
/* #endif */

                  sMasterData.u8Channel = 11;
            }
            break;

        case RV_STOPSTART:
            if(sMasterData.u8Mode != E_JPT_SSPT_MODE_STOPPED){
                sMasterData.u8Mode = E_JPT_SSPT_MODE_STOPPED;
            } else {
                sMasterData.u8Mode = u8RunMode;
            }
            break;

        }

        bEndpointOk = bJPT_SSPT_MasterSetState(&sMasterData);

    }
}


PUBLIC void AppWarmStart(void)
{
    AppColdStart();
}

/****************************************************************************/
/***        Local Functions                                               ***/
/****************************************************************************/

PRIVATE void vChooseOptions(void)
{

    uint8  n;
    uint8  u8Key;
    bool_t bOuterLoopFinished = FALSE;

    static uint32 u32RadioMode = E_JPT_MODE_LOPOWER;
    static uint8  u8Channel = 11;
    static uint8  u8PowerLevel = 47;
    static uint8  u8Retries = 3;
	static uint8  u8Leds = 0;

    bool_t bUpdateDisplay = TRUE;
    uint8  u8Selection = 0;

    char acPowerMode[5][11] = {"Low  \0","Boost\0","High \0","0dBm \0"};
    char acLeds[2][11]      = {"Disabled \0","Enabled \0"};
    char acOption[]         = {"      "};

    /* Initialise radio */
    bJPT_RadioInit(u32RadioMode);

    /* Get default power level */

    u8PowerLevel = u8JPT_RadioGetPowerFine();

    vJPT_RadioDeInit();

    do {

        if(bUpdateDisplay){
        /* Ascii clear screen */
         vPrintf("\x1b[2J");
            for(n = 0; n < 6; n++){
                if(n == u8Selection){
                    acOption[n] = '>';
                } else {
                    acOption[n] = ' ';
                }
            }
            vPrintf("\x1b[H");

            vPrintf("\x1b[H");
            vPrintf("Options\n"
                    "%cChannel: %d\n"
                    "%cPower Mode: %s\n"
                    "%cPower Level: %d\n"
                    "%cRetries: %d\n"
                    "%cLEDs: %s\n"
                    "%cDone\n"

                    "\n Up(1)   Down(2)  Select(3)     "
												,acOption[0],u8Channel
                                                ,acOption[1],acPowerMode[u32RadioMode]
                                                ,acOption[2],u8PowerLevel
                                                ,acOption[3],u8Retries
                                                ,acOption[4],acLeds[u8Leds]
                                                ,acOption[5]);

           bUpdateDisplay = FALSE;
        } 


        /* Handle any button presses */
        if (bUartRxDataAvailable(UART_TO_PC) && (bReadButtons(&u8Key)))
		{

            switch(u8Key){

            case '1':       /* Up */
            u8Selection--;
                if(u8Selection == 0xff) u8Selection = 5;
                break;

            case '2':       /* Down */
                u8Selection++;
                if(u8Selection > 5) u8Selection = 0;
                break;

            case '3':       /* Change */
                switch(u8Selection){

                case 0:		/* channel */
                    u8Channel++;

                    if(u8Channel > 26)
					{

                        u8Channel = 11;
                    }
                    break;

                case 1:		/* power mode */
                    u32RadioMode++;
                    if(u32RadioMode>E_JPT_MODE_0DBM)
					{
						u32RadioMode = E_JPT_MODE_LOPOWER;
					}

                    break;

                case 2:		/* power level */
 /* JN516x uses values of 0, 8, 16 and 24 for power level,Note: These values represent PAP = 0, 1, 2 and 3
    If values other than those 4 are selected then the software puts the power level to full = 24.  */


                    switch(u8PowerLevel)
					{
                        case 0:
                            u8PowerLevel = 8;
                            break;

                        case 8:
                            u8PowerLevel = 16;
                            break;

                        case 16:
                            u8PowerLevel = 24;
                            break;

                        case 24:
                            u8PowerLevel = 0;
                            break;

                        default:
                            u8PowerLevel = 24;
                            break;

                    }

                    break;

                case 3:		/* retries */
                    u8Retries++;
                    if(u8Retries > 7){
                        u8Retries = 0;
                    }
                    break;

                case 4:		/* led enable */

                    u8Leds++;
                    if(u8Leds > 1)
					 {
                        u8Leds = 0;
                    }


				break;

                case 5:    /* done sw1 (dk2) or ok (dk3) */
                    bOuterLoopFinished = TRUE;
                    break;

                }
                break;

            }

            bUpdateDisplay = TRUE;

        }

    } while (bOuterLoopFinished == FALSE);


    if(u8Leds)
	{
        bLedsEnabled = TRUE;
    } else 
	{
        bLedsEnabled = FALSE;
    }


    bJPT_RadioInit(u32RadioMode);

    vJPT_RadioSetPowerFine(u8PowerLevel);

    sMasterData.u8Channel = u8Channel;
    sMasterData.u8Retries = u8Retries;

    vInitPrintf((void*)vPutC);

}

/****************************************************************************
 *
 * NAME:       u8UpdateUI
 *
 * DESCRIPTION:
 * Updates the user interface (Updates LCD and console displays and checks
 * the buttons for any presses. Returns when a button has been pressed.
 *
 * PARAMETERS: Name     RW  Usage
 * none
 *
 * RETURNS:
 * uint8    Keypress value
 *
 ****************************************************************************/

PRIVATE uint8 u8UpdateUI(void)
{

    uint8  u8Key;
    bool_t bOuterLoopFinished = FALSE;
    uint8  u8Count = 0;
    uint8  u8RetVal = 0;

    do
    {

        /* Doze CPU until an interrupt occurs */
        vAHI_CpuDoze();

        /* get updated per test data */
        vJPT_SSPT_MasterGetState(&sMasterData);

        if (u8Count++ < 100) {
            u8Count = 0;
            vUpdateDisplays(TRUE, TRUE);

        }

        if(sMasterData.u32Total > 50000){
            sMasterData.u8Mode = E_JPT_SSPT_MODE_RESTART;
            bEndpointOk = bJPT_SSPT_MasterSetState(&sMasterData);
        }

        /* Handle any button presses */
        if (bUartRxDataAvailable(UART_TO_PC) && (bReadButtons(&u8Key))){
            switch(u8Key){

            case '1':
                /* Change channel */
                bOuterLoopFinished = TRUE;
                u8RetVal = RV_CHANGE_CHANNEL;
                break;

            case '2':
                /* Stop/Start */
                bOuterLoopFinished = TRUE;
                u8RetVal = RV_STOPSTART;
                break;

            case '3':
               /* Ack/No ack */
                bOuterLoopFinished = TRUE;
                u8RetVal = RV_CHANGE_MODE;
                break;

            }

        }

    } while (bOuterLoopFinished == FALSE);

    return u8RetVal;
}


/****************************************************************************
 *
 * NAME:       vUpdateDisplays
 *
 * DESCRIPTION:
 * Updates the displays generated on the LCD module and the serial console.
 *
 * PARAMETERS: Name         RW  Usage
 *          bUpdateLcd      R   Set to TRUE to update LCD
 *          bUpdateSerial   R   Set to TRUE to update serial console
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/

PRIVATE void vUpdateDisplays(bool_t bUpdateLcd, bool_t bUpdateSerialConsole)
{

	#define STARTING_COLUMN 0

    char   acStart[] = "Start";
    char   acStop[] = "Stop ";

    char *pcStart;

/*    uint8 x; */
    uint16 u16Per;
    uint16 u16Bad;

    /* If we are receiving packets, turn LED 0 on */
    if(sMasterData.u32Seen != u32LastSeen)
	{

#ifdef  JENNIC_PCB_DEVKIT2
        vLedControl(0, TRUE & bLedsEnabled);
#else
		vGenericLEDSetOutput(1, TRUE&bLedsEnabled);
#endif
    }


    else
	{

#ifdef  JENNIC_PCB_DEVKIT2
        vLedControl(0, FALSE);
#else
		vGenericLEDSetOutput(1, FALSE);
#endif

        /*
         * Since the LQI value holds the level for the last received packet,
         * we need to set it to 0 ourselves if we stop seeing any.
         */
        sMasterData.u8Lqi = 0;
    }
    u32LastSeen = sMasterData.u32Seen;

    /* Each Lqi step is very roughly 2.5dB */

/* 4 leds available only on dk2 board */
#ifdef JENNIC_PCB_DEVKIT2

    /* If > 20dB above sensitivity, turn LED 1 on */
    if(sMasterData.u8Lqi > 63)
	{
        vLedControl(1, TRUE & bLedsEnabled);
    } 
	else 
	{
        vLedControl(1, FALSE);
    }

    /* If > 40dB above sensitivity, turn LED 2 on */
    if(sMasterData.u8Lqi > 126)
	{
        vLedControl(2, TRUE & bLedsEnabled);
    } 
	else 
	{
        vLedControl(2, FALSE);
    }

    /* If > 60dB above sensitivity, turn LED 3 on */
    if(sMasterData.u8Lqi > 189)
	{
        vLedControl(3, TRUE & bLedsEnabled);
    } 
	else 
	{
        vLedControl(3, FALSE);
    }
#endif

    /* Calculate new PER and CCA Fail values */
    if(sMasterData.u32Total == 0)
	{
        u16Per = 0;
        u16Bad = 0;
    } 
	else 
	{
        u16Per = (uint16)(1000 - ((sMasterData.u32Seen * 1000) / sMasterData.u32Total));
        u16Bad = (uint16)((sMasterData.u32Errors * 1000) / sMasterData.u32Total);
    }

    /*
     * Figure out which mode we're in and point to
     * relavent text for start/stop button
     */
    if(sMasterData.u8Mode == E_JPT_SSPT_MODE_STOPPED)
	{
        pcStart = acStart;
    } 
	else 
	{
        pcStart = acStop;
    }

    /* Update the serial console content if required */
    if(bUpdateSerialConsole)
	{
        vPrintf("\x1b[H");
        switch(u8RunMode)
		{

        case E_JPT_SSPT_MODE_LOCATE:
            break;

        case E_JPT_SSPT_MODE_RUNNING_ACKS:
        
            /* write the static parts of the display */
            vPrintf("Channel      %d     \n"
                    "Mode         Ack     \n"
                    "Seen         %d    \n"
                    "Total        %d    \n"
                    "PER %%        %d.%d    \n"
                    "CCA Fail %%   %d.%d    \n"
                    "Retries      %d    \n"
                    "LQI          %d    \n"
                    "Chan (1)  %s (2)  Mode (3)  "
                    ,sMasterData.u8Channel,
                                        sMasterData.u32Seen,
                                        sMasterData.u32Total,
                                        u16Per / 10, u16Per % 10,
                                        u16Bad / 10, u16Bad % 10,
                                        sMasterData.u8Retries,
                                        sMasterData.u8Lqi, pcStart);
            break;

        case E_JPT_SSPT_MODE_RUNNING_NO_ACKS:
            /* write the static parts of the display */
            vPrintf("Channel      %d        \n"
                    "Mode         No Ack     \n"
                    "Seen         %d    \n"
                    "Total        %d    \n"
                    "PER %%        %d.%d    \n"
                    "LQI          %d    \n"
                    "                     \n"
                    "                     \n"
                     "Chan (1)  %s (2)  Mode (3) "
                    ,sMasterData.u8Channel,
                                        sMasterData.u32Seen,
                                        sMasterData.u32Total,
                                        u16Per / 10, u16Per % 10,
                                        sMasterData.u8Lqi, pcStart);
            break;

        }

    }


}

/****************************************************************************
 *
 * NAME:       bReadButtons
 *
 * DESCRIPTION:
 * Checks for button presses.
 *
 * PARAMETERS:  Name    RW  Usage
 *              u8Key   W   Pointer to a memory location that will receive the
 *                          key press value.
 *
 * RETURNS:
 * bool_t, TRUE if a button was pressed, FALSE if not
 *
 ****************************************************************************/
PRIVATE bool_t bReadButtons(uint8 *u8Key)
{
    bool_t bKeyPress = FALSE;

       char acKey = 0;
        acKey = acGetC();

        switch(acKey){

    case '1': /* Switch 1 */
        *u8Key = '1';
        bKeyPress = TRUE;
        break;

    case '2': /* Switch 2 */
        *u8Key = '2';
        bKeyPress = TRUE;
        break;

    case '3': /* Switch 3 */
        *u8Key = '3';
        bKeyPress = TRUE;
        break;

    default:
        u32Debounce = KEY_DEBOUNCE;
        break;

    }


    return(bKeyPress);
}



/****************************************************************************
 *
 * NAME:       vTickTimerISR
 *
 * DESCRIPTION:
 * Interrupt handler for the tick timer
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vTickTimerISR(void)
{

	if(u32Debounce){
        u32Debounce--;
    }

}



/****************************************************************************
 *
 * NAME:       vPutC
 *
 * DESCRIPTION:
 * Writes characters to the UART connected to the PC
 *
 * PARAMETERS: Name     RW  Usage
 *          u8Data      R   Character to write to the UART
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/

PRIVATE void vPutC(uint8 u8Data)
{
    vUartWrite(UART_TO_PC, u8Data);
}
/****************************************************************************
 *
 * NAME:       acGetC
 *
 * DESCRIPTION:
 * Reads a character from the uart connected to the pc. If no character is
 * waiting in the rx buffer, it will wait until there is.
 *
 *
 * RETURNS:
 * char, Character read from the UART
 *
 ****************************************************************************/
PRIVATE char acGetC(void)
{
    return(u8UartRead(UART_TO_PC));
}



/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
