/*****************************************************************************
 *
 * MODULE:              PER Test Slave
 *
 ****************************************************************************
 *
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN5168, JN5179].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2015. All rights reserved
 ****************************************************************************/

/****************************************************************************/
/***        Include Files                                                 ***/
/****************************************************************************/

#include <jendefs.h>
#include <AppHardwareApi.h>
#include <AppQueueApi.h>
#include <mac_sap.h>
#include <mac_pib.h>

#include "LedControl.h"

#include "PerTest.h"

#include "UartBuffered.h"
#include "Printf.h"

/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/

#define UART_TO_PC          E_AHI_UART_0            /* Uart to PC           */
#define BAUD_RATE           E_AHI_UART_RATE_115200  /* Baud rate to use     */

#define LED_STOPPED         (1 << 0)
#define LED_MODE            (1 << 1)

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/

PRIVATE void vPutC(uint8 u8Data);

/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/

tsPER_SlaveState sSlaveData;       /* will hold slave state data */

tsUartFifo sFifo;                       /* TX & RX Buffers for the UART */

uint32 u32RadioMode      = 0;
uint32 u32JPT_RadioModes = 0;

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

PUBLIC void AppColdStart(void)
{
    volatile int i;

    /* Disable watchdog if enabled by default */
    #ifdef WATCHDOG_ENABLED
    vAHI_WatchdogStop();
    #endif

    /* Initialise the hardware API */
    u32AHI_Init();

    /* Set up the serial console */
    vUartInit(UART_TO_PC, BAUD_RATE, &sFifo);   /* uart for console */
    vInitPrintf((void*)vPutC);
    vPrintf(CLR_SCREEN HOME "AN1212 PER Test Slave JN" STR(JENNIC_CHIP) " " STR(JENNIC_MAC) "\n");

    vAHI_DioSetOutput(LED_STOPPED | LED_MODE, 0);
    vAHI_DioSetDirection(0, LED_STOPPED | LED_MODE);

    /* Initialise the site survey PER test slave */
    vPER_SlaveInit();

    while(1){

        /* Doze CPU until an interrupt occurs */
        vAHI_CpuDoze();

        /* wait here a while */
        for(i = 0; i < 1000000; i++);

        /* get current PER test state */
        vPER_SlaveGetState(&sSlaveData);

        /* Display status of the test on the UART console and LED's */
        vPrintf(CURSOR(2, 1) CLR_DOWN);
        vPrintf("Channel      = %d\n"
        		"Packet Len   = %d\n"
#ifdef RXPOWERADJUST_SUPPORT
        		"Input level  = %d\n"
#endif
        		"TX Power     = %i\n",
        		sSlaveData.u8Channel,
        		sSlaveData.u8PacketLength,
#ifdef RXPOWERADJUST_SUPPORT
        		sSlaveData.u8InputLevel,
#endif
        		sSlaveData.i8PowerLevel);

        switch(sSlaveData.eMode){

        case E_PER_MODE_STOPPED:
            vPrintf("Stopped");
            vAHI_DioSetOutput(0, LED_STOPPED);
            break;

        case E_PER_MODE_RUNNING_ACKS:
            vPrintf("Running (Acks)");
            vAHI_DioSetOutput(LED_STOPPED, LED_MODE);
            break;

        case E_PER_MODE_RUNNING_NO_ACKS:
            vPrintf("Running (No Acks)");
            vAHI_DioSetOutput(LED_STOPPED | LED_MODE, 0);
            break;

        default:
        	sSlaveData.eMode = E_PER_MODE_STOPPED;
        	break;
        }

    }

}


PUBLIC void AppWarmStart(void)
{
    AppColdStart();
}

/****************************************************************************/
/***        Local Functions                                               ***/
/****************************************************************************/

/****************************************************************************
 *
 * NAME:       vPutC
 *
 * DESCRIPTION:
 * Writes characters to the UART connected to the PC
 *
 * PARAMETERS: Name     RW  Usage
 *          u8Data      R   Character to write to the UART
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/

PRIVATE void vPutC(uint8 u8Data)
{
    vUartWrite(UART_TO_PC, u8Data);
}

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
