/****************************************************************************
 *
 * MODULE:             Buffered, interrupt driven serial I/O
 *
 ****************************************************************************
 *
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN5168, JN5179].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2015. All rights reserved
 ****************************************************************************/

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/

#include <jendefs.h>
#include <AppHardwareApi.h>
#include "UartBuffered.h"

/****************************************************************************/
/***        Macro Definitions                                             ***/
/****************************************************************************/
#ifndef E_AHI_UART1
#define UART_NUM_UARTS  1
#else
#define UART_NUM_UARTS  2
#endif

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Function Prototypes                                     ***/
/****************************************************************************/

PRIVATE void vUartISR(uint32 u32DeviceId, uint32 u32ItemBitmap);
PRIVATE void vUartTxIsr(uint8 u8Uart);
PRIVATE void vUartRxIsr(uint8 u8Uart);

/****************************************************************************/
/***        Exported Variables                                            ***/
/****************************************************************************/

/****************************************************************************/
/***        Local Variables                                               ***/
/****************************************************************************/

tsUartFifo *apsUartFifo[UART_NUM_UARTS];

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

/****************************************************************************
 *
 * NAME:       vUartInit
 *
 * DESCRIPTION:
 * Initialises the specified UART.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8Uart          R   UART to initialise, eg, E_AHI_UART_0
 *                  u8BaudRate      R   Baudrate to use,
 *                                      eg,E_AHI_UART_RATE_9600
 *                  psFifo          R   Pointer to a tsUartFifo struct holding
 *                                      all data for this UART
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PUBLIC void vUartInit(uint8 u8Uart, uint8 u8BaudRate, tsUartFifo *psFifo)
{

    apsUartFifo[u8Uart] = psFifo;

    psFifo->bTxInProgress = FALSE;
    psFifo->u32TxInPtr = 0;
    psFifo->u32TxOutPtr = 0;
    psFifo->u32RxInPtr = 0;
    psFifo->u32RxOutPtr = 0;

    vAHI_UartEnable(u8Uart);

    vAHI_UartReset(u8Uart, TRUE, TRUE);
    vAHI_UartReset(u8Uart, FALSE, FALSE);

    vAHI_UartSetClockDivisor(u8Uart, u8BaudRate);
    vAHI_UartSetControl(u8Uart, FALSE, FALSE, E_AHI_UART_WORD_LEN_8, TRUE, FALSE);

    /* install interrupt service calback */
    if(u8Uart == E_AHI_UART_0){
        vAHI_Uart0RegisterCallback((void*)vUartISR);
    }
#ifdef E_AHI_UART1
    else {
        vAHI_Uart1RegisterCallback((void*)vUartISR);
    }
#endif
    /* Enable TX Fifo empty and Rx data interrupts */
    vAHI_UartSetInterrupt(u8Uart, FALSE, FALSE, TRUE, TRUE, E_AHI_UART_FIFO_LEVEL_1);

}


/****************************************************************************
 *
 * NAME:       vUartDeInit
 *
 * DESCRIPTION:
 * Disables the specified UART.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8Uart          R   UART to disable, eg, E_AHI_UART_0
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PUBLIC void vUartDeInit(uint8 u8Uart)
{

    /* Disable TX Fifo empty and Rx data interrupts */
    vAHI_UartSetInterrupt(u8Uart, FALSE, FALSE, FALSE, FALSE, E_AHI_UART_FIFO_LEVEL_1);

    /* remove interrupt service callback */
    if(u8Uart == E_AHI_UART_0){
        vAHI_Uart0RegisterCallback((void*)NULL);
    }
#ifdef E_AHI_UART1
    else {
        vAHI_Uart1RegisterCallback((void*)NULL);
    }
#endif

    vAHI_UartDisable(u8Uart);
}


/****************************************************************************
 *
 * NAME:       bUartReadBinary
 *
 * DESCRIPTION:
 * Reads a specified number of bytes from the uart and stores them in an area
 * of memory. Times out if no data is received for a specified time.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8Uart          R   UART to read from, eg. E_AHI_UART_0
 *                  pu8Ptr          W   Pointer to an area of memory that
 *                                      will receive the data.
 *                  u32Len          R   Number of bytes to receive
 *                  u32TimeoutTime  R   How long to wait for data before
 *                                      timeout
 *
 * RETURNS:
 * bool_t, TRUE if specified number of bytes were read, FALSE if timed out
 *
 ****************************************************************************/
PUBLIC bool_t bUartReadBinary(uint8 u8Uart, uint8 *pu8Ptr, uint32 u32Len, uint32 u32TimeoutTime)
{

    uint32 n;

    for(n = 0; n < u32Len; n++){
        if(!bUartReadWithTimeout(u8Uart, pu8Ptr++, u32TimeoutTime)){
            return(FALSE);
        }
    }

    return(TRUE);
}


/****************************************************************************
 *
 * NAME:       bUartReadWithTimeout
 *
 * DESCRIPTION:
 * Attempts to read 1 byte from the RX buffer. If there is no data in the
 * buffer, then it will wait for u32TimeoutTime, and then exit.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8Uart          R   UART to use, eg, E_AHI_UART_0
 *                  pu8Data         W   pointer to 8 bit RX data destination
 *                  u32TimeoutTime  R   Time to wait for data to if there
 *                                      is none already in the buffer
 *
 * RETURNS:
 * bool_t: TRUE if data was read, data is left in *pu8Data,
 *         FALSE if no data was read
 *
 ****************************************************************************/
PUBLIC bool_t bUartReadWithTimeout(uint8 u8Uart, uint8 *pu8Data, uint32 u32TimeoutTime)
{

    uint32 u32Time;

    tsUartFifo *pasUartFifo = apsUartFifo[u8Uart];

    for(u32Time = 0; u32Time < u32TimeoutTime; u32Time++){

        if(pasUartFifo->u32RxInPtr != pasUartFifo->u32RxOutPtr){

            *pu8Data = pasUartFifo->u8UartRxBuff[pasUartFifo->u32RxOutPtr];

            pasUartFifo->u32RxOutPtr++;          /* increment out pointer */
            if(pasUartFifo->u32RxOutPtr == UART_RX_BUFFER_LEN){     /* if end of buffer */
                pasUartFifo->u32RxOutPtr = 0;                   /* wrap arount to start */
            }

            return(TRUE);
        }
    }

    *pu8Data = 0;
    return(FALSE);
}


/****************************************************************************
 *
 * NAME:       u8UartRead
 *
 * DESCRIPTION:
 * Reads 1 byte from the RX buffer. If there is no data in the
 * buffer, then it will wait until there is.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8Uart          R   UART to use, eg, E_AHI_UART_0
 *
 * RETURNS:
 * uint8: received data
 *
 ****************************************************************************/
PUBLIC uint8 u8UartRead(uint8 u8Uart)
{

    uint8 u8RxChar = 0;

    tsUartFifo *pasUartFifo = apsUartFifo[u8Uart];

    /* wait until there is data in the receive buffer */
    while(pasUartFifo->u32RxOutPtr == pasUartFifo->u32RxInPtr);

    u8RxChar = pasUartFifo->u8UartRxBuff[pasUartFifo->u32RxOutPtr];

    pasUartFifo->u32RxOutPtr++;          /* increment out pointer */
    if(pasUartFifo->u32RxOutPtr == UART_RX_BUFFER_LEN){     /* if end of buffer */
        pasUartFifo->u32RxOutPtr = 0;                   /* wrap arount to start */
    }
    return(u8RxChar);

}


/****************************************************************************
 *
 * NAME:       bUartTxInProgress
 *
 * DESCRIPTION:
 * Returns the state of data transmission
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8Uart          R   UART to use, eg, E_AHI_UART_0
 *
 * RETURNS:
 * bool_t: TRUE if data in buffer is being transmitted
 *         FALSE if all data in buffer has been transmitted by the UART
 *
 ****************************************************************************/
PUBLIC bool_t bUartTxInProgress(uint8 u8Uart)
{

    tsUartFifo *pasUartFifo = apsUartFifo[u8Uart];

    if(pasUartFifo->u32TxInPtr == pasUartFifo->u32TxOutPtr){

        if((u8AHI_UartReadLineStatus(u8Uart) & E_AHI_UART_LS_TEMT ) != 0){
            return(FALSE);
        }

    }

    return(TRUE);

}


/****************************************************************************
 *
 * NAME:       bUartRxDataAvailable
 *
 * DESCRIPTION:
 * Returns state of data reception
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8Uart          R   UART to use, eg, E_AHI_UART_0
 *
 * RETURNS:
 * bool_t: TRUE if there is received data in the buffer
 *         FALSE if there is no received data
 *
 ****************************************************************************/
PUBLIC bool_t bUartRxDataAvailable(uint8 u8Uart)
{

    tsUartFifo *pasUartFifo = apsUartFifo[u8Uart];

    /* see if data has become available in rx buffer */
    if(pasUartFifo->u32RxInPtr == pasUartFifo->u32RxOutPtr){
        return(FALSE);
    } else {
        return(TRUE);
    }

}


/****************************************************************************
 *
 * NAME:       vUartWriteBinary
 *
 * DESCRIPTION:
 * Writes a specified number of bytes to the uart.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8Uart          R   UART to write to, eg. E_AHI_UART_0
 *                  pu8Ptr          W   Pointer to an area of memory that
 *                                      contains the data to send
 *                  u32Len          R   Number of bytes to send
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PUBLIC void vUartWriteBinary(uint8 u8Uart, uint8 *pu8Ptr, uint32 u32Len)
{

    uint32 n;

    for(n = 0; n < u32Len; n++){
        vUartWrite(u8Uart, *pu8Ptr++);
    }

}


/****************************************************************************
 *
 * NAME:       bUartReadBinary
 *
 * DESCRIPTION:
 * Writes a null terminated string to the specified UART
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8Uart          R   UART to write to, eg. E_AHI_UART_0
 *                  pu8String       W   Pointer to a null terminated string
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PUBLIC void vUartWriteString(uint8 u8Uart, uint8 *pu8String)
{

    while(*pu8String != '\0'){
        vUartWrite(u8Uart, *pu8String);
        if(*pu8String++ == '\r'){
            vUartWrite(u8Uart, '\n');
        }
    }

}


/****************************************************************************
 *
 * NAME:       vUartWrite
 *
 * DESCRIPTION:
 * Writes one byte to the specified uart for transmission
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8Uart          R   UART to disable, eg, E_AHI_UART_0
 *                  u8Data          R   data to transmit
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PUBLIC void vUartWrite(uint8 u8Uart, uint8 u8Data)
{

    uint32 u32NewInPtr;

    tsUartFifo *pasUartFifo = apsUartFifo[u8Uart];

    u32NewInPtr = pasUartFifo->u32TxInPtr;  /* new in pointer */

    u32NewInPtr++;
    if(u32NewInPtr == UART_TX_BUFFER_LEN){   /* if end of buffer */
        u32NewInPtr = 0;                     /* wrap around to start */
    }

    /* wait here until there is space in the buffer to accept the data */
    while(u32NewInPtr == pasUartFifo->u32TxOutPtr);

    pasUartFifo->u8UartTxBuff[pasUartFifo->u32TxInPtr] = u8Data;    /* put tx byte in buffer */
    pasUartFifo->u32TxInPtr = u32NewInPtr;              /* save new in pointer */


    /*
     * if there is already a tx in progress, we can expect a TX interrupt
     * some time in the future, in which case the data we wrote to the tx
     * buffer will get sent in due course to the UART in the interrupt
     * service routine.
     * if there is no tx in progress, there won't be a tx interrupt, and the
     * byte won't get read from the buffer in the ISR, so we must write it
     * to the UART tx FIFO here instead, and since there can't be any more
     * data in the buffer other than the data we just wrote to it, we can reset
     * the buffer pointers (quicker than reading the byte back out of the buffer
     * properly), and just write it to the UART fifo instead.
     */
    if(pasUartFifo->bTxInProgress == FALSE){

        pasUartFifo->bTxInProgress = TRUE;
        pasUartFifo->u32TxInPtr = 0;
        pasUartFifo->u32TxOutPtr = 0;
        vAHI_UartWriteData(u8Uart,u8Data);

    }

}


/****************************************************************************
 *
 * NAME:       vUartFlush
 *
 * DESCRIPTION:
 * Flushes the buffers of the specified UART
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8Uart          R   UART to disable, eg, E_AHI_UART_0
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PUBLIC void vUartFlush(uint8 u8Uart)
{

    tsUartFifo *pasUartFifo = apsUartFifo[u8Uart];

    /* Disable TX Fifo empty and Rx data interrupts */
    vAHI_UartSetInterrupt(u8Uart, FALSE, FALSE, FALSE, FALSE, E_AHI_UART_FIFO_LEVEL_1);

    /* flush software buffer */
    pasUartFifo->bTxInProgress = FALSE;
    pasUartFifo->u32TxInPtr = 0;
    pasUartFifo->u32TxOutPtr = 0;
    pasUartFifo->u32RxInPtr = 0;
    pasUartFifo->u32RxOutPtr = 0;

    /* flush hardware buffer */
    vAHI_UartReset(u8Uart, TRUE, TRUE);
    vAHI_UartReset(u8Uart, FALSE, FALSE);

    /* Re-enable TX Fifo empty and Rx data interrupts */
    vAHI_UartSetInterrupt(u8Uart, FALSE, FALSE, TRUE, TRUE, E_AHI_UART_FIFO_LEVEL_1);

}


/****************************************************************************/
/***        Local Functions                                               ***/
/****************************************************************************/

/****************************************************************************
 *
 * NAME:       vUartISR
 *
 * DESCRIPTION:
 * Interrupt service callback for UART's
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u32DeviceId     R   Device ID of whatever generated the
 *                                      interrupt
 *                  u32ItemBitmap   R   Which part of the device generated
 *                                      the interrupt
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vUartISR(uint32 u32DeviceId, uint32 u32ItemBitmap)
{

    uint8 u8Uart;


    switch(u32DeviceId){

    case E_AHI_DEVICE_UART0:
        u8Uart = 0;
        break;
#ifdef E_AHI_UART1
    case E_AHI_DEVICE_UART1:
        u8Uart = 1;
        break;
#endif
    default:
        return;
    }


    switch(u32ItemBitmap){

    case E_AHI_UART_INT_TX:
        vUartTxIsr(u8Uart);
        break;

    case E_AHI_UART_INT_RXDATA:
        vUartRxIsr(u8Uart);
        break;

    }

}


/****************************************************************************
 *
 * NAME:       vUartTxISR
 *
 * DESCRIPTION:
 * Interrupt service callback for UART data transmission. Checks the tx buffer
 * for any data waiting for transmission, and if any is available, will write
 * up to 16 bytes of it to the UART's hardware fifo, set the tx in progress
 * flag then exit.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8Uart          R   Uart to write to
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vUartTxIsr(uint8 u8Uart)
{

    uint8 u8Data;
    uint32 u32Bytes = 0;

    tsUartFifo *pasUartFifo = apsUartFifo[u8Uart];

    /*
     * if there is data in buffer waiting for tx and we've not filled the
     * hardware fifo up
     */
    while(pasUartFifo->u32TxOutPtr != pasUartFifo->u32TxInPtr && u32Bytes < 16){

        u8Data = pasUartFifo->u8UartTxBuff[pasUartFifo->u32TxOutPtr];
        vAHI_UartWriteData(u8Uart,u8Data); /* write one byte to the UART */

        pasUartFifo->u32TxOutPtr++;          /* increment out pointer */
        if(pasUartFifo->u32TxOutPtr == UART_TX_BUFFER_LEN){     /* if end of buffer */
            pasUartFifo->u32TxOutPtr = 0;                   /* wrap arount to start */
        }
        u32Bytes++;
    }

    if(u32Bytes){
        pasUartFifo->bTxInProgress = TRUE;
    } else {
        pasUartFifo->bTxInProgress = FALSE;
    }

}


/****************************************************************************
 *
 * NAME:       vUartRxISR
 *
 * DESCRIPTION:
 * Interrupt service callback for UART data reception. Reads a received
 * byte from the UART and writes it to the reception buffer if it is not
 * full.
 *
 * PARAMETERS:      Name            RW  Usage
 *                  u8Uart          R   Uart to read from
 *
 * RETURNS:
 * void
 *
 ****************************************************************************/
PRIVATE void vUartRxIsr(uint8 u8Uart)
{
    uint8 u8Data;

    uint32 u32NewInPtr;

    tsUartFifo *pasUartFifo = apsUartFifo[u8Uart];

    u8Data = u8AHI_UartReadData(u8Uart);

    u32NewInPtr = pasUartFifo->u32RxInPtr;  /* new in pointer */

    u32NewInPtr++;
    if(u32NewInPtr == UART_RX_BUFFER_LEN){   /* if end of buffer */
        u32NewInPtr = 0;                     /* wrap around to start */
    }

    /*
     * Place the received data into the RX buffer if it is not full.
     * If it is full, nothing we can do other than discard it.
     */
    if(u32NewInPtr != pasUartFifo->u32RxOutPtr){    /* if buffer not full */
        pasUartFifo->u8UartRxBuff[pasUartFifo->u32RxInPtr] = u8Data;    /* put tx byte in buffer */
        pasUartFifo->u32RxInPtr = u32NewInPtr;              /* save new in pointer */
    } else {                                /* buffer is full */
        /* buffer is full, nowhere to store incoming data ! */
    }

}

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/

