/*****************************************************************************
 *
 * MODULE:             JN-AN-xxxx
 *
 * COMPONENT:          ntag.c
 *
 * DESCRIPTION:        NTAG I2C driver
 *                     (NT3H1101 has 1Kbyte flash, NT3H1201 has 2Kbyte flash)
 *
 *****************************************************************************
 *
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN5168, JN5164,
 * JN5161, JN5148, JN5142, JN5139].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2015. All rights reserved
 *
 ****************************************************************************/

/* Doxygen info, do not remove! */
/**
 * @file  ntag.c
 * @brief This file implements the NTAG-I2C driver.
 * \image html SW-LAYERS.JPG
 */

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/

#include "app_config.h"

#ifdef NFC_SUPPORT

#include "ntag.h"
#include "AppHardwareApi.h"
#include "dbg.h"
#include "Timer.h"
#include "I2C_Driver.h"

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

#ifdef DEBUG_NTAG
    #define TRACE_NTAG            TRUE
#else
    #define TRACE_NTAG            FALSE
#endif

#define NTAG_I2C_ADDR             0x55      // 7-bit address

typedef enum {
    NTAG_NC_REG  = 0,
    NTAG_LAST_NDEF_BLOCK,
    NTAG_SRAM_MIRROR_BLOCK,
    NTAG_WDT_LS,
    NTAG_WDT_MS,
    NTAG_I2C_CLOCK_STR,
    NTAG_NS_REG
} ntag_session_reg_t;

#define NC_REG_I2C_RST_ON_OFF     (1 << 7)
#define NC_REG_PHTRU_ON_OFF       (1 << 6)
#define NC_REG_FD_OFF             (3 << 4)
#define NC_REG_FD_ON              (3 << 2)
#define NC_REG_SRAM_MIRROR_ON_OFF (1 << 1)
#define NC_REG_PHTRU_DIR          (1 << 0)

// memory block organization of NTAG I2C IC (seen from I2C interface)
#define NTAG_TAG_HEADER_BYTES     0x00
#define NTAG_USERMEM_START        0x01
#define NTAG_CONFIG_REGISTERS_1K  0x3A
#define NTAG_CONFIG_REGISTERS_2K  0x7A
#define NTAG_SRAM_FIXED_START     0xF8
#define NTAG_SRAM_FIXED_END       0xFB
#define NTAG_SESSION_REGISTERS    0xFE

#if defined(ZoneThermostatV2)
    #define HW_SETTLE_TIME        50        // 50 milli seconds
#else
    #define HW_SETTLE_TIME        30        // 30 milli seconds
#endif

/****************************************************************************/
/***        Global variables                                              ***/
/****************************************************************************/

static bool_t bNtagFound = FALSE;
static bool_t bNtagDataAvailable = TRUE;      // On reset start by reading NTAG

static uint32 rd_block = NTAG_USERMEM_START;  // current NTAG block being read
static uint8  rd_pos;                         // index in rd_buf
static uint8  rd_buf[NTAG_BLOCK_SIZE];

static uint32 wr_block = NTAG_USERMEM_START;  // current NTAG block being written
static uint8  wr_pos;                         // index in wr_buf
static uint8  wr_buf[NTAG_BLOCK_SIZE];

/****************************************************************************/
/***        Local Functions                                               ***/
/****************************************************************************/

static bool bNtagReadBlock(uint8 block, uint8 * pbuf)
{
    bool bRetval = i2c_BusReadData(NTAG_I2C_ADDR, block, NTAG_BLOCK_SIZE, pbuf);
    return bRetval;
}

static bool bNtagWriteBlock(uint8 block, uint8 * pbuf)
{
    uint8 buf[NTAG_BLOCK_SIZE];
    bool  bRetval = i2c_BusReadData(NTAG_I2C_ADDR, block, NTAG_BLOCK_SIZE, buf);
    if (bRetval)
    {
        uint8 *ptmp = pbuf;
        uint32 i;
        for (i=0; i<NTAG_BLOCK_SIZE; i++)
        {
            if (*ptmp++ != buf[i])
            {
                // passed data is different from EEPROM contents
                bRetval = FALSE;
                break;
            }
        }
    }
    if (!bRetval)
    {
        bRetval = i2c_BusWriteReg(NTAG_I2C_ADDR, block, NTAG_BLOCK_SIZE, pbuf);
        vDelayMs(5);  // writing in EEPROM, so give HW some time to settle
    }
    return bRetval;
}

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

/* Doxygen info, do not remove! */
/**
 * @brief   This function checks the availability of the NTAG-I2C, which
 *          is useful when it is an optional module in the device.
 *          In some devices the NTAG IC can be powered down to save energy.
 *          When the NTAG IC is powered on, it is checked whether the
 *          manufacturer ID can be read, and whether the NS register holds
 *          a valid value.
 * @param   cmd - depending on the design of the PCB, the caller can pass
 *          the value NTAG_POWER_OFF, NTAG_POWER_OFF_ON or NTAG_POWER_ON
 * @return  This function returns TRUE when the NTAG is detected and could be
 *          powered on, and FALSE when no NTAG is available or could not be
 *          read out.
 * @note    The NTAG-I2C IC always allows I2C read/write access to the
 *          session registers. Getting exclusive access to the I2C bus
 *          must be done by the caller of this function.
 * @warning THIS FUNCTION MUST NOT BE CALLED BY THE APPLICATION BECAUSE
 *          IT IS CALLED BY FUNCTION bNfcResetNtag() (see nfc.c) ALREADY !!!
 */

bool bNtagPower(ntag_power_cmd_t cmd)
{
    bool bRetval = FALSE;

    // Now check whether we can access the NTAG I2C IC
    if (!bNtagFound)
    {
#ifdef NTAG_ENABLE_PIN
        // In these devices the NTAG I2C chip can be powered down to save energy
        // Now initialize the IO pin which is used for this
        vAHI_DioSetDirection(0, 1 << NTAG_ENABLE_PIN);   // set as output
        vAHI_DioSetPullup(0, 1 << NTAG_ENABLE_PIN);      // disable the pull-up
        vAHI_DioSetOutput(1 << NTAG_ENABLE_PIN, 0);      // drive output HIGH
#endif
        // After power-up of the device, give HW some time to settle
        vDelayMs(HW_SETTLE_TIME);
#if 0
        // The next block of source code is for analyzing purposes
        {
            uint8 addr;
            for (addr=0; addr < 0x80; addr++)
            {
                if (i2c_CheckSlaveAvailable(addr))
                {
                    DBG_vPrintf(TRACE_NTAG, "NTAG: Detected I2C slave at address 0x%x\n", addr);
                }
            }
        }
#endif
#ifdef FIX_NTAG_I2C_ADDR
        // The next block of code can be used to fix the I2C address of the NTAG-I2C IC,
        // unless you know what its current I2C slave address (FIX_NTAG_I2C_ADDR) is.
        {
            uint8 buf[NTAG_BLOCK_SIZE] = {NTAG_I2C_ADDR * 2,    // Restore the factory NTAG I2C address.
                                          0,0,0,0,0,0,          // Serial number bytes can't be changed.
                                          0,0,0,                // Internal data can't be changed.
                                          0,0,                  // Reset the lock bytes to default values.
                                          0xE1,0x10,0xEA,0x00}; // Reset the CC bytes to default values (2k version).
            i2c_BusWriteReg(FIX_NTAG_I2C_ADDR, NTAG_TAG_HEADER_BYTES, NTAG_BLOCK_SIZE, buf);
            vDelayMs(5);  // writing in EEPROM, so give HW some time to settle
        }
#endif
        if (i2c_CheckSlaveAvailable(NTAG_I2C_ADDR))
        {
            DBG_vPrintf(TRACE_NTAG, "NTAG: ==>  NTAG I2C detected  <==\n");
            bNtagFound = TRUE;
        }
        else
        {
            DBG_vPrintf(TRACE_NTAG, "NTAG: ==>  NTAG I2C not detected  <==\n");
        }
    }

    if (bNtagFound)
    {
        if ((NTAG_POWER_OFF == cmd) || (NTAG_POWER_OFF_ON == cmd))
        {
#ifdef NTAG_ENABLE_PIN
            // In these devices the NTAG I2C chip can be powered down to save energy
            DBG_vPrintf(TRACE_NTAG, "NTAG: NTAG_POWER_OFF\n");
            vAHI_DioSetOutput(0, 1 << NTAG_ENABLE_PIN);      // drive output LOW
            // Give HW some time to settle (drain capacitor)
            vDelayMs(HW_SETTLE_TIME);
#endif
        }

        if ((NTAG_POWER_ON == cmd) || (NTAG_POWER_OFF_ON == cmd))
        {
            uint8 i;
            uint8 regval;
            uint8 pbuf[3];
            uint8 buf[NTAG_BLOCK_SIZE];
            bool  bSucces;
            // Try to read the NTAG I2C manufacturer ID
            for (i=0;i<3;i++)
            {
#ifdef NTAG_ENABLE_PIN
                vAHI_DioSetOutput(1 << NTAG_ENABLE_PIN, 0);      // drive output HIGH
                // Give HW some time to settle
                vDelayMs(HW_SETTLE_TIME);
#endif
                bSucces = bNtagReadBlock(NTAG_TAG_HEADER_BYTES, buf);
                // check if SN0 holds the Manufacturer ID for NXP Semiconductors, being value 0x04
                if (bSucces && (0x04 == buf[0]))
                {
                    regval = 0;
                    if (bNtagGetNsReg(&regval))
                    {
                        DBG_vPrintf(TRACE_NTAG, "NTAG: NS_REG=0x%02x after NTAG power cycle %d\n", regval, i);
// nlv10675: commented out the next check, because some older NTAG samples didn't set this status bit
//                        if (regval & NS_REG_I2C_LOCKED)
                        {
                            DBG_vPrintf(TRACE_NTAG, "NTAG: NTAG_POWER_ON\n");
#if 0
                            uint64 u64NtagUID;
                            u64NtagUID  = ((uint64)buf[0] << 48) | ((uint64)buf[1] << 40) | ((uint64)buf[2] << 32);
                            u64NtagUID |= (buf[3] << 24) | (buf[4] << 16) | (buf[5] <<  8) | (buf[6] <<  0);
                            uint16 u16NtagLockBytes = (buf[10] << 8) | (buf[11] << 0);
                            uint32 u32NtagCC = (buf[12] << 24) | (buf[13] << 16) | (buf[14] <<  8) | (buf[15] <<  0);
                            DBG_vPrintf(TRACE_NTAG, "NTAG: UID = 0x%016llx, Lock bytes = 0x%04x, CC = 0x%08x\n",
                                                    u64NtagUID, u16NtagLockBytes, u32NtagCC);
#endif
                            bRetval = TRUE;
                            break;
                        }
                    }
                }
#ifdef NTAG_ENABLE_PIN
                vAHI_DioSetOutput(0, 1 << NTAG_ENABLE_PIN);      // drive output LOW
                // Give HW some time to settle (drain capacitor)
                vDelayMs(HW_SETTLE_TIME);
#endif
            }
            pbuf[0] = NTAG_NS_REG;
            pbuf[1] = NS_REG_I2C_LOCKED;
            pbuf[2] = 0;
            i2c_BusWriteReg(NTAG_I2C_ADDR, NTAG_SESSION_REGISTERS, sizeof(pbuf), pbuf);
        }
    }
    return bRetval;
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function sets the NTAG watch dog, plus the LAST_NDEF_BLOCK
 *          which when read from RF side results in the flag NDEF_DATA_READ
 *          in NS register to be set (this is used to trigger a call of the
 *          function vNdefParser() from vNfcLoop() by this node).
 * @param   u16Wdt - Setting for the watch dog timer (in steps of 9.43usec)
 * @param   end_block - Value to be programmed in register LAST_NDEF_BLOCK
 * @return  This function returns TRUE when NTAG setup is successful, and FALSE
 *          when not successful.
 * @note    The NTAG-I2C IC always allows I2C read/write access to the
 *          session registers. Getting exclusive access to the I2C bus
 *          must be done by the caller of this function.
 * @warning THIS FUNCTION MUST NOT BE CALLED BY THE APPLICATION BECAUSE
 *          IT IS CALLED BY FUNCTION bNfcResetNtag() (see nfc.c) ALREADY !!!
 */

bool bNtagSetup(uint16 u16Wdt, uint8 end_block)
{
    bool bSucces = FALSE;
    if (bNtagFound)
    {
        uint8 u8WdtLs = (u16Wdt >> 0) & 0xFF;
        uint8 u8WdtMs = (u16Wdt >> 8) & 0xFF;
        uint8 pbuf[3];
        // NTAG always allows I2C read/write access to the session registers
        pbuf[0] = NTAG_WDT_LS;
        pbuf[1] = 0xFF;
        pbuf[2] = u8WdtLs;
        if (i2c_BusWriteReg(NTAG_I2C_ADDR, NTAG_SESSION_REGISTERS, sizeof(pbuf), pbuf))
        {
            pbuf[0] = NTAG_WDT_MS;
            pbuf[1] = 0xFF;
            pbuf[2] = u8WdtMs;
            if (i2c_BusWriteReg(NTAG_I2C_ADDR, NTAG_SESSION_REGISTERS, sizeof(pbuf), pbuf))
            {
                // Reset the (volatile) SRAM mirror address session registers
                pbuf[0] = NTAG_SRAM_MIRROR_BLOCK;
                pbuf[1] = 0xFF;
                pbuf[2] = NTAG_SRAM_FIXED_START;
                if (i2c_BusWriteReg(NTAG_I2C_ADDR, NTAG_SESSION_REGISTERS, sizeof(pbuf), pbuf))
                {
                    pbuf[0] = NTAG_NC_REG;
                    pbuf[1] = 0xFF;
                    pbuf[2] = NC_REG_PHTRU_DIR;
                    if (i2c_BusWriteReg(NTAG_I2C_ADDR, NTAG_SESSION_REGISTERS, sizeof(pbuf), pbuf))
                    {
                        pbuf[0] = NTAG_LAST_NDEF_BLOCK;
                        pbuf[1] = 0xFF;
                        pbuf[2] = end_block;
                        if (i2c_BusWriteReg(NTAG_I2C_ADDR, NTAG_SESSION_REGISTERS, sizeof(pbuf), pbuf))
                        {
                            bSucces = TRUE;
                        }
                    }
                }
            }
        }
    }
    return bSucces;
}

/* Doxygen info, do not remove! */
/**
 * @brief   Using the I2C driver, read the NS register of the NTAG-I2C. Since the flag
 *          NS_REG_NDEF_DATA_READ is automatically reset on reading the NS_REG, the bool
 *          bNtagDataAvailable is used to remember that NS_REG_NDEF_DATA_READ was set.
 *          After reading out the NTAGs EEPROM the function vNtagClearNdefDataRead() is
 *          used to clear bool bNtagDataAvailable.
 * @param   pu8regval - Pointer to callers storage to hold the read NS_REG contents
 * @return  This function returns TRUE, plus the current contents of the NS register,
 *          when NS_REG could be read. It returns FALSE when NS_REG couldn't be read.
 * @note    The NTAG-I2C IC always allows I2C read/write access to the
 *          session registers. Getting exclusive access to the I2C bus
 *          must be done by the caller of this function.
 */

bool bNtagGetNsReg(uint8 *pu8regval)
{
    bool bSucces = FALSE;
    uint8 regval = 0xFF;   // default error value
    if (bNtagFound)
    {
        // The actual reading of the NTAG NS_REG.
        // Keep it at only this one place, to be able to keep track of the NDEF_DATA_READ flag.
        // This flag is reset to 0 by the NTAG IC when NS_REG is read.
        bSucces = i2c_ReadNtagReg(NTAG_I2C_ADDR, NTAG_SESSION_REGISTERS, NTAG_NS_REG, &regval);
        if (!bSucces)
        {
            DBG_vPrintf(TRACE_NTAG, "NTAG: Could not read NS_REG\n");
        }
        else
        {
#if 0
            //---------------------------------------
            // Print debug output of the NS_REG flags
            //---------------------------------------
            if (regval & NS_REG_NDEF_DATA_READ)
            {
                DBG_vPrintf(TRACE_NTAG, "NTAG: NDEF_DATA_READ\n");
            }
            if (regval & NS_REG_I2C_LOCKED)
            {
                DBG_vPrintf(TRACE_NTAG, "NTAG: I2C_LOCKED\n");
            }
            if (regval & NS_REG_RF_LOCKED)
            {
                DBG_vPrintf(TRACE_NTAG, "NTAG: RF_LOCKED\n");
            }
            if (regval & NS_REG_SRAM_I2C_READY)
            {
                DBG_vPrintf(TRACE_NTAG, "NTAG: SRAM_I2C_READY\n");
            }
            if (regval & NS_REG_SRAM_RF_READY)
            {
                DBG_vPrintf(TRACE_NTAG, "NTAG: SRAM_RF_READY\n");
            }
            if (regval & NS_REG_EEPROM_WR_ERR)
            {
                DBG_vPrintf(TRACE_NTAG, "NTAG: EEPROM_WR_ERR\n");
            }
            if (regval & NS_REG_EEPROM_WR_BUSY)
            {
                DBG_vPrintf(TRACE_NTAG, "NTAG: EEPROM_WR_BUSY\n");
            }
            if (regval & NS_REG_RF_FIELD_PRESENT)
            {
                DBG_vPrintf(TRACE_NTAG, "NTAG: RF_FIELD_PRESENT\n");
            }
#endif
            if (regval & NS_REG_NDEF_DATA_READ)
            {
                // Remember that this flag was set,
                // because it is cleared by this read of the NS_REG.
                bNtagDataAvailable = TRUE;
            }
            else if (bNtagDataAvailable)
            {
                regval |= NS_REG_NDEF_DATA_READ;
            }
        }
    }
    *pu8regval = regval;
    return bSucces;
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function clears the bNtagDataAvailable boolean.
 * @return  None.
 */

void vNtagClearNdefDataRead(void)
{
    bNtagDataAvailable = FALSE;
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function can be used to try to set the I2C_LOCKED bit,
 *          which means that the NTAG EEPROM is locked for read/write access
 *          from the I2C side.
 *          The function bNtagReleaseI2cLock() can be used to release the
 *          lock from the I2C side on the NTAG EEPROM.
 * @return  This function returns TRUE when the lock on the EEPROM by I2C
 *          side was established, and FALSE when EEPROM couldn't be locked
 *          by I2C side.
 * @note    Getting exclusive access to the I2C bus must be done by the
 *          caller of this function.
 */

bool bNtagTry2SetI2cLock(void)
{
    bool bSucces = FALSE;
    if (bNtagFound)
    {
        uint8 regval = 0xFF;
        if (bNtagGetNsReg(&regval))
        {
            if (0 == (regval & NS_REG_RF_LOCKED))
            {
                if (NS_REG_I2C_LOCKED == (regval & NS_REG_I2C_LOCKED))
                {
                    bSucces = TRUE;
                }
                else
                {
                    uint8 buf[NTAG_BLOCK_SIZE];
                    if (bNtagReadBlock(NTAG_USERMEM_START, buf))
                    {
                        if (bNtagGetNsReg(&regval))
                        {
                            if (NS_REG_I2C_LOCKED == (regval & NS_REG_I2C_LOCKED))
                            {
                                bSucces = TRUE;
                            }
                        }
                    }
                }
            }
        }
    }
    return bSucces;
}

/* Doxygen info, do not remove! */
/**
 * @brief   Set the reader to u16Offset of the start of the stream.
 * @param   u16Offset - the offset in bytes from the start of the user
 *          memory of the NTAG
 * @return  This function returns TRUE when successful, and FALSE if not.
 * @note    The NTAG-I2C IC only allows I2C read/write access to the
 *          EEPROM when it is not locked from the RF side. This right
 *          to access the EEPROM is not acquired in this function.
 *          Getting exclusive access to the I2C bus must be done by the
 *          caller of this function.
 */

bool bNtagStartRead(uint16 u16Offset)
{
    bool bSucces = FALSE;
    if (bNtagFound)
    {
        rd_block = NTAG_USERMEM_START;
        rd_pos   = 0;
        while (u16Offset >= NTAG_BLOCK_SIZE)
        {
            rd_block += 1;
            u16Offset -= NTAG_BLOCK_SIZE;
        }
        if (0 == u16Offset)
        {
            // block will be read by u32NtagRead()
            bSucces = TRUE;
        }
        else
        {
            bSucces = bNtagReadBlock(rd_block, rd_buf);
            if (bSucces)
            {
                rd_pos = (uint8)u16Offset;
            }
        }
    }
    return bSucces;
}

/* Doxygen info, do not remove! */
/**
 * @brief   Read or Skip the requested bytes from the NTAG-I2C IC.
 *          When pu8Buffer is equal to NULL, requested bytes will be
 *          skipped in the byte stream.
 * @param   pu8Buffer - Pointer to callers buffer in which read data
 *          will be returned
 * @param   u32Nbyte - Number of bytes to read
 * @return  This function returns the number of bytes read.
 * @note    The NTAG-I2C IC only allows I2C read/write access to the
 *          EEPROM when it is not locked from the RF side. This right
 *          to access the EEPROM is not acquired in this function.
 *          Getting exclusive access to the I2C bus must be done by the
 *          caller of this function.
 */

uint32 u32NtagRead(uint8 *pu8Buffer, uint32 u32Nbyte)
{
    uint32 u32BytesRd = 0;
    bool   bSucces = TRUE;

    if (pu8Buffer)
    {
        while (bSucces && (u32BytesRd < u32Nbyte))
        {
            if (0 == rd_pos)
            {
                bSucces = bNtagReadBlock(rd_block, rd_buf);
            }
            if (bSucces)
            {
                *pu8Buffer++ = rd_buf[rd_pos++];
                u32BytesRd += 1;
                if (NTAG_BLOCK_SIZE <= rd_pos)
                {
                    rd_block += 1;
                    rd_pos    = 0;
                }
            }
        }
    }
    else
    {
        uint32 u32NSkipByte = u32Nbyte;
        while (u32NSkipByte >= (NTAG_BLOCK_SIZE - rd_pos))
        {
            u32NSkipByte -= (NTAG_BLOCK_SIZE - rd_pos);
            rd_pos    = 0;
            rd_block += 1;
        }
        rd_pos += u32NSkipByte;
        bSucces = bNtagReadBlock(rd_block, rd_buf);
        if (bSucces)
        {
            u32BytesRd = u32Nbyte;
        }
    }
    return u32BytesRd;
}

/* Doxygen info, do not remove! */
/**
 * @brief   Set the writer to u16Offset of the start of the stream.
 * @param   u16Offset - the offset in bytes from the start of the user
 *          memory of the NTAG
 * @return  This function returns TRUE when successful, and FALSE if not.
 * @note    The NTAG-I2C IC only allows I2C read/write access to the
 *          EEPROM when it is not locked from the RF side. This right
 *          to access the EEPROM is not acquired in this function.
 *          Getting exclusive access to the I2C bus must be done by the
 *          caller of this function.
 */

bool bNtagStartWrite(uint16 u16Offset)
{
    bool bSucces = FALSE;
    if (bNtagFound)
    {
        wr_block = NTAG_USERMEM_START;
        wr_pos   = 0;
        while (u16Offset >= NTAG_BLOCK_SIZE)
        {
            wr_block += 1;
            u16Offset -= NTAG_BLOCK_SIZE;
        }
        if (0 == u16Offset)
        {
            // block will be written from the start by u32NtagRead()
            // so no need to read this block from NTAG
            bSucces = TRUE;
        }
        else
        {
            bSucces = bNtagReadBlock(wr_block, wr_buf);
            if (bSucces)
            {
                wr_pos = (uint8)u16Offset;
            }
        }
    }
    return bSucces;
}

/* Doxygen info, do not remove! */
/**
 * @brief   Write the passed bytes to the NTAG-I2C IC.
 * @param   pu8Buffer - Pointer to buffer that holds the data to be
 *          written
 * @param   u32Nbyte - Number of bytes to write
 * @return  This function returns the number of bytes written.
 * @note    The NTAG-I2C IC only allows I2C read/write access to the
 *          EEPROM when it is not locked from the RF side. This right
 *          to access the EEPROM is not acquired in this function.
 *          Getting exclusive access to the I2C bus must be done by the
 *          caller of this function.
 */

uint32 u32NtagWrite(uint8 *pu8Buffer, uint32 u32Nbyte)
{
    uint32 u32BytesWr = 0;
    bool   bSucces = TRUE;

    while (bSucces && (u32BytesWr < u32Nbyte))
    {
        if (NTAG_BLOCK_SIZE > wr_pos)
        {
            wr_buf[wr_pos++] = *pu8Buffer++;
            u32BytesWr += 1;
        }
        if (NTAG_BLOCK_SIZE <= wr_pos)
        {
            bSucces = bNtagWriteBlock(wr_block, wr_buf);
            if (bSucces)
            {
                wr_block += 1;
                wr_pos    = 0;
            }
        }
    }
    return u32BytesWr;
}

/* Doxygen info, do not remove! */
/**
 * @brief   Writes the remaining bytes from the wr_buf to the NTAG-I2C IC.
 * @return  This function returns TRUE when the remaining bytes are written to
 *          NTAG, and FALSE when it failed to write remaining bytes to NTAG.
 * @note    The NTAG-I2C IC only allows I2C read/write access to the
 *          EEPROM when it is not locked from the RF side. This right
 *          to access the EEPROM is not acquired in this function.
 *          Getting exclusive access to the I2C bus must be done by the
 *          caller of this function.
 */

bool bNtagFlush(void)
{
    bool bSucces = TRUE;
    if (wr_pos > 0)
    {
        uint8 buf[NTAG_BLOCK_SIZE];
        bSucces = bNtagReadBlock(wr_block, buf);
        if (bSucces)
        {
            uint8 i;
            for (i=0; i<wr_pos; i++)
            {
                buf[i] = wr_buf[i];
            }
            bSucces = bNtagWriteBlock(wr_block, buf);
            wr_pos = 0;
        }
    }
    return bSucces;
}

/* Doxygen info, do not remove! */
/**
 * @brief   Instruct the NTAG to release the EEPROM lock by I2C side.
 * @return  This function returns TRUE when the NTAG has released the EEPROM
 *          lock by I2C side, and FALSE if it fails to release the lock.
 * @note    Getting exclusive access to the I2C bus must be done by the
 *          caller of this function.
 */

bool bNtagReleaseI2cLock(void)
{
    bool  bSucces = FALSE;
    uint8 regval = 0xFF;
    uint8 pbuf[3];

    if (bNtagGetNsReg(&regval))
    {
        if (NS_REG_I2C_LOCKED == (regval & NS_REG_I2C_LOCKED))
        {
            pbuf[0] = NTAG_NS_REG;
            pbuf[1] = NS_REG_I2C_LOCKED;
            pbuf[2] = 0;
            if (i2c_BusWriteReg(NTAG_I2C_ADDR, NTAG_SESSION_REGISTERS, sizeof(pbuf), pbuf))
            {
                if (bNtagGetNsReg(&regval))
                {
                    if (0 == (regval & NS_REG_I2C_LOCKED))
                    {
                        bSucces = TRUE;
                    }
                }
            }
        }
        else
        {
            bSucces = TRUE;
        }
    }
    return bSucces;
}

#endif  // NFC_SUPPORT

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
