/*****************************************************************************
 *
 * MODULE:             JN-AN-xxxx
 *
 * COMPONENT:          ndef.c
 *
 * DESCRIPTION:        NDEF layer driver
 *
 *****************************************************************************
 *
 * This software is owned by NXP B.V. and/or its supplier and is protected
 * under applicable copyright laws. All rights are reserved. We grant You,
 * and any third parties, a license to use this software solely and
 * exclusively on NXP products [NXP Microcontrollers such as JN5168, JN5164,
 * JN5161, JN5148, JN5142, JN5139].
 * You, and any third parties must reproduce the copyright and warranty notice
 * and any other legend of ownership on each copy or partial copy of the
 * software.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright NXP B.V. 2015. All rights reserved
 *
 ****************************************************************************/

/* Doxygen info, do not remove! */
/**
 * @file  ndef.c
 * @brief This file implements the NDEF layer driver. Functions in this layer
 *        can be used by the application(s) to read/write in the NTAG. To be
 *        able to do so, each application must register its own unique NDEF
 *        type and use the lock and unlock API functions of this layer to
 *        get/release exclusive access to the NTAG EEPROM. This layer will use
 *        the callback function to notify the application of possible
 *        modifications in the NDEF message in the NTAG after a NFC reader
 *        (eg. smart phone) has triggered the NDEF_DATA_READ flag of the NTAG.
 * \image html SW-LAYERS.JPG
 */

/****************************************************************************/
/***        Include files                                                 ***/
/****************************************************************************/

#include "app_config.h"

#ifdef NFC_SUPPORT

#include <string.h>
#include "dbg.h"
#include "ndef.h"
#include "ntag.h"
#include "I2C_Driver.h"
#ifdef NFC_COMMISSIONING
#include "nsc.h"
#endif  // NFC_COMMISSIONING

/****************************************************************************/
/***        Type Definitions                                              ***/
/****************************************************************************/

#define TLV_TYPE_NULL           0x00
#define TLV_TYPE_LOCK_CTRL      0x01
#define TLV_TYPE_MEM_CTRL       0x02
#define TLV_TYPE_NDEF_MSG       0x03
#define TLV_TYPE_PROPRIETARY    0xFD
#define TLV_TYPE_TERMINATOR     0xFE

#define NDEF_REC_HDR_MASK_MB    (1 << 7)
#define NDEF_REC_HDR_MASK_ME    (1 << 6)
#define NDEF_REC_HDR_MASK_CF    (1 << 5)
#define NDEF_REC_HDR_MASK_SR    (1 << 4)
#define NDEF_REC_HDR_MASK_IL    (1 << 3)
#define NDEF_REC_HDR_MASK_TNF   (0x07)

// The next define gives the maximum fixed NDEF header length
//   max 1 byte  for T
//   max 3 bytes for L
//   max 1 byte  for NDEF flags
//   max 1 byte  for NDEF type length field
//   max 4 bytes for NDEF payload length field
//   max 1 byte  for NDEF ID length field
#define NDEF_MAX_FIXED_LEN       (1 + 3 + 1 + 1 + 4 + 1)

typedef struct ndef_msg_admin_struct
{
    uint8 *pu8NdefType;
    uint8  u8NdefTypeLength;
    uint8 *pu8NdefId;
    uint8  u8NdefIdLength;
    uint32 u32NdefPayloadLength;
    void(*pCallbackFunc)(ndef_rec_info_t *);
    uint16 u16StartOfTlv;
    uint16 u16StartOfNdefPayload;
} ndef_msg_admin_t;

/****************************************************************************/
/***        Global variables                                              ***/
/****************************************************************************/

#ifdef DEBUG_NDEF
static bool_t           bTraceNdef  = TRUE;
#endif
static bool_t           bNtagLocked = FALSE;
static uint8           *pNdefMsg = NULL;

static uint8            u8ListIndex = 0xFF; // 0xFF=callback function not found
static uint8            u8NrOfRegisteredMsgs = 0;
static ndef_rec_info_t  NdefRecInfo;
static uint8            au8NdefId[CONFIG_NDEF_MAX_ID_LENGTH];

#ifndef NFC_COMMISSIONING

static ndef_msg_admin_t MsgList[CONFIG_NUMBER_OF_APP_NDEF_MSGS];
static uint8            au8NdefType[CONFIG_NDEF_MAX_TYPE_LENGTH];
static uint8            au8NdefHeader[NDEF_MAX_FIXED_LEN + CONFIG_NDEF_MAX_ID_LENGTH + CONFIG_NDEF_MAX_TYPE_LENGTH];

#else  // NFC_COMMISSIONING

static ndef_msg_admin_t MsgList[1 + CONFIG_NUMBER_OF_APP_NDEF_MSGS];
  #if (CONFIG_NDEF_MAX_TYPE_LENGTH > NSC_NDEF_TYPE_LENGTH)
static uint8            au8NdefType[CONFIG_NDEF_MAX_TYPE_LENGTH];
static uint8            au8NdefHeader[NDEF_MAX_FIXED_LEN + CONFIG_NDEF_MAX_ID_LENGTH + CONFIG_NDEF_MAX_TYPE_LENGTH];
  #else
static uint8            au8NdefType[NSC_NDEF_TYPE_LENGTH];
static uint8            au8NdefHeader[NDEF_MAX_FIXED_LEN + CONFIG_NDEF_MAX_ID_LENGTH + NSC_NDEF_TYPE_LENGTH];
  #endif

#endif  // NFC_COMMISSIONING

/****************************************************************************/
/***        Local Functions                                               ***/
/****************************************************************************/

static uint16 u16NdefGetTlvLength(uint16 * pu16Offset)
{
    uint16 u16TlvLength = 0;
    uint8  u8Byte;
    if (u32NtagRead(&u8Byte, sizeof(u8Byte)) == sizeof(u8Byte))
    {
        *pu16Offset += sizeof(u8Byte);
        if (0xFF != u8Byte)
        {
            // The TLV length field is one byte long
            // The TLV value field is from 0x00 till 0xFE bytes long
            u16TlvLength = u8Byte;
        }
        else
        {
            // The TLV length field is three bytes long
            // The TLV value field is from 0x00FF till 0xFFFE bytes long
            uint8 au8Bytes[2];
            if (u32NtagRead(au8Bytes, sizeof(au8Bytes)) == sizeof(au8Bytes))
            {
                *pu16Offset += sizeof(au8Bytes);
                u16TlvLength = ((uint16)au8Bytes[0] << 8) | au8Bytes[1];
            }
        }
    }
    return u16TlvLength;
}

static bool bNdefCheckTlvLength(uint16 * pu16TlvLength, uint16 * pu16Offset, uint16 u16Length)
{
    bool bTlvLengthOK = FALSE;
    if (*pu16TlvLength >= u16Length)
    {
        *pu16TlvLength -= u16Length;
        *pu16Offset    += u16Length;
        bTlvLengthOK    = TRUE;
    }
    return bTlvLengthOK;
}

static uint16 u16NdefCheckTlvType(uint16 * pu16Offset)
{
    uint16 u16TlvLength = 0;
    bool   bExitLoop  = FALSE;
    while (!bExitLoop)
    {
        uint8 u8TlvType = 0;
        if (u32NtagRead(&u8TlvType, sizeof(u8TlvType)) == sizeof(u8TlvType))
        {
            *pu16Offset += sizeof(u8TlvType);
            switch (u8TlvType)
            {
            case TLV_TYPE_NULL:
#ifdef DEBUG_NDEF
                DBG_vPrintf(bTraceNdef, "NDEF: TLV (T=NULL TLV)\n");
#endif
                // skip this TLV type field since it is NULL
                break;
#if 0
            case TLV_TYPE_LOCK_CTRL:
            case TLV_TYPE_MEM_CTRL:
#ifdef DEBUG_NDEF
                DBG_vPrintf(bTraceNdef, "NDEF: TLV (T=LOCK or MEM CTRL)\n");
#endif
                u16TlvLength = u16NdefGetTlvLength(pu16Offset);
#ifdef DEBUG_NDEF
                DBG_vPrintf(bTraceNdef, "NDEF: TLV (L=0x%04x)\n", u16TlvLength);
#endif
                if (3 != u16TlvLength)
                {
#ifdef DEBUG_NDEF
                    DBG_vPrintf(bTraceNdef, "NDEF: NFC ERROR - V Length should be 3 bytes\n");
#endif
                }
                // skip the following 'u16TlvLength' bytes (will not be interpreted)
                while (u16TlvLength)
                {
                    uint8 u8Byte;
                    if (u32NtagRead(&u8Byte, sizeof(u8Byte)) == sizeof(u8Byte))
                    {
                        *pu16Offset += sizeof(u8Byte);
                        u16TlvLength -= 1;
                    }
                }
                break;
#endif
            case TLV_TYPE_NDEF_MSG:
#ifdef DEBUG_NDEF
                DBG_vPrintf(bTraceNdef, "NDEF: TLV (T=NDEF msg)\n");
#endif
                u16TlvLength = u16NdefGetTlvLength(pu16Offset);
#ifdef DEBUG_NDEF
                DBG_vPrintf(bTraceNdef, "NDEF: TLV (L=0x%04x)\n", u16TlvLength);
#endif
                // check for empty NDEF message
                if (u16TlvLength > 0)
                {
                    // Yes, none empty NDEF message, start parsing from here
                    bExitLoop = TRUE;
                }
                break;
            case TLV_TYPE_PROPRIETARY:
#ifdef DEBUG_NDEF
                DBG_vPrintf(bTraceNdef, "NDEF: TLV (T=Proprietary data)\n");
#endif
                u16TlvLength = u16NdefGetTlvLength(pu16Offset);
#ifdef DEBUG_NDEF
                DBG_vPrintf(bTraceNdef, "NDEF: TLV (L=0x%04x)\n", u16TlvLength);
#endif
                // skip the proprietary value field of the TLV
                while (u16TlvLength)
                {
                    uint8 u8Byte;
                    if (u32NtagRead(&u8Byte, sizeof(u8Byte)) == sizeof(u8Byte))
                    {
                        *pu16Offset += sizeof(u8Byte);
                        u16TlvLength -= 1;
                    }
                    else
                    {
                        u16TlvLength = 0;
                        // Read error, stop further processing of the data in the NTAG
                        bExitLoop = TRUE;
                    }
                }
                break;
            case TLV_TYPE_TERMINATOR:
#ifdef DEBUG_NDEF
                DBG_vPrintf(bTraceNdef, "NDEF: TLV (T=Terminator)\n");
#endif
                // Terminator reached
                bExitLoop = TRUE;
                break;
            default:
#ifdef DEBUG_NDEF
                DBG_vPrintf(bTraceNdef, "NDEF: TLV (T=0x%02x - unknown)\n", u8TlvType);
#endif
                // Unknown TLV type, stop further processing of the data in the NTAG
                bExitLoop = TRUE;
                break;
            }
        }
        else
        {
            // Read error, stop further processing of the data in the NTAG
            bExitLoop = TRUE;
        }
    }
    return u16TlvLength;
}

static bool bNdefProcessRecord(uint16 * pu16TlvLength, uint16 * pu16Offset)
{
    bool   bRecordOK = FALSE;
    uint8  i;
    uint8  u8Byte;
    uint8  u8NdefRecHdr = 0;
    uint8  u8NdefTypeLen = 0;
    uint32 u32NdefPayloadLen = 0;
    uint16 u16FieldLen;

    u8ListIndex = 0xFF; // 0xFF=callback function not found
    memset(&NdefRecInfo, 0, sizeof(ndef_rec_info_t));
    memset(au8NdefType, 0, sizeof(au8NdefType));
    memset(au8NdefId, 0, sizeof(au8NdefId));
    // Read NDEF record header
    u16FieldLen = sizeof(u8NdefRecHdr);
    if (u32NtagRead(&u8NdefRecHdr, u16FieldLen) == u16FieldLen)
    {
        bRecordOK = bNdefCheckTlvLength(pu16TlvLength, pu16Offset, u16FieldLen);
        // Read NDEF type length field
        u16FieldLen = sizeof(u8Byte);
        if (bRecordOK && (u32NtagRead(&u8Byte, u16FieldLen) == u16FieldLen))
        {
            bRecordOK = bNdefCheckTlvLength(pu16TlvLength, pu16Offset, u16FieldLen);
            u8NdefTypeLen = u8Byte;
        }
        else
        {
            bRecordOK = FALSE;
        }
        if (NDEF_REC_HDR_MASK_MB == (u8NdefRecHdr & NDEF_REC_HDR_MASK_MB))
        {
            NdefRecInfo.bNdefMsgBegin = TRUE;     // Begin of NDEF message
        }
        if (NDEF_REC_HDR_MASK_ME == (u8NdefRecHdr & NDEF_REC_HDR_MASK_ME))
        {
            NdefRecInfo.bNdefMsgEnd = TRUE;       // End of NDEF message
        }
        if (NDEF_REC_HDR_MASK_CF == (u8NdefRecHdr & NDEF_REC_HDR_MASK_CF))
        {
            NdefRecInfo.bNdefChunkFlag = TRUE;    // First or middle chunk of NDEF record
        }
        NdefRecInfo.u8NdefTnf = u8NdefRecHdr & NDEF_REC_HDR_MASK_TNF;
        if (NDEF_REC_HDR_MASK_SR == (u8NdefRecHdr & NDEF_REC_HDR_MASK_SR))
        {
            // Read payload length field - Short NDEF record (max 255 bytes)
            u16FieldLen = sizeof(u8Byte);
            if (bRecordOK && (u32NtagRead(&u8Byte, u16FieldLen) == u16FieldLen))
            {
                bRecordOK = bNdefCheckTlvLength(pu16TlvLength, pu16Offset, u16FieldLen);
                u32NdefPayloadLen = u8Byte;
            }
            else
            {
                bRecordOK = FALSE;
            }
        }
        else
        {
            // Read payload length field - Long NDEF record (max 2^32 bytes)
            uint8 au8Bytes[4];
            u16FieldLen = sizeof(au8Bytes);
            if (bRecordOK && (u32NtagRead(au8Bytes, u16FieldLen) == u16FieldLen))
            {
                bRecordOK = bNdefCheckTlvLength(pu16TlvLength, pu16Offset, u16FieldLen);
                u32NdefPayloadLen = (au8Bytes[0] << 24) |
                                    (au8Bytes[1] << 16) |
                                    (au8Bytes[2] <<  8) |
                                    (au8Bytes[3] <<  0);
            }
            else
            {
                bRecordOK = FALSE;
            }
        }
        if (NDEF_REC_HDR_MASK_IL == (u8NdefRecHdr & NDEF_REC_HDR_MASK_IL))
        {
            // Read ID_LENGTH field because it is present
            u16FieldLen = sizeof(u8Byte);
            if (bRecordOK && (u32NtagRead(&u8Byte, u16FieldLen) == u16FieldLen))
            {
                bRecordOK = bNdefCheckTlvLength(pu16TlvLength, pu16Offset, u16FieldLen);
                NdefRecInfo.u8NdefIdLen = u8Byte;
            }
            else
            {
                bRecordOK = FALSE;
            }
        }
        if (u8NdefTypeLen > 0)
        {
            // Read the type field from the NDEF record
            if (u8NdefTypeLen <= sizeof(au8NdefType))
            {
                u16FieldLen = u8NdefTypeLen;
                if (bRecordOK && (u32NtagRead(au8NdefType, u16FieldLen) == u16FieldLen))
                {
                    bRecordOK = bNdefCheckTlvLength(pu16TlvLength, pu16Offset, u16FieldLen);
                }
                else
                {
                    bRecordOK = FALSE;
                }
                // Find out which callback function is interested in this NDEF message
                if (bRecordOK)
                {
                    for (i=0; i<u8NrOfRegisteredMsgs; i++)
                    {
                        if (MsgList[i].u8NdefTypeLength == u8NdefTypeLen)
                        {
                            if (0 == memcmp(MsgList[i].pu8NdefType, au8NdefType, u8NdefTypeLen))
                            {
                                u8ListIndex = i;
                                break;
                            }
                        }
                    }
                }
            }
            else
            {
                // Error: NDEF type field is longer than max configured!
                u16FieldLen = u8NdefTypeLen;
                if (bRecordOK && (u32NtagRead(NULL, u16FieldLen) == u16FieldLen))
                {
                    bRecordOK = bNdefCheckTlvLength(pu16TlvLength, pu16Offset, u16FieldLen);
                }
                else
                {
                    bRecordOK = FALSE;
                }
            }
        }
        if (NdefRecInfo.u8NdefIdLen > 0)
        {
            // Get the ID field (max 255 bytes) from the NDEF record
            if (NdefRecInfo.u8NdefIdLen <= sizeof(au8NdefId))
            {
                u16FieldLen = NdefRecInfo.u8NdefIdLen;
                if (bRecordOK && (u32NtagRead(au8NdefId, u16FieldLen) == u16FieldLen))
                {
                    bRecordOK = bNdefCheckTlvLength(pu16TlvLength, pu16Offset, u16FieldLen);
                }
                else
                {
                    bRecordOK = FALSE;
                }
            }
            else
            {
                // Remark: NDEF ID field is longer than max configured!
                u16FieldLen = sizeof(au8NdefId);
                if (bRecordOK && (u32NtagRead(au8NdefId, u16FieldLen) == u16FieldLen))
                {
                    bRecordOK = bNdefCheckTlvLength(pu16TlvLength, pu16Offset, u16FieldLen);
                }
                else
                {
                    bRecordOK = FALSE;
                }
                u16FieldLen = NdefRecInfo.u8NdefIdLen - sizeof(au8NdefId);
                if (bRecordOK && (u32NtagRead(NULL, u16FieldLen) == u16FieldLen))
                {
                    bRecordOK = bNdefCheckTlvLength(pu16TlvLength, pu16Offset, u16FieldLen);
                }
                else
                {
                    bRecordOK = FALSE;
                }
                NdefRecInfo.u8NdefIdLen = sizeof(au8NdefId);  // truncate the ID length
            }
            NdefRecInfo.pu8NdefId = au8NdefId;
        }
        if (u32NdefPayloadLen > 0)
        {
            // Get the payload from the NDEF record
            if ((0xFF != u8ListIndex) &&
                (u32NdefPayloadLen == MsgList[u8ListIndex].u32NdefPayloadLength))
            {
                u16FieldLen = u32NdefPayloadLen;
                bRecordOK = bNdefCheckTlvLength(pu16TlvLength, pu16Offset, u16FieldLen);
            }
            else
            {
                // Skip the payload from this NDEF record
                u16FieldLen = u32NdefPayloadLen;
                if (bRecordOK && (u32NtagRead(NULL, u16FieldLen) == u16FieldLen))
                {
                    bRecordOK = bNdefCheckTlvLength(pu16TlvLength, pu16Offset, u16FieldLen);
                }
            }
        }
    }
    return bRecordOK;
}

static uint16 u16NdefBuildHdr(bool bWriteHdr, uint8 u8Index)
{
    uint16 u16Length = 0;
    uint16 u16TlvValueLength;
    uint16 i;
    uint8  u8NdefFlags;
    uint8 *pu8Tmp;

    /* Write T field (1 byte) */
    /* ---------------------- */
    if (bWriteHdr) au8NdefHeader[u16Length] = TLV_TYPE_NDEF_MSG;
    u16Length++;
    /* Determine Length of TLV Value field */
    /* ----------------------------------- */
    u16TlvValueLength = 2;        // 1 byte NDEF header + 1 byte NDEF type length field
    if (MsgList[u8Index].u32NdefPayloadLength <= 0xFF)
    {
        u16TlvValueLength += 1;   // 1 byte payload length field
    }
    else
    {
        u16TlvValueLength += 4;   // 4 bytes payload length field
    }
    if (MsgList[u8Index].u8NdefIdLength > 0)
    {
        u16TlvValueLength += 1;   // 1 byte ID length field
    }
    u16TlvValueLength += MsgList[u8Index].u8NdefTypeLength +             // x bytes type field
                         MsgList[u8Index].u8NdefIdLength +               // x bytes ID field
                         (uint16)MsgList[u8Index].u32NdefPayloadLength;  // x bytes payload field
    /* Write L field (1 or 3 bytes) */
    /* ---------------------------- */
    if (u16TlvValueLength < 0xFF)
    {
        /* L field (1 byte) */
        if (bWriteHdr) au8NdefHeader[u16Length] = (uint8)u16TlvValueLength;      // Length in TLV
        u16Length++;
    }
    else
    {
        /* L field (3 bytes) */
        if (bWriteHdr) au8NdefHeader[u16Length] = 0xFF;    // Write 0xFF because we are using the three byte L format
        u16Length++;
        if (bWriteHdr) au8NdefHeader[u16Length] = (uint8)((u16TlvValueLength >> 8) & 0xFF);      // Length in TLV msb
        u16Length++;
        if (bWriteHdr) au8NdefHeader[u16Length] = (uint8)((u16TlvValueLength >> 0) & 0xFF);      // Length in TLV lsb
        u16Length++;
    }
    /* Write V field (value field follows) */
    /* ----------------------------------- */
    u8NdefFlags = 0xC4;         // set flags: MB=1,ME=1,CF=0,SR=0,IL=0,TNF=100
    if (MsgList[u8Index].u32NdefPayloadLength <= 0xFF)
    {
        u8NdefFlags |= 0x10;    // set flag SR=1
    }
    if (MsgList[u8Index].u8NdefIdLength > 0)
    {
        u8NdefFlags |= 0x08;    // set flag IL=1
    }
    // Write NDEF Flags byte
    if (bWriteHdr) au8NdefHeader[u16Length] = u8NdefFlags;
    u16Length++;
    // Write NDEF type length field
    if (bWriteHdr) au8NdefHeader[u16Length] = MsgList[u8Index].u8NdefTypeLength;
    u16Length++;
    // Write NDEF payload length field
    if (MsgList[u8Index].u32NdefPayloadLength <= 0xFF)
    {
        if (bWriteHdr) au8NdefHeader[u16Length] = (uint8)MsgList[u8Index].u32NdefPayloadLength;  // ndef payload length
        u16Length++;
    }
    else
    {
        if (bWriteHdr) au8NdefHeader[u16Length] = 0;       // zero, because payload length < 2^16
        u16Length++;
        if (bWriteHdr) au8NdefHeader[u16Length] = 0;       // zero, because payload length < 2^16
        u16Length++;
        if (bWriteHdr) au8NdefHeader[u16Length] = (uint8)(MsgList[u8Index].u32NdefPayloadLength >> 8);  // ndef payload length msb
        u16Length++;
        if (bWriteHdr) au8NdefHeader[u16Length] = (uint8)(MsgList[u8Index].u32NdefPayloadLength >> 0);  // ndef payload length lsb
        u16Length++;
    }
    // Write NDEF ID length field
    if (MsgList[u8Index].u8NdefIdLength > 0)
    {
        if (bWriteHdr) au8NdefHeader[u16Length] = MsgList[u8Index].u8NdefIdLength;  // ndef id length
        u16Length++;
    }
    // Write NDEF type field
    pu8Tmp = MsgList[u8Index].pu8NdefType;
    for (i=0; i < MsgList[u8Index].u8NdefTypeLength; i++)
    {
        if (bWriteHdr) au8NdefHeader[u16Length] = *pu8Tmp++;
        u16Length++;
    }
    // Write NDEF ID field
    pu8Tmp = MsgList[u8Index].pu8NdefId;
    for (i=0; i < MsgList[u8Index].u8NdefIdLength; i++)
    {
        if (bWriteHdr) au8NdefHeader[u16Length] = *pu8Tmp++;
        u16Length++;
    }
    return u16Length;
}

static bool bNdefWriteHdr(uint8 u8Index)
{
    bool   bSucces = FALSE;
    uint16 u16HdrLength = u16NdefBuildHdr(TRUE, u8Index);
    if (bNtagStartWrite(MsgList[u8Index].u16StartOfTlv))
    {
        if (u32NtagWrite(au8NdefHeader, u16HdrLength) == u16HdrLength)
        {
            if (bNtagFlush())
            {
                bSucces = TRUE;
            }
            else
            {
#ifdef DEBUG_NDEF
                DBG_vPrintf(bTraceNdef, "NDEF: bNdefWriteHdr - Error flush\n");
#endif
            }
        }
    }
    return bSucces;
}

static bool bNdefWriteTerminator(void)
{
    bool bSucces = FALSE;
    uint16 u16Offset = 0;
    if (u8NrOfRegisteredMsgs > 0)
    {
        uint8 index = u8NrOfRegisteredMsgs - 1;
        u16Offset = MsgList[index].u16StartOfNdefPayload + MsgList[index].u32NdefPayloadLength;
    }
    if (bNtagStartWrite(u16Offset))
    {
        uint8 terminator = TLV_TYPE_TERMINATOR;
        if (u32NtagWrite(&terminator, sizeof(terminator)) == sizeof(terminator))
        {
            if (bNtagFlush())
            {
                bSucces = TRUE;
            }
            else
            {
#ifdef DEBUG_NDEF
                DBG_vPrintf(bTraceNdef, "NDEF: vNdefWriteTerminator - Error flush\n");
#endif
            }
        }
        else
        {
#ifdef DEBUG_NDEF
            DBG_vPrintf(bTraceNdef, "NDEF: vNdefWriteTerminator - Error writing TLV TERMINATOR\n");
#endif
        }
    }
    return bSucces;
}

/****************************************************************************/
/***        Exported Functions                                            ***/
/****************************************************************************/

/* Doxygen info, do not remove! */
/**
 * @brief   This function tries to register the NDEF message with the specified
 *          NDEF payload type, optional NDEF payload identifier, and callback
 *          function pointer. It will check whether there is room in the local
 *          administration. All applications that want to read/write a NDEF msg
 *          in the NTAG should register its payload type as soon as possible
 *          after reset of the CPU (after JenOS is started). When the last
 *          application registers its NDEF payload type, then the function
 *          eNdefFormatNtag() is called by this function.
 * @param   pu8NdefType - Pointer to NDEF payload type
 * @param   u8NdefTypeLength - Length of the NDEF payload type
 * @param   pu8NdefId - Pointer to NDEF payload ID
 * @param   u8NdefIdLength - Length of NDEF payload ID
 * @param   u32NdefPayloadLength - Length of the payload of the NDEF message
 * @param   pCallbackFunc - Callback function pointer
 * @return  This function returns E_NDEF_SUCCESS when the NDEF message was
 *          successfully registered (and added) to the local administration.
 *          One of the following errors is returned when the message couldn't
 *          be registered: E_NDEF_ERR_TOO_MANY_MSGS, E_NDEF_ERR_TYPE_LENGTH,
 *          E_NDEF_ERR_ID_LENGTH, E_NDEF_ERR_NO_ROOM_IN_NTAG, E_NDEF_ERR_NTAG_FORMAT
 * @note    Passing a NULL pointer as callback function pointer is allowed, this
 *          can be useful if the application only wants to write in the NTAG.
 * @warning The define CONFIG_NUMBER_OF_APP_NDEF_MSGS must exactly match the number
 *          of NDEF messages that will be registered by the application,
 *          in order for the format of the NTAG to take place at startup of the
 *          application software.
 */

teNDEF_Status eNdefRegisterMsg(const uint8 *pu8NdefType,
                               uint8 u8NdefTypeLength,
                               uint8 *pu8NdefId,
                               uint8 u8NdefIdLength,
                               uint32 u32NdefPayloadLength,
                               void(*pCallbackFunc)(ndef_rec_info_t *))
{
    teNDEF_Status status = E_NDEF_SUCCESS;

    if (u8NrOfRegisteredMsgs >= (sizeof(MsgList) / sizeof(ndef_msg_admin_t)))
    {
        status = E_NDEF_ERR_TOO_MANY_MSGS;
    }
    else if (u8NdefTypeLength > sizeof(au8NdefType))
    {
        status = E_NDEF_ERR_TYPE_LENGTH;
    }
    else if (u8NdefIdLength > sizeof(au8NdefId))
    {
        status = E_NDEF_ERR_ID_LENGTH;
    }
    else
    {
        MsgList[u8NrOfRegisteredMsgs].pu8NdefType          = (uint8 *)pu8NdefType;
        MsgList[u8NrOfRegisteredMsgs].u8NdefTypeLength     = u8NdefTypeLength;
        MsgList[u8NrOfRegisteredMsgs].pu8NdefId            = pu8NdefId;
        MsgList[u8NrOfRegisteredMsgs].u8NdefIdLength       = u8NdefIdLength;
        MsgList[u8NrOfRegisteredMsgs].u32NdefPayloadLength = u32NdefPayloadLength;
        MsgList[u8NrOfRegisteredMsgs].pCallbackFunc        = pCallbackFunc;
        if (0 == u8NrOfRegisteredMsgs)
        {
            MsgList[u8NrOfRegisteredMsgs].u16StartOfTlv = 0;
        }
        else
        {
            MsgList[u8NrOfRegisteredMsgs].u16StartOfTlv = MsgList[u8NrOfRegisteredMsgs-1].u16StartOfNdefPayload +
                                                          MsgList[u8NrOfRegisteredMsgs-1].u32NdefPayloadLength;
        }
        uint16 u16HdrLength = u16NdefBuildHdr(FALSE, u8NrOfRegisteredMsgs);
        MsgList[u8NrOfRegisteredMsgs].u16StartOfNdefPayload = MsgList[u8NrOfRegisteredMsgs].u16StartOfTlv + u16HdrLength;
        if (MsgList[u8NrOfRegisteredMsgs].u16StartOfNdefPayload + u32NdefPayloadLength >= (((NTAG_LAST_USER_MEM_BLOCK-1) * NTAG_BLOCK_SIZE) - 1))
        {
            status = E_NDEF_ERR_NO_ROOM_IN_NTAG;
            memset(&MsgList[u8NrOfRegisteredMsgs], 0, sizeof(ndef_msg_admin_t));
        }
        else
        {
            u8NrOfRegisteredMsgs += 1;
            if (u8NrOfRegisteredMsgs == (sizeof(MsgList) / sizeof(ndef_msg_admin_t)))
            {
                if (E_NDEF_SUCCESS != eNdefFormatNtag())
                {
                    status = E_NDEF_ERR_NTAG_FORMAT;
                }
                else
                {
#ifdef DEBUG_NDEF
                    DBG_vPrintf(bTraceNdef, "NDEF: NTAG formatted\n");
#endif
                }
            }
        }
    }
    if (E_NDEF_SUCCESS != status)
    {
#ifdef DEBUG_NDEF
        DBG_vPrintf(bTraceNdef, "NDEF: eNdefRegisterMsg - return status %d\n", status);
#endif
    }
    return status;
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function reads the NTAG EEPROM (from start of user memory
 *          till the TLV terminator) and parses the TLVs that it finds.
 *          During this parsing it has exclusive R/W access to the NTAG.
 *          Before calling the callback function belonging to the parsed
 *          NDEF message, the lock on the NTAG is released.
 *          In the applications callback function, use the functions
 *          eNdefLockNtag() / eNdefReadMsg() / eNdefWriteMsg / eNdefUnlockNtag()
 *          to access the payload of the related NDEF.
 * @return  None.
 * @warning THIS FUNCTION MUST NOT BE CALLED BY THE APPLICATION BECAUSE
 *          IT IS CALLED BY FUNCTION vNfcLoop() (see nfc.c) ALREADY !!!
 */

void vNdefParser(void)
{
    bool   bStayInLoop = TRUE;
    uint16 u16NtagOffset = 0;
    i2c_LockBus();
    while (bStayInLoop)
    {
        if (bNtagTry2SetI2cLock())
        {
            vNtagClearNdefDataRead();
            bStayInLoop = bNtagStartRead(u16NtagOffset);
            if (bStayInLoop)
            {
                uint16 u16TlvLength = u16NdefCheckTlvType(&u16NtagOffset);
                if (0 == u16TlvLength)
                {
                    bStayInLoop = FALSE;
                }
                else
                {
                    bool bSucces = bNdefProcessRecord(&u16TlvLength, &u16NtagOffset);
                    if (!bSucces || (u16TlvLength > 0))
                    {
#ifdef DEBUG_NDEF
                        DBG_vPrintf(bTraceNdef, "NDEF: vNdefParser - Error reading NDEF record\n");
#endif
                        bStayInLoop = FALSE;
                    }
                    else if ((0xFF != u8ListIndex) && (NULL != MsgList[u8ListIndex].pCallbackFunc))
                    {
                        i2c_UnlockBus();
                        MsgList[u8ListIndex].pCallbackFunc(&NdefRecInfo);
                        i2c_LockBus();
                    }
                }
            }
            if (!bStayInLoop)
            {
                bNtagReleaseI2cLock();
            }
        }
        else
        {
#ifdef DEBUG_NDEF
            DBG_vPrintf(bTraceNdef, "NDEF: vNdefParser - Error setting I2C_LOCKED bit\n");
#endif
            bStayInLoop = FALSE;
        }
    }
    i2c_UnlockBus();
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function tries to get exclusive R/W access to the specified
 *          NDEF in the NTAG EEPROM. After having gained exclusive R/W access
 *          the functions eNdefReadMsg() and eNdefWriteMsg() can be used to
 *          read and/or write to the specified NDEF payload area.
 *          The exclusive R/W access must be released by calling the
 *          function eNdefUnlockNtag().
 * @param   pu8NdefType - Pointer to NDEF type
 * @return  This function returns E_NDEF_SUCCESS when exclusive R/W access to
 *          the NTAG has been established.
 *          One of the following errors is returned when the NTAG couldn't be
 *          be locked: E_NDEF_ERR_NTAG_ALREADY_LOCKED, E_NDEF_ERR_NTAG_NO_MEM_LOCK.
 * @note    This function first has to get a lock on the I2C bus before it tries
 *          to get the lock on the NTAG EEPROM. The lock on the I2C bus is needed
 *          because there can be multiple I2C slaves connected to the same I2C bus.
 *          The second reason to establish the I2C lock is to prevent re-entrance
 *          problems when multiple applications want to access the NTAG EEPROM at
 *          the same time.
 * @warning The NTAG-I2C IC has a built in watch dog timer that releases the I2C
 *          side lock on the NTAG EEPROM when the timer expires. However, the I2C
 *          bus lock (which isn't controlled by the NTAG) is still in place.
 *          The maximum watch dog timeout is ~600 msec, so the application should
 *          take care to release the lock within this maximum timeout. In the case
 *          that the application needs more time, it just has to unlock and lock
 *          the NTAG again, which reloads the watch dog timeout again.
 */

teNDEF_Status eNdefLockNtag(const uint8 *pu8NdefType)
{
    teNDEF_Status status = E_NDEF_SUCCESS;
    if (bNtagLocked)
    {
        if (pNdefMsg != (uint8 *)pu8NdefType)
        {
            status = E_NDEF_ERR_NTAG_ALREADY_LOCKED;
        }
    }
    else
    {
        i2c_LockBus();
        if (bNtagTry2SetI2cLock())
        {
            bNtagLocked = TRUE;
            pNdefMsg = (uint8 *)pu8NdefType;
        }
        else
        {
            i2c_UnlockBus();
            status = E_NDEF_ERR_NTAG_NO_MEM_LOCK;
#ifdef DEBUG_NDEF
            DBG_vPrintf(bTraceNdef, "NDEF: eNdefLockNtag - Error setting I2C_LOCKED bit\n");
#endif
        }
    }
    if (E_NDEF_SUCCESS != status)
    {
#ifdef DEBUG_NDEF
        DBG_vPrintf(bTraceNdef, "NDEF: eNdefLockNtag - return status %d\n", status);
#endif
    }
    return status;
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function releases the exclusive R/W access to the specified
 *          NDEF in the NTAG EEPROM, but only in the case that this application
 *          actually holds the lock.
 * @param   pu8NdefType - Pointer to NDEF type
 * @return  This function returns E_NDEF_SUCCESS when the exclusive R/W access
 *          to NTAG is released successfully.
 *          One of the following errors is returned when the NTAG couldn't be
 *          be unlocked: E_NDEF_ERR_NTAG_NOT_LOCKED, E_NDEF_ERR_WRONG_APP.
 */

teNDEF_Status eNdefUnlockNtag(const uint8 *pu8NdefType)
{
    teNDEF_Status status = E_NDEF_SUCCESS;
    if (!bNtagLocked)
    {
        status = E_NDEF_ERR_NTAG_NOT_LOCKED;
    }
    else if (pNdefMsg != (uint8 *)pu8NdefType)
    {
        status = E_NDEF_ERR_WRONG_APP;
    }
    else
    {
        pNdefMsg = NULL;
        bNtagLocked = FALSE;
        bNtagReleaseI2cLock();
        i2c_UnlockBus();
    }
    if (E_NDEF_SUCCESS != status)
    {
#ifdef DEBUG_NDEF
        DBG_vPrintf(bTraceNdef, "NDEF: eNdefUnlockNtag - return status %d\n", status);
#endif
    }
    return status;
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function reads NDEF payload data from the NTAG EEPROM.
 *          This function checks whether the caller only reads from his own
 *          NDEF payload area.
 * @param   pu8NdefType - Pointer to NDEF type
 * @param   u32NdefOffset - Offset in the payload area of this NDEF message
 * @param   pu8Data - Pointer to the callers buffer in which data will be stored
 * @param   u32DataLength - Length of the data to be read
 * @return  This function returns E_NDEF_SUCCESS when NDEF data is read from NTAG.
 *          One of the following errors is returned when the message couldn't
 *          be read: E_NDEF_ERR_NTAG_NOT_LOCKED, E_NDEF_ERR_WRONG_APP,
 *          E_NDEF_ERR_UNKNOWN_TYPE, E_NDEF_ERR_PAYLOAD_LENGTH,
 *          E_NDEF_ERR_READ_POSITION, E_NDEF_ERR_READ_PAYLOAD
 * @note    Exclusive R/W access to the NTAG IC must be acquired first by
 *          calling the function eNdefLockNtag(), and after finishing R/W
 *          to this NDEF payload the exclusive R/W access must be released
 *          by calling the function eNdefUnlockNtag().
 * @warning Trying to read data outside of your own NDEF payload area results
 *          in no data being read at all.
 */

teNDEF_Status eNdefReadMsg(const uint8 *pu8NdefType, uint32 u32NdefOffset, uint8 *pu8Data, uint32 u32DataLength)
{
    teNDEF_Status status = E_NDEF_ERR_UNKNOWN_TYPE;
    if (!bNtagLocked)
    {
        status = E_NDEF_ERR_NTAG_NOT_LOCKED;
    }
    else if (pNdefMsg != (uint8 *)pu8NdefType)
    {
        status = E_NDEF_ERR_WRONG_APP;
    }
    else
    {
        uint8 i;
        for (i=0; i < u8NrOfRegisteredMsgs; i++)
        {
            if ((uint8 *)pu8NdefType == MsgList[i].pu8NdefType)
            {
                if ((u32NdefOffset + u32DataLength) <= MsgList[i].u32NdefPayloadLength)
                {
                    if (bNtagStartRead(MsgList[i].u16StartOfNdefPayload + (uint16)u32NdefOffset))
                    {
                        if (u32NtagRead(pu8Data, u32DataLength) == u32DataLength)
                        {
                            status = E_NDEF_SUCCESS;
                        }
                        else
                        {
                            status = E_NDEF_ERR_READ_PAYLOAD;
                        }
                    }
                    else
                    {
                        status = E_NDEF_ERR_READ_POSITION;
                    }
                }
                else
                {
                    status = E_NDEF_ERR_PAYLOAD_LENGTH;
                }
                break;
            }
        }
    }
    if (E_NDEF_SUCCESS != status)
    {
#ifdef DEBUG_NDEF
        DBG_vPrintf(bTraceNdef, "NDEF: eNdefReadMsg - return status %d\n", status);
#endif
    }
    return status;
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function writes NDEF payload data into the NTAG EEPROM.
 *          This function checks whether the caller only writes to his own
 *          NDEF payload area.
 * @param   pu8NdefType - Pointer to NDEF type
 * @param   u32NdefOffset - Offset in the payload area of this NDEF message
 * @param   pu8Data - Pointer to the data to be written in the NTAG
 * @param   u32DataLength - Length of the data to write
 * @return  This function returns E_NDEF_SUCCESS when NDEF data is written
 *          in the NTAG.
 *          One of the following errors is returned when the message couldn't
 *          be written: E_NDEF_ERR_NTAG_NOT_LOCKED, E_NDEF_ERR_WRONG_APP,
 *          E_NDEF_ERR_UNKNOWN_TYPE, E_NDEF_ERR_PAYLOAD_LENGTH,
 *          E_NDEF_ERR_WRITE_HDR, E_NDEF_ERR_WRITE_POSITION,
 *          E_NDEF_ERR_WRITE_PAYLOAD, E_NDEF_ERR_FLUSH_DATA
 * @note    Exclusive R/W access to the NTAG IC must be acquired first by
 *          calling the function eNdefLockNtag(), and after finishing R/W
 *          to this NDEF payload the exclusive R/W access must be released
 *          by calling the function eNdefUnlockNtag().
 * @warning Trying to write data outside of your own NDEF payload area results
 *          in no data being written at all.
 */

teNDEF_Status eNdefWriteMsg(const uint8 *pu8NdefType, uint32 u32NdefOffset, uint8 *pu8Data, uint32 u32DataLength)
{
    teNDEF_Status status = E_NDEF_ERR_UNKNOWN_TYPE;
    if (!bNtagLocked)
    {
        status = E_NDEF_ERR_NTAG_NOT_LOCKED;
    }
    else if (pNdefMsg != (uint8 *)pu8NdefType)
    {
        status = E_NDEF_ERR_WRONG_APP;
    }
    else
    {
        uint8 i;
        for (i=0; i < u8NrOfRegisteredMsgs; i++)
        {
            if ((uint8 *)pu8NdefType == MsgList[i].pu8NdefType)
            {
                if ((u32NdefOffset + u32DataLength) <= MsgList[i].u32NdefPayloadLength)
                {
                    status = E_NDEF_SUCCESS;
                    if (0 == u32NdefOffset)
                    {
                        if (!bNdefWriteHdr(i))
                        {
                            status = E_NDEF_ERR_WRITE_HDR;
                        }
                    }
                    if (E_NDEF_SUCCESS == status)
                    {
                        if (!bNtagStartWrite(MsgList[i].u16StartOfNdefPayload + (uint16)u32NdefOffset))
                        {
                            status = E_NDEF_ERR_WRITE_POSITION;
                        }
                    }
                    if (E_NDEF_SUCCESS == status)
                    {
                        if (u32NtagWrite(pu8Data, u32DataLength) == u32DataLength)
                        {
                            if (!bNtagFlush())
                            {
                                status = E_NDEF_ERR_FLUSH_DATA;
#ifdef DEBUG_NDEF
                                DBG_vPrintf(bTraceNdef, "NDEF: eNdefWriteMsg - Error flush\n");
#endif
                            }
                        }
                        else
                        {
                            status = E_NDEF_ERR_WRITE_PAYLOAD;
                        }
                    }
                }
                else
                {
                    status = E_NDEF_ERR_PAYLOAD_LENGTH;
                }
                break;
            }
        }
    }
    if (E_NDEF_SUCCESS != status)
    {
#ifdef DEBUG_NDEF
        DBG_vPrintf(bTraceNdef, "NDEF: eNdefWriteMsg - return status %d\n", status);
#endif
    }
    return status;
}

/* Doxygen info, do not remove! */
/**
 * @brief   This function writes the TLVs, NDEF headers and TLV terminator
 *          into the NTAG EEPROM. This function can be used to repair the
 *          NTAG format as designed by the application.
 * RETURNS: This function returns E_NDEF_SUCCESS when the NTAG is formatted.
 *          One of the following errors is returned when it couldn't format
 *          the NTAG: E_NDEF_ERR_NTAG_ALREADY_LOCKED, E_NDEF_ERR_NTAG_NO_MEM_LOCK,
 *          E_NDEF_ERR_WRITE_HDR or E_NDEF_ERR_WRITE_TERMINATOR
 * @note    The function bNdefTypeRegisterCallback() calls this function when
 *          the last NDEF callback (defined by CONFIG_NUMBER_OF_APP_NDEF_MSGS)
 *          is registered directly after reset of the CPU (after JenOS is started).
 */

teNDEF_Status eNdefFormatNtag(void)
{
    teNDEF_Status status = E_NDEF_SUCCESS;
    if (bNtagLocked)
    {
        status = E_NDEF_ERR_NTAG_ALREADY_LOCKED;
    }
    else
    {
        i2c_LockBus();
        if (bNtagTry2SetI2cLock())
        {
            bool bSucces = TRUE;
            uint8 i;
            for (i=0; i < u8NrOfRegisteredMsgs; i++)
            {
                bSucces = bNdefWriteHdr(i);
                if (!bSucces)
                {
                    status = E_NDEF_ERR_WRITE_HDR;
                    break;
                }
            }
            if (bSucces)
            {
                if (!bNdefWriteTerminator())
                {
                    status = E_NDEF_ERR_WRITE_TERMINATOR;
                }
            }
            bNtagReleaseI2cLock();
        }
        else
        {
            status = E_NDEF_ERR_NTAG_NO_MEM_LOCK;
#ifdef DEBUG_NDEF
            DBG_vPrintf(bTraceNdef, "NDEF: eNdefFormatNtag - Error setting I2C_LOCKED bit\n");
#endif
        }
        i2c_UnlockBus();
    }
    if (E_NDEF_SUCCESS != status)
    {
#ifdef DEBUG_NDEF
        DBG_vPrintf(bTraceNdef, "NDEF: eNdefFormatNtag - return status %d\n", status);
#endif
    }
    return status;
}

void vNdefToggleDBG(void)
{
#ifdef DEBUG_NDEF
    bTraceNdef = !bTraceNdef;
#endif
}

#endif  // NFC_SUPPORT

/****************************************************************************/
/***        END OF FILE                                                   ***/
/****************************************************************************/
