/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of the copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*  Standard C Included Files */
#include <stdio.h>
#include <string.h>
#include <math.h>
#include "board.h"
#include "fsl_lcdc.h"
#include "app.h"
#include "fsl_mma.h"
#include "fsl_i2c.h"
#include "fsl_debug_console.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define APP_BIT_PER_PIXEL 2
#define APP_PIXEL_PER_BYTE 4
#define APP_PIXEL_MAX_VALUE 3
#define APP_PIXEL_MIN_VALUE 3

#define MIN_XPOS_TILT_ANGLE 10
#define MIN_XNEG_TILT_ANGLE -10
#define MIN_YPOS_TILT_ANGLE 10
#define MIN_YNEG_TILT_ANGLE -10
/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/

#if (defined(__CC_ARM) || defined(__GNUC__))
__attribute__((aligned(8)))
#elif defined(__ICCARM__)
#pragma data_alignment = 8
#else
#error Toolchain not support.
#endif
static uint8_t s_frameBuf0[IMG_HEIGHT][IMG_WIDTH / APP_PIXEL_PER_BYTE];

#if (defined(__CC_ARM) || defined(__GNUC__))
__attribute__((aligned(8)))
#elif defined(__ICCARM__)
#pragma data_alignment = 8
#else
#error Toolchain not support.
#endif
static uint8_t s_frameBuf1[IMG_HEIGHT][IMG_WIDTH / APP_PIXEL_PER_BYTE];

static const uint32_t s_frameBufAddr[] = {(uint32_t)s_frameBuf0, (uint32_t)s_frameBuf1};

/* RAM Palette look up for 2bpp */
static const uint32_t palette[] = {0x001F0000U, 0x7C0003E0U};

/* The index of the inactive buffer. */
static volatile uint8_t s_inactiveBufsIdx;

/* The new frame address already loaded to the LCD controller. */
static volatile bool s_frameAddrUpdated = false;

i2c_master_handle_t g_MasterHandle;
/* MMA8652 device address */
const uint8_t g_accel_address[] = {0x1CU, 0x1DU, 0x1EU, 0x1FU};

uint8_t accelGValue = 0;
uint8_t dataScale = 0;
int16_t accelXData = 0;
int16_t accelYData = 0;
int16_t accelZData = 0;
int16_t xAngle = 0;
int16_t yAngle = 0;
int16_t zAngle = 0;
bool startmov = false;
/*******************************************************************************
 * Code
 ******************************************************************************/
void APP_LCD_IRQHandler(void)
{
    uint32_t intStatus = LCDC_GetEnabledInterruptsPendingStatus(APP_LCD);

    LCDC_ClearInterruptsStatus(APP_LCD, intStatus);

    if (intStatus & kLCDC_BaseAddrUpdateInterrupt)
    {
        s_frameAddrUpdated = true;
    }
    __DSB();
}

/* Draw a line based on start and end pixel*/
static void APP_Draw2BPPLine(uint8_t *line, uint16_t start, uint16_t end, uint8_t color)
{
    uint16_t i;
    uint16_t startByte;
    uint16_t endByte;

    startByte = start / APP_PIXEL_PER_BYTE;
    endByte = end / APP_PIXEL_PER_BYTE;

    if (startByte == endByte)
    {
        for (i = (start & _2BPP_BYTE_MASK); i <= (end & _2BPP_BYTE_MASK); i++)
        {
            line[startByte] =
                (line[startByte] & ~(_2BPP_BYTE_MASK << (i * _2BPP_BYTE_SHIFT))) | (color << (i * _2BPP_BYTE_SHIFT));
        }
    }
    else
    {
        for (i = (start & _2BPP_BYTE_MASK); i < APP_PIXEL_PER_BYTE; i++)
        {
            line[startByte] =
                (line[startByte] & ~(_2BPP_BYTE_MASK << (i * _2BPP_BYTE_SHIFT))) | (color << (i * _2BPP_BYTE_SHIFT));
        }

        for (i = (startByte + 1U); i < endByte; i++)
        {
            line[i] = color * 0x55U;
        }

        for (i = 0U; i < (end & _2BPP_BYTE_MASK); i++)
        {
            line[endByte] =
                (line[endByte] & ~(_2BPP_BYTE_MASK << (i * _2BPP_BYTE_SHIFT))) | (color << (i * _2BPP_BYTE_SHIFT));
        }
    }
}

static void APP_FillBuffer(void *buffer)
{
    /* Background color of lcd display panel */
    static uint8_t bgColor = 0U;
    static uint16_t lcdPanelXOrigin = 0U;
    static uint16_t lcdPanelYOrigin = 0U;

    uint8_t colorToSet = 0U;

    /* Position of the foreground rectangle. */
    static uint16_t upperLeftX = 0U;
    static uint16_t upperLeftY = 0U;
    static uint16_t lowerRightX = (IMG_WIDTH - 1U) / 2U;
    static uint16_t lowerRightY = (IMG_HEIGHT - 1U) / 2U;
    /* Foreground color. */
    static uint8_t fgColor = 2U;

    /* Increase X and Y positions*/
    static int16_t incX = 5;
    static int16_t incY = 5;

    /* Change color in next frame or not. */
    static bool changeColor = false;
    static bool xDirChange = false;
    static bool yDirChange = false;

    uint32_t i, j;
    uint8_t(*buf)[IMG_WIDTH / APP_PIXEL_PER_BYTE] = buffer;

    /*
     +------------------------------------------------------------------------+
     |                                                                        |
     |                                                                        |
     |                                                                        |
     |                          Area 1                                        |
     |                                                                        |
     |                                                                        |
     |                  +---------------------------+                         |
     |                  |XXXXXXXXXXXXXXXXXXXXXXXXXXX|                         |
     |                  |XXXXXXXXXXXXXXXXXXXXXXXXXXX|                         |
     |    Area 2        |XXXXXXXXXXXXXXXXXXXXXXXXXXX|       Area 3            |
     |                  |XXXXXXXXXXXXXXXXXXXXXXXXXXX|                         |
     |                  |XXXXXXXXXXXXXXXXXXXXXXXXXXX|                         |
     |                  |XXXXXXXXXXXXXXXXXXXXXXXXXXX|                         |
     |                  +---------------------------+                         |
     |                                                                        |
     |                                                                        |
     |                                                                        |
     |                                                                        |
     |                         Area 4                                         |
     |                                                                        |
     |                                                                        |
     +------------------------------------------------------------------------+
     */

    /* Fill the frame buffer. */
    /* Fill area 1. */
    colorToSet = bgColor * 0x55U;
    /* Go row (line) wise*/
    for (i = lcdPanelYOrigin; i < upperLeftY; i++)
    {
        /*Go column wise(scanning pixel in every line)*/
        for (j = lcdPanelXOrigin; j < IMG_WIDTH / APP_PIXEL_PER_BYTE; j++)
        {
            /* Fill entire selected region with background color */
            buf[i][j] = colorToSet;
        }
    }

    /*Draw the rectangle object*/
    /* Part-1 fill one line of object*/
    APP_Draw2BPPLine((uint8_t *)buf[i], lcdPanelXOrigin, upperLeftX,
                     bgColor); /*Fill one line with background color till Object*/
    APP_Draw2BPPLine((uint8_t *)buf[i], upperLeftX, lowerRightX + 1,
                     fgColor); /*Fill selected pixels in line with foreground color*/
    APP_Draw2BPPLine((uint8_t *)buf[i], lowerRightX + 1, IMG_WIDTH,
                     bgColor); /*Fill remaining width of line with background color */

    /* Part-2 Copy the previously filled line to the entire rectangle object region*/
    for (i++; i <= lowerRightY; i++)
    {
        for (j = 0; j < (IMG_WIDTH / APP_PIXEL_PER_BYTE); j++)
        {
            buf[i][j] = buf[upperLeftY][j];
        }
    }

    /* Fill area 4. */
    /* Fill remaining area of panel with background color*/
    colorToSet = bgColor * 0x55U;
    for (i = lowerRightY + 1; i < IMG_HEIGHT; i++)
    {
        /*Go column wise(scanning pixel in every line)*/
        for (j = 0; j < IMG_WIDTH / APP_PIXEL_PER_BYTE; j++)
        {
            /* Fill entire selected region with background color */
            buf[i][j] = colorToSet;
        }
    }

    /* Update X position of object on LCD based on Accelerometer data.
     * X coordinates should increment when angle is positive.
     * Here if angle is greater than MIN_XPOS_TILT_ANGLE the object position is updated.
     */
    if ((xAngle > MIN_XPOS_TILT_ANGLE) && (lowerRightX < IMG_WIDTH - 1))
    {
        /* Update rectangle X position. */
        upperLeftX += incX;
        lowerRightX += incX;

        xDirChange = true; /*x axis direction has changed*/
    }

    /* X coordinates should decrement when angle is negative. Here if angle is less than MIN_XNEG_TILT_ANGLE the object
     * position is updated*/
    if ((xAngle < MIN_XNEG_TILT_ANGLE) && (upperLeftX > 0))
    {
        /* Update rectangle X position. */
        upperLeftX -= incX;
        lowerRightX -= incX;

        xDirChange = true; /*x axis direction has changed*/
    }

    /* Update Y position of object on LCD based on Accelerometer data.
     * Y coordinates should increment when angle is positive.
     * Here if angle is greater than MIN_YPOS_TILT_ANGLE the object position is updated
     */
    if ((yAngle > MIN_YPOS_TILT_ANGLE) && (lowerRightY < IMG_HEIGHT - 1))
    {
        /* Update rectangle Y position. */
        upperLeftY += incY;
        lowerRightY += incY;

        yDirChange = true; /*y axis direction has changed*/
    }

    /* Y coordinates should decrement when angle is negative. Here if angle is less than MIN_YNEG_TILT_ANGLE the object
     * position is updated*/
    if ((yAngle < MIN_YNEG_TILT_ANGLE) && (upperLeftY > 0))
    {
        /* Update rectangle Y position. */
        upperLeftY -= incY;
        lowerRightY -= incY;

        yDirChange = true; /*y axis direction has changed*/
    }

    changeColor = false;

    /* If the object touches the boundaries , change the color of the object*/
    if (lowerRightX == IMG_WIDTH - 1)
    {
        if (xDirChange)
        {
            changeColor = true;
            xDirChange = false;
        }
    }

    if (upperLeftX == 0)
    {
        if (xDirChange)
        {
            changeColor = true;
            xDirChange = false;
        }
    }

    if (lowerRightY >= IMG_HEIGHT - 1)
    {
        if (yDirChange)
        {
            changeColor = true;
            yDirChange = false;
        }
    }

    if (upperLeftY == 0)
    {
        if (yDirChange)
        {
            changeColor = true;
            yDirChange = false;
        }
    }

    if (changeColor)
    {
        if (APP_PIXEL_MAX_VALUE == fgColor)
        {
            fgColor = 1U;
        }
        else
        {
            fgColor++;
        }
    }
}

int main(void)
{
    lcdc_config_t lcdConfig;
    mma_handle_t mmaHandle = {0};
    mma_data_t accelSensorData = {0};
    i2c_master_config_t masterConfig;
    uint8_t numberOfAccelSlvAddr = 0;
    uint8_t i = 0;
    uint8_t readRegResult = 0;
    bool foundDevice = false;

    /* Initialize Board pins, clock and debug console unit*/
    BOARD_InitHardware();

    /*Initialize and setup LCD configuration*/
    s_frameAddrUpdated = false;
    s_inactiveBufsIdx = 1;

    /* Fill frame buffer before setting UP/LPBASE addr and enabling interrupts*/
    APP_FillBuffer((void *)(s_frameBufAddr[0]));

    LCDC_GetDefaultConfig(&lcdConfig);

    lcdConfig.panelClock_Hz = LCD_PANEL_CLK;                /* LCD Panel Clock*/
    lcdConfig.ppl = LCD_PPL;                                /* Pixels per line */
    lcdConfig.hsw = LCD_HSW;                                /* HSYNC signal pulse width*/
    lcdConfig.hfp = LCD_HFP;                                /* HSYNC front porch value*/
    lcdConfig.hbp = LCD_HBP;                                /* HSYNC back porch value*/
    lcdConfig.lpp = LCD_LPP;                                /* lines per panel*/
    lcdConfig.vsw = LCD_VSW;                                /* VSYNC signal pulse width*/
    lcdConfig.vfp = LCD_VFP;                                /* VSYNC front porch value*/
    lcdConfig.vbp = LCD_VBP;                                /* VSYNC back porch value*/
    lcdConfig.polarityFlags = LCD_POL_FLAGS;                /* Set clock and signal polarity flags */
    lcdConfig.bpp = kLCDC_2BPP;                             /* bits per pixel*/
    lcdConfig.display = kLCDC_DisplayTFT;                   /* display type is TFT display*/
    lcdConfig.swapRedBlue = false;                          /* Do not swap RGB to BGR*/
    lcdConfig.upperPanelAddr = (uint32_t)s_frameBufAddr[0]; /* Upper panel base address - 64-bit aligned*/

    /* Initialize LCD clock and registers*/
    LCDC_Init(APP_LCD, &lcdConfig, LCD_INPUT_CLK_FREQ);

    /* Set the RAM Palette entries*/
    LCDC_SetPalette(APP_LCD, palette, ARRAY_SIZE(palette));

    /*Enable LCD Interrupts*/
    LCDC_EnableInterrupts(APP_LCD, kLCDC_BaseAddrUpdateInterrupt);
    NVIC_EnableIRQ(APP_LCD_IRQn);

    /* Start Output LCD timing signals*/
    LCDC_Start(APP_LCD);

    /*Enable Power to LCD Panel*/
    LCDC_PowerUp(APP_LCD);

    /*Initialize and setup Accelerometer*/
    mmaHandle.base = BOARD_ACCEL_I2C_BASEADDR;
    mmaHandle.i2cHandle = &g_MasterHandle;

    /*
       * masterConfig->enableMaster = true;
     * masterConfig->baudRate_Bps = 100000U;
       * masterConfig->enableTimeout = false;
       */
    I2C_MasterGetDefaultConfig(&masterConfig);
    I2C_MasterInit(BOARD_ACCEL_I2C_BASEADDR, &masterConfig, I2C_MASTER_CLOCK_FREQUENCY);
    I2C_MasterTransferCreateHandle(BOARD_ACCEL_I2C_BASEADDR, &g_MasterHandle, NULL, NULL);

    numberOfAccelSlvAddr = sizeof(g_accel_address) / sizeof(g_accel_address[0]);
    /* Find the accelerometer device*/
    for (i = 0; i < numberOfAccelSlvAddr; i++)
    {
        mmaHandle.xfer.slaveAddress = g_accel_address[i];

        if (MMA_ReadReg(&mmaHandle, kMMA8652_WHO_AM_I, &readRegResult) == kStatus_Success)
        {
            foundDevice = true;
            break;
        }

        if ((!foundDevice) && (i == numberOfAccelSlvAddr - 1))
        {
            PRINTF("\n\r Accelerometer Device not found ! ");
            while (1)
                ;
        }
    }

    /*Initialize the accelerometer device:- Dynamic range : -4g to +4g; FIFO : disabled; F_READ : Normal Read mode*/
    if (MMA_Init(&mmaHandle) != kStatus_Success)
    {
        return kStatus_Fail;
    }

    /* Get Accelerometer dynamic range */
    if (MMA_ReadReg(&mmaHandle, kMMA8652_XYZ_DATA_CFG, &accelGValue) != kStatus_Success)
    {
        return kStatus_Fail;
    }
    /*Check FS bits[1:0] for dynamic range value : 2g = 0x00; 4g = 0x01; 8g = 0x02 */
    if ((accelGValue & 0x03) == 0x00)
    {
        dataScale = 2;
    }
    else if ((accelGValue & 0x03) == 0x01)
    {
        dataScale = 4;
    }
    else if ((accelGValue & 0x03) == 0x02)
    {
        dataScale = 8;
    }

    PRINTF("\n\r Accelerometer initialized");

    while (1)
    {
        /* Read Accelerometer Sensor Data*/
        if (MMA_ReadSensorData(&mmaHandle, &accelSensorData) != kStatus_Success)
        {
            return kStatus_Fail;
        }

        /* Save each sensor data as a 16 bit result.
        * The 16 bit result is in left-justified format.
        * Shift 4 bits to the right(12-bit resolution) to get the actual value.
        */
        accelXData =
            (int16_t)(((uint16_t)(accelSensorData.accelXMSB << 8)) | ((uint16_t)(accelSensorData.accelXLSB))) / 16U;
        accelYData =
            (int16_t)(((uint16_t)(accelSensorData.accelYMSB << 8)) | ((uint16_t)(accelSensorData.accelYLSB))) / 16U;
        accelZData =
            (int16_t)(((uint16_t)(accelSensorData.accelZMSB << 8)) | ((uint16_t)(accelSensorData.accelZLSB))) / 16U;

        /* Convert raw accelerometer sensor data to angle (normalize to 0-90 degrees). No negative angles. */
        xAngle = (int16_t)floor((double)accelXData * (double)dataScale * 180 / 4096);
        yAngle = (int16_t)floor((double)accelYData * (double)dataScale * 180 / 4096);

        /* Fill the inactive buffer. */
        APP_FillBuffer((void *)s_frameBufAddr[s_inactiveBufsIdx]);

        /*
         * The buffer address has been loaded to the LCD controller, now
         * set the inactive buffer to active buffer.
         */
        LCDC_SetPanelAddr(APP_LCD, kLCDC_UpperPanel, (uint32_t)(s_frameBufAddr[s_inactiveBufsIdx]));

        while (!s_frameAddrUpdated)
        {
        }

        s_frameAddrUpdated = false;
        s_inactiveBufsIdx ^= 1U;
    }
}
