/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2017 NXP
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of the copyright holder nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "fsl_device_registers.h"
#include "fsl_debug_console.h"
#include "board.h"
#include "app.h"
#include "fsl_adc.h"
#include "fsl_clock.h"
#include "fsl_power.h"
#include "fsl_dma.h"
#include "fsl_inputmux.h"
#include "fsl_usart.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
static void ADC_Configuration(void);
static void DMA_Configuration(void);
static void NVIC_Configuration(void);

dma_handle_t gDmaChannelHandleStruct[2]; /* Handler structure for using DMA. */
uint32_t gDemoAdcConvResult[2];          /* Keep the ADC conversion result moved from ADC data register by DMA. */
volatile bool bDmaTransferDone;          /* Flag of DMA transfer done trigger by ADC conversion. */

/*******************************************************************************
 * Code
 ******************************************************************************/
/*!
 * @brief Main function
 */
int main(void)
{
    /* Initialize board hardware. */
    BOARD_InitHardware();
    PRINTF("ADC DMA SequenceA/B example.\r\n");

    /* Configure NVIC */
    NVIC_Configuration();
    /* Configure DMA */
    DMA_Configuration();
    /* Configure ADC */
    ADC_Configuration();

    PRINTF("Configuration Done.\r\n");
    PRINTF("Type in any key to trigger the conversion ...\r\n");

    while (1)
    {
        /* Get the input from terminal and trigger the converter by software. */
        GETCHAR();

        bDmaTransferDone = false;
        /* Start DMA transfer */
        DMA_StartTransfer(&gDmaChannelHandleStruct[0]);
        /* Trigger the ADC and start the conversion. */
        ADC_DoSoftwareTriggerConvSeqA(DEMO_ADC_BASE);

        /* Wait for the conversion & transfer to be done. */
        while (!bDmaTransferDone)
        {
        }

        bDmaTransferDone = false;
        /* Start DMA transfer */
        DMA_StartTransfer(&gDmaChannelHandleStruct[1]);
        /* Trigger the ADC and start the conversion. */
        ADC_DoSoftwareTriggerConvSeqB(DEMO_ADC_BASE);

        /* Wait for the conversion & transfer to be done. */
        while (!bDmaTransferDone)
        {
        }

        PRINTF("Sequence A Conversion word : 0x%X\r\n", gDemoAdcConvResult[0]);
        PRINTF("Sequence A Conversion value: %d\r\n",
               (gDemoAdcConvResult[0] & ADC_DAT_RESULT_MASK) >> ADC_DAT_RESULT_SHIFT);
        PRINTF("\r\n");

        PRINTF("Sequence B Conversion word : 0x%X\r\n", gDemoAdcConvResult[1]);
        PRINTF("Sequence B Conversion value: %d\r\n",
               (gDemoAdcConvResult[1] & ADC_DAT_RESULT_MASK) >> ADC_DAT_RESULT_SHIFT);
        PRINTF("\r\n");
    }
}

/*!
 * @brief NVIC Configuration function
 */
static void NVIC_Configuration(void)
{
    NVIC_EnableIRQ(DMA0_IRQn);
}

/*!
 * @brief ADC Configuration function
 */
static void ADC_Configuration(void)
{
    adc_config_t adcConfigStruct;
    adc_conv_seq_config_t adcConvSeqAConfigStruct;
    adc_conv_seq_config_t adcConvSeqBConfigStruct;

    /* Configure the converter. */
    adcConfigStruct.clockMode = kADC_ClockSynchronousMode; /* Using sync clock source. */
    adcConfigStruct.clockDividerNumber = 1;                /* The divider for sync clock is 2. */
    adcConfigStruct.resolution = kADC_Resolution12bit;     /* 12-bit resolution */
    adcConfigStruct.enableBypassCalibration = false;       /*Calibration not bypassed */
    adcConfigStruct.sampleTimeNumber = 0U;                 /* Default sample time */

    /* Initialize ADC*/
    ADC_Init(DEMO_ADC_BASE, &adcConfigStruct);

    /* Calibration after power up. */
    if (ADC_DoSelfCalibration(DEMO_ADC_BASE))
    {
        PRINTF("ADC_DoSelfCalibration() Done.\r\n");
    }
    else
    {
        PRINTF("ADC_DoSelfCalibration() Failed.\r\n");
    }

#if !(defined(FSL_FEATURE_ADC_HAS_NO_INSEL) && FSL_FEATURE_ADC_HAS_NO_INSEL)
    /* Use the temperature sensor input to channel 0. */
    ADC_EnableTemperatureSensor(DEMO_ADC_BASE, true);
#endif /* FSL_FEATURE_ADC_HAS_NO_INSEL. */

    /* Enable channel DEMO_ADC_SAMPLE0_CHANNEL_NUMBER's conversion in Sequence A. */
    adcConvSeqAConfigStruct.channelMask =
        (1U << DEMO_ADC_SAMPLE0_CHANNEL_NUMBER); /* Includes channel DEMO_ADC_SAMPLE0_CHANNEL_NUMBER. */
    adcConvSeqAConfigStruct.triggerMask = 0U;
    adcConvSeqAConfigStruct.triggerPolarity = kADC_TriggerPolarityPositiveEdge;
    adcConvSeqAConfigStruct.enableSingleStep = false;
    adcConvSeqAConfigStruct.enableSyncBypass = false;
    adcConvSeqAConfigStruct.interruptMode = kADC_InterruptForEachSequence; /* Enable the interrupt/DMA trigger. */

    ADC_SetConvSeqAConfig(DEMO_ADC_BASE, &adcConvSeqAConfigStruct);
    /* Enable the conversion sequence A. */
    ADC_EnableConvSeqA(DEMO_ADC_BASE, true);

    /* Enable interrupts for conversion sequence A*/
    ADC_EnableInterrupts(DEMO_ADC_BASE, kADC_ConvSeqAInterruptEnable);

    /* Enable channel DEMO_ADC_SAMPLE1_CHANNEL_NUMBER's conversion in Sequence B. */
    adcConvSeqBConfigStruct.channelMask =
        (1U << DEMO_ADC_SAMPLE1_CHANNEL_NUMBER); /* Includes channel DEMO_ADC_SAMPLE1_CHANNEL_NUMBER. */
    adcConvSeqBConfigStruct.triggerMask = 0U;
    adcConvSeqBConfigStruct.triggerPolarity = kADC_TriggerPolarityPositiveEdge;
    adcConvSeqBConfigStruct.enableSingleStep = false;
    adcConvSeqBConfigStruct.enableSyncBypass = false;
    adcConvSeqBConfigStruct.interruptMode = kADC_InterruptForEachSequence; /* Enable the interrupt/DMA trigger. */

    ADC_SetConvSeqBConfig(DEMO_ADC_BASE, &adcConvSeqBConfigStruct);
    /* Enable the conversion sequence B. */
    ADC_EnableConvSeqB(DEMO_ADC_BASE, true);

    /* Enable interrupts for conversion sequence B*/
    ADC_EnableInterrupts(DEMO_ADC_BASE, kADC_ConvSeqBInterruptEnable);
}

/* Software ISR for DMA transfer done. */
void DEMO_DMA_Callback(dma_handle_t *handle, void *param, bool transferDone, uint32_t tcds)
{
    bDmaTransferDone = true;
}

/*!
 * @brief DMA Configuration function
 */
static void DMA_Configuration(void)
{
    dma_transfer_config_t dmaTransferConfigStruct[2];
    dma_channel_trigger_t dmaChannelTriggerStruct[2];

    /* Configure DMAMUX. */
    INPUTMUX_Init(INPUTMUX);
    INPUTMUX_AttachSignal(INPUTMUX, DEMO_DMA_ADC_SAMPLE0_CHANNEL, kINPUTMUX_Adc0SeqaIrqToDma);
    INPUTMUX_AttachSignal(INPUTMUX, DEMO_DMA_ADC_SAMPLE1_CHANNEL, kINPUTMUX_Adc0SeqbIrqToDma);

    /* Initialize DMA */
    DMA_Init(DMA0);

    /* Configure DMA for DMA channel 0*/
    DMA_EnableChannel(DMA0, DEMO_DMA_ADC_SAMPLE0_CHANNEL);
    DMA_CreateHandle(&gDmaChannelHandleStruct[0], DMA0, DEMO_DMA_ADC_SAMPLE0_CHANNEL);
    DMA_SetCallback(&gDmaChannelHandleStruct[0], DEMO_DMA_Callback, NULL);
    /*
     * Configure the DMA trigger:
     * The DATAVALID of ADC will trigger the interrupt. This signal is also for thie DMA triger, which is changed 0 ->
     * 1.
     */
    dmaChannelTriggerStruct[0].burst = kDMA_EdgeBurstTransfer1;
    dmaChannelTriggerStruct[0].type = kDMA_RisingEdgeTrigger;
    dmaChannelTriggerStruct[0].wrap = kDMA_NoWrap;
    DMA_ConfigureChannelTrigger(DMA0, DEMO_DMA_ADC_SAMPLE0_CHANNEL, &dmaChannelTriggerStruct[0]);

    /* Prepare and submit the transfer. */
    DMA_PrepareTransfer(&dmaTransferConfigStruct[0],     /* To keep the configuration. */
                        (void *)DEMO_ADC_DATA0_REG_ADDR, /* DMA transfer source address. */
                        (void *)&gDemoAdcConvResult[0],  /* DMA transfer destination address. */
                        sizeof(uint32_t),                /* DMA transfer destination address width(bytes). */
                        sizeof(uint32_t),                /* DMA transfer bytes to be transferred. */
                        kDMA_MemoryToMemory,             /* DMA transfer type. */
                        NULL                             /* nextDesc Chain custom descriptor to transfer. */
                        );
    DMA_SubmitTransfer(&gDmaChannelHandleStruct[0], &dmaTransferConfigStruct[0]);

    /* Configure DMA for DMA channel 1*/
    DMA_EnableChannel(DMA0, DEMO_DMA_ADC_SAMPLE1_CHANNEL);
    DMA_CreateHandle(&gDmaChannelHandleStruct[1], DMA0, DEMO_DMA_ADC_SAMPLE1_CHANNEL);
    DMA_SetCallback(&gDmaChannelHandleStruct[1], DEMO_DMA_Callback, NULL);
    /*
     * Configure the DMA trigger:
     * The DATAVALID of ADC will trigger the interrupt. This signal is also for thie DMA triger, which is changed 0 ->
     * 1.
     */
    dmaChannelTriggerStruct[1].burst = kDMA_EdgeBurstTransfer1;
    dmaChannelTriggerStruct[1].type = kDMA_RisingEdgeTrigger;
    dmaChannelTriggerStruct[1].wrap = kDMA_NoWrap;
    DMA_ConfigureChannelTrigger(DMA0, DEMO_DMA_ADC_SAMPLE1_CHANNEL, &dmaChannelTriggerStruct[1]);
    /* Prepare and submit the transfer. */
    DMA_PrepareTransfer(&dmaTransferConfigStruct[1],     /* To keep the configuration. */
                        (void *)DEMO_ADC_DATA1_REG_ADDR, /* DMA transfer source address. */
                        (void *)&gDemoAdcConvResult[1],  /* DMA transfer destination address. */
                        sizeof(uint32_t),                /* DMA transfer destination address width(bytes). */
                        sizeof(uint32_t),                /* DMA transfer bytes to be transferred. */
                        kDMA_MemoryToMemory,             /* DMA transfer type. */
                        NULL                             /* nextDesc Chain custom descriptor to transfer. */
                        );
    DMA_SubmitTransfer(&gDmaChannelHandleStruct[1], &dmaTransferConfigStruct[1]);
}
