/*
 * Copyright 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2019 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "acim_control.h"

#include "qs.h"
#include "gpio.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/

/* dead-time compensation voltage table */
const frac16_t f16UDtComp[] = DTCOMP_TABLE_DATA;

/*******************************************************************************
 * Code
 ******************************************************************************/
static void MCS_DTComp(GMCLIB_2COOR_ALBE_T_F16 *sUAlBeDTComp, 
                GMCLIB_3COOR_T_F16 *sIABC,
                frac16_t f16UDcBusFilt);
/*!
 * @brief ACIM Scalar Open Loop Control
 *
 * This function is used to ACIM Open Loop scalar mode
 *
 * @param psFocACIM     The pointer of the ACIM FOC parameters structure
 * @param psScalarACIM  The pointer of the ACIM scalar parameters structure
 *
 * @return None
 */
void MCS_ACIMOpenLoopScalarCtrlA1(mcs_acim_foc_a1_t *psFocACIM, 
                                  mcs_scalar_a1_t *psScalarACIM)
{		
	frac16_t f16TempScalar;
	
	/* required speed ramp */
    psScalarACIM->f16SpdMeRamp = MLIB_Conv_F16l(GFLIB_Ramp_F32(MLIB_Conv_F32s(psScalarACIM->f16SpdMeReq), 
    																	      &psScalarACIM->sRampParSpdMe));
        
    /* frequency generation by integrating required speed */
	psScalarACIM->f16PosEl = GFLIB_Integrator_F16(psScalarACIM->f16SpdMeRamp, &psScalarACIM->sSpeedMeInteg);	

    /* calculate required stator voltage */
	/* Udreq = 0; Uqreq = GainRpm2Volt * SpeedRamp, Uqreq limit = <Umin; DutyLim*DcBusAct> */
	psFocACIM->sUDQReq.f16D = FRAC16(0.0);
	f16TempScalar = MLIB_ShLSat_F16(MLIB_Mul_F16(psScalarACIM->f16GainRpm2Volt, 
									MLIB_AbsSat_F16(psScalarACIM->f16SpdMeRamp)), 
								    psScalarACIM->ui16ShiftRpm2Volt);
	
    psFocACIM->sUDQReq.f16Q = GFLIB_Limit_F16(f16TempScalar, 
    									      psScalarACIM->f16UMin, 
    									      MLIB_Mul_F16(psFocACIM->f16DutyLim, 
    									      psFocACIM->f16UDcBus));
    
    /* calculate sine & cosine for Park's transformation */
    psFocACIM->sSCFOC.f16Sin = GFLIB_Sin_F16(psScalarACIM->f16PosEl); 
    psFocACIM->sSCFOC.f16Cos = GFLIB_Cos_F16(psScalarACIM->f16PosEl); 
    
    /* call voltage control to calculate PWM duty cycles */
    MCS_ACIMFocCtrlVoltageA1(psFocACIM);
}

/*!
 * @brief ACIM field oriented voltage control
 *
 * This function is used to perform ACIM field oriented voltage control 
 * with position generated from Scalar control
 *
 * @param psFocACIM     The pointer of the ACIM FOC parameters structure
 *
 * @return None
 */
void MCS_ACIMFocCtrlVoltageA1(mcs_acim_foc_a1_t *psFocACIM)
{
    GMCLIB_2COOR_SINCOS_T_F16 sSCFOC;
    
    /* Clarke's transformation of stator currents */
    GMCLIB_Clark_F16(&psFocACIM->sIABC, 
    		         &psFocACIM->sIAlBe);
   
    /* run rotor flux observer on background to compare scalar and observed positions */
	AMCLIB_ACIMRotFluxObsrv_F16(&psFocACIM->sIAlBe, 
			 	 	 	 	 	&psFocACIM->sUAlBe, 
			 	 	 	 	 	&psFocACIM->sRFO);	

    /* run MRAS speed estimator on background */
	AMCLIB_ACIMSpeedMRAS_F16(&psFocACIM->sIAlBe, 
							 &psFocACIM->sRFO.sPsiRotSAlBe, 
							 psFocACIM->sSpdObs.f16RotPos, 
							 &psFocACIM->sSpdObs);

    /* calculate sine & cosine of rotor flux position */
    sSCFOC.f16Sin = GFLIB_Sin_F16(psFocACIM->sRFO.f16RotFluxPos);
    sSCFOC.f16Cos = GFLIB_Cos_F16(psFocACIM->sRFO.f16RotFluxPos);
	    
    /* Park's transformation of stator currents */
    GMCLIB_Park_F16(&psFocACIM->sIAlBe, 
    				&sSCFOC, 
    				&psFocACIM->sIDQ);
   
    /* transform required voltage into stator reference frame */
    GMCLIB_ParkInv_F16(&psFocACIM->sUDQReq, 
    				   &psFocACIM->sSCFOC,
    				   &psFocACIM->sUAlBe);
    
    GMCLIB_Park_F16(&psFocACIM->sUAlBe, 
    				&sSCFOC, 
    				&psFocACIM->sUDQReqScalar);
    
    /* dead-time compensation */
    psFocACIM->sUAlBeDTComp = psFocACIM->sUAlBe;
    
    if(psFocACIM->bFlagDTComp)
    {
        MCS_DTComp(&psFocACIM->sUAlBeDTComp, 
        		   &psFocACIM->sIABC, 
        		   psFocACIM->f16UDcBusFilt);
    }

    /* DC-bus voltage ripple elimination */
    GMCLIB_ElimDcBusRipFOC_F16(psFocACIM->f16UDcBus, &psFocACIM->sUAlBeDTComp, &psFocACIM->sUAlBeCmpFrac);
    
    /* space vector modulation */
    psFocACIM->ui16SectorSVM = GMCLIB_SvmStd_F16(&psFocACIM->sUAlBeCmpFrac, &psFocACIM->sDutyABC);
    
}

/*!
 * @brief ACIM field oriented voltage control
 *
 * This function is used to perform ACIM field oriented voltage control with 
 * preset sin and cos of rotor flux position.
 *
 * @param psFocACIM     The pointer of the ACIM FOC parameters structure
 *
 * @return None
 */
void MCS_ACIMFocCtrlVoltageA2(mcs_acim_foc_a1_t *psFocACIM)
{
    
    /* Clarke's transformation of stator currents */
    GMCLIB_Clark_F16(&psFocACIM->sIABC, &psFocACIM->sIAlBe);

    /* run rotor flux observer on background to compare scalar and observed positions */
	AMCLIB_ACIMRotFluxObsrv_F16(&psFocACIM->sIAlBe, 
								&psFocACIM->sUAlBe, 
								&psFocACIM->sRFO);	

	/* run MRAS speed estimator on background */
	AMCLIB_ACIMSpeedMRAS_F16(&psFocACIM->sIAlBe, 
							 &psFocACIM->sRFO.sPsiRotSAlBe, 
							 psFocACIM->sSpdObs.f16RotPos, 
							 &psFocACIM->sSpdObs);

    /* calculate sine & cosine of rotor flux position */
    psFocACIM->sSCFOC.f16Sin = GFLIB_Sin_F16(psFocACIM->sRFO.f16RotFluxPos);
    psFocACIM->sSCFOC.f16Cos = GFLIB_Cos_F16(psFocACIM->sRFO.f16RotFluxPos);

    /* Park's transformation of stator currents */
    GMCLIB_Park_F16(&psFocACIM->sIAlBe, &psFocACIM->sSCFOC, &psFocACIM->sIDQ);

    /* transform required voltage into stator reference frame */
    GMCLIB_ParkInv_F16(&psFocACIM->sUDQReq, &psFocACIM->sSCFOC, &psFocACIM->sUAlBe);
    
    /* dead-time compensation */
    psFocACIM->sUAlBeDTComp = psFocACIM->sUAlBe; 
    if(psFocACIM->bFlagDTComp)
    {
        MCS_DTComp(&psFocACIM->sUAlBeDTComp, &psFocACIM->sIABC, psFocACIM->f16UDcBusFilt);
    } 

    /* DC-bus voltage ripple elimination */
    GMCLIB_ElimDcBusRipFOC_F16(psFocACIM->f16UDcBus, &psFocACIM->sUAlBeDTComp, &psFocACIM->sUAlBeCmpFrac);
    
    /* space vector modulation */
    psFocACIM->ui16SectorSVM = GMCLIB_SvmStd_F16(&psFocACIM->sUAlBeCmpFrac, &psFocACIM->sDutyABC);
}

/*!
 * @brief ACIM field oriented current control
 *
 * This function is used to perform ACIM field oriented current control 
 *
 * @param psFocACIM     The pointer of the ACIM FOC parameters structure
 *
 * @return None
 */
void MCS_ACIMFocCtrlCurrentA1(mcs_acim_foc_a1_t *psFocACIM)
{	
    /* transform stator phase currents into stator reference frame */    
    GMCLIB_Clark_F16(&(psFocACIM->sIABC), &(psFocACIM->sIAlBe));
  
    /* do not estimate position if in startup*/
    if (!psFocACIM->bFlagSpdStart)
    {
    	/* run rotor flux observer */  	
    	AMCLIB_ACIMRotFluxObsrv_F16(&psFocACIM->sIAlBe, 
    			 	 	 	 	 	&psFocACIM->sUAlBe, 
    			 	 	 	 	 	&psFocACIM->sRFO);
    	
    	AMCLIB_ACIMSpeedMRAS_F16(&psFocACIM->sIAlBe, 
    							 &psFocACIM->sRFO.sPsiRotSAlBe, 
    							 psFocACIM->sSpdObs.f16RotPos, 
    							 &psFocACIM->sSpdObs);    	
					
        /* calculate sine & cosine of rotor flux position */
        psFocACIM->sSCFOC.f16Sin = GFLIB_Sin_F16(psFocACIM->sRFO.f16RotFluxPos);
        psFocACIM->sSCFOC.f16Cos = GFLIB_Cos_F16(psFocACIM->sRFO.f16RotFluxPos);
    }
    else
    {
        /* force rotor flux position to zero */
        psFocACIM->sSCFOC.f16Sin = FRAC16(0.0);
        psFocACIM->sSCFOC.f16Cos = FRAC16(1.0);
    }
    
    /* transform stator currents in stator ref. frame into rotating rotor flux ref. frame */
    GMCLIB_Park_F16(&psFocACIM->sIAlBe, &psFocACIM->sSCFOC, &psFocACIM->sIDQ);
    
    /* calculate d-axis voltage limitation */      
    psFocACIM->sPIParId.f16LowerLim = 
    		MLIB_MulNeg_F16(psFocACIM->f16DutyLim, psFocACIM->f16UDcBus);
    psFocACIM->sPIParId.f16UpperLim =   
    		MLIB_Mul_F16(psFocACIM->f16DutyLim, psFocACIM->f16UDcBus);
    
    /* calculate d-axis current control error */
    psFocACIM->sIDQErr.f16D = MLIB_Sub_F16(psFocACIM->sIDQReq.f16D, psFocACIM->sIDQ.f16D);
    
    /* calculate d-axis required voltage */
    psFocACIM->sUDQReq.f16D = GFLIB_CtrlPIpAW_F16(psFocACIM->sIDQErr.f16D, &psFocACIM->bFlagPIIdStopInt, &psFocACIM->sPIParId);
    
    /* calculate q-axis voltage limitation */
    psFocACIM->sPIParIq.f16UpperLim = 
    	GFLIB_Sqrt_F16l(MLIB_Msu4_F32ssss(
		psFocACIM->sPIParId.f16UpperLim, psFocACIM->sPIParId.f16UpperLim, 
		psFocACIM->sUDQReq.f16D, psFocACIM->sUDQReq.f16D));
    
    psFocACIM->sPIParIq.f16LowerLim = MLIB_Neg_F16(psFocACIM->sPIParIq.f16UpperLim);
      
    /* calculate q-axis current control error */
    psFocACIM->sIDQErr.f16Q = MLIB_SubSat_F16(psFocACIM->sIDQReq.f16Q, psFocACIM->sIDQ.f16Q);
    
    /* calculate q-axis required voltage */
    psFocACIM->sUDQReq.f16Q  = GFLIB_CtrlPIpAW_F16(psFocACIM->sIDQErr.f16Q, &psFocACIM->bFlagPIIqStopInt, &psFocACIM->sPIParIq); 
  
    /* transform required voltage into stator reference frame */
    GMCLIB_ParkInv_F16(&psFocACIM->sUDQReq, 
    		           &psFocACIM->sSCFOC, 
    		           &psFocACIM->sUAlBe);

    /* dead-time compensation */
    psFocACIM->sUAlBeDTComp = psFocACIM->sUAlBe;
    if(psFocACIM->bFlagDTComp)
    {
        MCS_DTComp(&psFocACIM->sUAlBeDTComp, 
        		   &psFocACIM->sIABC, 
        		   psFocACIM->f16UDcBusFilt);
    }

    /* DC-bus voltage ripple elimination */
    GMCLIB_ElimDcBusRipFOC_F16(psFocACIM->f16UDcBus, &psFocACIM->sUAlBeDTComp, &psFocACIM->sUAlBeCmpFrac);
    
    /* space vector modulation */
    psFocACIM->ui16SectorSVM = GMCLIB_SvmStd_F16(&psFocACIM->sUAlBeCmpFrac, &psFocACIM->sDutyABC);     
}

/*!
 * @brief ACIM FOC with speed and flux control
 *
 * This function is used to perform flux and speed control of ACIM
 *
 * @param psFocACIM     The pointer of the ACIM FOC parameters structure
 *
 * @return None
 */
void MCS_ACIMSpeedFluxCtrlA1(mcs_acim_foc_a1_t *psFocACIM, 
                             mcs_speed_flux_a1_t *psSpdFlux)
{ 	
	frac16_t f16IdTmp;
  
    /* check whether the startup is done */
    if(!psSpdFlux->bStartupDone)
    {
        /* still in startup - keep maximal rotor flux */
        psFocACIM->sIDQReq.f16D = psSpdFlux->f16IdStart;
        
        /* check for end of startup */
        if(psSpdFlux->f16SpdMeFilt > psSpdFlux->f16SpdMeReqMin)
            psSpdFlux->bStartupDone = TRUE;
    }
    else
    {
    	
    	/* startup done - employ MTPA and FW*/ 
    	/* field weakening */
    	f16IdTmp = AMCLIB_CtrlFluxWkng_F16(psFocACIM->sIDQErr.f16Q,
    	                                    psFocACIM->sUDQReq.f16Q,
                                            psFocACIM->sPIParIq.f16UpperLim,
                                            &psSpdFlux->sFluxWkng); 

    			
    	/* max torque per ampere calculation */
    	psSpdFlux->f16IdMTPA = AMCLIB_ACIMCtrlMTPA_F16(psFocACIM->sIDQ.f16Q,
                                           	   	   	   &psSpdFlux->sMTPA);

    	/* limit d-axis current in case of flux-weakening */
    	psFocACIM->sIDQReq.f16D = GFLIB_UpperLimit_F16(psSpdFlux->f16IdMTPA, 
    	                                               f16IdTmp);


    }
            
    /* Speed ramp generation */
    psSpdFlux->f16SpdMeRamp =  MLIB_Conv_F16l(GFLIB_Ramp_F32(MLIB_Conv_F32s(psSpdFlux->f16SpdMeReq), &psSpdFlux->sRampParSpdMe));    
        
    /* limit speed controller integrator in case the speed cannot be reached */
    psSpdFlux->bFlagPISpdMeStopInt = (psFocACIM->sPIParId.bLimFlag || 
                                      psFocACIM->sPIParIq.bLimFlag) && 
                                     (MLIB_AbsSat_F16(psSpdFlux->f16SpdMeRamp) >  
                                      MLIB_AbsSat_F16(psSpdFlux->f16SpdMeFilt));
    
    /* calculate control error */
    psSpdFlux->f16SpdMeErr = MLIB_SubSat_F16(psSpdFlux->f16SpdMeRamp, 
                                          psSpdFlux->f16SpdMeFilt);
 
    
    
    /* speed controller */	
	psFocACIM->sIDQReq.f16Q  = GFLIB_CtrlPIpAW_F16(psSpdFlux->f16SpdMeErr, &psSpdFlux->bFlagPISpdMeStopInt, &psSpdFlux->sPIParSpdMe);

}

/******************************************************************************
@brief   Dead-time compensation using LUT wit interpolation

@param   N/A

@return  N/A
******************************************************************************/
static void MCS_DTComp(GMCLIB_2COOR_ALBE_T_F16 *sUAlBeDTComp, 
                GMCLIB_3COOR_T_F16 *sIABC,
                frac16_t f16UDcBusFilt)
{
    register GMCLIB_3COOR_T_F16 sUABCErr;
    register frac16_t           f16UerrMax;
    register int16_t            i16CurrSign;

    /* maximal error voltage */
    f16UerrMax = f16UDtComp[0];

    /* compensate phase A */
    i16CurrSign = (sIABC->f16A > DTCOMP_I_RANGE) - 
                  (sIABC->f16A < -DTCOMP_I_RANGE);
    if (!i16CurrSign)
        sUABCErr.f16A = GFLIB_Lut1D_F16(sIABC->f16A,
                                        f16UDtComp, 
                                        DTCOMP_TABLE_SIZE);
    else
    	sUABCErr.f16A = i16CurrSign * 
    		MLIB_MulRnd_F16(MLIB_Sub_F16(MLIB_AbsSat_F16(sIABC->f16A), DTCOMP_I_RANGE), 
    						MLIB_SubSat_F16(DTCOMP_LINCOEFF, f16UerrMax));

    /* compensate phase B */
    i16CurrSign = (sIABC->f16B > DTCOMP_I_RANGE) - 
                  (sIABC->f16B < -DTCOMP_I_RANGE);
    if (!i16CurrSign)
        sUABCErr.f16B = GFLIB_Lut1D_F16(sIABC->f16B,
                                        f16UDtComp, 
                                        DTCOMP_TABLE_SIZE);
    else
    	sUABCErr.f16B = i16CurrSign * 
    		MLIB_MulRnd_F16(MLIB_Sub_F16(MLIB_AbsSat_F16(sIABC->f16B), DTCOMP_I_RANGE), 
    						MLIB_SubSat_F16(DTCOMP_LINCOEFF, f16UerrMax));

    /* compensate phase C */
    i16CurrSign = (sIABC->f16C > DTCOMP_I_RANGE) - 
                  (sIABC->f16C < -DTCOMP_I_RANGE);
    if (!i16CurrSign)
        sUABCErr.f16C = GFLIB_Lut1D_F16(sIABC->f16C,
                                        f16UDtComp, 
                                        DTCOMP_TABLE_SIZE);
    else
    	sUABCErr.f16C = i16CurrSign * 
    		MLIB_MulRnd_F16(MLIB_Sub_F16(MLIB_AbsSat_F16(sIABC->f16C), DTCOMP_I_RANGE), 
    						MLIB_SubSat_F16(DTCOMP_LINCOEFF, f16UerrMax));

    /* add compensation voltages */
    sUAlBeDTComp->f16Alpha = MLIB_MacRnd_F16(sUAlBeDTComp->f16Alpha,
    	FRAC16(0.333333333333), 
    	MLIB_MulRnd_F16(f16UDcBusFilt, MLIB_Add4Sat_F16(sUABCErr.f16A, 
    													sUABCErr.f16A, 
                             	 	 	 	 	 	 	MLIB_NegSat_F16(sUABCErr.f16B), 
                             	 	 	 	 	 	 	MLIB_NegSat_F16(sUABCErr.f16C))));

    sUAlBeDTComp->f16Beta = MLIB_MacRnd_F16(sUAlBeDTComp->f16Beta, 
    	FRAC16(0.5773502691896), 
    	MLIB_MulRnd_F16(f16UDcBusFilt,MLIB_SubSat_F16(sUABCErr.f16B, sUABCErr.f16C)));
}
