/*
 * Copyright 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2019 NXP
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include "m1_sm_ref_sol.h"
#include "mcdrv.h"


#include "qs.h"
#include "gpio.h"

/*******************************************************************************
 * Definitions
 ******************************************************************************/

#define M1_SVM_SECTOR_DEFAULT (2) /* default SVM sector */
#define M1_T_MAX (550.0F)
/*******************************************************************************
 * Prototypes
 ******************************************************************************/

static void M1_StateFaultFast(void);
static void M1_StateInitFast(void);
static void M1_StateStopFast(void);
static void M1_StateRunFast(void);

static void M1_StateFaultSlow(void);
static void M1_StateInitSlow(void);
static void M1_StateStopSlow(void);
static void M1_StateRunSlow(void);

static void M1_TransFaultStop(void);
static void M1_TransInitFault(void);
static void M1_TransInitStop(void);
static void M1_TransStopFault(void);
static void M1_TransStopRun(void);
static void M1_TransRunFault(void);
static void M1_TransRunStop(void);

static void M1_StateRunCalibFast(void);
static void M1_StateRunReadyFast(void);
static void M1_StateRunStartupFast(void);
static void M1_StateRunSpinFast(void);
static void M1_StateRunFreewheelFast(void);

static void M1_StateRunCalibSlow(void);
static void M1_StateRunReadySlow(void);
static void M1_StateRunStartupSlow(void);
static void M1_StateRunSpinSlow(void);
static void M1_StateRunFreewheelSlow(void);

static void M1_TransRunCalibReady(void);
static void M1_TransRunCalibMeasure(void);
static void M1_TransRunReadyStartup(void);
static void M1_TransRunStartupSpin(void);
static void M1_TransRunStartupFreewheel(void);
static void M1_TransRunSpinFreewheel(void);
static void M1_TransRunFreewheelReady(void);

static void M1_ClearFOCVariables(void);

static void M1_FaultDetection(void);

/*******************************************************************************
 * Variables
 ******************************************************************************/

/*! @brief Main control structure */
mcdef_acim_t g_sM1Drive;

/*! @brief Main application switch */
bool_t g_bM1SwitchAppOnOff;

/*! @brief M1 structure */
m1_run_substate_t g_eM1StateRun;

/*! @brief FreeMASTER scales */
volatile float g_fltM1voltageScale;
volatile float g_fltM1DCBvoltageScale;
volatile float g_fltM1currentScale;
volatile float g_fltM1speedScale;
volatile float g_fltM1temperatureScale;

/*! @brief Application state machine table - fast */
const sm_app_state_fcn_t s_STATE_FAST = { M1_StateFaultFast, M1_StateInitFast,
		M1_StateStopFast, M1_StateRunFast };

/*! @brief Application state machine table - slow */
const sm_app_state_fcn_t s_STATE_SLOW = { M1_StateFaultSlow, M1_StateInitSlow,
		M1_StateStopSlow, M1_StateRunSlow };

/*! @brief Application sub-state function field - fast */
static const pfcn_void_void s_M1_STATE_RUN_TABLE_FAST[6] = {
		M1_StateRunCalibFast, M1_StateRunReadyFast, M1_StateRunStartupFast,
		M1_StateRunSpinFast, M1_StateRunFreewheelFast };

/*! @brief Application sub-state function field - slow */
static const pfcn_void_void s_M1_STATE_RUN_TABLE_SLOW[6] = {
		M1_StateRunCalibSlow, M1_StateRunReadySlow, M1_StateRunStartupSlow,
		M1_StateRunSpinSlow, M1_StateRunFreewheelSlow };

/*! @brief Application state-transition functions field  */
static const sm_app_trans_fcn_t s_TRANS = { M1_TransFaultStop,
		M1_TransInitFault, M1_TransInitStop, M1_TransStopFault, M1_TransStopRun,
		M1_TransRunFault, M1_TransRunStop };

/*! @brief  State machine structure declaration and initialization */
sm_app_ctrl_t g_sM1Ctrl = {
	/* g_sM1Ctrl.psState, User state functions  */
	&s_STATE_FAST,
	
	/* g_sM1Ctrl.psState, User state functions  */
	&s_STATE_SLOW,
	
	/* g_sM1Ctrl..psTrans, User state-transition functions */
	&s_TRANS,
	
	/* g_sM1Ctrl.uiCtrl, Default no control command */
	SM_CTRL_NONE,
	
	/* g_sM1Ctrl.eState, Default state after reset */
kSM_AppInit };

/*******************************************************************************
 * Code
 ******************************************************************************/

/*!
 * @brief Fault state called in fast state machine
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_StateFaultFast(void) {
	/* read ADC results (ADC triggered by HW trigger from PDB) */
	/* get all adc samples - DC-bus voltage, current, and raw IPM temperature */
	M1_MCDRV_ADC_GET(&g_sM1AdcSensor);

	/* Sampled DC-Bus voltage filter */
	g_sM1Drive.sFocACIM.f16UDcBusFilt = GDFLIB_FilterIIR1_F16(g_sM1Drive.sFocACIM.f16UDcBus, &g_sM1Drive.sFocACIM.sFiltParUDcBus);

	/* Braking resistor control */
	if (g_sM1Drive.sFocACIM.f16UDcBusFilt > g_sM1Drive.sFaultThresholds.f16UDcBusTrip)
	{
		M1_BRAKE_SET(); 		
	} 
	else
	{
		M1_BRAKE_CLEAR();			
	}
 
	/* Disable user application switch */
	g_bM1SwitchAppOnOff = FALSE;

	/* clear fault state manually from FreeMASTER */
	if (g_sM1Drive.bFaultClearMan) {
		/* Clear fault state */
		g_sM1Ctrl.uiCtrl |= SM_CTRL_FAULT_CLEAR;
		g_sM1Drive.bFaultClearMan = FALSE;
	}

	/* PWM peripheral update */
	M1_MCDRV_PWM3PH_SET(&g_sM1Pwm3ph);

	/* Detects faults */
	M1_FaultDetection();
}

/*!
 * @brief State initialization routine called in fast state machine
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_StateInitFast(void) {
	/* Type the code to do when in the INIT state */

	/* init scalar control */
	g_sM1Drive.sScalarCtrl.f16GainRpm2Volt = M1_SCALAR_VHZ_FACTOR_GAIN;
	g_sM1Drive.sScalarCtrl.ui16ShiftRpm2Volt = M1_SCALAR_VHZ_FACTOR_SHIFT;	
	g_sM1Drive.sScalarCtrl.f16UMin = M1_SCALAR_U_MIN;
	g_sM1Drive.sScalarCtrl.sRampParSpdMe.f32RampUp = M1_SCALAR_RAMP_UP;
	g_sM1Drive.sScalarCtrl.sRampParSpdMe.f32RampDown = M1_SCALAR_RAMP_DOWN;
	g_sM1Drive.sScalarCtrl.sSpeedMeInteg.a32Gain = M1_SCALAR_INTEG_GAIN;

	/* init RFO */
	g_sM1Drive.sFocACIM.sRFO.a32KrInvGain       = M1_RFO_KR_INV_GAIN;
	g_sM1Drive.sFocACIM.sRFO.a32KrLsTotLeakGain = M1_RFO_LS_TOT_LEAK_GAIN;
	g_sM1Drive.sFocACIM.sRFO.f32KPsiRA1Gain = M1_RFO_PSI_RA1_GAIN;
	g_sM1Drive.sFocACIM.sRFO.f32KPsiRB1Gain = M1_RFO_PSI_RB1_GAIN;
	g_sM1Drive.sFocACIM.sRFO.f32KPsiSA1Gain = M1_RFO_PSI_SA1_GAIN;
	g_sM1Drive.sFocACIM.sRFO.f32KPsiSA2Gain = M1_RFO_PSI_SA2_GAIN;
	g_sM1Drive.sFocACIM.sRFO.f16KRsEst =      M1_RFO_RS_EST;
	g_sM1Drive.sFocACIM.sRFO.sCtrl.a32PGain = M1_RFO_COMP_KP_GAIN;
	g_sM1Drive.sFocACIM.sRFO.sCtrl.a32IGain = M1_RFO_COMP_KI_GAIN;
	g_sM1Drive.sFocACIM.sRFO.a32TorqueGain =  M1_RFO_TRQ_CNST;
		    
    /* init  MRAS */
    g_sM1Drive.sFocACIM.sSpdObs.f32KPsiRA1Gain = M1_MRAS_PSI_RA1;
    g_sM1Drive.sFocACIM.sSpdObs.f32KPsiRB1Gain = M1_MRAS_PSI_RB1;
    g_sM1Drive.sFocACIM.sSpdObs.f32KImaxGain = M1_MRAS_IMAX_GAIN;
    g_sM1Drive.sFocACIM.sSpdObs.sCtrl.a32PGain = M1_MRAS_KP_GAIN;
    g_sM1Drive.sFocACIM.sSpdObs.sCtrl.a32IGain = M1_MRAS_KI_GAIN;
    g_sM1Drive.sFocACIM.sSpdObs.sSpeedElIIR1Param.sFltCoeff.f32B0 = M1_SPEED_IIR_B0;
    g_sM1Drive.sFocACIM.sSpdObs.sSpeedElIIR1Param.sFltCoeff.f32B1 = M1_SPEED_IIR_B1;
    g_sM1Drive.sFocACIM.sSpdObs.sSpeedElIIR1Param.sFltCoeff.f32A1 = M1_SPEED_IIR_A1;
    g_sM1Drive.sFocACIM.sSpdObs.sSpeedInteg.a32Gain = M1_SCALAR_INTEG_GAIN;
    /* default MCAT control mode after reset */
    g_sM1Drive.eControl = kControlMode_SpeedFOC;

	/* Timing control and general variables */
	g_sM1Drive.ui32CounterState = M1_FAULT_DURATION;
	g_sM1Drive.ui32TimeFullSpeedFreeWheel = M1_FREEWHEEL_DURATION;
	g_sM1Drive.ui32TimeCalibration = M1_CALIB_DURATION;
	g_sM1Drive.ui32TimeFaultRelease = M1_FAULT_DURATION;
	g_bM1SwitchAppOnOff = FALSE;

	/* fault set to init states */
	M1_FAULT_CLEAR_ALL(g_sM1Drive.sFaultIdCaptured);
	M1_FAULT_CLEAR_ALL(g_sM1Drive.sFaultIdPending);

	/* fault thresholds */
	g_sM1Drive.sFaultThresholds.f16UDcBusOver = M1_U_DCB_OVERVOLTAGE;
	g_sM1Drive.sFaultThresholds.f16UDcBusUnder = M1_U_DCB_UNDERVOLTAGE;
	g_sM1Drive.sFaultThresholds.f16UDcBusTrip = M1_U_DCB_TRIP;
	g_sM1Drive.sFaultThresholds.f16SpeedOver = M1_N_OVERSPEED;
		
	/* defined scaling for FreeMASTER */
	g_fltM1voltageScale = M1_U_MAX;
	g_fltM1currentScale = M1_I_MAX;
	g_fltM1DCBvoltageScale = M1_U_DCB_MAX;
	g_fltM1speedScale = M1_N_MAX;
	g_fltM1temperatureScale = M1_T_MAX;

	/* setup PWM control */
	g_sM1Drive.sFocACIM.ui16SectorSVM = M1_SVM_SECTOR_DEFAULT;
	g_sM1Drive.sFocACIM.sDutyABC.f16A = FRAC16(0.5);
	g_sM1Drive.sFocACIM.sDutyABC.f16B = FRAC16(0.5);
	g_sM1Drive.sFocACIM.sDutyABC.f16C = FRAC16(0.5);
	g_sM1Drive.sFocACIM.f16DutyLim = M1_CLOOP_LIMIT;

	/* enable dead-time compensation */
	g_sM1Drive.sFocACIM.bFlagDTComp = FALSE;

	/* DC-bus voltage measurement*/
	g_sM1Drive.sFocACIM.f16UDcBus = FRAC16(0.0);
	g_sM1Drive.sFocACIM.f16UDcBusFilt = FRAC16(0.0);
	g_sM1Drive.sFocACIM.sFiltParUDcBus.sFltCoeff.f32B0 = M1_UDCB_IIR_B0;
	g_sM1Drive.sFocACIM.sFiltParUDcBus.sFltCoeff.f32B1 = M1_UDCB_IIR_B1;
	g_sM1Drive.sFocACIM.sFiltParUDcBus.sFltCoeff.f32A1 = M1_UDCB_IIR_A1;
	g_sM1Drive.sFocACIM.sFiltParUDcBus.f16FltBfrX[0] = FRAC16(M1_U_DCB_UNDERVOLTAGE / 2.0 + M1_U_DCB_OVERVOLTAGE / 2.0);
	g_sM1Drive.sFocACIM.sFiltParUDcBus.f32FltBfrY[0] = MLIB_Conv_F32s(g_sM1Drive.sFocACIM.sFiltParUDcBus.f16FltBfrX[0]);

	g_sM1Drive.sSpdFlux.sRampParSpdMe.f32RampUp = M1_SPEED_RAMP_UP;
	g_sM1Drive.sSpdFlux.sRampParSpdMe.f32RampDown = M1_SPEED_RAMP_DOWN;	

	g_sM1Drive.sSpdFlux.f16SpdMeReqMin = M1_SPEED_REQ_MIN;
	g_sM1Drive.sSpdFlux.f16SpdMeReqMax = M1_SPEED_REQ_MAX;

	g_sM1Drive.sSpdFlux.sPIParSpdMe.a32PGain = M1_SPEED_PI_KP_GAIN;
	g_sM1Drive.sSpdFlux.sPIParSpdMe.a32IGain = M1_SPEED_PI_KI_GAIN;
	g_sM1Drive.sSpdFlux.sPIParSpdMe.f16UpperLim = M1_SPEED_LIMIT_HIGH;
	g_sM1Drive.sSpdFlux.sPIParSpdMe.f16LowerLim = M1_SPEED_LIMIT_LOW;


	g_sM1Drive.sSpdFlux.f16IdReqMin = M1_FLUX_ID_MIN; 
	g_sM1Drive.sSpdFlux.f16IdReqMax = M1_FLUX_ID_MAX; 
	
	
	g_sM1Drive.sSpdFlux.f16IdStart = M1_FLUX_ID_START;
    g_sM1Drive.sSpdFlux.f16IdStartMinPct = M1_FLUX_ID_START_MINPCT;
	g_sM1Drive.sSpdFlux.sMTPA.f16LowerLim = M1_FLUX_ID_MIN;
	g_sM1Drive.sSpdFlux.sMTPA.f16UpperLim = M1_FLUX_ID_MAX;
	g_sM1Drive.sSpdFlux.sMTPA.sIdExpParam.f32A = M1_FLUX_MTPA_FILT_COEFF;
	
	g_sM1Drive.sSpdFlux.sFluxWkng.sFWPiParam.a32PGain = M1_FLUX_FW_KP_GAIN;
	g_sM1Drive.sSpdFlux.sFluxWkng.sFWPiParam.a32IGain = M1_FLUX_FW_KI_GAIN;
	g_sM1Drive.sSpdFlux.sFluxWkng.sFWPiParam.f16UpperLim = M1_FLUX_ID_MAX;
	g_sM1Drive.sSpdFlux.sFluxWkng.sFWPiParam.f16LowerLim = M1_FLUX_FW_ID_MIN;
	g_sM1Drive.sSpdFlux.sFluxWkng.sIqErrIIR1Param.sFltCoeff.f32B0 = M1_FLUX_IIR_B0;
	g_sM1Drive.sSpdFlux.sFluxWkng.sIqErrIIR1Param.sFltCoeff.f32B1 = M1_FLUX_IIR_B1;
	g_sM1Drive.sSpdFlux.sFluxWkng.sIqErrIIR1Param.sFltCoeff.f32A1 = M1_FLUX_IIR_A1;
	g_sM1Drive.sSpdFlux.sFluxWkng.pbStopIntegFlag = &g_sM1Drive.sSpdFlux.bFlagPIFWStopInt;

	/* current control */
	g_sM1Drive.sFocACIM.sPIParId.a32PGain = M1_D_KP_GAIN;
	g_sM1Drive.sFocACIM.sPIParId.a32IGain = M1_D_KI_GAIN;
	g_sM1Drive.sFocACIM.sPIParIq.a32PGain = M1_Q_KP_GAIN;
	g_sM1Drive.sFocACIM.sPIParIq.a32IGain = M1_Q_KI_GAIN;

	/* Clear rest of variables  */
	M1_ClearFOCVariables();

	/* Application timing */
	g_sM1Drive.ui16FastCtrlLoopFreq = (M1_PWM_FREQ/M1_FOC_FREQ_VS_PWM_FREQ);
	g_sM1Drive.ui16SlowCtrlLoopFreq = M1_SLOW_LOOP_FREQ;
	
	/* Init sensors/actuators pointers */
	M1_SET_PTR_DUTY(g_sM1Drive.sFocACIM.sDutyABC);
	M1_SET_PTR_U_DC_BUS(g_sM1Drive.sFocACIM.f16UDcBus);
	M1_SET_PTR_I_ABC(g_sM1Drive.sFocACIM.sIABCFrac);
	M1_SET_PTR_SECTOR(g_sM1Drive.sFocACIM.ui16SectorSVM);
	M1_SET_PTR_AUX_CHAN(g_sM1Drive.f16AdcAuxSample);

	/* INIT_DONE command */
	g_sM1Ctrl.uiCtrl |= SM_CTRL_INIT_DONE;
}

/*!
 * @brief Stop state routine called in fast state machine
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_StateStopFast(void) {
	/* read 3-phase motor currents, DC-bus voltage and raw IPM temperature */
	M1_MCDRV_ADC_GET(&g_sM1AdcSensor);
	/* Sampled DC-Bus voltage filter */
	g_sM1Drive.sFocACIM.f16UDcBusFilt = GDFLIB_FilterIIR1_F16(g_sM1Drive.sFocACIM.f16UDcBus, &g_sM1Drive.sFocACIM.sFiltParUDcBus);

	/* If the user switches on  or set non-zero speed*/
	if ((g_bM1SwitchAppOnOff != 0)) {
		/* Set the switch on */
		g_bM1SwitchAppOnOff = TRUE;

		/* Start command */
		g_sM1Ctrl.uiCtrl |= SM_CTRL_START;
	}
	/* Braking resistor control */
	if (g_sM1Drive.sFocACIM.f16UDcBusFilt > g_sM1Drive.sFaultThresholds.f16UDcBusTrip)
	{
		M1_BRAKE_SET();		
	} 
	else
	{
		M1_BRAKE_CLEAR();		
	}

	/* Detect fault */
	M1_FaultDetection();

	/* If a fault occurred */
	if (g_sM1Drive.sFaultIdPending) {
		/* Switches to the FAULT state */
		g_sM1Ctrl.uiCtrl |= SM_CTRL_FAULT;
	}

	/* PWM peripheral update */
	M1_MCDRV_PWM3PH_SET(&g_sM1Pwm3ph);
}

/*!
 * @brief Run state routine called in fast state machine
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_StateRunFast(void) {
	/* read 3-phase motor currents, DC-bus voltage and raw IPM temperature */
	M1_MCDRV_ADC_GET(&g_sM1AdcSensor);

	/* If the user switches off */
	if (!g_bM1SwitchAppOnOff) {
		/* Stop command */
		g_sM1Ctrl.uiCtrl |= SM_CTRL_STOP;
	}

	/* detect fault */
	M1_FaultDetection();

	/* If a fault occurred */
	if (g_sM1Drive.sFaultIdPending) {
		/* Switches to the FAULT state */
		g_sM1Ctrl.uiCtrl |= SM_CTRL_FAULT;
	}

	/* convert phase currents from fractional measured values to float */
	g_sM1Drive.sFocACIM.sIABC.f16A = g_sM1Drive.sFocACIM.sIABCFrac.f16A;
	g_sM1Drive.sFocACIM.sIABC.f16B = g_sM1Drive.sFocACIM.sIABCFrac.f16B;
	g_sM1Drive.sFocACIM.sIABC.f16C = g_sM1Drive.sFocACIM.sIABCFrac.f16C;

	/* convert voltages from fractional measured values to float */
	g_sM1Drive.sFocACIM.f16UDcBus = g_sM1Drive.sFocACIM.f16UDcBus;

	/* filter DC-bus voltage */
	g_sM1Drive.sFocACIM.f16UDcBusFilt = GDFLIB_FilterIIR1_F16(
			g_sM1Drive.sFocACIM.f16UDcBus, &g_sM1Drive.sFocACIM.sFiltParUDcBus);

	/* Braking resistor control */
	if (g_sM1Drive.sFocACIM.f16UDcBusFilt > g_sM1Drive.sFaultThresholds.f16UDcBusTrip)
	{
		M1_BRAKE_SET();		
	} 
	else
	{
		M1_BRAKE_CLEAR();		
	}

	/* read speed */
	g_sM1Drive.sSpdFlux.f16SpdMeFilt = g_sM1Drive.sFocACIM.sSpdObs.f16SpeedElIIR1;

	/* Run sub-state function */
	s_M1_STATE_RUN_TABLE_FAST[g_eM1StateRun]();

	/* PWM peripheral update */
	M1_MCDRV_PWM3PH_SET(&g_sM1Pwm3ph);

	/* set current sensor for  sampling */
	M1_MCDRV_CURR_3PH_CHAN_ASSIGN(&g_sM1AdcSensor);
}

/*!
 * @brief Fault state routine called in slow state machine
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_StateFaultSlow(void) {
	/* after fault condition ends wait defined time to clear fault state */
	if (!M1_FAULT_ANY(g_sM1Drive.sFaultIdPending)) {
		if (--g_sM1Drive.ui32CounterState == 0) {
			/* Clear fault state */
			g_sM1Ctrl.uiCtrl |= SM_CTRL_FAULT_CLEAR;			
		}
	} else {
		g_sM1Drive.ui32CounterState = g_sM1Drive.ui32TimeFaultRelease;
	}
	
    /* get IPM temperature in Celsius degrees */
    g_sM1Drive.f16IPMTemperature = GetIPMTemperature(g_sM1Drive.f16AdcAuxSample);			
}

/*!
 * @brief Fault state routine called in slow state machine
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_StateInitSlow(void) {
}

/*!
 * @brief Stop state routine called in slow state machine
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_StateStopSlow(void) {
	
    /* get IPM temperature in Celsius degrees */
    g_sM1Drive.f16IPMTemperature = GetIPMTemperature(g_sM1Drive.f16AdcAuxSample);	
	
}

/*!
 * @brief Run state routine called in slow state machine
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_StateRunSlow(void) {
	/* Run sub-state function */
	s_M1_STATE_RUN_TABLE_SLOW[g_eM1StateRun]();
	
    /* get IPM temperature in Celsius degrees */
    g_sM1Drive.f16IPMTemperature = GetIPMTemperature(g_sM1Drive.f16AdcAuxSample);	
    
}

/*!
 * @brief Transition from Fault to Stop state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_TransFaultStop(void) {
	/* Type the code to do when going from the FAULT to the INIT state */
	M1_FAULT_CLEAR_ALL(g_sM1Drive.sFaultIdCaptured);

	/* Clear all FOC variables, init filters, etc. */
	M1_ClearFOCVariables();
}

/*!
 * @brief Transition from Init to Fault state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_TransInitFault(void) {
	/* disable PWM outputs */
	M1_MCDRV_PWM3PH_DIS(&g_sM1Pwm3ph);
	g_sM1Drive.ui32CounterState = g_sM1Drive.ui32TimeFaultRelease;
}

/*!
 * @brief Transition from Init to Stop state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_TransInitStop(void) {
	/* type the code to do when going from the INIT to the STOP state */
	/* disable PWM outputs */
	M1_MCDRV_PWM3PH_DIS(&g_sM1Pwm3ph);
}

/*!
 * @brief Transition from Stop to Fault state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_TransStopFault(void) {
	/* type the code to do when going from the STOP to the FAULT state */
	/* load the fault release time to counter */
	g_sM1Drive.ui32CounterState = g_sM1Drive.ui32TimeFaultRelease;
}

/*!
 * @brief Transition from Stop to Run state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_TransStopRun(void) {
	/* type the code to do when going from the STOP to the RUN state */
	/* 50% duty cycle */
	g_sM1Drive.sFocACIM.sDutyABC.f16A = FRAC16(0.5);
	g_sM1Drive.sFocACIM.sDutyABC.f16B = FRAC16(0.5);
	g_sM1Drive.sFocACIM.sDutyABC.f16C = FRAC16(0.5);

	/* PWM duty cycles calculation and update */
	M1_MCDRV_PWM3PH_SET(&g_sM1Pwm3ph);

	/* Clear offset filters */
	M1_MCDRV_CURR_3PH_CALIB_INIT(&g_sM1AdcSensor);

	/* Enable PWM output */
	M1_MCDRV_PWM3PH_EN(&g_sM1Pwm3ph);

	/* pass calibration routine duration to state counter*/
	g_sM1Drive.ui32CounterState = g_sM1Drive.ui32TimeCalibration;

	/* Calibration sub-state when transition to RUN */
	g_eM1StateRun = kRunState_Calib;

	/* Acknowledge that the system can proceed into the RUN state */
	g_sM1Ctrl.uiCtrl |= SM_CTRL_RUN_ACK;
}

/*!
 * @brief Transition from Run to Fault state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_TransRunFault(void) {
	/* type the code to do when going from the RUN to the FAULT state */
	/* disable PWM output */
	M1_MCDRV_PWM3PH_DIS(&g_sM1Pwm3ph);
	
	g_sM1Drive.ui32CounterState = g_sM1Drive.ui32TimeFaultRelease;	

	/* Clear FOC variables */
	M1_ClearFOCVariables();
}

/*!
 * @brief Transition from Run to Stop state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_TransRunStop(void) {
	/* type the code to do when going from the RUN to the STOP state */
	/* disable PWM outputs */
	M1_MCDRV_PWM3PH_DIS(&g_sM1Pwm3ph);

	/* clear FOC variables */
	M1_ClearFOCVariables();

	/* Acknowledge that the system can proceed into the STOP state */
	g_sM1Ctrl.uiCtrl |= SM_CTRL_STOP_ACK;
}

/*!
 * @brief Calibration process called in fast state machine as Run sub state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_StateRunCalibFast(void) {
	/* Type the code to do when in the RUN CALIB sub-state
	   performing ADC offset calibration */

	/* call offset measurement */
	M1_MCDRV_CURR_3PH_CALIB(&g_sM1AdcSensor);

	/* change SVM sector in range <1;6> to measure all AD channel mapping combinations */
	if (++g_sM1Drive.sFocACIM.ui16SectorSVM > 6)
		g_sM1Drive.sFocACIM.ui16SectorSVM = 1;
}

/*!
 * @brief Ready state called in fast state machine as Run sub state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_StateRunReadyFast(void) {
	/* Type the code to do when in the RUN READY sub-state */
	frac16_t f16SpdMeReqAbs;

	/* MCAT control structure switch */
	switch (g_sM1Drive.eControl) {
	case kControlMode_Scalar:
		if (g_sM1Drive.sScalarCtrl.f16SpdMeReq != 0) {
			g_sM1Drive.sFocACIM.sUDQReq.f16D = FRAC16(0.0);
			g_sM1Drive.sFocACIM.sUDQReq.f16Q = FRAC16(0.0);

			/* Transition to the RUN STARTUP sub-state */
			M1_TransRunReadyStartup();
		}
		break;

	case kControlMode_VoltageFOC:
		if ((g_sM1Drive.sFocACIM.sUDQReq.f16D != 0)
				&& (g_sM1Drive.sFocACIM.sUDQReq.f16Q != 0)) {
			/* Transition to the RUN STARTUP sub-state */
			M1_TransRunReadyStartup();
		}
		break;

	case kControlMode_CurrentFOC:
		if ((g_sM1Drive.sFocACIM.sIDQReq.f16D != 0)
				&& (g_sM1Drive.sFocACIM.sIDQReq.f16Q != 0)) {
			/* Transition to the RUN STARTUP sub-state */
			M1_TransRunReadyStartup();
		}
		break;

	case kControlMode_SpeedFOC:
	default:
		/* speed FOC */
		f16SpdMeReqAbs = MLIB_AbsSat_F16(g_sM1Drive.sSpdFlux.f16SpdMeReq);
		if (f16SpdMeReqAbs < g_sM1Drive.sSpdFlux.f16SpdMeReqMin) {
			g_sM1Drive.sSpdFlux.f16SpdMeReq = FRAC16(0.0);
		} else {
			/* limit to maximal speed */
			if (f16SpdMeReqAbs > g_sM1Drive.sSpdFlux.f16SpdMeReqMax) {
				g_sM1Drive.sSpdFlux.f16SpdMeReq = MLIB_MulRndSat_F16(
						g_sM1Drive.sSpdFlux.f16SpdMeReqMax,
						MLIB_Sign_F16(g_sM1Drive.sSpdFlux.f16SpdMeReq));
			}

			/* transition to the RUN STARTUP sub-state */
			M1_TransRunReadyStartup();
		}
	}
}

/*!
 * @brief Start-up process called in fast state machine as Run sub state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_StateRunStartupFast(void) {
	/* Type the code to do when in the RUN STARTUP sub-state */

	/* MCAT control structure switch */
	switch (g_sM1Drive.eControl) {
	case kControlMode_Scalar:
	case kControlMode_VoltageFOC:
	case kControlMode_CurrentFOC:
		/* switch directly to SPIN state */
		M1_TransRunStartupSpin();
		break;

	case kControlMode_SpeedFOC:
	default:

		/* if user changes speed out of limits */
		if ((MLIB_AbsSat_F16(g_sM1Drive.sSpdFlux.f16SpdMeReq)
				< g_sM1Drive.sSpdFlux.f16SpdMeReqMin)
				|| (MLIB_AbsSat_F16(g_sM1Drive.sSpdFlux.f16SpdMeReq)
						>= g_sM1Drive.sSpdFlux.f16SpdMeReqMax)) {
			/* switch to FREEWHEEL state */
			M1_TransRunStartupFreewheel();
			return;
		}

		/* reset RFO and MRAS */
		AMCLIB_ACIMRotFluxObsrvInit_F16(&g_sM1Drive.sFocACIM.sRFO);
		AMCLIB_ACIMSpeedMRASInit_F16(&g_sM1Drive.sFocACIM.sSpdObs);

		/* generate no torque and magnetize rotor */
		g_sM1Drive.sFocACIM.sIDQReq.f16Q = 0;
		g_sM1Drive.sFocACIM.sIDQReq.f16D = g_sM1Drive.sSpdFlux.f16IdStart;

		/* current control */
		MCS_ACIMFocCtrlCurrentA1(&g_sM1Drive.sFocACIM);

		/* if sufficient time elapsed go to spin */
		if (g_sM1Drive.sFocACIM.sIDQ.f16D
				> (MLIB_MulRnd_F16(g_sM1Drive.sSpdFlux.f16IdStartMinPct,
						g_sM1Drive.sSpdFlux.f16IdStart))) {
			M1_TransRunStartupSpin();
		}
		break;
	}
}

/*!
 * @brief Spin state called in fast state machine as Run sub state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_StateRunSpinFast(void) {
	/* Type the code to do when in the RUN SPIN sub-state */
	/* MCAT control structure switch */
	switch (g_sM1Drive.eControl) {
	case kControlMode_Scalar:
				
		/* scalar control */
		MCS_ACIMOpenLoopScalarCtrlA1(&g_sM1Drive.sFocACIM, &g_sM1Drive.sScalarCtrl);
		
		/* check whether the speed is below minimum */
		if (g_sM1Drive.sScalarCtrl.f16SpdMeReq == 0) {
			/* Sub-state RUN FREEWHEEL */
			M1_TransRunSpinFreewheel();
		}
		break;

	case kControlMode_VoltageFOC:
		/* voltage FOC */
		MCS_ACIMFocCtrlVoltageA2(&g_sM1Drive.sFocACIM);

		/* check whether the required voltages are non-zero */
		if ((g_sM1Drive.sFocACIM.sUDQReq.f16D == 0)
				|| (g_sM1Drive.sFocACIM.sUDQReq.f16Q == 0)) {
			/* sub-state RUN FREEWHEEL */
			M1_TransRunSpinFreewheel();
		}
		break;

	case kControlMode_CurrentFOC:
		/* current FOC */
		MCS_ACIMFocCtrlCurrentA1(&g_sM1Drive.sFocACIM);

		/* check whether the required currents are non-zero */
		if ((g_sM1Drive.sFocACIM.sIDQReq.f16D == 0)
				|| (g_sM1Drive.sFocACIM.sIDQReq.f16Q == 0)) {
			/* sub-state RUN FREEWHEEL */
			M1_TransRunSpinFreewheel();
		}
		break;

	case kControlMode_SpeedFOC:
	default:
		/* full (speed) FOC */
		MCS_ACIMFocCtrlCurrentA1(&g_sM1Drive.sFocACIM);
		break;
	}
}

/*!
 * @brief Free-wheel process called in fast state machine as Run sub state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_StateRunFreewheelFast(void) {
	/* Type the code to do when in the RUN FREEWHEEL sub-state */

	/* clear actual speed values */
	g_sM1Drive.sSpdFlux.f16SpdMeRamp = FRAC16(0.0);
	
	/* Clear scalar speed ramp */
	g_sM1Drive.sScalarCtrl.f16SpdMeRamp = FRAC16(0.0);  
	g_sM1Drive.sScalarCtrl.f16PosEl 	= FRAC16(0.0); 
	
	/* control variables */
	g_sM1Drive.sFocACIM.sIDQReq.f16D = FRAC16(0.0);
	g_sM1Drive.sFocACIM.sIDQReq.f16Q = FRAC16(0.0);
	g_sM1Drive.sFocACIM.sUDQReq.f16D = FRAC16(0.0);
	g_sM1Drive.sFocACIM.sUDQReq.f16Q = FRAC16(0.0);
}

/*!
 * @brief Calibration process called in slow state machine as Run sub state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_StateRunCalibSlow(void) {
	if (--g_sM1Drive.ui32CounterState == 0) {
		/* write calibrated offset values */
		M1_MCDRV_CURR_3PH_CALIB_SET(&g_sM1AdcSensor);

		/* To switch to the RUN READY sub-state */
		M1_TransRunCalibReady();
	}
}

/*!
 * @brief Ready state called in slow state machine as Run sub state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_StateRunReadySlow(void) {
	/* type the code to do when in the RUN READY sub-state */
}

/*!
 * @brief Start-up process called in slow state machine as Run sub state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_StateRunStartupSlow(void) {
	/* type the code to do when in the RUN STARTUP sub-state */
}

/*!
 * @brief Spin state called in slow state machine as Run sub state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_StateRunSpinSlow(void) {
	frac16_t f16SpdMeReqAbs;

	/* type the code to do when in the RUN SPIN sub-state */

	if (g_sM1Drive.eControl == kControlMode_SpeedFOC) {
		/* calculate absolute value of required speed */
		f16SpdMeReqAbs = MLIB_AbsSat_F16(g_sM1Drive.sSpdFlux.f16SpdMeReq);

		/* speed reverse */
		if (f16SpdMeReqAbs > g_sM1Drive.sSpdFlux.f16SpdMeReqMin) {
			/* pass required speed values lower than nominal speed */
			if (f16SpdMeReqAbs > g_sM1Drive.sSpdFlux.f16SpdMeReqMax) {
				/* set required speed to nominal speed if over speed command > speed nominal */
				g_sM1Drive.sSpdFlux.f16SpdMeReq = MLIB_MulRndSat_F16(
						g_sM1Drive.sSpdFlux.f16SpdMeReqMax,
						MLIB_Sign_F16(g_sM1Drive.sSpdFlux.f16SpdMeReq));
			}
		} else {
			g_sM1Drive.sSpdFlux.f16SpdMeRamp = 0;
			if (MLIB_AbsSat_F32(g_sM1Drive.sSpdFlux.sRampParSpdMe.f32State)
					< MLIB_Conv_F32s(g_sM1Drive.sSpdFlux.f16SpdMeReqMin)) {
				/* sub-state RUN FREEWHEEL */
				M1_TransRunSpinFreewheel();
			}
		}

		/* call speed control */
		MCS_ACIMSpeedFluxCtrlA1(&g_sM1Drive.sFocACIM, &g_sM1Drive.sSpdFlux);
	}
}

/*!
 * @brief Free-wheel process called in slow state machine as Run sub state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_StateRunFreewheelSlow(void) {
	/* wait until free-wheel time passes */
	if (--g_sM1Drive.ui32CounterState == 0) {
		/* switch to sub state READY */
		M1_TransRunFreewheelReady();
	}
}

/*!
 * @brief Transition from Calib to Ready state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_TransRunCalibReady(void) {
	/* Type the code to do when going from the RUN CALIB to the RUN READY sub-state */

	/* set 50% PWM duty cycle */
	g_sM1Drive.sFocACIM.sDutyABC.f16A = FRAC16(0.5);
	g_sM1Drive.sFocACIM.sDutyABC.f16B = FRAC16(0.5);
	g_sM1Drive.sFocACIM.sDutyABC.f16C = FRAC16(0.5);
	
	g_sM1Drive.ui32CounterState = g_sM1Drive.ui32TimeFaultRelease;	

	/* switch to sub state READY */
	g_eM1StateRun = kRunState_Ready;
}

/*!
 * @brief Transition from Ready to Startup state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_TransRunReadyStartup(void) {
	/* Type the code to do when going from the RUN READY to the RUN STARTUP sub-state */

	/* set the startup flag */
	g_sM1Drive.sFocACIM.bFlagSpdStart = TRUE;

	/* Go to sub-state RUN STARTUP */
	g_eM1StateRun = kRunState_Startup;
}

/*!
 * @brief Transition from Startup to Spin state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_TransRunStartupSpin(void) {
	/* Type the code to do when going from the RUN STARTUP to the RUN SPIN sub-state */

	/* clear the startup flag */
	g_sM1Drive.sFocACIM.bFlagSpdStart = FALSE;

	/* To switch to the RUN kRunState_Spin sub-state */
	g_eM1StateRun = kRunState_Spin;
}

/*!
 * @brief Transition from Startup to Free-wheel state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_TransRunStartupFreewheel(void) {
	/* type the code to do when going from the RUN STARTUP to the RUN FREEWHEEL 
	 sub-state */

	/* turn off all transistors */
	M1_MCDRV_PWM3PH_DIS(&g_sM1Pwm3ph);

	/* clear application parameters */
	M1_ClearFOCVariables();

	/* Free-wheel duration set-up */
	g_sM1Drive.ui32CounterState = g_sM1Drive.ui32TimeFullSpeedFreeWheel;

	/* enter FREEWHEEL sub-state */
	g_eM1StateRun = kRunState_Freewheel;
}

/*!
 * @brief Transition from Spin to Free-wheel state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_TransRunSpinFreewheel(void) {
	/* Type the code to do when going from the RUN SPIN to the RUN FREEWHEEL sub-state */
	/* set 50% PWM duty cycle */

	/* turn off all transistors */
	M1_MCDRV_PWM3PH_DIS(&g_sM1Pwm3ph);

	/* clear application parameters */
	M1_ClearFOCVariables();

	/* Generates a time gap to assure the rotor is not rotating */
	g_sM1Drive.ui32CounterState = g_sM1Drive.ui32TimeFullSpeedFreeWheel;

	/* enter FREEWHEEL sub-state */
	g_eM1StateRun = kRunState_Freewheel;
}

/*!
 * @brief Transition from Free-wheel to Ready state
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_TransRunFreewheelReady(void) {
	/* Type the code to do when going from the RUN kRunState_FreeWheel to the RUN kRunState_Ready sub-state */

	/* enable PWM output */
	M1_MCDRV_PWM3PH_EN(&g_sM1Pwm3ph);

	/* Sub-state RUN READY */
	g_eM1StateRun = kRunState_Ready;
}

/*!
 * @brief Clear FOc variables in global variable
 *
 * @param void  No input parameter
 *
 * @return None
 */
static void M1_ClearFOCVariables(void) {
	/* Reset ACIM FOC algorithm state variables */
	
	/* Clear scalar speed ramp */
	g_sM1Drive.sScalarCtrl.f16SpdMeRamp = FRAC16(0.0);  
	g_sM1Drive.sScalarCtrl.f16SpdMeReq  = FRAC16(0.0); 
	g_sM1Drive.sScalarCtrl.f16PosEl 	= FRAC16(0.0); 	

	/* Clear voltages */
	g_sM1Drive.sFocACIM.sUDQReq.f16D = 0;
	g_sM1Drive.sFocACIM.sUDQReq.f16Q = 0;
	g_sM1Drive.sFocACIM.sUAlBe.f16Alpha = 0;
	g_sM1Drive.sFocACIM.sUAlBe.f16Beta = 0;
	g_sM1Drive.sFocACIM.sDutyABC.f16A = FRAC16(0.5);
	g_sM1Drive.sFocACIM.sDutyABC.f16B = FRAC16(0.5);
	g_sM1Drive.sFocACIM.sDutyABC.f16C = FRAC16(0.5);
	g_sM1Drive.sFocACIM.sUAlBeDTComp.f16Alpha = 0;
	g_sM1Drive.sFocACIM.sUAlBeDTComp.f16Beta = 0;

	/* Clear currents */
	g_sM1Drive.sFocACIM.sIAlBe.f16Alpha = 0;
	g_sM1Drive.sFocACIM.sIAlBe.f16Beta = 0;
	g_sM1Drive.sFocACIM.sIDQ.f16D = 0;
	g_sM1Drive.sFocACIM.sIDQ.f16Q = 0;
	g_sM1Drive.sFocACIM.sIDQReq.f16D = 0;
	g_sM1Drive.sFocACIM.sIDQReq.f16Q = 0;
	g_sM1Drive.sFocACIM.sIABC.f16A = 0;
	g_sM1Drive.sFocACIM.sIABC.f16B = 0;
	g_sM1Drive.sFocACIM.sIABC.f16C = 0;

	/* Clear d-axis current controller */
	g_sM1Drive.sFocACIM.sPIParId.f32IAccK_1 = FRAC32(0.0);
	g_sM1Drive.sFocACIM.sPIParId.f16InErrK_1 = FRAC16(0.0);
	g_sM1Drive.sFocACIM.sPIParId.bLimFlag = FALSE;
	g_sM1Drive.sFocACIM.sIDQErr.f16D = FRAC16(0.0);
	g_sM1Drive.sFocACIM.bFlagPIIdStopInt = FALSE;
	GFLIB_CtrlPIpAWInit_F16(FRAC16(0.0), &g_sM1Drive.sFocACIM.sPIParId);

	/* Clear q-axis current controller */
	g_sM1Drive.sFocACIM.sPIParIq.f32IAccK_1 = FRAC32(0.0);
	g_sM1Drive.sFocACIM.sPIParIq.f16InErrK_1 = FRAC16(0.0);
	g_sM1Drive.sFocACIM.sPIParIq.bLimFlag = FALSE;
	g_sM1Drive.sFocACIM.sIDQErr.f16Q = FRAC16(0.0);
	g_sM1Drive.sFocACIM.bFlagPIIqStopInt = FALSE;
	GFLIB_CtrlPIpAWInit_F16(FRAC16(0.0), &g_sM1Drive.sFocACIM.sPIParIq);

	/* Clear rotor flux position and rotor speed estimation algorithms*/
	g_sM1Drive.sFocACIM.sSCFOC.f16Sin = FRAC16(0.0);
	g_sM1Drive.sFocACIM.sSCFOC.f16Cos = FRAC16(1.0);
	g_sM1Drive.sFocACIM.sSpdObs.f16SpeedElIIR1 = FRAC16(0.0);
	
	AMCLIB_ACIMRotFluxObsrvInit_F16(&g_sM1Drive.sFocACIM.sRFO);
	AMCLIB_ACIMSpeedMRASInit_F16(&g_sM1Drive.sFocACIM.sSpdObs);
    GDFLIB_FilterIIR1Init_F16(&g_sM1Drive.sFocACIM.sSpeedElIIR1Param);

	/* Speed and flux control reset */
	g_sM1Drive.sSpdFlux.f16SpdMeReq = FRAC16(0.0);
	g_sM1Drive.sSpdFlux.f16SpdMeRamp = FRAC16(0.0);
	g_sM1Drive.sSpdFlux.f16SpdMeFilt = FRAC16(0.0);
	g_sM1Drive.sSpdFlux.f16IdMTPA = FRAC16(0.0);
	g_sM1Drive.sSpdFlux.sPIParSpdMe.f32IAccK_1 = FRAC32(0.0);
	g_sM1Drive.sSpdFlux.sPIParSpdMe.f16InErrK_1 = FRAC16(0.0);
	g_sM1Drive.sSpdFlux.sPIParSpdMe.bLimFlag = FALSE;
	g_sM1Drive.sSpdFlux.f16SpdMeErr = FRAC16(0.0);
	g_sM1Drive.sSpdFlux.bFlagPISpdMeStopInt = FALSE;
	g_sM1Drive.sSpdFlux.bFlagPIFWStopInt = FALSE;
	g_sM1Drive.sSpdFlux.bStartupDone = FALSE;
	/* Clear speed controller */
	g_sM1Drive.sSpdFlux.sPIParSpdMe.f32IAccK_1 = FRAC32(0.0);
	g_sM1Drive.sSpdFlux.sPIParSpdMe.f16InErrK_1 = FRAC16(0.0);
	g_sM1Drive.sSpdFlux.sPIParSpdMe.bLimFlag = FALSE;
	g_sM1Drive.sSpdFlux.f16SpdMeErr = FRAC16(0.0);
	g_sM1Drive.sSpdFlux.bFlagPISpdMeStopInt = FALSE;
	GFLIB_CtrlPIpAWInit_F16(FRAC16(0.0), &g_sM1Drive.sSpdFlux.sPIParSpdMe);
	
	AMCLIB_CtrlFluxWkngInit_F16(g_sM1Drive.sSpdFlux.f16IdReqMax,
								&g_sM1Drive.sSpdFlux.sFluxWkng);
	AMCLIB_ACIMCtrlMTPAInit_F16(g_sM1Drive.sSpdFlux.sMTPA.f16LowerLim,
								g_sM1Drive.sSpdFlux.sMTPA.f16UpperLim,
								&g_sM1Drive.sSpdFlux.sMTPA);
}

/*******************************************************************************
 * API
 ******************************************************************************/

/*!
 * @brief Set application switch value to On or Off mode
 *
 * @param bValue  bool value, true - On of false - Off
 *
 * @return None
 */
void M1_SetAppSwitch(bool_t bValue) {
	g_bM1SwitchAppOnOff = bValue;
}

/*!
 * @brief Get application switch value
 *
 * @param void  No input parameter
 *
 * @return bool_t Return bool value, true or false
 */
bool_t M1_GetAppSwitch(void) {
	return (g_bM1SwitchAppOnOff);
}

/*!
 * @brief Get application state
 *
 * @param void  No input parameter
 *
 * @return uint16_t Return current application state
 */uint16_t M1_GetAppState() {
	return ((uint16_t) g_sM1Ctrl.eState);
}

/*!
 * @brief Set spin speed of the motor in fractional value
 *
 * @param f16SpeedCmd  Speed command - set speed
 *
 * @return None
 */
void M1_SetSpeed(frac16_t f16SpdMeReq) {
	if (g_bM1SwitchAppOnOff) {
		/* Set speed */
		if (MLIB_AbsSat_F16(f16SpdMeReq) < g_sM1Drive.sSpdFlux.f16SpdMeReqMin) {
			g_sM1Drive.sSpdFlux.f16SpdMeReq = 0;
		} else if (MLIB_AbsSat_F16(f16SpdMeReq)
				> g_sM1Drive.sSpdFlux.f16SpdMeReqMax) {
			g_sM1Drive.sSpdFlux.f16SpdMeReq =
					g_sM1Drive.sSpdFlux.f16SpdMeReqMax;
		} else {
			g_sM1Drive.sSpdFlux.f16SpdMeReq = f16SpdMeReq;
		}
	} else {
		/* Set zero speed */
		g_sM1Drive.sSpdFlux.f16SpdMeRamp = 0;
	}
}

/*!
 * @brief Get spin speed of the motor in fractional value
 *
 * @param void  No input parameter
 *
 * @return frac16_t Fractional value of the current speed
 */
frac16_t M1_GetSpeed(void) {
	/* return speed */
	return g_sM1Drive.sSpdFlux.f16SpdMeReq;
}

/*!
 * @brief Fault detention routine - check various faults
 *
 * @param void  No input parameter
 *
 * @return None
 */
void M1_FaultDetection(void) {
	/* clearing actual faults before detecting them again */

	/* clear all faults */
	M1_FAULT_CLEAR_ALL(g_sM1Drive.sFaultIdPending);

	/* fault: DC-bus over-current */
	if (M1_MCDRV_PWM3PH_FLT_GET(&g_sM1Pwm3ph))
		M1_FAULT_SET(g_sM1Drive.sFaultIdPending, M1_FAULT_I_DCBUS_OVER);

	/* fault: DC-bus over-voltage */
	if (g_sM1Drive.sFocACIM.f16UDcBusFilt
			> g_sM1Drive.sFaultThresholds.f16UDcBusOver)
		M1_FAULT_SET(g_sM1Drive.sFaultIdPending, M1_FAULT_U_DCBUS_OVER);

	/* fault: DC-bus under-voltage */
	if (g_sM1Drive.sFocACIM.f16UDcBusFilt
			< g_sM1Drive.sFaultThresholds.f16UDcBusUnder)
		M1_FAULT_SET(g_sM1Drive.sFaultIdPending, M1_FAULT_U_DCBUS_UNDER);

	/* fault: over-speed  */
	if ((MLIB_AbsSat_F16(g_sM1Drive.sSpdFlux.f16SpdMeFilt) > g_sM1Drive.sFaultThresholds.f16SpeedOver)
			&& g_sM1Drive.sSpdFlux.bStartupDone
			&& (g_eM1StateRun == kRunState_Spin))
	{
		M1_FAULT_SET(g_sM1Drive.sFaultIdPending, M1_FAULT_SPEED_OVER);
		g_sM1Drive.sSpdFlux.f16SpdMeFilt = FRAC16(0.0);
	}
		

	/* pass fault to fault ID */
	g_sM1Drive.sFaultIdCaptured |= g_sM1Drive.sFaultIdPending;
}

