/*******************************************************************************
*
* Copyright 2004-2016, Freescale Semiconductor, Inc.
* Copyright 2016-2019 NXP
* All rights reserved.
*
********************************************************************************
*
* File Name: dma.h
*
* Description: Header file for DMA low-level driver 
*
* This file supports different hardware versions of the module. A version
* of module used in your project is defined in the "arch.h" processor 
* architecture definition file.
*
*  DMA_VERSION_1 .. 56F84xxx and 56F827xx devices, see MC56F84xxxRM.pdf and MC56F827xxRM.pdf
*  DMA_VERSION_2 .. 56F83xxx devices, see MC56F83xxxRM.pdf
*
*****************************************************************************/

#ifndef __EDMA_H
#define __EDMA_H

/* qs.h is a master header file, which must be included */
#if !defined(__ARCH_H) || !defined(__PERIPH_H) || !defined(__APPCONFIG_H)
#error Please include qs.h before dma.h
#endif

#ifdef __cplusplus
extern "C" {
#endif

/************************************************************
* DMA module identifier, corresponds to module base address
*************************************************************/

#ifdef EDMA_BASE
#define EDMA (&ArchIO.Edma)
#define EDMA_0 (&ArchIO.Edma.channel_0)
#define EDMA_1 (&ArchIO.Edma.channel_1)
#define EDMA_2 (&ArchIO.Edma.channel_2)
#define EDMA_3 (&ArchIO.Edma.channel_3)
#endif
#ifdef DMA_MUX_BASE
#define DMA_MUX (&ArchIO.Dmamux)
#endif


/*************************************************************
* arch.h should define device-specific information about DMA
**************************************************************/

#if !defined(DMA_VERSION)
#error DMA not properly described in arch.h (old Quick_Start version?)
#endif

/******************************************************************************
*
*                      General Interface Description
*
*
* The DMA controller module enables fast transfers of data, providing an efficient way to
* move blocks of data with minimal processor interaction. The DMA module has four channels 
* that allow byte, word, or longword data transfers.
* Each channel has a dedicated source address register (SARn), destination address register
* (DARn), status register (DSRn), byte count register (BCRn), and control register
* (DCRn). Collectively, the combined program-visible registers associated with each
* channel define a transfer control descriptor (TCD). All transfers are dual address, moving
* data from a source memory location to a destination memory location with the module
* operating as a 32-bit bus master connected to the system bus. The programming model is
* accessed through a 32-bit connection with the slave peripheral bus. DMA data transfers
* may be explicitly initiated by software or by peripheral hardware requests.
*
* For more information, please refer to the user manual for the particular
* chip that you are using.
*
* The DMA device driver interface uses "ioctl" call to control specific
* DMA functions during operation. For details see "ioctl" call.
*
******************************************************************************/

/***********************************************************************
* DMA static configuration items for appconfig.h
***********************************************************************/
/*
    #define EDMA_CR_INIT
    #define EDMA_ERQ_INIT
    #define EDMA_EEI_INIT
    #define EDMA_EARS_INIT
    #define EDMA_DCHPRI0_INIT
    #define EDMA_DCHPRI1_INIT
    #define EDMA_DCHPRI2_INIT
    #define EDMA_DCHPRI3_INIT

    #define EDMA_TCD0_SARDR_INIT
    #define EDMA_TCD0_SOFF_INIT
    #define EDMA_TCD0_ATTR_INIT
    #define EDMA_TCD0_NBYTES_MLNO_INIT
    #define EDMA_TCD0_SLAST_INIT
    #define EDMA_TCD0_DADDR_INIT
    #define EDMA_TCD0_DOFF_INIT
    #define EDMA_TCD0_CISTER_ELINKNO_INIT
    #define EDMA_TCD0_DLASTSGA_INIT
    #define EDMA_TCD0_CSR_INIT
    #define EDMA_TCD0_BITER_ELINKNO_INIT

    #define EDMA_TCD1_SARDR_INIT
    #define EDMA_TCD1_SOFF_INIT
    #define EDMA_TCD1_ATTR_INIT
    #define EDMA_TCD1_NBYTES_MLNO_INIT
    #define EDMA_TCD1_SLAST_INIT
    #define EDMA_TCD1_DADDR_INIT
    #define EDMA_TCD1_DOFF_INIT
    #define EDMA_TCD1_CISTER_ELINKNO_INIT
    #define EDMA_TCD1_DLASTSGA_INIT
    #define EDMA_TCD1_CSR_INIT
    #define EDMA_TCD1_BITER_ELINKNO_INIT

    #define EDMA_TCD2_SARDR_INIT
    #define EDMA_TCD2_SOFF_INIT
    #define EDMA_TCD2_ATTR_INIT
    #define EDMA_TCD2_NBYTES_MLNO_INIT
    #define EDMA_TCD2_SLAST_INIT
    #define EDMA_TCD2_DADDR_INIT
    #define EDMA_TCD2_DOFF_INIT
    #define EDMA_TCD2_CISTER_ELINKNO_INIT
    #define EDMA_TCD2_DLASTSGA_INIT
    #define EDMA_TCD2_CSR_INIT
    #define EDMA_TCD2_BITER_ELINKNO_INIT

    #define EDMA_TCD3_SARDR_INIT
    #define EDMA_TCD3_SOFF_INIT
    #define EDMA_TCD3_ATTR_INIT
    #define EDMA_TCD3_NBYTES_MLNO_INIT
    #define EDMA_TCD3_SLAST_INIT
    #define EDMA_TCD3_DADDR_INIT
    #define EDMA_TCD3_DOFF_INIT
    #define EDMA_TCD3_CISTER_ELINKNO_INIT
    #define EDMA_TCD3_DLASTSGA_INIT
    #define EDMA_TCD3_CSR_INIT
    #define EDMA_TCD3_BITER_ELINKNO_INIT

    #define EDMA_MUX_CHCFG0_INIT
    #define EDMA_MUX_CHCFG1_INIT
    #define EDMA_MUX_CHCFG2_INIT
    #define EDMA_MUX_CHCFG3_INIT

*/
/*****************************************************************
* DMA common ioctl() commands
*
*  Word16 ioctl( _module_identifier, Command_name, Parameter );
*
******************************************************************

 COMMAND NAME                         PARAMETERS, COMMENTS
----------------------------------------------------------------*/
#define DMA_INIT                            /* NULL, Initialisation of the DMA peripheral registers using appconfig.h _INIT values */


/*   DMA commands used common configuration */

#define DMA_READ_CR_REG                     /* NULL, read and return DMA Control register (CR->_) */
#define DMA_WRITE_CR_REG                    /* UWord32, write the parameter value into the DMA Control register (CR->_) */

#define DMA_GET_ACTIVE_STATUS               /* NULL, read and return not null value when any DMA channel is executing (CR->ACTIVE) */
#define DMA_SET_CANCEL_TRANSFER             /* NULL, Cancel remaining data transfer (CR->CX) */
#define DMA_SET_CANCEL_TRANSFER_WITH_ERROR  /* NULL, Cancel remaining data transfer treat the cancel as error in DMAx_ES (CR->CX) */

#define DMA_SET_MINOR_LOOP_MAPPING          /* DMA_ENABLE/DMA_DISABLE, Enable/disable the minor loop mapping, The bit changes meaning of bits in NBYTES register (CR->EMLM) */
#define DMA_SET_CONTINUOUS_LINK_MODE        /* DMA_ENABLE/DMA_DISABLE, Enable/disable the continues mode for minor loop channel link (CR->CLM) */
#define DMA_SET_HALT_OPERATION              /* DMA_ENABLE/DMA_DISABLE, Enable/disable the halt mode which stall the start of any new channels (CR->HALT) */
#define DMA_SET_HALT_ON_ERROR               /* DMA_ENABLE/DMA_DISABLE, Enable/disable setting up halt bit of channels on any error of channel (CR->HOE) */
#define DMA_SET_ROUND_ROBIN_CHAN_ARBITRATION /* DMA_ENABLE/DMA_DISABLE, Enable/disable round robin arbitration for channel selection  (CR->ERCA) */
#define DMA_SET_DEBUG_MODE                  /* DMA_ENABLE/DMA_DISABLE, Enable/disable debugging of DMA channels, DMA stalls the start of a new channel (CR->EDBG) */
#define DMA_SET_BUFFERED_WRITES             /* DMA_ENABLE/DMA_DISABLE, Enable/disable buffered writes (CR->EBWR) */

#define DMA_READ_ES_REG                     /* NULL, read and return DMA Control register (ES->_) */
#define DMA_TEST_LAST_DMA_ERROR             /* DMA_xxx (xxx=ANY_ERROR|TRANSFER_CANCEL|CHANNEL_PRIORITY|SOURCE_ADDRESS|
                                                        SOURCE_OFFSET|DESTINATION_ADDRESS|DESTINATION_OFFSET|NBYTES_CIFER_CONFIG|
                                                        SCATTER_GATHER_CONFIG|SOURCE_BUS|DESTINATION_BUS), read and return values of selected error flags  (ES->_) */
#define DMA_GET_CHANNEL_OF_LAST_DMA_ERROR   /* NULL, read and return value of index of last channel with error (ES->ERRCHN) */

#define DMA_READ_ERQ_REG                    /* NULL, read and return Enable Request Register (ERQ->_) */
#define DMA_WRITE_ERQ_REG                   /* UWord32, write the parameter value into the Enable Request Register (ERQ->_) */
#define DMA_SET_DMA_REQUEST_0               /* DMA_ENABLE/DMA_DISABLE, Enable/disable DMA request signal for the corresponding channel (ERQ->ERQ0) */
#define DMA_SET_DMA_REQUEST_1               /* DMA_ENABLE/DMA_DISABLE, Enable/disable DMA request signal for the corresponding channel (ERQ->ERQ1) */
#define DMA_SET_DMA_REQUEST_2               /* DMA_ENABLE/DMA_DISABLE, Enable/disable DMA request signal for the corresponding channel (ERQ->ERQ2) */
#define DMA_SET_DMA_REQUEST_3               /* DMA_ENABLE/DMA_DISABLE, Enable/disable DMA request signal for the corresponding channel (ERQ->ERQ3) */

#define DMA_READ_EEI_REG                    /* NULL, read and return Enable Error Interrupt Register (EEI->_) */
#define DMA_WRITE_EEI_REG                   /* UWord32, write the parameter value into the Enable Error Interrupt Register (EEI->_) */
#define DMA_SET_DMA_ERROR_INTERRUPT_0       /* DMA_ENABLE/DMA_DISABLE, Enable/disable DMA error interrupt for the corresponding channel (EEI->EEI0) */
#define DMA_SET_DMA_ERROR_INTERRUPT_1       /* DMA_ENABLE/DMA_DISABLE, Enable/disable DMA error interrupt for the corresponding channel (EEI->EEI1) */
#define DMA_SET_DMA_ERROR_INTERRUPT_2       /* DMA_ENABLE/DMA_DISABLE, Enable/disable DMA error interrupt for the corresponding channel (EEI->EEI2) */
#define DMA_SET_DMA_ERROR_INTERRUPT_3       /* DMA_ENABLE/DMA_DISABLE, Enable/disable DMA error interrupt for the corresponding channel (EEI->EEI3) */

#define DMA_WRITE_CEEI_REG                  /* UWord32, write the parameter value into the Clear Enable Error Interrupt Register (CEEI->_) */
#define DMA_SET_DISABLE_DMA_ERROR_INTERRUPT /* DMA_xxx (xxx=ALL_CHANNELS/CHANNEL_0/CHANNEL_1/CHANNEL_2/CHANNEL_3), Disables the DMA error interrupt for the corresponding channel (CEEI->_) */
#define DMA_WRITE_SEEI_REG                  /* UWord32, write the parameter value into the Set Enable Error Interrupt Register (SEEI->_) */
#define DMA_SET_ENABLE_DMA_ERROR_INTERRUPT  /* DMA_xxx (xxx=ALL_CHANNELS/CHANNEL_0/CHANNEL_1/CHANNEL_2/CHANNEL_3), Enables the DMA error interrupt for the corresponding channel (SEEI->_) */

#define DMA_WRITE_CERQ_REG                  /* UWord32, write the parameter value into the Clear Enable Error Request Register (CERQ->_) */
#define DMA_SET_DISABLE_DMA_REQUEST         /* DMA_xxx (xxx=ALL_CHANNELS/CHANNEL_0/CHANNEL_1/CHANNEL_2/CHANNEL_3), Disables the DMA error request for the corresponding channel (CERQ->_) */
#define DMA_WRITE_SERQ_REG                  /* UWord32, write the parameter value into the Set Enable Error Request Register (SERQ->_) */
#define DMA_SET_ENABLE_DMA_REQUEST          /* DMA_xxx (xxx=ALL_CHANNELS/CHANNEL_0/CHANNEL_1/CHANNEL_2/CHANNEL_3), Enables the DMA error request for the corresponding channel (SERQ->_) */

#define DMA_WRITE_CDNE_REG                  /* UWord32, write the parameter value into the Clear DONE Status Bit Register (CDNE->_) */
#define DMA_CLEAR_DMA_DONE_BIT              /* DMA_xxx (xxx=ALL_CHANNELS/CHANNEL_0/CHANNEL_1/CHANNEL_2/CHANNEL_3), Clears the DONE bit for the corresponding channel (CDNE->_) */
#define DMA_WRITE_SSRT_REG                  /* UWord32, write the parameter value into the Set Start Bit Register (SSRT->_) */
#define DMA_SET_DMA_START_BIT               /* DMA_xxx (xxx=ALL_CHANNELS/CHANNEL_0/CHANNEL_1/CHANNEL_2/CHANNEL_3), Set START bit for the corresponding channel (SSRT->_) */
#define DMA_WRITE_CDNE_REG                  /* UWord32, write the parameter value into the Clear ERROR Register (CERR->_) */
#define DMA_CLEAR_DMA_ERROR_BIT             /* DMA_xxx (xxx=ALL_CHANNELS/CHANNEL_0/CHANNEL_1/CHANNEL_2/CHANNEL_3), Clears the ERROR bit for the corresponding channel (CERR->_) */
#define DMA_WRITE_CINT_REG                  /* UWord32, write the parameter value into the Clear Interrupt Request Register (CINT->_) */
#define DMA_CLEAR_INTERRUPT_REQUEST         /* DMA_xxx (xxx=ALL_CHANNELS/CHANNEL_0/CHANNEL_1/CHANNEL_2/CHANNEL_3), Clears the interrupt request bit for the corresponding channel (CINT->_) */
#define DMA_READ_INT_REG                    /* NULL, read and return Interrupt Register (INT->_) */
#define DMA_WRITE_INT_REG                   /* UWord32, write the parameter value into the Interrupt Request Register (INT->_) */
#define DMA_TEST_INTERRUPT_REQUEST_BIT      /* DMA_xxx (xxx=CHANNEL_0_MASK|CHANNEL_1_MASK|CHANNEL_2_MASK|CHANNEL_3_MASK), reads and returns value of interrupt request for DMA channel (INT->_) */
#define DMA_CLEAR_INTERRUPT_REQUEST_BIT     /* DMA_xxx (xxx=CHANNEL_0_MASK|CHANNEL_1_MASK|CHANNEL_2_MASK|CHANNEL_3_MASK), Clears the interrupt request bit for the corresponding channel (INT->_) */
#define DMA_READ_ERR_REG                    /* NULL, read and return Error Interrupt Register (ERR->_) */
#define DMA_WRITE_ERR_REG                   /* UWord32, write the parameter value into the Interrupt Request Register (ERR->_) */
#define DMA_TEST_ERROR_INTERRUPT_BIT        /* DMA_xxx (xxx=CHANNEL_0_MASK|CHANNEL_1_MASK|CHANNEL_2_MASK|CHANNEL_3_MASK), reads and returns value of interrupt error request for DMA channel (ERR->_) */
#define DMA_CLEAR_ERROR_INTERRUPT_BIT       /* DMA_xxx (xxx=CHANNEL_0_MASK|CHANNEL_1_MASK|CHANNEL_2_MASK|CHANNEL_3_MASK), Clears the interrupt error bit for the corresponding channel (ERR->_) */
#define DMA_READ_HRS_REG                    /* NULL, read and return Hardware Request Status Register (HRS->_) */
#define DMA_TEST_HARDWARE_REQUEST_BIT       /* DMA_xxx (xxx=CHANNEL_0_MASK|CHANNEL_1_MASK|CHANNEL_2_MASK|CHANNEL_3_MASK), reads and returns value of hardware request status for DMA channel (HRS->_) */
#define DMA_READ_EARS_REG                   /* NULL, read and return Enable Asynchronous Request In Stop Register (EARS->_) */
#define DMA_WRITE_EARS_REG                  /* UWord32, write the parameter value into the Enable Asynchronous Request In Stop Register (EARS->_) */
#define DMA_SET_ASYNC_DMA_REQ_IN_STOP_CH0   /* DMA_ENABLE/DMA_DISABLE, Enable/disable asynchronous DMA request in stop mode for the corresponding channel (EARS->EDREQ_0) */
#define DMA_SET_ASYNC_DMA_REQ_IN_STOP_CH1   /* DMA_ENABLE/DMA_DISABLE, Enable/disable asynchronous DMA request in stop mode for the corresponding channel (EARS->EDREQ_1) */
#define DMA_SET_ASYNC_DMA_REQ_IN_STOP_CH2   /* DMA_ENABLE/DMA_DISABLE, Enable/disable asynchronous DMA request in stop mode for the corresponding channel (EARS->EDREQ_2) */
#define DMA_SET_ASYNC_DMA_REQ_IN_STOP_CH3   /* DMA_ENABLE/DMA_DISABLE, Enable/disable asynchronous DMA request in stop mode for the corresponding channel (EARS->EDREQ_3) */

#define DMA_READ_DCHPRI0_REG                /* NULL, read and return Channel 0 priority Register (DCHPRI0->_) */
#define DMA_WRITE_DCHPRI0_REG               /* UWord32, write the parameter value into the Channel 0 priority Register (DCHPRI0->_) */
#define DMA_SET_CHANNEL_0_PREEMPTION        /* DMA_ENABLE/DMA_DISABLE, Enable/disable channel 0 preemption, which enables break service request by higher priority (DCHPRI0->ECP) */
#define DMA_SET_CHANNEL_0_PREEMPT_ABILITY   /* DMA_ENABLE/DMA_DISABLE, Enable/disable channel 0 preempt ability, which enables break other service request with lower priority (DCHPRI0->DPA) */
#define DMA_SET_CHANNEL_0_ARB_PRIORITY      /* UWord16 0..3, Sets the DMA channel 0 arbitration priority (DCHPRI0->CHPRI) */

#define DMA_READ_DCHPRI1_REG                /* NULL, read and return Channel 1 priority Register (DCHPRI1->_) */
#define DMA_WRITE_DCHPRI1_REG               /* UWord32, write the parameter value into the Channel 1 priority Register (DCHPRI1->_) */
#define DMA_SET_CHANNEL_1_PREEMPTION        /* DMA_ENABLE/DMA_DISABLE, Enable/disable channel 1 preemption, which enables break service request by higher priority (DCHPRI1->ECP) */
#define DMA_SET_CHANNEL_1_PREEMPT_ABILITY   /* DMA_ENABLE/DMA_DISABLE, Enable/disable channel 1 preempt ability, which enables break other service request with lower priority (DCHPRI1->DPA) */
#define DMA_SET_CHANNEL_1_ARB_PRIORITY      /* UWord16 0..3, Sets the DMA channel 1 arbitration priority (DCHPRI1->CHPRI) */

#define DMA_READ_DCHPRI2_REG                /* NULL, read and return Channel 2 priority Register (DCHPRI2->_) */
#define DMA_WRITE_DCHPRI2_REG               /* UWord32, write the parameter value into the Channel 2 priority Register (DCHPRI2->_) */
#define DMA_SET_CHANNEL_2_PREEMPTION        /* DMA_ENABLE/DMA_DISABLE, Enable/disable channel 2 preemption, which enables break service request by higher priority (DCHPRI2->ECP) */
#define DMA_SET_CHANNEL_2_PREEMPT_ABILITY   /* DMA_ENABLE/DMA_DISABLE, Enable/disable channel 2 preempt ability, which enables break other service request with lower priority (DCHPRI2->DPA) */
#define DMA_SET_CHANNEL_2_ARB_PRIORITY      /* UWord16 0..3, Sets the DMA channel 2 arbitration priority (DCHPRI2->CHPRI) */

#define DMA_READ_DCHPRI3_REG                /* NULL, read and return Channel 3 priority Register (DCHPRI3->_) */
#define DMA_WRITE_DCHPRI3_REG               /* UWord32, write the parameter value into the Channel 3 priority Register (DCHPRI3->_) */
#define DMA_SET_CHANNEL_3_PREEMPTION        /* DMA_ENABLE/DMA_DISABLE, Enable/disable channel 3 preemption, which enables break service request by higher priority (DCHPRI3->ECP) */
#define DMA_SET_CHANNEL_3_PREEMPT_ABILITY   /* DMA_ENABLE/DMA_DISABLE, Enable/disable channel 3 preempt ability, which enables break other service request with lower priority (DCHPRI3->DPA) */
#define DMA_SET_CHANNEL_3_ARB_PRIORITY      /* UWord16 0..3, Sets the DMA channel 3 arbitration priority (DCHPRI3->CHPRI) */


/*****************************************************************
* DMA channel ioctl() commands
*
*  Word16 ioctl( DMA_channel_identifier, Command_name, Parameter );
*
******************************************************************

 COMMAND NAME                         PARAMETERS, COMMENTS
----------------------------------------------------------------*/
#define DMACH_SET_SOURCE_ADDRESS            /* UWord32, Sets the DMA channel Source address (TCDx_SADDR->_) */
#define DMACH_GET_SOURCE_ADDRESS            /* NULL, read and return the DMA channel Source address (TCDx_SADDR->_) */
#define DMACH_SET_SOURCE_ADDRESS_OFFSET     /* UWord32, Sets the DMA channel Source address offset (TCDx_SOFF->_) */
#define DMACH_GET_SOURCE_ADDRESS_OFFSET     /* NULL, read and return the DMA channel Source address offset (TCDx_SOFF->_) */

#define DMACH_READ_ATTRIBUTES_REG           /* NULL, read and return Channel Transfer Attributes Register (TCDx_ATTR->_) */
#define DMACH_WRITE_ATTRIBUTES_REG          /* UWord32, write the parameter value into the Channel Transfer Attributes Register (TCDx_ATTR->_) */
#define DMACH_SET_SOURCE_ADDRESS_MODULO     /* DMA_xxx (xxx=DISABLE/1B/2B/4B/8B/16B/32B/64B/128B/256B/512B/1KB/2KB/4KB/8KB/16KB/32KB/64KB/128KB/256KB/512KB/1MB/2MB/4MB/8MB/16MB/32MB/64MB/128MB/256MB/512MB/1GB/2GB), set the size of the source data circular buffer used by the DMA channel (TCDx_ATTR->SMOD) */
#define DMACH_SET_SOURCE_SIZE               /* DMA_xxx (xxx=BYTE/WORD/LONGWORD/16BYTE), set the data size of the source bus cycle for the DMA channel (TCDx_ATTR->SSIZE) */
#define DMACH_SET_DESTINATION_ADDRESS_MODULO /* DMA_xxx (xxx=DISABLE/1B/2B/4B/8B/16B/32B/64B/128B/256B/512B/1KB/2KB/4KB/8KB/16KB/32KB/64KB/128KB/256KB/512KB/1MB/2MB/4MB/8MB/16MB/32MB/64MB/128MB/256MB/512MB/1GB/2GB), set the size of the destination data circular buffer used by the DMA channel (TCDx_ATTR->DMOD) */
#define DMACH_SET_DESTINATION_SIZE          /* DMA_xxx (xxx=BYTE/WORD/LONGWORD/16BYTE), set the data size of the destination bus cycle for the DMA channel (TCDx_ATTR->DSIZE) */

#define DMACH_READ_NBYTES_MLOFF_REG         /* NULL, read and return Channel Signed Minor Loop Offset Register (TCDx_NBYTES_MLOFF->_) */
#define DMACH_WRITE_NBYTES_MLOFF_REG        /* UWord32, write the parameter value into the Channel Signed Minor Loop Offset Register (TCDx_NBYTES_MLOFF->_) */
#define DMACH_SET_SOURCE_MINOR_LOOP_OFFSET  /* DMA_ENABLE/DMA_DISABLE, Enable/disable Source minor loop offset applied to SADDR (TCDx_NBYTES_MLOFF->SMLOE) */
#define DMACH_SET_DESTINATION_MINOR_LOOP_OFFSET /* DMA_ENABLE/DMA_DISABLE, Enable/disable Destination minor loop offset applied to DADDR (TCDx_NBYTES_MLOFF->DMLOE) */
#define DMACH_SET_NBYTES_WITH_MLOFF         /* UWord32, write the parameter value into the Minor byte transfer count (TCDx_NBYTES_MLOFF->NBYTES) */
#define DMACH_SET_NBYTES_WITH_MLON          /* UWord32, write the parameter value into the Minor byte transfer count  (TCDx_NBYTES_MLOFF->NBYTES) */
#define DMACH_SET_OFFSET_WITH_MLON          /* UWord32, write the parameter value into the sign-extended offset applied to source or destination address after minor loop complete  (TCDx_NBYTES_MLOFF->MLOFF) */
#define DMACH_SET_NBYTES_WITH_MLNO          /* UWord32, write the parameter value into the Minor byte transfer count  (TCDx_NBYTES_MLOFF->_) */

#define DMACH_GET_LAST_SOURCE_ADDR_ADJ      /* NULL, read and return Last Source Address Adjustment Register (TCDx_SLAST->_) */
#define DMACH_SET_LAST_SOURCE_ADDR_ADJ      /* UWord32, write the parameter value into the Last Source Address Adjustment Register (TCDx_SLAST->_) */

#define DMACH_GET_DESTINATION_ADDRESS       /* NULL, read and return the destination address used by the DMA channel (TCDx_DADDR->_) */
#define DMACH_SET_DESTINATION_ADDRESS       /* UWord32 destination address, set the destination address used by the DMA channel to write data (TCDx_DADDR->_) */
#define DMACH_GET_DESTINATION_ADDRESS_OFFSET /* NULL, read and return the DMA channel Destination address offset (TCDx_DOFF->_) */
#define DMACH_SET_DESTINATION_ADDRESS_OFFSET /* UWord32, Sets the DMA channel Destination address offset (TCDx_DOFF->_) */

#define DMACH_READ_CITER_ELINK_REG          /* NULL, read and return Current Minor Loop Link, Major Loop Count Register (TCDx_CITER_ELINKNO->_) */
#define DMACH_WRITE_CITER_ELINK_REG         /* UWord32, write the parameter value into the Current Minor Loop Link, Major Loop Count Register (TCDx_CITER_ELINKNO->_) */
#define DMACH_SET_CHANNEL_TO_CHANNEL_LINKING      /* DMA_ENABLE/DMA_DISABLE, Enable/disable Channel to channel linking (TCDx_CITER_ELINKNO->ELINK) */
#define DMACH_SET_CURRENT_MAJOR_ITER_CNT_ELINKNO  /* UWord32, write the parameter value into the Current Major Iteration count (TCDx_CITER_ELINKNO->CITER) */
#define DMACH_SET_CURRENT_MAJOR_ITER_CNT_ELINKYES /* UWord32, write the parameter value into the Current Major Iteration count (TCDx_CITER_ELINKYES->CITER) */
#define DMACH_SET_MINOR_LOOP_LINK_CHANNEL_ELINKYES /* UWord32, write the parameter value into the Minor Loop Link Channel Number (TCDx_CITER_ELINKYES->LINKCH) */

#define DMACH_READ_DLASTGA_REG              /* NULL, read and return Last Destination Address Adjustment/Scatter Gather Address Register (TCDx_DLASTSGA->_) */
#define DMACH_WRITE_DLASTGA_REG             /* UWord32, write the parameter value into the Last Destination Address Adjustment/Scatter Gather Address Register (TCDx_DLASTSGA->_) */

#define DMACH_READ_CONTROL_REG              /* NULL, read and return Control and Status (TCDx_CSR->_) */
#define DMACH_WRITE_CONTROL_REG             /* UWord32, write the parameter value into the Control and Status Register (TCDx_CSR->_) */
#define DMACH_SET_BANDWITH_CONTROL          /* DMA_xxx (xxx=NO_STALL/4CYCLES_STALL/8CYCLES_STALL), set the amount of bus bandwidth consumed by the eDMA (TCDx_CSR->BWC) */
#define DMACH_SET_MAJOR_LOOP_CHANNEL        /* UWord32 (0..3), set the major loop link channel number (TCDx_CSR->MAJORLINKCH) */
#define DMACH_TEST_CHANNEL_DONE             /* NULL, read and return the Channel Done flag (TCDx_CSR->DONE) */
#define DMACH_CLEAR_CHANNEL_DONE            /* NULL, Clear the Channel Done flag (TCDx_CSR->DONE) */
#define DMACH_TEST_CHANNEL_ACTIVE           /* NULL, read and return the Channel Active flag (TCDx_CSR->ACTIVE) */
#define DMACH_SET_CHANNEL_TO_CHANNEL_LINKING_ON_MAJOR_LOOP_COMPLETE /* DMA_ENABLE/DMA_DISABLE, Enable/disable channel-to-channel linking on major loop complete (TCDx_CSR->MAJORELINK) */
#define DMACH_SET_CHANNEL_TO_SCATTER_GATHER_PROCESSING /* DMA_ENABLE/DMA_DISABLE, Enable/disable Scatter/Gather Processing (TCDx_CSR->ESG) */
#define DMACH_SET_ERQ_REQUEST               /* DMA_ENABLE/DMA_DISABLE, Enable/disable clearing of corresponding ERQ bit on major iteration count is zero (TCDx_CSR->DREQ) */
#define DMACH_SET_HALF_COMPLETE_INTERRUPT   /* DMA_ENABLE/DMA_DISABLE, Enable/disable interrupt when major counter is half complete (TCDx_CSR->INTHALF) */
#define DMACH_SET_MAJOR_ITER_COUNT_COMPLETE_INTERRUPT   /* DMA_ENABLE/DMA_DISABLE, Enable/disable interrupt when major iteration count completes (TCDx_CSR->INTMAJOR) */
#define DMACH_SET_CHANNEL_START               /* DMA_ENABLE/DMA_DISABLE, Enable/disable if channel is requesting service (TCDx_CSR->START) */

#define DMACH_READ_BITER_ELINK_REG          /* NULL, read and return Beginning Minor Loop Link, Major Loop Count Register (TCDx_BITER_ELINKYES->_) */
#define DMACH_WRITE_BITER_ELINK_REG         /* UWord32, write the parameter value into the Beginning Minor Loop Link, Major Loop Count Register (TCDx_BITER_ELINKYES->_) */
#define DMACH_SET_BITER_CHANNEL_TO_CHANNEL_LINKING /* DMA_ENABLE/DMA_DISABLE, Enable/disable channel-to-channel linking on minor loop complete (TCDx_CITER_ELINKNO->ELINK) */
#define DMACH_SET_STARTING_MAJOR_ITER_CNT_ELINKNO  /* UWord32, write the parameter value into the Starting major iteration count (TCDx_BITER_ELINKNO->BITER) */
#define DMACH_SET_STARTING_MAJOR_ITER_CNT_ELINKYES /* UWord32, write the parameter value into the Starting major iteration count (TCDx_BITER_ELINKYES->BITER) */
#define DMACH_SET_STARTING_MINOR_LOOP_LINK_CHANNEL_ELINKYES /* UWord32, write the parameter value into the Link Channel Number (TCDx_BITER_ELINKYES->LINKCH) */


/*****************************************************************
* DMA multiplexer ioctl() commands
*
*  Word16 ioctl( DMA_channel_identifier, Command_name, Parameter );
*
******************************************************************

 COMMAND NAME                         PARAMETERS, COMMENTS
----------------------------------------------------------------*/
#define DMAMUX_READ_CHANNEL_0_CONFIG_REG           /* NULL, read and return Channel Configuration Register (CHCFGx->_) */
#define DMAMUX_WRITE_CHANNEL_0_CONFIG_REG          /* UWord32, write the parameter value into the Channel Configuration Register (CHCFGx->_) */
#define DMAMUX_SET_CHANNEL_0                       /* DMA_ENABLE/DMA_DISABLE, Enable/disable  the DMA channel(CHCFGx->ENBL) */
#define DMAMUX_SET_CHANNEL_0_SOURCE                /* DMA_MUX_XXX, (DISABLED/SCI0_RF/SCI0_TE/SCI1_RF/SCI1_TE/SCI2_RF/
                                                      SCI2_TE/SPI0_RF/SPI0_TE/SPI1_RF/SPI1_TE/CAN_FD/IIC0/IIC1/
                                                      PWMA0_CP/PWMA1_CP/PWMA2_CP/PWMA3_CP/PWMA0_WR/PWMA1_WR/PWMA2_WR/PWMA3_WR/
                                                      PWMB0_CP/PWMB1_CP/PWMB2_CP/PWMB3_CP/PWMB0_WR/PWMB1_WR/PWMB2_WR/PWMB3_WR/
                                                      TMRA0_CP/TMRA0_CMP1/TMRA0_CMP2/TMRA1_CP/TMRA1_CMP1/TMRA1_CMP2/
                                                      TMRA2_CP/TMRA2_CMP1/TMRA2_CMP2/TMRA3_CP/TMRA3_CMP1/TMRA3_CMP2/
                                                      TMRB0_CP/TMRB0_CMP1/TMRB0_CMP2/TMRB1_CP/TMRB1_CMP1/TMRB1_CMP2/
                                                      TMRB2_CP/TMRB2_CMP1/TMRB2_CMP2/TMRB3_CP/TMRB3_CMP1/TMRB3_CMP2/
                                                      ADCA_ES/ADCB_ES/DACA_FIFO/DACB_FIFO/CMPA/CMPB/CMPC/CMPD/
                                                      XBAR_DSC0/XBAR_DSC1/XBAR_DSC2/XBAR_DSC3/DMA_MUX0/DMA_MUX1/DMA_MUX2/DMA_MUX3)  the DMA channel(CHCFGx->SOURCE) */
#define DMAMUX_READ_CHANNEL_1_CONFIG_REG           /* NULL, read and return Channel Configuration Register (CHCFGx->_) */
#define DMAMUX_WRITE_CHANNEL_1_CONFIG_REG          /* UWord32, write the parameter value into the Channel Configuration Register (CHCFGx->_) */
#define DMAMUX_SET_CHANNEL_1                       /* DMA_ENABLE/DMA_DISABLE, Enable/disable  the DMA channel(CHCFGx->ENBL) */
#define DMAMUX_SET_CHANNEL_1_SOURCE                /* DMA_MUX_XXX, (DISABLED/SCI0_RF/SCI0_TE/SCI1_RF/SCI1_TE/SCI2_RF/
                                                      SCI2_TE/SPI0_RF/SPI0_TE/SPI1_RF/SPI1_TE/CAN_FD/IIC0/IIC1/
                                                      PWMA0_CP/PWMA1_CP/PWMA2_CP/PWMA3_CP/PWMA0_WR/PWMA1_WR/PWMA2_WR/PWMA3_WR/
                                                      PWMB0_CP/PWMB1_CP/PWMB2_CP/PWMB3_CP/PWMB0_WR/PWMB1_WR/PWMB2_WR/PWMB3_WR/
                                                      TMRA0_CP/TMRA0_CMP1/TMRA0_CMP2/TMRA1_CP/TMRA1_CMP1/TMRA1_CMP2/
                                                      TMRA2_CP/TMRA2_CMP1/TMRA2_CMP2/TMRA3_CP/TMRA3_CMP1/TMRA3_CMP2/
                                                      TMRB0_CP/TMRB0_CMP1/TMRB0_CMP2/TMRB1_CP/TMRB1_CMP1/TMRB1_CMP2/
                                                      TMRB2_CP/TMRB2_CMP1/TMRB2_CMP2/TMRB3_CP/TMRB3_CMP1/TMRB3_CMP2/
                                                      ADCA_ES/ADCB_ES/DACA_FIFO/DACB_FIFO/CMPA/CMPB/CMPC/CMPD/
                                                      XBAR_DSC0/XBAR_DSC1/XBAR_DSC2/XBAR_DSC3/DMA_MUX0/DMA_MUX1/DMA_MUX2/DMA_MUX3)  the DMA channel(CHCFGx->SOURCE)  the DMA channel(CHCFGx->SOURCE) */
#define DMAMUX_READ_CHANNEL_2_CONFIG_REG           /* NULL, read and return Channel Configuration Register (CHCFGx->_) */
#define DMAMUX_WRITE_CHANNEL_2_CONFIG_REG          /* UWord32, write the parameter value into the Channel Configuration Register (CHCFGx->_) */
#define DMAMUX_SET_CHANNEL_2                       /* DMA_ENABLE/DMA_DISABLE, Enable/disable  the DMA channel(CHCFGx->ENBL) */
#define DMAMUX_SET_CHANNEL_2_SOURCE                /* DMA_MUX_XXX, (DISABLED/SCI0_RF/SCI0_TE/SCI1_RF/SCI1_TE/SCI2_RF/
                                                      SCI2_TE/SPI0_RF/SPI0_TE/SPI1_RF/SPI1_TE/CAN_FD/IIC0/IIC1/
                                                      PWMA0_CP/PWMA1_CP/PWMA2_CP/PWMA3_CP/PWMA0_WR/PWMA1_WR/PWMA2_WR/PWMA3_WR/
                                                      PWMB0_CP/PWMB1_CP/PWMB2_CP/PWMB3_CP/PWMB0_WR/PWMB1_WR/PWMB2_WR/PWMB3_WR/
                                                      TMRA0_CP/TMRA0_CMP1/TMRA0_CMP2/TMRA1_CP/TMRA1_CMP1/TMRA1_CMP2/
                                                      TMRA2_CP/TMRA2_CMP1/TMRA2_CMP2/TMRA3_CP/TMRA3_CMP1/TMRA3_CMP2/
                                                      TMRB0_CP/TMRB0_CMP1/TMRB0_CMP2/TMRB1_CP/TMRB1_CMP1/TMRB1_CMP2/
                                                      TMRB2_CP/TMRB2_CMP1/TMRB2_CMP2/TMRB3_CP/TMRB3_CMP1/TMRB3_CMP2/
                                                      ADCA_ES/ADCB_ES/DACA_FIFO/DACB_FIFO/CMPA/CMPB/CMPC/CMPD/
                                                      XBAR_DSC0/XBAR_DSC1/XBAR_DSC2/XBAR_DSC3/DMA_MUX0/DMA_MUX1/DMA_MUX2/DMA_MUX3)  the DMA channel(CHCFGx->SOURCE)  the DMA channel(CHCFGx->SOURCE) */
#define DMAMUX_READ_CHANNEL_3_CONFIG_REG           /* NULL, read and return Channel Configuration Register (CHCFGx->_) */
#define DMAMUX_WRITE_CHANNEL_3_CONFIG_REG          /* UWord32, write the parameter value into the Channel Configuration Register (CHCFGx->_) */
#define DMAMUX_SET_CHANNEL_3                       /* DMA_ENABLE/DMA_DISABLE, Enable/disable  the DMA channel(CHCFGx->ENBL) */
#define DMAMUX_SET_CHANNEL_3_SOURCE                /* DMA_MUX_XXX, (DISABLED/SCI0_RF/SCI0_TE/SCI1_RF/SCI1_TE/SCI2_RF/
                                                      SCI2_TE/SPI0_RF/SPI0_TE/SPI1_RF/SPI1_TE/CAN_FD/IIC0/IIC1/
                                                      PWMA0_CP/PWMA1_CP/PWMA2_CP/PWMA3_CP/PWMA0_WR/PWMA1_WR/PWMA2_WR/PWMA3_WR/
                                                      PWMB0_CP/PWMB1_CP/PWMB2_CP/PWMB3_CP/PWMB0_WR/PWMB1_WR/PWMB2_WR/PWMB3_WR/
                                                      TMRA0_CP/TMRA0_CMP1/TMRA0_CMP2/TMRA1_CP/TMRA1_CMP1/TMRA1_CMP2/
                                                      TMRA2_CP/TMRA2_CMP1/TMRA2_CMP2/TMRA3_CP/TMRA3_CMP1/TMRA3_CMP2/
                                                      TMRB0_CP/TMRB0_CMP1/TMRB0_CMP2/TMRB1_CP/TMRB1_CMP1/TMRB1_CMP2/
                                                      TMRB2_CP/TMRB2_CMP1/TMRB2_CMP2/TMRB3_CP/TMRB3_CMP1/TMRB3_CMP2/
                                                      ADCA_ES/ADCB_ES/DACA_FIFO/DACB_FIFO/CMPA/CMPB/CMPC/CMPD/
                                                      XBAR_DSC0/XBAR_DSC1/XBAR_DSC2/XBAR_DSC3/DMA_MUX0/DMA_MUX1/DMA_MUX2/DMA_MUX3)  the DMA channel(CHCFGx->SOURCE)  the DMA channel(CHCFGx->SOURCE) */


/*****************************************************************
* Software Layer ioctl() commands
*
*  Word16 ioctl( DMA_module_identifier, Command_name, Parameter );
*
******************************************************************

 COMMAND NAME                         PARAMETERS, COMMENTS
----------------------------------------------------------------*/

/****************************************************************************
* DMA constants and type declarations used in the ioctl functions
*****************************************************************************/

#define DMA_ENABLE    1
#define DMA_DISABLE   0



#define DMAMUX_CHCFG_ENABLE_MASK 0x80
#define DMAMUX_CHCFG_SOURCE_MASK 0x3F

#define DMAMUX_DISABLED         0x00
#define DMAMUX_SCI0_RF          0x02
#define DMAMUX_SCI0_TE          0x03
#define DMAMUX_SCI1_RF          0x04
#define DMAMUX_SCI1_TE          0x05
#define DMAMUX_SCI2_RF          0x06
#define DMAMUX_SCI2_TE          0x07
#define DMAMUX_SPI0_RF          0x0C
#define DMAMUX_SPI0_TE          0x0D
#define DMAMUX_SPI1_RF          0x0E
#define DMAMUX_SPI1_TE          0x0F
#define DMAMUX_CAN_FD           0x10
#define DMAMUX_IIC0             0x12
#define DMAMUX_IIC1             0x13
#define DMAMUX_PWMA0_CP         0x16
#define DMAMUX_PWMA1_CP         0x17
#define DMAMUX_PWMA2_CP         0x18
#define DMAMUX_PWMA3_CP         0x19
#define DMAMUX_PWMA0_WR         0x1A
#define DMAMUX_PWMA1_WR         0x1A
#define DMAMUX_PWMA2_WR         0x1A
#define DMAMUX_PWMA3_WR         0x1A
#define DMAMUX_PWMB0_CP         0x1B
#define DMAMUX_PWMB1_CP         0x1C
#define DMAMUX_PWMB2_CP         0x1D
#define DMAMUX_PWMB3_CP         0x1E
#define DMAMUX_PWMB0_WR         0x1F
#define DMAMUX_PWMB1_WR         0x1F
#define DMAMUX_PWMB2_WR         0x1F
#define DMAMUX_PWMB3_WR         0x1F
#define DMAMUX_TMRA0_CP         0x20
#define DMAMUX_TMRA0_CMP1       0x20
#define DMAMUX_TMRA0_CMP2       0x21
#define DMAMUX_TMRA1_CP         0x22
#define DMAMUX_TMRA1_CMP1       0x22
#define DMAMUX_TMRA1_CMP2       0x23
#define DMAMUX_TMRA2_CP         0x24
#define DMAMUX_TMRA2_CMP1       0x24
#define DMAMUX_TMRA2_CMP2       0x25
#define DMAMUX_TMRA3_CP         0x26
#define DMAMUX_TMRA3_CMP1       0x26
#define DMAMUX_TMRA3_CMP2       0x27
#define DMAMUX_TMRB0_CP         0x28
#define DMAMUX_TMRB0_CMP1       0x28
#define DMAMUX_TMRB0_CMP2       0x29
#define DMAMUX_TMRB1_CP         0x2A
#define DMAMUX_TMRB1_CMP1       0x2A
#define DMAMUX_TMRB1_CMP2       0x2B
#define DMAMUX_TMRB2_CP         0x2C
#define DMAMUX_TMRB2_CMP1       0x2C
#define DMAMUX_TMRB2_CMP2       0x2D
#define DMAMUX_TMRB3_CP         0x2E
#define DMAMUX_TMRB3_CMP1       0x2E
#define DMAMUX_TMRB3_CMP2       0x2F
#define DMAMUX_ADCA_ES          0x30
#define DMAMUX_ADCB_ES          0x31
#define DMAMUX_DACA_FIFO        0x32
#define DMAMUX_DACB_FIFO        0x33
#define DMAMUX_CMPA             0x34
#define DMAMUX_CMPB             0x35
#define DMAMUX_CMPC             0x36
#define DMAMUX_CMPD             0x37
#define DMAMUX_XBAR_DSC0        0x38
#define DMAMUX_XBAR_DSC1        0x39
#define DMAMUX_XBAR_DSC2        0x3A
#define DMAMUX_XBAR_DSC3        0x3B
#define DMAMUX_DMA_MUX0         0x3C
#define DMAMUX_DMA_MUX1         0x3D
#define DMAMUX_DMA_MUX2         0x3E
#define DMAMUX_DMA_MUX3         0x3F

//CR register
#define DMA_CR_ACTIVE_MASK      (UWord32)0x80000000
#define DMA_CR_CX_MASK          (UWord32)0x00020000
#define DMA_CR_ECX_MASK         (UWord32)0x00010000
#define DMA_CR_EMLM_MASK        (UWord32)0x00000080
#define DMA_CR_CLM_MASK         (UWord32)0x00000040
#define DMA_CR_HALT_MASK        (UWord32)0x00000020
#define DMA_CR_HOE_MASK         (UWord32)0x00000010
#define DMA_CR_ERCA_MASK        (UWord32)0x00000004
#define DMA_CR_EDBG_MASK        (UWord32)0x00000002
#define DMA_CR_EBWR_MASK        (UWord32)0x00000001

//ES register
#define DMA_ANY_ERROR             (UWord32)0x80000000     //VLD
#define DMA_TRANSFER_CANCEL       (UWord32)0x00010000     //ECX
#define DMA_CHANNEL_PRIORITY      (UWord32)0x00004000     //CPE
#define DMA_SOURCE_ADDRESS        (UWord32)0x00000080     //SAE
#define DMA_SOURCE_OFFSET         (UWord32)0x00000040     //SOE
#define DMA_DESTINATION_ADDRESS   (UWord32)0x00000020     //DAE
#define DMA_DESTINATION_OFFSET    (UWord32)0x00000010     //DOE
#define DMA_NBYTES_CIFER_CONFIG   (UWord32)0x00000008     //NCE
#define DMA_SCATTER_GATHER_CONFIG (UWord32)0x00000004     //SGE
#define DMA_SOURCE_BUS            (UWord32)0x00000002     //SBE
#define DMA_DESTINATION_BUS       (UWord32)0x00000001     //DBE
#define DMA_ES_ERRCHN_MASK        (UWord32)0x00000300     //ERRCHN
#define DMA_ES_ERRCHN_SHIFT       8

//ERQ register
#define DMA_ERQ_ERQ0_MASK       (UWord32)0x00000001
#define DMA_ERQ_ERQ1_MASK       (UWord32)0x00000002
#define DMA_ERQ_ERQ2_MASK       (UWord32)0x00000004
#define DMA_ERQ_ERQ3_MASK       (UWord32)0x00000008

//EEI register
#define DMA_EEI_EEI0_MASK       (UWord32)0x00000001
#define DMA_EEI_EEI1_MASK       (UWord32)0x00000002
#define DMA_EEI_EEI2_MASK       (UWord32)0x00000004
#define DMA_EEI_EEI3_MASK       (UWord32)0x00000008

#define DMA_ALL_CHANNELS        0x00000040
#define DMA_CHANNEL_0           0x00000000
#define DMA_CHANNEL_1           0x00000001
#define DMA_CHANNEL_2           0x00000002
#define DMA_CHANNEL_3           0x00000003
#define DMA_CHANNEL_0_MASK      0x00000001
#define DMA_CHANNEL_1_MASK      0x00000002
#define DMA_CHANNEL_2_MASK      0x00000004
#define DMA_CHANNEL_3_MASK      0x00000008


#define DMA_DCHPRIX_ECP_MASK    0x00000080
#define DMA_DCHPRIX_DPA_MASK    0x00000040
#define DMA_DCHPRIX_CHPRI_MASK  0x00000003


#define DMA_ATTR_SMOD_MASK      0xf800
#define DMA_ATTR_SSIZE_MASK     0x0700
#define DMA_ATTR_DMOD_MASK      0x00f8
#define DMA_ATTR_DSIZE_MASK     0x0007

#define DMA_DISABLE             0
#define DMA_1B                  1UL
#define DMA_2B                  2UL
#define DMA_4B                  3UL
#define DMA_8B                  4UL
#define DMA_16B                 5UL
#define DMA_32B                 6UL
#define DMA_64B                 7UL
#define DMA_128B                8UL
#define DMA_256B                9UL
#define DMA_512B                10UL
#define DMA_1K                  11UL
#define DMA_2K                  12UL
#define DMA_4K                  13UL
#define DMA_8K                  14UL
#define DMA_16K                 15UL
#define DMA_32K                 16UL
#define DMA_64K                 17UL
#define DMA_128K                18UL
#define DMA_256K                19UL
#define DMA_512K                20UL
#define DMA_1M                  21UL
#define DMA_2M                  22UL
#define DMA_4M                  23UL
#define DMA_8M                  24UL
#define DMA_16M                 25UL
#define DMA_32M                 26UL
#define DMA_64M                 27UL
#define DMA_128M                28UL
#define DMA_256M                29UL
#define DMA_512M                30UL
#define DMA_1G                  31UL

#define DMA_BYTE                0UL
#define DMA_WORD                1UL
#define DMA_LONGWORD            2UL
#define DMA_16BYTE              5UL

#define DMA_NBYTES_SMLOE_MASK   (UWord32)0x80000000
#define DMA_NBYTES_DMLOE_MASK   (UWord32)0x40000000
#define DMA_NBYTES_MLOFF_MASK   (UWord32)0x3FFFFC00
#define DMA_NBYTES_NBYTES_MLOFF_MASK  (UWord32)0x3FFFFFFF
#define DMA_NBYTES_NBYTES_MLON_MASK   (UWord32)0x000003FF
#define DMA_NBYTES_NBYTES_MLNO_MASK   (UWord32)0xFFFFFFFF

#define DMA_CITER_ELINK_MASK           0x8000
#define DMA_CITER_ELINKNO_CITER_MASK   0x7FFF
#define DMA_CITER_ELINKYES_CITER_MASK  0x01FF
#define DMA_CITER_ELINKYES_LINKCH_MASK 0x0600

//CSR register
#define DMA_CSR_BWC_MASK        0xC000
#define DMA_CSR_MAJORLINKCH_MASK 0x0300
#define DMA_CSR_DONE_MASK       0x0080
#define DMA_CSR_ACTIVE_MASK     0x0040
#define DMA_CSR_MAJORELINK_MASK 0x0020
#define DMA_CSR_ESG_MASK        0x0010
#define DMA_CSR_DREQ_MASK       0x0008
#define DMA_CSR_INTHALF_MASK    0x0004
#define DMA_CSR_INTMAJOR_MASK   0x0002
#define DMA_CSR_START_MASK      0x0001

#define DMA_NO_STALL            0x0000
#define DMA_4CYCLES_STALL       0x8000
#define DMA_8CYCLES_STALL       0xC000





#define DMA_REQUEST_0           (UWord32)0x00
#define DMA_REQUEST_1           (UWord32)0x01
#define DMA_REQUEST_2           (UWord32)0x02
#define DMA_REQUEST_3           (UWord32)0x03
#define DMA_REQUEST_4           (UWord32)0x04
#define DMA_REQUEST_5           (UWord32)0x05
#define DMA_REQUEST_6           (UWord32)0x06
#define DMA_REQUEST_7           (UWord32)0x07
#define DMA_REQUEST_8           (UWord32)0x08
#define DMA_REQUEST_9           (UWord32)0x09
#define DMA_REQUEST_10          (UWord32)0x0A
#define DMA_REQUEST_11          (UWord32)0x0B
#define DMA_REQUEST_12          (UWord32)0x0C
#define DMA_REQUEST_13          (UWord32)0x0D
#define DMA_REQUEST_14          (UWord32)0x0E
#define DMA_REQUEST_15          (UWord32)0x0F


#define DMA_REQC_DMAC0_MASK     (UWord32)0x00F000000
#define DMA_REQC_DMAC1_MASK     (UWord32)0x0000F0000
#define DMA_REQC_DMAC2_MASK     (UWord32)0x000000F00
#define DMA_REQC_DMAC3_MASK     (UWord32)0x00000000F
#define DMA_REQC_CFSM0          (UWord32)0x080000000
#define DMA_REQC_CFSM1          (UWord32)0x000800000
#define DMA_REQC_CFSM2          (UWord32)0x000008000
#define DMA_REQC_CFSM3          (UWord32)0x000000080

#define DMA_DSR_BCR_CE_MASK     (UWord32)0x040000000
#define DMA_DSR_BCR_BES_MASK    (UWord32)0x020000000
#define DMA_DSR_BCR_BED_MASK    (UWord32)0x010000000
#define DMA_DSR_BCR_REQ_MASK    (UWord32)0x004000000
#define DMA_DSR_BCR_BSY_MASK    (UWord32)0x002000000
#define DMA_DSR_BCR_DONE_MASK   (UWord32)0x001000000
#define DMA_DSR_BCR_BCR_MASK    (UWord32)0x000FFFFFF

#define DMA_DCR_EINT_MASK       (UWord32)0x080000000
#define DMA_DCR_ERQ_MASK        (UWord32)0x040000000
#define DMA_DCR_CS_MASK         (UWord32)0x020000000
#define DMA_DCR_AA_MASK         (UWord32)0x010000000
#define DMA_DCR_SINC_MASK       (UWord32)0x000400000
#define DMA_DCR_SSIZE_MASK      (UWord32)0x000300000
#define DMA_DCR_DINC_MASK       (UWord32)0x000080000
#define DMA_DCR_DSIZE_MASK      (UWord32)0x000060000
#define DMA_DCR_START_MASK      (UWord32)0x000010000
#define DMA_DCR_SMOD_MASK       (UWord32)0x00000F000
#define DMA_DCR_DMOD_MASK       (UWord32)0x000000F00
#define DMA_DCR_D_REQ_MASK      (UWord32)0x000000080
#define DMA_DCR_LINKCC_MASK     (UWord32)0x000000030
#define DMA_DCR_LCH1_MASK       (UWord32)0x00000000C
#define DMA_DCR_LCH2_MASK       (UWord32)0x000000003

/***********************************************************************
*  control command macros
***********************************************************************/

/* DMA initialization */
void Dma_common_init(arch_sEDMA *pDmaBase);
void Dma_common_init_last(arch_sEDMA *pDmaBase);

void Dma_channel_init(arch_sEDMAchannel *pDmaBase);

void Dma_init(void);
#define ioctlDMA_INIT(pDmaBase, param) Dma_init()


#define ioctlDMAMUX_READ_CHANNEL_0_CONFIG_REG(pDmaMuxBase, param) \
        periphMemRead(&((pDmaMuxBase)->chcfg[0]))

#define ioctlDMAMUX_WRITE_CHANNEL_0_CONFIG_REG(pDmaMuxBase, param) \
        periphMemWrite( param, &((pDmaMuxBase)->chcfg[0]))

#define ioctlDMAMUX_SET_CHANNEL_0(pDmaMuxBase, param) \
        if (param)periphBitSet(DMAMUX_CHCFG_ENABLE_MASK, &((pDmaMuxBase)->chcfg[0])); \
        else periphBitClear(DMAMUX_CHCFG_ENABLE_MASK, &((pDmaMuxBase)->chcfg[0]))        

#define ioctlDMAMUX_SET_CHANNEL_0_SOURCE(pDmaBase, param) \
        periphBitGrpSet(DMAMUX_CHCFG_SOURCE_MASK, param, &((pDmaBase)->chcfg[0])) 

#define ioctlDMAMUX_READ_CHANNEL_1_CONFIG_REG(pDmaMuxBase, param) \
        periphMemRead(&((pDmaMuxBase)->chcfg[1]))

#define ioctlDMAMUX_WRITE_CHANNEL_1_CONFIG_REG(pDmaMuxBase, param) \
        periphMemWrite( param, &((pDmaMuxBase)->chcfg[1]))

#define ioctlDMAMUX_SET_CHANNEL_1(pDmaMuxBase, param) \
        if (param)periphBitSet(DMAMUX_CHCFG_ENABLE_MASK, &((pDmaMuxBase)->chcfg[1])); \
        else periphBitClear(DMAMUX_CHCFG_ENABLE_MASK, &((pDmaMuxBase)->chcfg[1]))        

#define ioctlDMAMUX_SET_CHANNEL_1_SOURCE(pDmaBase, param) \
        periphBitGrpSet(DMAMUX_CHCFG_SOURCE_MASK<<8, param<<8, (UWord16*)&((pDmaBase)->chcfg[0])) 

#define ioctlDMAMUX_READ_CHANNEL_2_CONFIG_REG(pDmaMuxBase, param) \
        periphMemRead(&((pDmaMuxBase)->chcfg[2]))

#define ioctlDMAMUX_WRITE_CHANNEL_2_CONFIG_REG(pDmaMuxBase, param) \
        periphMemWrite( param, &((pDmaMuxBase)->chcfg[2]))

#define ioctlDMAMUX_SET_CHANNEL_2(pDmaMuxBase, param) \
        if (param)periphBitSet(DMAMUX_CHCFG_ENABLE_MASK, &((pDmaMuxBase)->chcfg[2])); \
        else periphBitClear(DMAMUX_CHCFG_ENABLE_MASK, &((pDmaMuxBase)->chcfg[2]))        

#define ioctlDMAMUX_SET_CHANNEL_2_SOURCE(pDmaBase, param) \
        periphBitGrpSet(DMAMUX_CHCFG_SOURCE_MASK, param, &((pDmaBase)->chcfg[2]))

#define ioctlDMAMUX_READ_CHANNEL_3_CONFIG_REG(pDmaMuxBase, param) \
        periphMemRead(&((pDmaMuxBase)->chcfg[3]))

#define ioctlDMAMUX_WRITE_CHANNEL_3_CONFIG_REG(pDmaMuxBase, param) \
        periphMemWrite( param, &((pDmaMuxBase)->chcfg[3]))

#define ioctlDMAMUX_SET_CHANNEL_3(pDmaMuxBase, param) \
        if (param)periphBitSet(DMAMUX_CHCFG_ENABLE_MASK, &((pDmaMuxBase)->chcfg[3])); \
        else periphBitClear(DMAMUX_CHCFG_ENABLE_MASK, &((pDmaMuxBase)->chcfg[3]))        

#define ioctlDMAMUX_SET_CHANNEL_3_SOURCE(pDmaBase, param) \
        periphBitGrpSet(DMAMUX_CHCFG_SOURCE_MASK<<8, param<<8, (UWord16*)&((pDmaBase)->chcfg[2]))



#define ioctlDMA_READ_CR_REG(pDmaBase, param) \
        periphMemRead(&((pDmaBase)->cr))

#define ioctlDMA_WRITE_CR_REG(pDmaBase, param) \
        periphMemWrite( param, &((pDmaBase)->cr))

#define ioctlDMA_GET_ACTIVE_STATUS(pDmaBase, param) \
        periphBitTest(DMA_CR_ACTIVE_MASK, &((pDmaBase)->cr))

#define ioctlDMA_SET_CANCEL_TRANSFER(pDmaBase, param) \
        periphBitSet(DMA_CR_CX_MASK, &((pDmaBase)->cr))

#define ioctlDMA_SET_CANCEL_TRANSFER_WITH_ERROR(pDmaBase, param) \
        periphBitSet(DMA_CR_ECX_MASK, &((pDmaBase)->cr))

#define ioctlDMA_SET_MINOR_LOOP_MAPPING(pDmaBase, param) \
        if (param)periphBitSet(DMA_CR_EMLM_MASK, &((pDmaBase)->cr)); \
        else periphBitClear(DMA_CR_EMLM_MASK, &((pDmaBase)->cr))  

#define ioctlDMA_SET_CONTINUOUS_LINK_MODE(pDmaBase, param) \
        if (param)periphBitSet(DMA_CR_CLM_MASK, &((pDmaBase)->cr)); \
        else periphBitClear(DMA_CR_CLM_MASK, &((pDmaBase)->cr)) 

#define ioctlDMA_SET_HALT_OPERATION(pDmaBase, param) \
        if (param)periphBitSet(DMA_CR_HALT_MASK, &((pDmaBase)->cr)); \
        else periphBitClear(DMA_CR_HALT_MASK, &((pDmaBase)->cr))

#define ioctlDMA_SET_HALT_ON_ERROR(pDmaBase, param) \
        if (param)periphBitSet(DMA_CR_HOE_MASK, &((pDmaBase)->cr)); \
        else periphBitClear(DMA_CR_HOE_MASK, &((pDmaBase)->cr))

#define ioctlDMA_SET_ROUND_ROBIN_CHAN_ARBITRATION(pDmaBase, param) \
        if (param)periphBitSet(DMA_CR_ERCA_MASK, &((pDmaBase)->cr)); \
        else periphBitClear(DMA_CR_ERCA_MASK, &((pDmaBase)->cr))

#define ioctlDMA_SET_DEBUG_MODE(pDmaBase, param) \
        if (param)periphBitSet(DMA_CR_EDBG_MASK, &((pDmaBase)->cr)); \
        else periphBitClear(DMA_CR_EDBG_MASK, &((pDmaBase)->cr))

#define ioctlDMA_SET_BUFFERED_WRITES(pDmaBase, param) \
        if (param)periphBitSet(DMA_CR_EBWR_MASK, &((pDmaBase)->cr)); \
        else periphBitClear(DMA_CR_EBWR_MASK, &((pDmaBase)->cr))


#define ioctlDMA_READ_ES_REG(pDmaBase, param) \
        periphMemRead(&((pDmaBase)->es))

#define ioctlDMA_TEST_LAST_DMA_ERROR(pDmaBase, param) \
        periphBitTest(param, &((pDmaBase)->es))

#define ioctlDMA_GET_CHANNEL_OF_LAST_DMA_ERROR(pDmaBase, param) \
        (periphBitTest(DMA_ES_ERRCHN_MASK, &((pDmaBase)->es))>>DMA_ES_ERRCHN_SHIFT)


#define ioctlDMA_READ_ERQ_REG(pDmaBase, param) \
        periphMemRead(&((pDmaBase)->erq))

#define ioctlDMA_WRITE_ERQ_REG(pDmaBase, param) \
        periphMemWrite( param, &((pDmaBase)->erq))

#define ioctlDMA_SET_DMA_REQUEST_0(pDmaBase, param) \
        if (param)periphBitSet(DMA_ERQ_ERQ0_MASK, &((pDmaBase)->erq)); \
        else periphBitClear(DMA_ERQ_ERQ0_MASK, &((pDmaBase)->erq))

#define ioctlDMA_SET_DMA_REQUEST_1(pDmaBase, param) \
        if (param)periphBitSet(DMA_ERQ_ERQ1_MASK, &((pDmaBase)->erq)); \
        else periphBitClear(DMA_ERQ_ERQ1_MASK, &((pDmaBase)->erq))

#define ioctlDMA_SET_DMA_REQUEST_2(pDmaBase, param) \
        if (param)periphBitSet(DMA_ERQ_ERQ2_MASK, &((pDmaBase)->erq)); \
        else periphBitClear(DMA_ERQ_ERQ2_MASK, &((pDmaBase)->erq))

#define ioctlDMA_SET_DMA_REQUEST_3(pDmaBase, param) \
        if (param)periphBitSet(DMA_ERQ_ERQ3_MASK, &((pDmaBase)->erq)); \
        else periphBitClear(DMA_ERQ_ERQ3_MASK, &((pDmaBase)->erq))

#define ioctlDMA_READ_EEI_REG(pDmaBase, param) \
        periphMemRead(&((pDmaBase)->eei))

#define ioctlDMA_WRITE_EEI_REG(pDmaBase, param) \
        periphMemWrite( param, &((pDmaBase)->eei))

#define ioctlDMA_SET_DMA_ERROR_INTERRUPT_0(pDmaBase, param) \
        if (param)periphBitSet(DMA_EEI_EEI0_MASK, &((pDmaBase)->eei)); \
        else periphBitClear(DMA_EEI_EEI0_MASK, &((pDmaBase)->eei))

#define ioctlDMA_SET_DMA_ERROR_INTERRUPT_1(pDmaBase, param) \
        if (param)periphBitSet(DMA_EEI_EEI1_MASK, &((pDmaBase)->eei)); \
        else periphBitClear(DMA_EEI_EEI1_MASK, &((pDmaBase)->eei))

#define ioctlDMA_SET_DMA_ERROR_INTERRUPT_2(pDmaBase, param) \
        if (param)periphBitSet(DMA_EEI_EEI2_MASK, &((pDmaBase)->eei)); \
        else periphBitClear(DMA_EEI_EEI2_MASK, &((pDmaBase)->eei))

#define ioctlDMA_SET_DMA_ERROR_INTERRUPT_3(pDmaBase, param) \
        if (param)periphBitSet(DMA_EEI_EEI3_MASK, &((pDmaBase)->eei)); \
        else periphBitClear(DMA_EEI_EEI3_MASK, &((pDmaBase)->eei))

#define ioctlDMA_WRITE_CEEI_REG(pDmaBase, param) \
        periphMemWrite( param, &((pDmaBase)->ceei))

#define ioctlDMA_SET_DISABLE_DMA_ERROR_INTERRUPT(pDmaBase, param) \
        periphMemWrite(param, &((pDmaBase)->ceei))

#define ioctlDMA_WRITE_SEEI_REG(pDmaBase, param) \
        periphMemWrite( param, &((pDmaBase)->seei))

#define ioctlDMA_SET_ENABLE_DMA_ERROR_INTERRUPT(pDmaBase, param) \
        periphMemWrite(param, &((pDmaBase)->seei))

#define ioctlDMA_WRITE_CERQ_REG(pDmaBase, param) \
        periphMemWrite( param, &((pDmaBase)->cerq))

#define ioctlDMA_SET_DISABLE_DMA_REQUEST(pDmaBase, param) \
        periphMemWrite(param, &((pDmaBase)->cerq))

#define ioctlDMA_WRITE_SERQ_REG(pDmaBase, param) \
        periphMemWrite( param, &((pDmaBase)->serq))

#define ioctlDMA_SET_ENABLE_DMA_REQUEST(pDmaBase, param) \
        periphMemWrite(param, &((pDmaBase)->serq))

#define ioctlDMA_WRITE_CDNE_REG(pDmaBase, param) \
        periphMemWrite( param, &((pDmaBase)->cdne))

#define ioctlDMA_CLEAR_DMA_DONE_BIT(pDmaBase, param) \
        periphMemWrite(param, &((pDmaBase)->cdne))

#define ioctlDMA_WRITE_SSRT_REG(pDmaBase, param) \
        periphMemWrite( param, &((pDmaBase)->ssrt))

#define ioctlDMA_SET_DMA_START_BIT(pDmaBase, param) \
        periphMemWrite(param, &((pDmaBase)->ssrt))

#define ioctlDMA_WRITE_CERR_REG(pDmaBase, param) \
        periphMemWrite( param, &((pDmaBase)->cerr))

#define ioctlDMA_CLEAR_DMA_ERROR_BIT(pDmaBase, param) \
        periphMemWrite(param, &((pDmaBase)->cerr))

#define ioctlDMA_WRITE_CINT_REG(pDmaBase, param) \
        periphMemWrite( param, &((pDmaBase)->cint))

#define ioctlDMA_CLEAR_INTERRUPT_REQUEST(pDmaBase, param) \
        periphMemWrite(param, &((pDmaBase)->cint))

#define ioctlDMA_READ_INT_REG(pDmaBase, param) \
        periphMemRead(&((pDmaBase)->int_req))

#define ioctlDMA_WRITE_INT_REG(pDmaBase, param) \
        periphMemWrite(param, &((pDmaBase)->int_req))

#define ioctlDMA_TEST_INTERRUPT_REQUEST_BIT(pDmaBase, param) \
        periphBitTest(param, &((pDmaBase)->int_req))

#define ioctlDMA_CLEAR_INTERRUPT_REQUEST_BIT(pDmaBase, param) \
        periphMemWrite(param, &((pDmaBase)->int_req))


#define ioctlDMA_READ_ERR_REG(pDmaBase, param) \
        periphMemRead(&((pDmaBase)->err))

#define ioctlDMA_WRITE_ERR_REG(pDmaBase, param) \
        periphMemWrite(param, &((pDmaBase)->err))

#define ioctlDMA_TEST_ERROR_INTERRUPT_BIT(pDmaBase, param) \
        periphBitTest(param, &((pDmaBase)->err))

#define ioctlDMA_CLEAR_ERROR_INTERRUPT_BIT(pDmaBase, param) \
        periphMemWrite(param, &((pDmaBase)->err))

#define ioctlDMA_READ_HRS_REG(pDmaBase, param) \
        periphMemRead(&((pDmaBase)->hrs))

#define ioctlDMA_TEST_HARDWARE_REQUEST_BIT(pDmaBase, param) \
        periphBitTest(param, &((pDmaBase)->hrs))

#define ioctlDMA_READ_EARS_REG(pDmaBase, param) \
        periphMemRead(&((pDmaBase)->ears))

#define ioctlDMA_WRITE_EARS_REG(pDmaBase, param) \
        periphMemWrite(param, &((pDmaBase)->ears))

#define ioctlDMA_SET_ASYNC_DMA_REQ_IN_STOP_CH0(pDmaBase, param) \
        if (param)periphBitSet(DMA_CHANNEL_0_MASK, &((pDmaBase)->ears)); \
        else periphBitClear(DMA_CHANNEL_0_MASK, &((pDmaBase)->ears))

#define ioctlDMA_SET_ASYNC_DMA_REQ_IN_STOP_CH1(pDmaBase, param) \
        if (param)periphBitSet(DMA_CHANNEL_1_MASK, &((pDmaBase)->ears)); \
        else periphBitClear(DMA_CHANNEL_1_MASK, &((pDmaBase)->ears))

#define ioctlDMA_SET_ASYNC_DMA_REQ_IN_STOP_CH2(pDmaBase, param) \
        if (param)periphBitSet(DMA_CHANNEL_2_MASK, &((pDmaBase)->ears)); \
        else periphBitClear(DMA_CHANNEL_2_MASK, &((pDmaBase)->ears))

#define ioctlDMA_SET_ASYNC_DMA_REQ_IN_STOP_CH3(pDmaBase, param) \
        if (param)periphBitSet(DMA_CHANNEL_3_MASK, &((pDmaBase)->ears)); \
        else periphBitClear(DMA_CHANNEL_3_MASK, &((pDmaBase)->ears))

#define ioctlDMA_READ_DCHPRI0_REG(pDmaBase, param) \
        periphMemRead(&((pDmaBase)->dchpri[0]))

#define ioctlDMA_WRITE_DCHPRI0_REG(pDmaBase, param) \
        periphMemWrite(param, &((pDmaBase)->dchpri[0]))

#define ioctlDMA_SET_CHANNEL_0_PREEMPTION(pDmaBase, param) \
        if (param)periphBitSet(DMA_DCHPRIX_ECP_MASK, &((pDmaBase)->dchpri[0])); \
        else periphBitClear(DMA_DCHPRIX_ECP_MASK, &((pDmaBase)->dchpri[0]))

#define ioctlDMA_SET_CHANNEL_0_PREEMPT_ABILITY(pDmaBase, param) \
        if (param)periphBitSet(DMA_DCHPRIX_DPA_MASK, &((pDmaBase)->dchpri[0])); \
        else periphBitClear(DMA_DCHPRIX_DPA_MASK, &((pDmaBase)->dchpri[0]))

#define ioctlDMA_SET_CHANNEL_0_ARB_PRIORITY(pDmaBase, param) \
        periphBitGrpSet(DMA_DCHPRIX_CHPRI_MASK, param, &((pDmaBase)->dchpri[0]))

#define ioctlDMA_READ_DCHPRI1_REG(pDmaBase, param) \
        periphMemRead(&((pDmaBase)->dchpri[1]))

#define ioctlDMA_WRITE_DCHPRI1_REG(pDmaBase, param) \
        periphMemWrite(param, &((pDmaBase)->dchpri[1]))

#define ioctlDMA_SET_CHANNEL_1_PREEMPTION(pDmaBase, param) \
        if (param)periphBitSet(DMA_DCHPRIX_ECP_MASK, &((pDmaBase)->dchpri[1])); \
        else periphBitClear(DMA_DCHPRIX_ECP_MASK, &((pDmaBase)->dchpri[1]))

#define ioctlDMA_SET_CHANNEL_1_PREEMPT_ABILITY(pDmaBase, param) \
        if (param)periphBitSet(DMA_DCHPRIX_DPA_MASK, &((pDmaBase)->dchpri[1])); \
        else periphBitClear(DMA_DCHPRIX_DPA_MASK, &((pDmaBase)->dchpri[1]))

#define ioctlDMA_SET_CHANNEL_1_ARB_PRIORITY(pDmaBase, param) \
        periphBitGrpSet(DMA_DCHPRIX_CHPRI_MASK<<8, (param)<<8, &((pDmaBase)->dchpri[0]))

#define ioctlDMA_READ_DCHPRI2_REG(pDmaBase, param) \
        periphMemRead(&((pDmaBase)->dchpri[2]))

#define ioctlDMA_WRITE_DCHPRI2_REG(pDmaBase, param) \
        periphMemWrite(param, &((pDmaBase)->dchpri[2]))

#define ioctlDMA_SET_CHANNEL_2_PREEMPTION(pDmaBase, param) \
        if (param)periphBitSet(DMA_DCHPRIX_ECP_MASK, &((pDmaBase)->dchpri[2])); \
        else periphBitClear(DMA_DCHPRIX_ECP_MASK, &((pDmaBase)->dchpri[2]))

#define ioctlDMA_SET_CHANNEL_2_PREEMPT_ABILITY(pDmaBase, param) \
        if (param)periphBitSet(DMA_DCHPRIX_DPA_MASK, &((pDmaBase)->dchpri[2])); \
        else periphBitClear(DMA_DCHPRIX_DPA_MASK, &((pDmaBase)->dchpri[2]))

#define ioctlDMA_SET_CHANNEL_2_ARB_PRIORITY(pDmaBase, param) \
        periphBitGrpSet(DMA_DCHPRIX_CHPRI_MASK, param, &((pDmaBase)->dchpri[2]))

#define ioctlDMA_READ_DCHPRI3_REG(pDmaBase, param) \
        periphMemRead(&((pDmaBase)->dchpri[3]))

#define ioctlDMA_WRITE_DCHPRI3_REG(pDmaBase, param) \
        periphMemWrite(param, &((pDmaBase)->dchpri[3]))

#define ioctlDMA_SET_CHANNEL_3_PREEMPTION(pDmaBase, param) \
        if (param)periphBitSet(DMA_DCHPRIX_ECP_MASK, &((pDmaBase)->dchpri[3])); \
        else periphBitClear(DMA_DCHPRIX_ECP_MASK, &((pDmaBase)->dchpri[3]))

#define ioctlDMA_SET_CHANNEL_3_PREEMPT_ABILITY(pDmaBase, param) \
        if (param)periphBitSet(DMA_DCHPRIX_DPA_MASK, &((pDmaBase)->dchpri[3])); \
        else periphBitClear(DMA_DCHPRIX_DPA_MASK, &((pDmaBase)->dchpri[3]))

#define ioctlDMA_SET_CHANNEL_3_ARB_PRIORITY(pDmaBase, param) \
        periphBitGrpSet(DMA_DCHPRIX_CHPRI_MASK<<8, (param)<<8, &((pDmaBase)->dchpri[2]))

#define ioctlDMACH_SET_SOURCE_ADDRESS(pDmaChBase, param) \
        periphMemWrite(param, &((pDmaChBase)->sadr))

#define ioctlDMACH_GET_SOURCE_ADDRESS(pDmaChBase, param) \
        periphMemRead(&((pDmaChBase)->sadr))

#define ioctlDMACH_SET_SOURCE_ADDRESS_OFFSET(pDmaChBase, param) \
        periphMemWrite(param, &((pDmaChBase)->soff))

#define ioctlDMACH_GET_SOURCE_ADDRESS_OFFSET(pDmaChBase, param) \
        periphMemRead(&((pDmaChBase)->soff))

#define ioctlDMACH_WRITE_ATTRIBUTES_REG(pDmaChBase, param) \
        periphMemWrite(param, &((pDmaChBase)->attr))

#define ioctlDMACH_READ_ATTRIBUTES_REG(pDmaChBase, param) \
        periphMemRead(&((pDmaChBase)->attr))

#define ioctlDMACH_SET_SOURCE_ADDRESS_MODULO(pDmaChBase, param) \
        periphBitGrpSet(DMA_ATTR_SMOD_MASK, (param)<<11, &((pDmaChBase)->attr))

#define ioctlDMACH_SET_SOURCE_SIZE(pDmaChBase, param) \
        periphBitGrpSet(DMA_ATTR_SSIZE_MASK, (param)<<8, &((pDmaChBase)->attr))

#define ioctlDMACH_SET_DESTINATION_ADDRESS_MODULO(pDmaChBase, param) \
        periphBitGrpSet(DMA_ATTR_DMOD_MASK, (param)<<3, &((pDmaChBase)->attr))

#define ioctlDMACH_SET_DESTINATION_SIZE(pDmaChBase, param) \
        periphBitGrpSet(DMA_ATTR_DSIZE_MASK, param, &((pDmaChBase)->attr))

#define ioctlDMACH_WRITE_NBYTES_MLOFF_REG(pDmaChBase, param) \
        periphMemWrite(param, &((pDmaChBase)->nbytes_mlno))

#define ioctlDMACH_READ_NBYTES_MLOFF_REG(pDmaChBase, param) \
        periphMemRead(&((pDmaChBase)->nbytes_mlno))

#define ioctlDMACH_SET_SOURCE_MINOR_LOOP_OFFSET(pDmaChBase, param) \
        if (param)periphBitSet(DMA_NBYTES_SMLOE_MASK, &((pDmaChBase)->nbytes_mlno)); \
        else periphBitClear(DMA_NBYTES_SMLOE_MASK, &((pDmaChBase)->nbytes_mlno))

#define ioctlDMACH_SET_DESTINATION_MINOR_LOOP_OFFSET(pDmaChBase, param) \
        if (param)periphBitSet(DMA_NBYTES_DMLOE_MASK, &((pDmaChBase)->nbytes_mlno)); \
        else periphBitClear(DMA_NBYTES_DMLOE_MASK, &((pDmaChBase)->nbytes_mlno))

#define ioctlDMACH_WRITE_NBYTES_MLOFF_REG(pDmaChBase, param) \
        periphMemWrite(param, &((pDmaChBase)->nbytes_mlno))

#define ioctlDMACH_SET_NBYTES_WITH_MLON(pDmaBase, param) \
        periphBitGrpSet32(DMA_NBYTES_NBYTES_MLON_MASK, param,  &((pDmaBase)->nbytes_mloffyes))

#define ioctlDMACH_SET_OFFSET_WITH_MLON(pDmaBase, param) \
        periphBitGrpSet32(DMA_NBYTES_MLOFF_MASK, (param)<<10,  &((pDmaBase)->nbytes_mloffyes))

#define ioctlDMACH_SET_NBYTES_WITH_MLOFF(pDmaBase, param) \
        periphBitGrpSet32(DMA_NBYTES_NBYTES_MLOFF_MASK, param,  &((pDmaBase)->nbytes_mloffno))

#define ioctlDMACH_SET_LAST_SOURCE_ADDR_ADJ(pDmaChBase, param) \
        periphMemWrite(param, &((pDmaChBase)->slast))

#define ioctlDMACH_GET_LAST_SOURCE_ADDR_ADJ(pDmaChBase, param) \
        periphMemRead(&((pDmaChBase)->slast))

#define ioctlDMACH_SET_DESTINATION_ADDRESS(pDmaChBase, param) \
        periphMemWrite(param, &((pDmaChBase)->daddr))

#define ioctlDMACH_GET_DESTINATION_ADDRESS(pDmaChBase, param) \
        periphMemRead(&((pDmaChBase)->daddr))

#define ioctlDMACH_SET_DESTINATION_ADDRESS_OFFSET(pDmaChBase, param) \
        periphMemWrite(param, &((pDmaChBase)->doff))

#define ioctlDMACH_GET_DESTINATION_ADDRESS_OFFSET(pDmaChBase, param) \
        periphMemRead(&((pDmaChBase)->doff))

#define ioctlDMACH_WRITE_CITER_ELINK_REG(pDmaChBase, param) \
        periphMemWrite(param, &((pDmaChBase)->citer_elinkno))

#define ioctlDMACH_READ_CITER_ELINK_REG(pDmaChBase, param) \
        periphMemRead(&((pDmaChBase)->citer_elinkno))

#define ioctlDMACH_SET_CHANNEL_TO_CHANNEL_LINKING(pDmaChBase, param) \
        if (param)periphBitSet(DMA_CITER_ELINK_MASK, &((pDmaChBase)->citer_elinkno)); \
        else periphBitClear(DMA_CITER_ELINK_MASK, &((pDmaChBase)->citer_elinkno))

#define ioctlDMACH_SET_CURRENT_MAJOR_ITER_CNT_ELINKNO(pDmaChBase, param) \
        periphBitGrpSet(DMA_CITER_ELINKNO_CITER_MASK, param, &((pDmaChBase)->citer_elinkno))

#define ioctlDMACH_SET_CURRENT_MAJOR_ITER_CNT_ELINKYES(pDmaChBase, param) \
        periphBitGrpSet(DMA_CITER_ELINKYES_CITER_MASK, param, &((pDmaChBase)->citer_elinkno))

#define ioctlDMACH_SET_MINOR_LOOP_LINK_CHANNEL_ELINKYES(pDmaChBase, param) \
        periphBitGrpSet(DMA_CITER_ELINKYES_LINKCH_MASK, (param)<<9, &((pDmaChBase)->citer_elinkyes))


#define ioctlDMACH_WRITE_DLASTGA_REG(pDmaChBase, param) \
        periphMemWrite(param, &((pDmaChBase)->dlastsga))

#define ioctlDMACH_READ_DLASTGA_REG(pDmaChBase, param) \
        periphMemRead(&((pDmaChBase)->dlastsga))

#define ioctlDMACH_WRITE_CONTROL_REG(pDmaChBase, param) \
        periphMemWrite(param, &((pDmaChBase)->csr))

#define ioctlDMACH_READ_CONTROL_REG(pDmaChBase, param) \
        periphMemRead(&((pDmaChBase)->csr))

#define ioctlDMACH_SET_BANDWITH_CONTROL(pDmaChBase, param) \
        periphBitGrpSet(DMA_CSR_BWC_MASK, param, &((pDmaChBase)->csr))

#define ioctlDMACH_SET_MAJOR_LOOP_CHANNEL(pDmaChBase, param) \
        periphBitGrpSet(DMA_CSR_MAJORLINKCH_MASK, (param)<<8, &((pDmaChBase)->csr))

#define ioctlDMACH_CLEAR_CHANNEL_DONE(pDmaChBase, param) \
        periphBitClear(DMA_CSR_DONE_MASK, &((pDmaChBase)->csr))

#define ioctlDMACH_TEST_CHANNEL_DONE(pDmaChBase, param) \
        periphBitTest(DMA_CSR_DONE_MASK, &((pDmaChBase)->csr))

#define ioctlDMACH_TEST_CHANNEL_ACTIVE(pDmaChBase, param) \
        periphBitTest(DMA_CSR_ACTIVE_MASK, &((pDmaChBase)->csr))

#define ioctlDMACH_SET_CHANNEL_TO_CHANNEL_LINKING_ON_MAJOR_LOOP_COMPLETE(pDmaChBase, param) \
        if (param)periphBitSet(DMA_CSR_MAJORELINK_MASK, &((pDmaChBase)->csr)); \
        else periphBitClear(DMA_CSR_MAJORELINK_MASK, &((pDmaChBase)->csr))

#define ioctlDMACH_SET_CHANNEL_TO_SCATTER_GATHER_PROCESSING(pDmaChBase, param) \
        if (param)periphBitSet(DMA_CSR_ESG_MASK, &((pDmaChBase)->csr)); \
        else periphBitClear(DMA_CSR_ESG_MASK, &((pDmaChBase)->csr))

#define ioctlDMACH_SET_ERQ_REQUEST(pDmaChBase, param) \
        if (param)periphBitSet(DMA_CSR_DREQ_MASK, &((pDmaChBase)->csr)); \
        else periphBitClear(DMA_CSR_DREQ_MASK, &((pDmaChBase)->csr))

#define ioctlDMACH_SET_HALF_COMPLETE_INTERRUPT(pDmaChBase, param) \
        if (param)periphBitSet(DMA_CSR_INTHALF_MASK, &((pDmaChBase)->csr)); \
        else periphBitClear(DMA_CSR_INTHALF_MASK, &((pDmaChBase)->csr))

#define ioctlDMACH_SET_MAJOR_ITER_COUNT_COMPLETE_INTERRUPT(pDmaChBase, param) \
        if (param)periphBitSet(DMA_CSR_INTMAJOR_MASK, &((pDmaChBase)->csr)); \
        else periphBitClear(DMA_CSR_INTMAJOR_MASK, &((pDmaChBase)->csr))

#define ioctlDMACH_SET_CHANNEL_START(pDmaChBase, param) \
        periphBitSet(DMA_CSR_START_MASK, &((pDmaChBase)->csr))


#define ioctlDMACH_WRITE_BITER_ELINK_REG(pDmaChBase, param) \
        periphMemWrite(param, &((pDmaChBase)->biter_elinkno))

#define ioctlDMACH_READ_BITER_ELINK_REG(pDmaChBase, param) \
        periphMemRead(&((pDmaChBase)->biter_elinkno))

#define ioctlDMACH_SET_BITER_CHANNEL_TO_CHANNEL_LINKING(pDmaChBase, param) \
        if (param)periphBitSet(DMA_CITER_ELINK_MASK, &((pDmaChBase)->biter_elinkno)); \
        else periphBitClear(DMA_CITER_ELINK_MASK, &((pDmaChBase)->biter_elinkno))

#define ioctlDMACH_SET_STARTING_MAJOR_ITER_CNT_ELINKNO(pDmaChBase, param) \
        periphBitGrpSet(DMA_CITER_ELINKNO_CITER_MASK, param, &((pDmaChBase)->biter_elinkno))

#define ioctlDMACH_SET_STARTING_MAJOR_ITER_CNT_ELINKYES(pDmaChBase, param) \
        periphBitGrpSet(DMA_CITER_ELINKYES_CITER_MASK, param, &((pDmaChBase)->biter_elinkno))

#define ioctlDMACH_SET_STARTING_MINOR_LOOP_LINK_CHANNEL_ELINKYES(pDmaChBase, param) \
        periphBitGrpSet(DMA_CITER_ELINKYES_LINKCH_MASK, (param)<<9, &((pDmaChBase)->biter_elinkno))


#ifdef __cplusplus
}
#endif

#endif
