/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*  Standard C Included Files */
#include <stdio.h>
#include <string.h>
#include "fsl_debug_console.h"
#include "board.h"
#include "fsl_lcdc.h"
#include "fsl_debug_console.h"
#include "pin_mux.h"
#include "fsl_sctimer.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/

#define LCD_PANEL_CLK 8000000
#define LCD_PPL 480		// pixel per line
#define LCD_HSW 2		// HSync width
#define LCD_HFP 8		// Horizontal front porch
#define LCD_HBP 43		// Horizontal back porch
#define LCD_LPP 272		// Line per panel
#define LCD_VSW 10		// VSync width
#define LCD_VFP 4		// Vertical front porch
#define LCD_VBP 12		// Vertical back porch
#define LCD_POL_FLAGS kLCDC_InvertVsyncPolarity | kLCDC_InvertHsyncPolarity
#define IMG_HEIGHT 		LCD_LPP
#define IMG_WIDTH 	LCD_PPL
#define APP_LCD_IRQHandler LCD_IRQHandler
#define APP_LCD_IRQn LCD_IRQn
#define APP_BIT_PER_PIXEL 16
#define APP_PIXEL_PER_BYTE 4

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
// utility union to ease RGB565 bit-field manipulation
typedef union
{
	struct {
		uint16_t r:5;
		uint16_t g:6;
		uint16_t b:5;
	}rgb;
	uint16_t d;
}PIXEL565_t;

/*******************************************************************************
 * Variables
 ******************************************************************************/
// Define framebuffer (FB). FB is specified to put in "FB" section, which is in SDRAM
#if (defined(__CC_ARM) || defined(__GNUC__))
	__attribute__((section("FB"), zero_init)) uint16_t s_FB[IMG_HEIGHT][IMG_WIDTH];
#elif defined(__ICCARM__)
	#pragma data_alignment = 8
	uint16_t s_FB[IMG_HEIGHT][IMG_WIDTH];
#else
	#error Toolchain not support.
#endif

/*******************************************************************************
 * Code
 ******************************************************************************/
static void BOARD_InitPWM(void)
{
    sctimer_config_t config;
    sctimer_pwm_signal_param_t pwmParam;
    uint32_t event;

    CLOCK_AttachClk(kMCLK_to_SCT_CLK);

    CLOCK_SetClkDiv(kCLOCK_DivSctClk, 2, true);

    SCTIMER_GetDefaultConfig(&config);

    SCTIMER_Init(SCT0, &config);

    pwmParam.output = kSCTIMER_Out_5;
    pwmParam.level = kSCTIMER_HighTrue;
    pwmParam.dutyCyclePercent = 5;

    SCTIMER_SetupPwm(SCT0, &pwmParam, kSCTIMER_CenterAlignedPwm, 1000U, CLOCK_GetFreq(kCLOCK_Sct), &event);
}

// tick counter as the time basis for the program
volatile uint8_t s_isNewTick;
void SysTick_Handler(void)
{
	s_isNewTick = 1; // Signal the new tick flag
}

#define RGB(r,g,b) ((r>>3) + ((g>>2)<<5) + ((b>>3)<<11))
uint16_t colTab[8] = { RGB(0,0,0), RGB(255,0,0), RGB(255,255,0), RGB(0,255,0), RGB(0,255,255), RGB(0,0,255), RGB(255,0,255), RGB(255,255,255)};
	
int main(void)
{
	lcdc_config_t lcdConfig;
	uint32_t colRem = 0, colNdx = 0;	// lines remaining in a color stripe and current color
	// Length, in lines, per color stripe, "+ 1" to make the 8 color stripes wider than LCD width, so rotating effect appears. 
	uint32_t colLen = IMG_HEIGHT / 8 + 1;
	uint32_t *pFB32, col;
	uint32_t lcdcInFreq;
    /* Hardware Initialization */
    /* attach 12 MHz clock to FLEXCOMM0 (debug console) */
    CLOCK_AttachClk(BOARD_DEBUG_UART_CLK_ATTACH);
    BOARD_InitPins();
    BOARD_BootClockFROHF96M();
    BOARD_InitDebugConsole();
    BOARD_InitSDRAM();
	BOARD_InitPWM();
    /* Route Main clock to LCD. */
    CLOCK_AttachClk(kMCLK_to_LCD_CLK);

    CLOCK_SetClkDiv(kCLOCK_DivLcdClk, 1, true);
	// >>> configure LCD controller
    LCDC_GetDefaultConfig(&lcdConfig);

    lcdConfig.panelClock_Hz = LCD_PANEL_CLK;
    lcdConfig.ppl = LCD_PPL;
    lcdConfig.hsw = LCD_HSW;
    lcdConfig.hfp = LCD_HFP;
    lcdConfig.hbp = LCD_HBP;
    lcdConfig.lpp = LCD_LPP;
    lcdConfig.vsw = LCD_VSW;
    lcdConfig.vfp = LCD_VFP;
    lcdConfig.vbp = LCD_VBP;
    lcdConfig.polarityFlags = LCD_POL_FLAGS;
    lcdConfig.upperPanelAddr = (uint32_t)s_FB[0];
    lcdConfig.bpp = kLCDC_16BPP565;
    lcdConfig.display = kLCDC_DisplayTFT;
    lcdConfig.swapRedBlue = false;
	
	lcdcInFreq = CLOCK_GetFreq(kCLOCK_LCD);
    LCDC_Init(LCD, &lcdConfig, lcdcInFreq);
	// <<<
    LCDC_Start(LCD);
    LCDC_PowerUp(LCD);
	SysTick_Config(SystemCoreClock / 13);	// Configure systick 10Hz, limit the max draw rate

	PRINTF("LCD + SDRAM RGB565 HOT started!\r\n");
    while (1)
    {
		uint32_t x, y, cnt;
		s_isNewTick = 0;
		cnt++;
		pFB32 = (uint32_t*) s_FB[0];		
		for (y=0; y<IMG_HEIGHT; y++)
		{	
			col = colTab[colNdx];
			col |= col<<16;		// Fill 2 pixels per write
			// >>> draw color stripe
			for (x=0; x<IMG_WIDTH/2; x++) 
			{
				*pFB32++ = col;
			}
			// <<<
			// >>> rotate color stripe
			if (++colRem >= colLen) 
			{
				colRem = 0 , colNdx = (colNdx + 1) % 8;
			}
			// <<<
		}
		while (!s_isNewTick) {}	// Wait until next tick comes to control drawing speed
    }

}
