/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this list
 *   of conditions and the following disclaimer.
 *
 * o Redistributions in binary form must reproduce the above copyright notice, this
 *   list of conditions and the following disclaimer in the documentation and/or
 *   other materials provided with the distribution.
 *
 * o Neither the name of Freescale Semiconductor, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*  Standard C Included Files */
#include <stdio.h>
#include <string.h>
#include "board.h"
#include "fsl_lcdc.h"
#include "fsl_debug_console.h"
#include "pin_mux.h"
#include "fsl_sctimer.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/
#define APP_LCD LCD
#define LCD_PANEL_CLK 7500000
#define LCD_PPL 480
#define LCD_HSW 2
#define LCD_HFP 8
#define LCD_HBP 43
#define LCD_LPP 272
#define LCD_VSW 10
#define LCD_VFP 4
#define LCD_VBP 12
#define LCD_POL_FLAGS kLCDC_InvertVsyncPolarity | kLCDC_InvertHsyncPolarity
#define IMG_HEIGHT 272
#define IMG_WIDTH 480
#define LCD_INPUT_CLK_FREQ CLOCK_GetFreq(kCLOCK_LCD)
#define APP_LCD_IRQHandler LCD_IRQHandler
#define APP_LCD_IRQn LCD_IRQn
#define APP_BIT_PER_PIXEL 2
#define APP_PIXEL_PER_BYTE 4
#define APP_PIXEL_MAX_VALUE 3
#define APP_PIXEL_MIN_VALUE 3

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*******************************************************************************
 * Variables
 ******************************************************************************/

// defines 2 framebuffers (FBs), since we use 2bpp, we put both FBs in on-chip RAM
// We use tool-chain extensions to align FB to 8 byte
// >>> define 1st FB
#if (defined(__CC_ARM) || defined(__GNUC__))
__attribute__((aligned(8)))
#elif defined(__ICCARM__)
#pragma data_alignment = 8
#else
#error Toolchain not support.
#endif
static uint8_t s_frameBuf0[IMG_HEIGHT][IMG_WIDTH / APP_PIXEL_PER_BYTE];
// <<<
// >>> define 2nd FB
#if (defined(__CC_ARM) || defined(__GNUC__))
__attribute__((aligned(8)))
#elif defined(__ICCARM__)
#pragma data_alignment = 8
#else
#error Toolchain not support.
#endif
static uint8_t s_frameBuf1[IMG_HEIGHT][IMG_WIDTH / APP_PIXEL_PER_BYTE];
// <<<
// Save addresses of FBs for convienence.
static const uint32_t s_frameBufAddr[] = {(uint32_t)s_frameBuf0, (uint32_t)s_frameBuf1};

// >>> define palette colors (4 colors)
#define RGB(r,g,b) ((r>>3) + ((g>>2)<<5) + ((b>>3)<<11))
static const uint32_t palette[] = {RGB(0,0,0)|RGB(255,220,0)<<16,RGB(120,255,0)|RGB(70,206,255)<<16};// 0x001F0000U, 0x7C0003E0U};
// <<<

/* The index of the inactive buffer. */
static volatile uint8_t s_inactiveBufsIdx;

/* IRQ and background shared flag that means the new frame address already loaded to the LCD controller. */
static volatile bool s_frameAddrUpdated = false;

/*******************************************************************************
 * Code
 ******************************************************************************/
static void BOARD_InitPWM(void)
{
    sctimer_config_t config;
    sctimer_pwm_signal_param_t pwmParam;
    uint32_t event;

    CLOCK_AttachClk(kMCLK_to_SCT_CLK);

    CLOCK_SetClkDiv(kCLOCK_DivSctClk, 2, true);

    SCTIMER_GetDefaultConfig(&config);

    SCTIMER_Init(SCT0, &config);

    pwmParam.output = kSCTIMER_Out_5;
    pwmParam.level = kSCTIMER_HighTrue;
    pwmParam.dutyCyclePercent = 5;

    SCTIMER_SetupPwm(SCT0, &pwmParam, kSCTIMER_CenterAlignedPwm, 1000U, CLOCK_GetFreq(kCLOCK_Sct), &event);
}

void APP_LCD_IRQHandler(void)
{
    uint32_t intStatus = LCDC_GetEnabledInterruptsPendingStatus(APP_LCD);

    LCDC_ClearInterruptsStatus(APP_LCD, intStatus);

    if (intStatus & kLCDC_BaseAddrUpdateInterrupt)
    {
        s_frameAddrUpdated = true;
		s_inactiveBufsIdx ^= 1;
    }
    __DSB();
}

// Draw a line in 2bpp mode framebuffer.
static void APP_Draw2BPPLine(uint8_t *line, uint16_t start, uint16_t end, uint8_t color)
{
    uint8_t i;
    uint16_t startByte;
    uint16_t endByte;

    startByte = start / APP_PIXEL_PER_BYTE;
    endByte = end / APP_PIXEL_PER_BYTE;

    if (startByte == endByte)
    {
        for (i = (start & 0x03U); i < (end & 0x03U); i++)
        {
            line[startByte] = (line[startByte] & ~(0x03U << (i * 2U))) | (color << (i * 2U));
        }
    }
    else
    {
        for (i = (start & 0x03U); i < APP_PIXEL_PER_BYTE; i++)
        {
            line[startByte] = (line[startByte] & ~(0x03U << (i * 2U))) | (color << (i * 2U));
        }

        for (i = (startByte + 1U); i < endByte; i++)
        {
            line[i] = color * 0x55U;
        }

        for (i = 0U; i < (end & 0x03U); i++)
        {
            line[endByte] = (line[endByte] & ~(0x03U << (i * 2U))) | (color << (i * 2));
        }
    }
}

// Fill 2bpp uffer. This user function maintains static locals as state and implements a moving rectangle effect.
static void APP_FillBuffer(void *buffer)
{
    /* Background color. */
    static uint8_t bgColor = 0U;
    /* Foreground color. */
    static uint8_t fgColor = 1U;
    uint8_t colorToSet = 0U;
    /* Position of the foreground rectangle. */
    static uint16_t upperLeftX = 0U;
    static uint16_t upperLeftY = 0U;
    static uint16_t lowerRightX = (IMG_WIDTH - 1U) / 2U;
    static uint16_t lowerRightY = (IMG_HEIGHT - 1U) / 2U;
    static int8_t incX = 2;
    static int8_t incY = 2;
    /* Change color in next forame or not. */
    static bool changeColor = false;

    uint32_t i, j;
    uint8_t(*buf)[IMG_WIDTH / APP_PIXEL_PER_BYTE] = buffer;

    /*
     +------------------------------------------------------------------------+
     |                                                                        |
     |                                                                        |
     |                                                                        |
     |                          Area 1                                        |
     |                                                                        |
     |                                                                        |
     |                  +---------------------------+                         |
     |                  |XXXXXXXXXXXXXXXXXXXXXXXXXXX|                         |
     |                  |XXXXXXXXXXXXXXXXXXXXXXXXXXX|                         |
     |    Area 2        |XXXXXXXXXXXXXXXXXXXXXXXXXXX|       Area 3            |
     |                  |XXXXXXXXXXXXXXXXXXXXXXXXXXX|                         |
     |                  |XXXXXXXXXXXXXXXXXXXXXXXXXXX|                         |
     |                  |XXXXXXXXXXXXXXXXXXXXXXXXXXX|                         |
     |                  +---------------------------+                         |
     |                                                                        |
     |                                                                        |
     |                                                                        |
     |                                                                        |
     |                         Area 4                                         |
     |                                                                        |
     |                                                                        |
     +------------------------------------------------------------------------+
     */

    /* Fill the frame buffer. */
    /* Fill area 1. */
    colorToSet = bgColor * 0x55U;
    for (i = 0; i < upperLeftY; i++)
    {
        for (j = 0; j < IMG_WIDTH / APP_PIXEL_PER_BYTE; j++)
        {
            /* Background color. */
            buf[i][j] = colorToSet;
        }
    }

    APP_Draw2BPPLine((uint8_t *)buf[i], 0, upperLeftX, bgColor);
    APP_Draw2BPPLine((uint8_t *)buf[i], upperLeftX, lowerRightX + 1, fgColor);
    APP_Draw2BPPLine((uint8_t *)buf[i], lowerRightX + 1, IMG_WIDTH, bgColor);

    for (i++; i <= lowerRightY; i++)
    {
        for (j = 0; j < (IMG_WIDTH / APP_PIXEL_PER_BYTE); j++)
        {
            buf[i][j] = buf[upperLeftY][j];
        }
    }

    /* Fill area 4. */
    colorToSet = bgColor * 0x55U;
    for (; i < IMG_HEIGHT; i++)
    {
        for (j = 0; j < IMG_WIDTH / APP_PIXEL_PER_BYTE; j++)
        {
            /* Background color. */
            buf[i][j] = colorToSet;
        }
    }

    /* Update the format: color and rectangle position. */
    upperLeftX += incX;
    upperLeftY += incY;
    lowerRightX += incX;
    lowerRightY += incY;

    changeColor = false;
	// >>> "colision" detect
    if (0U == upperLeftX)
    {
        incX = -incX;
        changeColor = true;
    }
    else if (IMG_WIDTH - 1 == lowerRightX)
    {
        incX = -incX;
        changeColor = true;
    }

    if (0U == upperLeftY)
    {
        incY = -incY;
        changeColor = true;
    }
    else if (IMG_HEIGHT - 1 == lowerRightY)
    {
        incY = -incY;
        changeColor = true;
    }
	// <<<
    if (changeColor)
    {
        if (APP_PIXEL_MAX_VALUE == fgColor)
        {
            fgColor = 1U;
        }
        else
        {
            fgColor++;
        }
    }
}

int main(void)
{
    lcdc_config_t lcdConfig;

    /* Route Main clock to LCD. */
    CLOCK_AttachClk(kMCLK_to_LCD_CLK);

    CLOCK_SetClkDiv(kCLOCK_DivLcdClk, 1, true);

    BOARD_InitPins();
    BOARD_BootClockFROHF48M();

    /* Set the back light PWM. */
    BOARD_InitPWM();
    /* attach 12 MHz clock to FLEXCOMM0 (debug console) */
    CLOCK_AttachClk(BOARD_DEBUG_UART_CLK_ATTACH);
    BOARD_InitDebugConsole();
    s_frameAddrUpdated = false;
    s_inactiveBufsIdx = 1;

	// >>> configure LCD controller
    LCDC_GetDefaultConfig(&lcdConfig);

    lcdConfig.panelClock_Hz = LCD_PANEL_CLK;
    lcdConfig.ppl = LCD_PPL;
    lcdConfig.hsw = LCD_HSW;
    lcdConfig.hfp = LCD_HFP;
    lcdConfig.hbp = LCD_HBP;
    lcdConfig.lpp = LCD_LPP;
    lcdConfig.vsw = LCD_VSW;
    lcdConfig.vfp = LCD_VFP;
    lcdConfig.vbp = LCD_VBP;
    lcdConfig.polarityFlags = LCD_POL_FLAGS;
    lcdConfig.upperPanelAddr = (uint32_t)s_frameBufAddr[0];
    lcdConfig.bpp = kLCDC_2BPP;
    lcdConfig.display = kLCDC_DisplayTFT;
    lcdConfig.swapRedBlue = false;

    LCDC_Init(APP_LCD, &lcdConfig, LCD_INPUT_CLK_FREQ);
	// <<<

    LCDC_SetPalette(APP_LCD, palette, ARRAY_SIZE(palette));

	// >>> Let LCD controller to signal IRQ when it loads new FB
    LCDC_EnableInterrupts(APP_LCD, kLCDC_BaseAddrUpdateInterrupt);
    NVIC_EnableIRQ(APP_LCD_IRQn);
	// <<<

    LCDC_Start(APP_LCD);
    LCDC_PowerUp(APP_LCD);
	PRINTF("LCD palette HOT started!\r\n");
    while (1)
    {
		// wait until LCD controller completes refreshing current FB and fetched next FB.
        while (!s_frameAddrUpdated)
        {
        }
		s_frameAddrUpdated = false;
		// update next active FB.
		LCDC_SetPanelAddr(APP_LCD, kLCDC_UpperPanel, (uint32_t)(s_frameBufAddr[s_inactiveBufsIdx]));

		// app's main logic: Draw next frame on background FB (inactive FB)
        APP_FillBuffer((void *)s_frameBufAddr[s_inactiveBufsIdx]);
    }
}
