/*
 * Copyright (c) 2001-2004 Swedish Institute of Computer Science.
 * All rights reserved. 
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
 * SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
 * OF SUCH DAMAGE.
 *
 * This file is part of and a contribution to the lwIP TCP/IP stack.
 *
 * Credits go to Adam Dunkels (and the current maintainers) of this software.
 *
 * Christiaan Simons rewrote this file to get a more stable echo example.
 */

/**
 * @file
 * TCP Socket example using raw API.
 *
 */

#include "lwip/opt.h"
#include "lwip/debug.h"
#include "lwip/stats.h"
#include "lwip/tcp.h"
#include "lwip/sys.h"
#include "tcpSocket_raw.h"
#include "fsl_gpio.h"
#include "fsl_adc.h"
#include "app_gui.h"

#if LWIP_TCP

/* TCP socket states */
enum tcpSocket_raw_states
{
    S_NONE = 0,
    S_CONNECTED,
    S_CLOSING
};

struct tcpSocket_connection *g_ConnectListHead = NULL, *g_ConnectListTail = NULL;
static uint32_t active_conns = 0;

/* Function to free a node from the connection list */
static void tcpSocket_raw_free(struct tcpSocket_connection *conn)
{
    struct tcpSocket_connection *curr = g_ConnectListHead, *prev = NULL;
    /* Find the matching node from the list */
    while(curr && curr != conn) {
        prev = curr;
        curr = curr->next;
    }
    /* Check if valid node was found */
    if((curr!= NULL) || (prev!= NULL)) {
        if(prev== NULL) {
            /* Head node is being freed */
            g_ConnectListHead = curr->next;
            if(g_ConnectListHead == NULL) {
                g_ConnectListTail = NULL;
            }
        }
        else if(curr != NULL) {
            /*Any other node other than head */
            prev->next = curr->next;
            /* If tail is being freed then update global tail pointer */
            if(curr->next == NULL) {
                g_ConnectListTail = prev;
            }
        }
    }
    /* Free node if a matching node was found */
    if(curr) {
        if (curr->pending_tx) {
          /* free the buffer chain if present */
          pbuf_free(curr->pending_tx);
        }
        mem_free(curr);
    }
}

/* Function to close a TCP connection */
static void tcpSocket_raw_close(struct tcp_pcb *tpcb, struct tcpSocket_connection *conn)
{   
    tcp_arg(tpcb, NULL);
    tcp_sent(tpcb, NULL);
    tcp_recv(tpcb, NULL);
    tcp_err(tpcb, NULL);
    tcp_poll(tpcb, NULL, 0);

    tcpSocket_raw_free(conn);

    tcp_close(tpcb);
    Clear_LCD();
    if(active_conns) {
        active_conns--;
    }
    Update_LCD(active_conns);
#if(NODE==CLIENT1)||(NODE==CLIENT2)
    if(active_conns == 0) {
        /* If client node got disconnected, then retry connection */
        tcpSocket_raw_init();
    }
#endif
}

/* Callback function when an error occurs */
static void tcpSocket_raw_error(void *arg, err_t err)
{
    struct tcpSocket_connection *conn;

    LWIP_UNUSED_ARG(err);

    conn = (struct tcpSocket_connection *)arg;
    
    if(conn) {
        tcpSocket_raw_close(conn->pcb, conn);
    }
    else {
#if(NODE==CLIENT1)||(NODE==CLIENT2)
        if((err == ERR_ABRT) && (active_conns == 0)) {
            /* Previous connection attempt failed, so retry */
            tcpSocket_raw_init();
        }
#endif        
    }
}

static void convertHexToAscii(char *dest, uint16_t data)
{
    uint8_t i;
    for(i = 0; i < 4; i++) {
        if((data & 0x0f) < 0x0a) {
            dest[3-i] = '0' + (data & 0x0f);
        }
        else {
            dest[3-i] = 'a' + ((data & 0x0f) - 0x0a);
        }
        data >>= 4;
    }
}

static void convertAsciiToHex(uint16_t *dest, char *data)
{
    uint8_t i;
    *dest = 0;
    for(i = 0; i < 4; i++) {
        if((data[i] >= '0') && (data[i] <= '9')) {
            *dest = (*dest << 4) | (data[i] - '0');
        }
        else {
            *dest = (*dest << 4) | ((data[i] - 'a') + 0xa);
        }
    }
}

/* Periodic polling function called every 500ms */
static err_t tcpSocket_raw_poll(void *arg, struct tcp_pcb *tpcb)
{
    err_t ret_err;
    struct tcpSocket_connection *conn;
    char hb_msg[6] = "Alive";
    static char adc_msg[] = "ADC Data 0000";
    adc_result_info_t adcResultInfoStruct;

    conn = (struct tcpSocket_connection *)arg;
    if (conn != NULL) {
        if (conn->pending_tx != NULL) {
            /* there is a remaining pbuf (chain)  */
            if (tcp_write(tpcb, conn->pending_tx->payload, conn->pending_tx->len, 1) == ERR_OK) {
                struct pbuf *ptr = conn->pending_tx;
                conn->pending_tx = conn->pending_tx->next;
                pbuf_free(ptr);
            }
        } 
        else {
            /* no remaining pbuf (chain)  */
            if(conn->state == S_CLOSING) {
                tcpSocket_raw_close(tpcb, conn);
            }
        }
        /* Read ADC value and start conversion only for the first node */
        if(g_ConnectListHead->adcCount == 0) {
            ADC_GetChannelConversionResult(ADC0, 0, &adcResultInfoStruct);
            convertHexToAscii(&adc_msg[9], adcResultInfoStruct.result);
            ADC_DoSoftwareTriggerConvSeqA(ADC0);
        }
        /* If the connected node sent heartbeat message then clear flags and counts */
        if(conn->hbFlag) {
            conn->hbFlag = 0;
            conn->hbMissCount = 0;
        }
        else {
            conn->hbMissCount++;
            /* Close connection after missing heart beat for 3s */
            if(conn->hbMissCount >= 6) {
                conn->state = S_CLOSING;
                tcpSocket_raw_close(conn->pcb, conn);
            }
        }
        /* Send heart beat message at a different slot than ADC data */
        if(conn->hbCount == 1) {
            tcpSocket_raw_send(conn, hb_msg, 6);
        }
        /* Send ADC data */
        if(conn->adcCount == 0) {
            tcpSocket_raw_send(conn, adc_msg, 14);
        }
        /* Send ADC data every 2s */
        if(++conn->adcCount >= 4) {
            conn->adcCount = 0;
        }
        /* Send heart beat message every 1s */
        if(++conn->hbCount >= 2) {
            conn->hbCount = 0;
        }
        ret_err = ERR_OK;
    } 
    else {
        /* nothing to be done */
        tcp_abort(tpcb);
        ret_err = ERR_ABRT;
    }
    
    return ret_err;
}

/* Callback function when a TCP packet was successfully transmitted */
static err_t tcpSocket_raw_sent(void *arg, struct tcp_pcb *tpcb, u16_t len)
{
    struct tcpSocket_connection *conn;
    err_t wr_err = ERR_OK;

    LWIP_UNUSED_ARG(len);

    conn = (struct tcpSocket_connection *)arg;

    if(conn->pending_tx != NULL) {
        /* still got pbufs to send */
        tcp_sent(tpcb, tcpSocket_raw_sent);
          
        wr_err = tcp_write(tpcb, conn->pending_tx->payload, conn->pending_tx->len, 1);
        if (wr_err == ERR_OK) {
            struct pbuf *ptr = conn->pending_tx;
            conn->pending_tx = conn->pending_tx->next;
            pbuf_free(ptr);
        }
    } 
    else {
        /* no more pbufs to send */
        if(conn->state == S_CLOSING) {
          tcpSocket_raw_close(tpcb, conn);
        }
    }
    return ERR_OK;
}

/* Function to process the received data for a connected node */
static void process_rx_data(struct tcp_pcb *tpcb, struct pbuf *p, struct tcpSocket_connection *conn, uint8_t index)
{
    struct pbuf *next;
    uint16_t len;
    
    /* Parse all received pBufs */
    while (p != NULL) {
        len = p->len;
        /* For each pBuf parse all the string data that was received */
        while(len > 0) {
            if((strncmp(p->payload, "ADC Data ", strlen("ADC Data ")) == 0)) {
                /* Update ADC value after receiving ADC data from node */
                convertAsciiToHex(&(conn->adcData), &(((char *)p->payload)[9])); 
                Update_ADC_Data(conn, index);
            }
            /* Check if LED off message was received */
            else if(strcmp(p->payload, "LED Off") == 0) {
                switch(index) {
                    case 0:
                        GPIO_SetPinsOutput(GPIO, 2, 1u << 2);
                        break;
                    case 1:
                        GPIO_SetPinsOutput(GPIO, 3, 1u << 3);
                        break;
                    default:
                        break;
                }
                conn->ledStatus = 0;
                Update_LED_Status(index, 0);
            }
            /* Check if LED On message was received */
            else if(strcmp(p->payload, "LED On") == 0) {
                switch(index) {
                    case 0:
                        GPIO_ClearPinsOutput(GPIO, 2, 1u << 2);
                        break;
                    case 1:
                        GPIO_ClearPinsOutput(GPIO, 3, 1u << 3);
                        break;
                    default:
                        break;
                }
                conn->ledStatus = 1;
                Update_LED_Status(index, 1);
            }
            /* Check if heart beat message was received */
            else if(strcmp(p->payload, "Alive") == 0) {
                conn->hbFlag = 1;
                conn->hbMissCount = 0;
            }
            len -= (strlen(p->payload) + 1);
            p->payload = (void *)((uint32_t)p->payload + strlen(p->payload) + 1);
            
        }
        if(p->next != NULL) {
            pbuf_ref(p->next);
        }
        /* After parsing the pBuf, flush it out */
        next = p->next;
        len = p->len;
        pbuf_free(p);
        p = next;
        tcp_recved(tpcb, len);
    }
}

/* Callback function when a TCP packet is received */
static err_t tcpSocket_raw_recv(void *arg, struct tcp_pcb *tpcb, struct pbuf *p, err_t err)
{
    struct tcpSocket_connection *conn, *curr = g_ConnectListHead;
    uint8_t count = 0;
    err_t ret_err = ERR_VAL;

    LWIP_ASSERT("arg != NULL",arg != NULL);
    conn = (struct tcpSocket_connection *)arg;
    if (p == NULL) {
        /* remote host closed connection */
        conn->state = S_CLOSING;
        /* we're done sending, close it */
        tcpSocket_raw_close(tpcb, conn);
        ret_err = ERR_OK;
    } 
    else if(err != ERR_OK) {
        /* cleanup, for unknown reason */
        if (p != NULL) {
          pbuf_free(p);
        }
        ret_err = err;
    }
    else if(conn->state == S_CONNECTED) {
        /* If valid data was recived then find the node from which the data was received to process it */
        while(curr && (curr != conn)) {
            count++;
            curr = curr->next;
        }
        if(curr) {
            process_rx_data(tpcb, p, conn, count);
            ret_err = ERR_OK;
        }
    } 
    else {
        /* trash data if node is not connected */
        tcp_recved(tpcb, p->tot_len);
        pbuf_free(p);
        ret_err = ERR_OK;
    }
    return ret_err;
}

#if(NODE==SERVER)
/* Callback function when the server accepts a new connection request */
static err_t tcpSocket_raw_accept(void *arg, struct tcp_pcb *newpcb, err_t err)
{
    err_t ret_err;
    struct tcpSocket_connection *curr;

    LWIP_UNUSED_ARG(arg);
    if ((err != ERR_OK) || (newpcb == NULL)) {
        return ERR_VAL;
    }

    /* Unless this pcb should have NORMAL priority, set its priority now.
     When running out of pcbs, low priority pcbs can be aborted to create
     new pcbs of higher priority. */
    tcp_setprio(newpcb, TCP_PRIO_MIN);

    /* Create a node and add the PCB to the list */
    if(g_ConnectListTail) {
        curr = g_ConnectListTail->next = (struct tcpSocket_connection *)mem_malloc(sizeof(struct tcpSocket_connection));
        if(g_ConnectListTail->next != NULL) {
            g_ConnectListTail = g_ConnectListTail->next;
        }
        
    }
    else {
        curr = g_ConnectListHead = g_ConnectListTail = (struct tcpSocket_connection *)mem_malloc(sizeof(struct tcpSocket_connection));
    }
    if (curr != NULL) {
        memset(curr, 0, sizeof(struct tcpSocket_connection));
        Clear_LCD();
        active_conns++;
        curr->state = S_CONNECTED;
        curr->pcb = newpcb;
        /* pass newly allocated es to our callbacks */
        tcp_arg(newpcb, curr);
        tcp_recv(newpcb, tcpSocket_raw_recv);
        tcp_err(newpcb, tcpSocket_raw_error);
        tcp_poll(newpcb, tcpSocket_raw_poll, 0);
        tcp_sent(newpcb, tcpSocket_raw_sent);
        Update_LCD(active_conns);
        ret_err = ERR_OK;
    } 
    else {
        ret_err = ERR_MEM;
    }
    return ret_err;
}

#elif(NODE==CLIENT1)||(NODE==CLIENT2)
/* Callback function when the client's connection request is successful */
static err_t tcpSocket_raw_connected(void *arg, struct tcp_pcb *tpcb, err_t err)
{
    err_t ret_err;
    struct tcpSocket_connection *curr;

    LWIP_UNUSED_ARG(arg);
    if ((err != ERR_OK) || (tpcb == NULL)) {
        return ERR_VAL;
    }

    /* Unless this pcb should have NORMAL priority, set its priority now.
     When running out of pcbs, low priority pcbs can be aborted to create
     new pcbs of higher priority. */
    tcp_setprio(tpcb, TCP_PRIO_MIN);    

    /* Create a node and add the PCB to the list */
    if(g_ConnectListTail) {
        curr = g_ConnectListTail->next = (struct tcpSocket_connection *)mem_malloc(sizeof(struct tcpSocket_connection));
        if(g_ConnectListTail->next != NULL) {
            g_ConnectListTail = g_ConnectListTail->next;
        }
        
    }
    else {
        curr = g_ConnectListHead = g_ConnectListTail = (struct tcpSocket_connection *)mem_malloc(sizeof(struct tcpSocket_connection));
    }
    if (curr != NULL) {
        memset(curr, 0, sizeof(struct tcpSocket_connection));
        Clear_LCD();
        active_conns++;
        curr->state = S_CONNECTED;
        curr->pcb = tpcb;
        /* pass newly allocated es to our callbacks */
        tcp_arg(tpcb, curr);
        tcp_recv(tpcb, tcpSocket_raw_recv);
        tcp_err(tpcb, tcpSocket_raw_error);
        tcp_poll(tpcb, tcpSocket_raw_poll, 0);
        tcp_sent(tpcb, tcpSocket_raw_sent);
        Update_LCD(active_conns);
        ret_err = ERR_OK;
    } 
    else {
        ret_err = ERR_MEM;
    }
    return ret_err;
}
#endif

/* Function called by App GUI when a checkbox state is changed by user */
void tcpSocket_checkbox_event(uint8_t node_index, uint8_t data)
{
    struct tcpSocket_connection *curr = g_ConnectListHead;
    uint8_t curr_index = 0;
    char strOn[] = "LED On";
    char strOff[] = "LED Off";
    /* Find the node based on the index */
    while(curr && (curr_index < node_index)) {
        curr = curr->next;
        curr_index++;
    }
    if(curr) {
        /* Update and send the led message */
        curr->ledSwitch = data;
        if(data) {
            tcpSocket_raw_send(curr, strOn, 7);
        }
        else {
            tcpSocket_raw_send(curr, strOff, 8);
        }
    }
}

/* Function to transmit a TCP packet to a particular connection node */
void tcpSocket_raw_send(struct tcpSocket_connection *conn, void *data, uint16_t len)
{
    struct pbuf *ptr;
    err_t wr_err = ERR_OK;
    
    if(conn && (conn->state == S_CONNECTED)) {
        /* If the connection is valid then allocate pBuf and copy data */
        ptr = pbuf_alloc(PBUF_TRANSPORT, len, PBUF_RAM);
        if((ptr != NULL) && len <= tcp_sndbuf(conn->pcb)) {
            memcpy(ptr->payload, data, len);
            ptr->len = len;
            ptr->tot_len = len;
            if(conn->pending_tx != NULL) {
                pbuf_cat(conn->pending_tx, ptr);
            }
            else {
                conn->pending_tx = ptr;
            }
            wr_err = tcp_write(conn->pcb, conn->pending_tx->payload, conn->pending_tx->len, 1);
            if (wr_err == ERR_OK) {
                ptr = conn->pending_tx;
                conn->pending_tx = conn->pending_tx->next;
                pbuf_free(ptr);
            } 
        }
    }
}

/* Returns the number of active connections */
uint32_t tcpSocket_raw_getActiveConns(void)
{
    return active_conns;
}

/* Initialization function for the TCP socket application */
void tcpSocket_raw_init(void)
{
    struct tcp_pcb *tcpSocket_raw_pcb = tcp_new_ip_type(IPADDR_TYPE_ANY);
    if (tcpSocket_raw_pcb != NULL) {
        err_t err;
#if(NODE==SERVER)
        err = tcp_bind(tcpSocket_raw_pcb, IP_ANY_TYPE, 54321);
        if (err == ERR_OK) {
            tcpSocket_raw_pcb = tcp_listen(tcpSocket_raw_pcb);
            tcp_accept(tcpSocket_raw_pcb, tcpSocket_raw_accept);
        } 
        else {
            /* abort? output diagnostic? */
        }
#elif(NODE==CLIENT1)||(NODE==CLIENT2)
        ip4_addr_t server_ipaddr;
        IP4_ADDR(&server_ipaddr, serverIP_ADDR0, serverIP_ADDR1, serverIP_ADDR2, serverIP_ADDR3);
        tcp_err(tcpSocket_raw_pcb, tcpSocket_raw_error);
        err = tcp_connect(tcpSocket_raw_pcb, (const ip4_addr_t *)&server_ipaddr, 54321, tcpSocket_raw_connected);
        if(err != ERR_OK) {
            while(1); /* Throw error and abort */
        }
#endif        
    } 
    else {
        /* abort? output diagnostic? */
    }
}

#endif /* LWIP_TCP */
