/*
 * Copyright (c) 2001-2004 Swedish Institute of Computer Science.
 * All rights reserved. 
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
 * SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
 * OF SUCH DAMAGE.
 *
 * This file is part of the lwIP TCP/IP stack.
 * 
 */
#ifndef LWIP_TCPSOCKET_RAW_H
#define LWIP_TCPSOCKET_RAW_H

#include "lwip/tcp.h"
#include "app.h"

struct tcpSocket_connection
{
    /* State of TCP connection */
    uint8_t state;
    uint8_t unused;
    /* Flag to indicate that a heart beat was received from this node */
    uint8_t hbFlag;
    /* Count of the number of missed heartbeat messages */
    uint8_t hbMissCount;
    /* The TCP PCB for this connection */
    struct tcp_pcb *pcb;
    /* List of pending pBuf's */
    struct pbuf *pending_tx;
    /* Received ADC data from the Rx node */
    uint16_t adcData;
    /* State of the LED checkbox for the Rx node */
    uint8_t ledSwitch;
    /* State of the LED for this node */
    uint8_t ledStatus;
    /* Heart beat counter for this node */
    uint16_t hbCount;
    /* ADC counter for this node */
    uint16_t adcCount;
    struct tcpSocket_connection *next;
};

/*!
 * @brief Initialization function for the TCP socket application.
 *
 *  This function initialize the TCP socket application.
 *  Based on whether the node is complied as a Server or a client
 *  it is initialized differently.
 *
 * @param none
 * @retval none
 */
void tcpSocket_raw_init(void);

/*!
 * @brief Function to transmit a TCP packet to a particular connection node
 *
 * @param conn  Handle of the node for which the packet needs to be send.
 * @param data  Pointer to the transmit data.
 * @param len   length of the TCP packet.
 * @retval None
 */
void tcpSocket_raw_send(struct tcpSocket_connection *conn, void *data, uint16_t len);

/*!
 * @brief Function called by App GUI when a checkbox state is changed by user.
 *
 * @param node_index    Index of the node in the connection list.
 * @param data    Checkbox state, 0 (unchecked) or 1 (checked).
 * @retval None
 */
void tcpSocket_checkbox_event(uint8_t node_index, uint8_t data);

/*!
 * @brief Returns the number of active connections.
 *
 * @param None
 * @retval Number of active connections with this node
 */
uint32_t tcpSocket_raw_getActiveConns(void);

#endif /* LWIP_TCPSOCKET_RAW_H */
