;**************************************************************
; 
;  (c) copyright Freescale Semiconductor. 2006
;  ALL RIGHTS RESERVED
; 
;**************************************************************
;**************************************************************
;* DC Fan Coding for 9RS08KA2
;*
;* Author:		Vincent Ko
;* Date:      Jan 2006
;*
;* PTA0/KBI0/ACMP+             RC input
;* PTA1/KBI1/ACMP-             Temp sensor input
;* PTA2/KBI2/TCLK/RESETb/VPP   Hall input
;* PTA3/ACMPO/BKGD/MS          Buzzer
;* PTA4/KBI4                   PWM+
;* PTA5/KBI5                   PWM-
;*
;**************************************************************
; include derivative specific macros
    XDEF    Entry
    
		include "MC9RS08KA2.inc"

;=========================================================================
; ICS Definition
;=========================================================================
ICS_DIV_1          equ	$00
ICS_DIV_2          equ	$40
ICS_DIV_4          equ	$80
ICS_DIV_8          equ	$c0

;=========================================================================
; MTIM Definition
;=========================================================================
MTIM_DIV_1          equ	$00
MTIM_DIV_2          equ	$01
MTIM_DIV_4          equ	$02
MTIM_DIV_8          equ	$03
MTIM_DIV_16         equ	$04
MTIM_DIV_32         equ	$05
MTIM_DIV_64         equ	$06
MTIM_DIV_128        equ	$07
MTIM_DIV_256        equ	$08

MTIM_BUS_CLK        equ	$00
MTIM_XCLK           equ	$10
MTIM_TCLK_FALLING   equ	$20
MTIM_TCLK_RISING    equ	$30

;=========================================================================
; ACMP Definition
;=========================================================================
ACMP_OUTPUT_FALLING   equ	$00
ACMP_OUTPUT_RAISING   equ	$01
ACMP_OUTPUT_BOTH      equ	$03

;=========================================================================
; RTI Definition
;=========================================================================
RTI_DISABLE         equ	 $00
RTI_8MS             equ	 $01
RTI_32MS            equ	 $02
RTI_64MS            equ	 $03
RTI_128MS           equ	 $04
RTI_256MS           equ	 $05
RTI_512MS           equ	 $06
RTI_1024MS          equ	 $07

;=========================================================================
; Application Definition
;=========================================================================
RC                  equ	PTAD_PTAD0
mRC                 equ	mPTAD_PTAD0
TEMPSEN             equ	PTAD_PTAD1
mTEMPSEN            equ	mPTAD_PTAD1
HALL                equ	PTAD_PTAD2
mHALL               equ	mPTAD_PTAD2
BUZZER              equ	PTAD_PTAD3
mBUZZER             equ	mPTAD_PTAD3
PWM2                equ	PTAD_PTAD4
mPWM2               equ	mPTAD_PTAD4
PWM1                equ	PTAD_PTAD5
mPWM1               equ	mPTAD_PTAD5

MinDeadTime         equ 2
MaxDeadTime         equ 150

TableStart:         equ   $00003E00
;=========================================================================
; Application Macro
;=========================================================================
StartTimer: macro
			mov		DeadTime, MTIMMOD                                       ; OF period
			mov		#(mMTIMSC_TRST|mMTIMSC_TOIE), MTIMSC                    ; Reset and Start Timer
			endm

			org			TINY_RAMStart
; variable/data section
DeadTime            ds.b 1
TargetPeriod        ds.b 1
ActualPeriod        ds.b 1
DriveTime           ds.b 1     			
SensorReading       ds.b 1
MotorRunning				ds.b 1

			org			RAMStart
; variable/data section			

			org			ROMStart
; code section
main:
Entry:
			;-------------------------------------------------------
      ; Config ICS
      ; Device is pre-trim to 16MHz ICLK frequency
      ; TRIM value are stored in $3FFA:$3FFB
			;-------------------------------------------------------
      mov   #HIGH_6_13(NV_ICSTRM), PAGESEL               
      mov   MAP_ADDR_6(NV_FTRIM), ICSSC                  ; $3FFB
      mov   MAP_ADDR_6(NV_ICSTRM), ICSTRM                ; $3FFA
			mov   #ICS_DIV_2, ICSC2													   ; Use 4MHz

			;-------------------------------------------------------
			;Config System
			;-------------------------------------------------------
			mov		#HIGH_6_13(SOPT), PAGESEL                         ; Init Page register
;			mov		#(mSOPT_COPT|mSOPT_STOPE|mSOPT_BKGDPE), SOPT      ; BKGD enable, COP disabled
			mov		#(mSOPT_COPT|mSOPT_STOPE), MAP_ADDR_6(SOPT)       ; BKGD disable, COP disabled
			mov		#(mSPMSC1_LVDE|mSPMSC1_LVDRE), MAP_ADDR_6(SPMSC1) ; LVI enable
			mov		#(RTI_128MS), MAP_ADDR_6(SRTISC)                  ; 128ms RTI

			;-------------------------------------------------------
      ; Init RAM
			;-------------------------------------------------------
      mov   #MaxDeadTime, DeadTime
      mov   #232, TargetPeriod                           ; 1000 rpm
      mov   #232, ActualPeriod													 ; 1000 rpm
      clr   SensorReading
      clr   MotorRunning
			
			;-------------------------------------------------------
			;Config GPIO
			; RC - init L
			; Buzzer - init L
			; PWMn/PWMp - init L
			;-------------------------------------------------------
			clr   PTAD                                         ; Initial low
			mov		#(mRC|mPWM1|mPWM2), PTADD                    ; Set Output pins		
			
			;-------------------------------------------------------
			;Config KBI
			;-------------------------------------------------------
			lda		#mHALL
			sta   KBIES                                       ;HALL rising Edge Trigger
			sta		KBIPE                                       ;KBI Enable

			;-------------------------------------------------------
			;Config MTIM
			;
			;Timer prescalar=256 -> Timer clk = 16kHz
      ;Bus = 4MHz
      ;Max OF period = 16.384ms 
      ;Timer resolution = 64us
			;-------------------------------------------------------
			mov		#(MTIM_BUS_CLK|MTIM_DIV_256), MTIMCLK
			mov		#255, MTIMMOD
      
			;-------------------------------------------------------
			;Motor Start Sequence
			;-------------------------------------------------------       
ResetPosition:   
      mov   #mPWM1, PTAD                                 ; Lock FAN in reset position
      lda   #30																					 ;
Dly1  bsr   Delay																				 ; for Delay 0.5s
      dbnza Dly1																				 ; 
      clr   PTAD                                         ; de-energize coils 
      bsr   Delay

      ; Drive L2
      ldx   #mPWM2         															 ; Select L2 Coils
      bsr   SetPWM                                       ; Drive coil  
      bsr   Delay																				 ; De-energize coils
      inc   MotorRunning                                 ; otherwise Update Software flag

			;-------------------------------------------------------
			;Fan Control Loop
			;-------------------------------------------------------
FanControlLoop:

      ;1) Drive L1 coil
      clr   KBIES                                        ; HALL falling edge trigger 
      ldx   #mPWM1         															 ; Select L1 Coil
      bsr   SetPWM                                       ; Drive coil  

      ;2) Read Temp Sensor
      jsr   ReadSensor                                   ; Read Sensor value

      ;3) Dead time control
      StartTimer  													             ; Wait dead time period
      wait
			mov   #(mMTIMSC_TSTP|mMTIMSC_TRST), MTIMSC         ; mask interrupt and clear flag      

      ;4) Drive L2 coil
      bset  HALL, KBIES                                  ; HALL rising edge trigger 
      ldx   #mPWM2         															 ; Select L2 Coil
      bsr   SetPWM     													         ; Drive coil 

      ;5) Read Temp Sensor Again
      bsr   ReadSensor                                   ; Read Sensor value

      ;6) Dead time control
      StartTimer

      ;7) During the dead time, update dead time period every 128ms
			brclr	SRTISC_RTIF, MAP_ADDR_6(SRTISC), UpdateLater ; Update PWM duty cycle
      jsr   TableLookup
UpdateLater:
      lda   ActualPeriod
      sub   TargetPeriod                                 ; Actual-Target
      blo   IncPeriod												
      beq   WaitAgain     							                 ; if same, Fan speed reach target then exit

DecPeriod:																							 ; if bigger, decrement DeadTime
      lda   DeadTime
      cmp   #MinDeadTime
      blo   WaitAgain
			dec   DeadTime
			bra   WaitAgain

IncPeriod:      																				 ; if smaller, increment DeadTime
      lda   DeadTime
      cmp   #MaxDeadTime
      bhs   WaitAgain
      inc   DeadTime
      bra   WaitAgain
      
WaitAgain:
			;8) Bump COP
			sta		MAP_ADDR_6(SRS)                              ; Bump COP
      wait
			mov   #(mMTIMSC_TSTP|mMTIMSC_TRST), MTIMSC         ; mask interrupt and clear flag      

			;9) Repeat the control cycle
			bra   FanControlLoop

           
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; Delay 16ms
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Delay:
			mov		#255, MTIMMOD                                ; OF period
			mov		#(mMTIMSC_TRST|mMTIMSC_TOIE), MTIMSC         ; Reset and Start Timer
      wait
			mov   #(mMTIMSC_TSTP|mMTIMSC_TRST), MTIMSC         ; mask interrupt and clear flag
			sta		MAP_ADDR_6(SRS)                              ; Bump COP
      rts
      
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; Drive coil
;
; X indicate the coil to be driven
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
SetPWM:
			mov		#255, MTIMMOD                                           ; OF period
			mov		#(mMTIMSC_TRST|mMTIMSC_TOIE), MTIMSC                    ; Reset and Start Timer

      lda   #20
			mov		#(mKBISC_KBIE), KBISC                                   ; Enable Interrupt & Edge only
			bset	KBISC_KBACK, KBISC                                      ; Clear Flag
      stx   PTAD															          						; Drive coil
TimingLoop:                           
      bclr  MTIMSC_TOF, MTIMSC                                      ; Clear TOF
			wait                                    
			brset	KBISC_KBF, KBISC, HallFound                             ; HALL sensor edge found
      dbnza TimingLoop
      jmp   MotorHang																								; If no HALL output, Stop the driving
HallFound:
      mov   MTIMCNT, DriveTime                                                       
      cbeqa #20, StableDrive
      mov   #MaxDeadTime, DriveTime
StableDrive:
      lda   DeadTime
      add   DriveTime
      sta   ActualPeriod      
      clr   PTAD	    																							; Disconnect coil
			mov   #(mKBISC_KBACK), KBISC                                  ; Clear Flag and mask interrupt
			mov   #(mMTIMSC_TSTP|mMTIMSC_TRST), MTIMSC                    ; mask interrupt and clear flag      
      rts

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; Read Temperature Sensor Value 
; Timer prescalar=8 -> Timer clk~250kHz
; Bus = 2MHz
; Max OF period = 1.02ms 
; Timer resolution = 4us
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
ReadSensor:
			mov		#(MTIM_BUS_CLK|MTIM_DIV_8), MTIMCLK                          ; Change Timer resolution
			mov		#63, MTIMMOD                                                 ; OF period
			mov		#(mMTIMSC_TRST|mMTIMSC_TOIE), MTIMSC                         ; Reset and Start Timer

			mov		#(mACMPSC_ACME|mACMPSC_ACIE|ACMP_OUTPUT_RAISING), ACMPSC	   ; Enable ACMP, start RC rise
			bset	ACMPSC_ACF, ACMPSC                                           ; Clear ACMP Flag
			wait																 															 ; delay to OF and make the read process deterministic
      brclr ACMPSC_ACF, ACMPSC, NoReading
      mov   MTIMCNT, SensorReading                                                       
			bset	ACMPSC_ACF, ACMPSC                                           ; Clear ACMP Flag
      clr   ACMPSC                                                       ; disable ACMP
      wait
			mov   #(mMTIMSC_TSTP|mMTIMSC_TRST), MTIMSC                         ; mask interrupt and clear flag      
			mov		#(MTIM_BUS_CLK|MTIM_DIV_256), MTIMCLK                        ; Reset Timer resolution
      rts

NoReading:
      mov   #$00, SensorReading                                          ; Smallest Number
      clr   ACMPSC                                                       ; disable ACMP
			mov   #(mMTIMSC_TSTP|mMTIMSC_TRST), MTIMSC                         ; mask interrupt and clear flag      
			mov		#(MTIM_BUS_CLK|MTIM_DIV_256), MTIMCLK                        ; Reset Timer resolution
      rts
      
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; 6-bit Table Lookup
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
TableLookup:
			bset	SRTISC_RTIACK, MAP_ADDR_6(SRTISC)															 ;5			
      mov   #HIGH_6_13(TableStart), PAGESEL	  														 ;5 Calculate the PAGE
      lda   SensorReading																									 ;3
      add   #$c0                              														 ;2 Reference to paging window
      tax																																	 ;2
      lda   ,x																														 ;3
      sta   TargetPeriod   																								 ;2
      mov   #HIGH_6_13(SOPT), PAGESEL																			 ;5
      rts																																	 ;3
      
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; Error Handling
; Stop the motor
; Sound the buzzer (about 520Hz) 
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
MotorHang:
      clr   PTAD                                                           ; clear PWMp and PWMn
      lda   MotorRunning																									 ; Check software flag
      bne   SoundBuzzer                                                    ; =1, Motor is running
      jmp   ResetPosition   

SoundBuzzer:
			mov   #(mMTIMSC_TSTP|mMTIMSC_TRST), MTIMSC                           ; mask interrupt and clear flag      
      clr   KBISC                                                          ; mask KBI

      lda   #255      
			sta		MAP_ADDR_6(SRS)                                                ; Bump COP
Beep:																													             ; a 20% duty cycle loop
      bset  BUZZER, PTAD										                               ; Drive buzzer
			mov		#6, MTIMMOD
			mov		#(mMTIMSC_TRST|mMTIMSC_TOIE), MTIMSC                           ; Reset and Start Timer
      wait
			mov   #(mMTIMSC_TSTP|mMTIMSC_TRST), MTIMSC                           ; mask interrupt and clear flag
			sta		MAP_ADDR_6(SRS)                                                ; Bump COP

      bclr  BUZZER, PTAD										                               ; Clear buzzer
			mov		#24, MTIMMOD
			mov		#(mMTIMSC_TRST|mMTIMSC_TOIE), MTIMSC                           ; Reset and Start Timer
      wait
			mov   #(mMTIMSC_TSTP|mMTIMSC_TRST), MTIMSC                           ; mask interrupt and clear flag
			sta		MAP_ADDR_6(SRS)                                                ; Bump COP
      dbnza Beep

      lda   #255      
Quiet:
      bclr  BUZZER, PTAD										                               ; Clear buzzer
			mov		#30, MTIMMOD
			mov		#(mMTIMSC_TRST|mMTIMSC_TOIE), MTIMSC                           ; Reset and Start Timer
      wait
			mov   #(mMTIMSC_TSTP|mMTIMSC_TRST), MTIMSC                           ; mask interrupt and clear flag
			sta		MAP_ADDR_6(SRS)                                                ; Bump COP
      dbnza Quiet
      
      bra   SoundBuzzer
      
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; Lookup Table
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
      org  TableStart
       
       dc.b   57, 57, 57, 57, 57, 60, 63, 67, 71, 76, 82, 82, 88, 88, 96, 96
       dc.b  105,105,115,115,115,128,128,128,128,144,144,144,144,165,165,165
       dc.b  165,193,193,193,193,193,232,232,232,232,232,232,232,232,232,232
       dc.b  232,232,232,232,232,232,232,232,232,232,232,232,232,232,232,232
       
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; Reset Vector			
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
			org		$3ffc
Security:
      dc.b  $FF						
			jmp		main