


/*
 * Copyright (c) 2001,2002 Florian Schulze.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the authors nor the names of the contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * test.c - This file is part of lwIP test
 *
 */

/**
 * @page misra_violations MISRA-C:2012 violations
 *
 * @section [global]
 * Violates MISRA 2012 Required Rule 1.3,  Taking address of near auto variable.
 * The code is not dynamically linked. An absolute stack address is obtained
 * when taking the address of the near auto variable.
 *
 * @section [global]
 * Violates MISRA 2012 Required Rule 2.1, Unreachable code.
 * Statement is intented to be unreachable. If the line is executed,
 * then there was insufficient FreeRTOS heap memory available for the idle and/or
 * timer tasks to be created.
 *
 * @section [global]
 * Violates MISRA 2012 Required Rule 8.2, old-style function definition for function
 * declaration.
 * This is done for compatibility with all hardware platforms.
 *
 * @section [global]
 * Violates MISRA 2012 Required Rule 8.4, A compatible declaration shall be
 * visible when an object or function with external linkage is defined.
 * These are symbols weak symbols defined in platform startup files (.s).
 *
 * @section [global]
 * Violates MISRA 2012 Required Rule 8.3, Symbol redeclared.
 * Declarations are guarded by preprocessor directives.
 *
 * @section [global]
 * Violates MISRA 2012 Required Rule 8.5, object/function previously declared.
 * This requirement is fulfilled since the function is declared as external.
 *
 * @section [global]
 * Violates MISRA 2012 Required Rule 10.1, Unpermitted operand to operator '||'
 * Variable is of essential boolean type
 *
 * @section [global]
 * Violates MISRA 2012 Required Rule 11.3, Cast performed between a pointer to object type
 * and a pointer to a different object type.
 * This is needed to respect the lwip API which requires argument to be passed as void *.
 *
 * @section [global]
 * Violates MISRA 2012 Advisory Rule 11.5, Conversion from pointer to void to pointer to other type
 * This is needed to respect the lwip API which requires argument to be passed as void *.
 *
 * @section [global]
 * Violates MISRA 2012 Required Rule 14.3, Controlling expressions shall not be invariant
 * Constant value used in LWIP_ASSERT to easily stop the program in that point if
 * needed.
 *
 * @section [global]
 * Violates MISRA 2012 Required Rule 14.4, Conditional expression should have essentially Boolean type.
 * This is required for macro constructs in form do {...} while(0).
 *
 * @section [global]
 * Violates MISRA 2012 Advisory Rule 15.5, Return statement before end of function.
 * The return statement before end of function is used for simpler code
 * structure and better readability.
 *
 * @section [global]
 * Violates MISRA 2012 Mandatory Rule 17.3, Symbol undeclared, assumed
 * to return int.
 * The symbol is defined in another file.
 *
 * @section [global]
 * Violates MISRA 2012 Required Rule 20.9, Undefined preprocessor variable, assumed 0
 * Preprocessor variable will be defined in further implementations or not defined intentionally.
 *
 */

/* C runtime includes */
#include <stdio.h>
#include <string.h>
#include <stdint.h>

#if defined(USING_OS_FREERTOS)
/* FreeRTOS kernel includes. */
#include "FreeRTOS.h"
#include "task.h"
#endif /* defined(USING_OS_FREERTOS) */

#include "osif.h"

/* lwIP core includes */
#include "lwip/opt.h"

#include "lwip/sys.h"
#include "lwip/timeouts.h"
#include "lwip/debug.h"
#include "lwip/stats.h"
#include "lwip/init.h"
#include "lwip/tcpip.h"
#include "lwip/netif.h"
#include "lwip/api.h"
#include "lwip/arch.h"

#include "lwip/tcp.h"
#include "lwip/udp.h"
#include "lwip/dns.h"
#include "lwip/dhcp.h"
#include "lwip/autoip.h"

/* lwIP netif includes */
#include "lwip/etharp.h"
#include "netif/ethernet.h"

/* include the port-dependent configuration */
#include "lwipcfg.h"

/* applications includes */
#include "lwip/sockets.h"
#include "lwip/err.h"
#include "lwip/sys.h"

#if LWIP_HTTPD_APP
#if LWIP_HTTPD_APP_NETCONN
#include "apps/httpserver/httpserver-netconn.h"
#else
#include "lwip/apps/httpd.h"
#endif /* LWIP_HTTPD_APP_NETCONN */
#endif /* LWIP_HTTPD_APP */
#if LWIP_CHARGEN_APP
#include "apps/chargen/chargen.h"
#endif /* LWIP_CHARGEN_APP */
#if LWIP_LWIPERF_APP
#include "lwip/apps/lwiperf.h"
#endif /* LWIP_LWIPERF_APP */
#if LWIP_NETBIOS_APP
#include "lwip/apps/netbiosns.h"
#endif /* LWIP_NETBIOS_APP */
#if LWIP_SNTP_APP
#include "lwip/apps/sntp.h"
#endif /* LWIP_SNTP_APP */
#include "lwip/apps/mdns.h"
#if LWIP_NETIO_APP
#include "apps/netio/netio.h"
#endif /* LWIP_NETIO_APP */
#if LWIP_RTP_APP
#include "apps/rtp/rtp.h"
#endif /* LWIP_RTP_APP */
#if LWIP_SSL_ECHO_APP
#include "apps/secure_socket/secure_socket.h"
#endif /* LWIP_SSL_ECHO_APP */
#if LWIP_SHELL_APP
#include "apps/shell/shell.h"
#endif /* LWIP_SHELL_APP */
#if LWIP_TCPECHO_APP
#if LWIP_TCPECHO_APP_NETCONN
#include "apps/tcpecho/tcpecho.h"
#else
#include "apps/tcpecho_raw/tcpecho_raw.h"
#endif /* LWIP_TCPECHO_APP_NETCONN */
#endif /* LWIP_TCPECHO_APP */
#if LWIP_UDPECHO_APP
#if LWIP_UDPECHO_APP_NETCONN
#include "apps/udpecho/udpecho.h"
#else
#include "apps/udpecho_raw/udpecho_raw.h"
#endif /* LWIP_UDPECHO_APP_NETCONN */
#endif /* LWIP_UDPECHO_APP */
#if LWIP_SOCKET_EXAMPLES_APP
#include "apps/socket_examples/socket_examples.h"
#endif /* LWIP_SOCKET_EXAMPLES_APP */

#if NO_SYS
/* ... then we need information about the timer intervals: */
#include "lwip/ip4_frag.h"
#include "lwip/igmp.h"
#endif /* NO_SYS */

#include "enetif.h"
#include "cpu.h"

#define TJA1101    /*enable this macro to use TJA1101 as ENET transceiver*/

#ifdef TJA1101
#define MASTER  1
#define SLAVE   0
#define TJA1101_WORK_MODE SLAVE  /*modify this macro define to change the TJA1101 work mode*/

#endif /*end of TJA1101*/

/* globales variables for netifs */
/* THE ethernet interface */
struct netif netif;
#if LWIP_DHCP
/* dhcp struct for the ethernet netif */
struct dhcp netif_dhcp;
#endif /* LWIP_DHCP */
#if LWIP_AUTOIP
/* autoip struct for the ethernet netif */
struct autoip netif_autoip;
#endif /* LWIP_AUTOIP */

#if defined(USING_OS_FREERTOS)
/*
 * Prototypes for the standard FreeRTOS application hook (callback) functions
 * implemented within this file.  See www.freertos.org/a00016.html .
 */
void vApplicationMallocFailedHook(void);
void vApplicationStackOverflowHook(TaskHandle_t pxTask, char *pcTaskName);

/* Priorities at which the tasks are created. */
#define	mainQUEUE_SEND_TASK_PRIORITY         (tskIDLE_PRIORITY + 1)
#endif /* defined(USING_OS_FREERTOS) */

/*-----------------------------------------------------------*/

/*
* The tasks as described in the comments at the top of this file.
*/
static void mainLoopTask(void *pvParameters);

void start_example(void);

/*-----------------------------------------------------------*/

#if LWIP_NETIF_STATUS_CALLBACK
static void
status_callback(struct netif *state_netif)
{
  if (netif_is_up(state_netif)) {
#if LWIP_IPV4
    /* status_callback UP, local interface IP is ip4addr_ntoa(netif_ip4_addr(state_netif)) */
#else
    /* status_callback UP */
#endif
#if LWIP_MDNS_RESPONDER
    mdns_resp_netif_settings_changed(state_netif);
#endif
  } else {
    /* status_callback DOWN */
  }
}
#endif /* LWIP_NETIF_STATUS_CALLBACK */

#if LWIP_NETIF_LINK_CALLBACK
static void
link_callback(struct netif *state_netif)
{
  if (netif_is_link_up(state_netif)) {
    /* link_callback UP */
  } else {
    /* link_callback DOWN */
  }
}
#endif /* LWIP_NETIF_LINK_CALLBACK */

/* This function initializes all network interfaces
 * Implements enetif_init_Activity
 */
static void enetif_init(void)
{
#if LWIP_IPV4
  ip4_addr_t ipaddr, netmask, gw;
#endif /* LWIP_IPV4 */
#if LWIP_DHCP || LWIP_AUTOIP
  err_t err;
#endif

#if LWIP_IPV4
#define NETIF_ADDRS &ipaddr, &netmask, &gw,
  ip4_addr_set_zero(&gw);
  ip4_addr_set_zero(&ipaddr);
  ip4_addr_set_zero(&netmask);
#if (!LWIP_DHCP) && (!LWIP_AUTOIP)
  LWIP_PORT_INIT_GW(&gw);
  LWIP_PORT_INIT_IPADDR(&ipaddr);
  LWIP_PORT_INIT_NETMASK(&netmask);
#endif /* (!LWIP_DHCP) && (!LWIP_AUTOIP) */
#else /* LWIP_IPV4 */
#define NETIF_ADDRS
  /* Starting lwIP, IPv4 disable */
#endif /* LWIP_IPV4 */

#if NO_SYS
  netif_set_default(netif_add(&netif, NETIF_ADDRS NULL, enet_ethernetif_init, netif_input));
#else /* NO_SYS */
  netif_set_default(netif_add(&netif, NETIF_ADDRS NULL, enet_ethernetif_init, tcpip_input));
#endif /* NO_SYS */

#if LWIP_IPV6
  netif_create_ip6_linklocal_address(&netif, 1);
#if PRINTF_SUPPORT
  printf("ip6 linklocal address: ");
#endif
  ip6_addr_debug_print(0xFFFFFFFFU & ~LWIP_DBG_HALT, netif_ip6_addr(&netif, 0));
#endif /* LWIP_IPV6 */
#if LWIP_NETIF_STATUS_CALLBACK
  netif_set_status_callback(&netif, status_callback);
#endif /* LWIP_NETIF_STATUS_CALLBACK */
#if LWIP_NETIF_LINK_CALLBACK
  netif_set_link_callback(&netif, link_callback);
#endif /* LWIP_NETIF_LINK_CALLBACK */

#if LWIP_AUTOIP
  autoip_set_struct(&netif, &netif_autoip);
#endif /* LWIP_AUTOIP */
#if LWIP_DHCP
  dhcp_set_struct(&netif, &netif_dhcp);
#endif /* LWIP_DHCP */
  netif_set_up(&netif);
#if LWIP_DHCP
  err = dhcp_start((struct netif *)&netif);
  LWIP_ASSERT("dhcp_start failed", err == (err_t)ERR_OK);
#elif LWIP_AUTOIP
  err = autoip_start(&netif);
  LWIP_ASSERT("autoip_start failed", err == (err_t)ERR_OK);
#endif /* LWIP_DHCP */
}

#if LWIP_DNS_APP && LWIP_DNS
static void
dns_found(const char *name, const ip_addr_t *addr, void *arg)
{
  LWIP_UNUSED_ARG(arg);
#if PRINTF_SUPPORT
  printf("%s: %s\n", name, addr ? ipaddr_ntoa(addr) : "<not found>");
#else
  LWIP_UNUSED_ARG(name);
  LWIP_UNUSED_ARG(addr);
#endif
}

static void
dns_dorequest(void *arg)
{
  const char* dnsname = "3com.com";
  ip_addr_t dnsresp;
  LWIP_UNUSED_ARG(arg);

  if (dns_gethostbyname(dnsname, &dnsresp, dns_found, 0) == ERR_OK) {
    dns_found(dnsname, &dnsresp, 0);
  }
}
#endif /* LWIP_DNS_APP && LWIP_DNS */

#if LWIP_LWIPERF_APP
static void
lwiperf_report(void *arg, enum lwiperf_report_type report_type,
  const ip_addr_t* local_addr, u16_t local_port, const ip_addr_t* remote_addr, u16_t remote_port,
  u32_t bytes_transferred, u32_t ms_duration, u32_t bandwidth_kbitpsec)
{
  LWIP_UNUSED_ARG(arg);
  LWIP_UNUSED_ARG(local_addr);
  LWIP_UNUSED_ARG(local_port);

#if PRINTF_SUPPORT
  printf("IPERF report: type=%d, remote: %s:%d, total bytes: %lu, duration in ms: %lu, kbits/s: %lu\n",
    (int)report_type, ipaddr_ntoa(remote_addr), (int)remote_port, bytes_transferred, ms_duration, bandwidth_kbitpsec);
#else
  LWIP_UNUSED_ARG(report_type);
  LWIP_UNUSED_ARG(remote_addr);
  LWIP_UNUSED_ARG(remote_port);
  LWIP_UNUSED_ARG(bytes_transferred);
  LWIP_UNUSED_ARG(ms_duration);
  LWIP_UNUSED_ARG(bandwidth_kbitpsec);
#endif
}
#endif /* LWIP_LWIPERF_APP */

#if LWIP_MDNS_RESPONDER && LWIP_HTTPD_APP
static void srv_txt(struct mdns_service *service, void *txt_userdata)
{
   err_t res = mdns_resp_add_service_txtitem(service, "path=/", 6);
   LWIP_ERROR("mdns add service txt failed\n", (res == (err_t)ERR_OK), return);
   LWIP_UNUSED_ARG(txt_userdata);
}
#endif

/* This function initializes applications
 * Implements apps_init_Activity
 */
static void
apps_init(void)
{
#if LWIP_DNS_APP && LWIP_DNS
  /* wait until the netif is up (for dhcp, autoip or ppp) */
  sys_timeout(5000, dns_dorequest, NULL);
#endif /* LWIP_DNS_APP && LWIP_DNS */

#if LWIP_CHARGEN_APP && LWIP_SOCKET
  chargen_init();
#endif /* LWIP_CHARGEN_APP && LWIP_SOCKET */

#if LWIP_NETBIOS_APP && LWIP_UDP
  netbiosns_init();
#ifndef NETBIOS_LWIP_NAME
#if LWIP_NETIF_HOSTNAME
  netbiosns_set_name(netif_default->hostname);
#else
  netbiosns_set_name("NETBIOSLWIPDEV");
#endif
#endif
#endif /* LWIP_NETBIOS_APP && LWIP_UDP */

#if LWIP_HTTPD_APP && LWIP_TCP
#if LWIP_HTTPD_APP_NETCONN
  http_server_netconn_init();
#else /* LWIP_HTTPD_APP_NETCONN */
  httpd_init();
#endif /* LWIP_HTTPD_APP_NETCONN */
#endif /* LWIP_HTTPD_APP && LWIP_TCP */

#if LWIP_MDNS_RESPONDER
  mdns_resp_init();
#if LWIP_NETIF_HOSTNAME
  (void)mdns_resp_add_netif(netif_default, netif_default->hostname, 3600);
#else
  (void)mdns_resp_add_netif(netif_default, "lwip", 3600);
#endif
#if LWIP_HTTPD_APP
  (void)mdns_resp_add_service(netif_default, "lwipweb", "_http", DNSSD_PROTO_TCP, (u16_t)HTTPD_SERVER_PORT, 3600, srv_txt, NULL);
#endif
#endif

#if LWIP_NETIO_APP && LWIP_TCP
  netio_init();
#endif /* LWIP_NETIO_APP && LWIP_TCP */

#if LWIP_RTP_APP && LWIP_SOCKET && LWIP_IGMP
  rtp_init();
#endif /* LWIP_RTP_APP && LWIP_SOCKET && LWIP_IGMP */

#if LWIP_SNTP_APP && LWIP_SOCKET
  sntp_init();
#endif /* LWIP_SNTP_APP && LWIP_SOCKET */

#if LWIP_SHELL_APP && LWIP_NETCONN
  shell_init();
#endif /* LWIP_SHELL_APP && LWIP_NETCONN */
#if LWIP_SSL_ECHO_APP && LWIP_SOCKET
  secure_socket_init();
#endif
#if LWIP_TCPECHO_APP
#if LWIP_NETCONN && LWIP_TCPECHO_APP_NETCONN
  tcpecho_init();
#else /* LWIP_NETCONN && LWIP_TCPECHO_APP_NETCONN */
  tcpecho_raw_init();
#endif
#endif /* LWIP_TCPECHO_APP && LWIP_NETCONN */
#if LWIP_UDPECHO_APP
#if LWIP_NETCONN && LWIP_UDPECHO_APP_NETCONN
  udpecho_init(&netif);
#else /* LWIP_NETCONN && LWIP_UDPECHO_APP_NETCONN */
  udpecho_raw_init();
#endif /* LWIP_NETCONN && LWIP_UDPECHO_APP_NETCONN */
#endif /* LWIP_UDPECHO_APP */
#if LWIP_LWIPERF_APP
  (void)lwiperf_start_tcp_server_default(lwiperf_report, NULL);
#endif
#if LWIP_SOCKET_EXAMPLES_APP && LWIP_SOCKET
  socket_examples_init();
#endif /* LWIP_SOCKET_EXAMPLES_APP && LWIP_SOCKET */
#ifdef LWIP_APP_INIT
  LWIP_APP_INIT();
#endif
}

/* This function initializes this lwIP test. When NO_SYS=1, this is done in
 * the main_loop context (there is no other one), when NO_SYS=0, this is done
 * in the tcpip_thread context */
static void
test_init(void* arg)
{
/* remove compiler warning */
#if NO_SYS
  LWIP_UNUSED_ARG(arg);
#else /* NO_SYS */
  sys_sem_t* init_sem = (sys_sem_t*)arg;
  LWIP_ASSERT("init_sem != NULL", init_sem != NULL);
#endif /* NO_SYS */

  /* init network interfaces */
  enetif_init();

  /* init apps */
  apps_init();

#if !NO_SYS
  sys_sem_signal(init_sem);
#endif /* !NO_SYS */
}

static void tcp_client(void *arg)
{
		int client_sockfd;
		int err;
		struct sockaddr_in remote_addr;
		char buf[20];
		memset(&remote_addr,0,sizeof(remote_addr)); //set zeros for initialization
		remote_addr.sin_family=AF_INET; //set as IP communication
		remote_addr.sin_addr.s_addr=inet_addr("192.168.0.3");//set the server IP address
		remote_addr.sin_port=htons(7000); //server port

		char* data="hello,server\r\n";


		while(1)
		{
			while(1)/*try to connect*/
			{
				client_sockfd=socket(AF_INET,SOCK_STREAM,0);

				if(client_sockfd==-1)
				{
					 LWIP_ASSERT("create socket failed",0);
				}

				err=connect(client_sockfd,(struct sockaddr *)&remote_addr,sizeof(struct sockaddr));

				if(err<0)
				{
					close(client_sockfd);
					vTaskDelay(100);
				}
				else
				{
					break;
				}
			}
			while(1)/*normal communication loop*/
			{
				err = send(client_sockfd, data, strlen(data), 0);
				if(err<0)/*if any error, try reconnect*/
				{
					close(client_sockfd);//close current socket
					break;
				}
				vTaskDelay(1000);
			}

		}

		vTaskDelete(NULL);
}
static void tcp_server()
{
	struct sockaddr_in server_addr;
	struct sockaddr_in conn_addr;
	int sock_fd;                /* server socked */
	int sock_conn;          /* request socked */
	socklen_t addr_len;
	int err;
	int length;
	char* data="hello,client\r\n";

	sock_fd = socket(AF_INET, SOCK_STREAM, 0);
	if (sock_fd == -1) {
		LWIP_ASSERT("create socket failed",0);
	}

	memset(&server_addr, 0, sizeof(server_addr));
	server_addr.sin_family = AF_INET;
	server_addr.sin_addr.s_addr =htonl(INADDR_ANY);
	server_addr.sin_port = htons(7001);

	err = bind(sock_fd, (struct sockaddr *)&server_addr, sizeof(server_addr));
	if (err < 0) {
		LWIP_ASSERT("bind port failed",0);
	}

	err = listen(sock_fd, 1);
	if (err < 0) {
		LWIP_ASSERT("listen failed",0);
	}

	addr_len = sizeof(struct sockaddr_in);

	while(1)
	{
		/************************************************************
		 * wait for accept connection
		 * this task will be blocked until connection accept
		 ***********************************************************/
		sock_conn = accept(sock_fd, (struct sockaddr *)&conn_addr, &addr_len);

		while (1) /*normal communication loop*/
		{
			err = send(sock_conn, data, strlen(data), 0);
			if(err<0)/*if any error happen, try to re-accept*/
			{
				break;
			}

			vTaskDelay(1000);/**delay 1 second */

		}

	}
}

/**************************************************************************
 * call this API to configure the TJA1101 Enternet transceiver work mode
 * via HW RESET pin generate a reset and read the configure PINs state
 * if ModeSelect = 1, select master mode;
 * if ModeSelect = 0, select slave mode
 *************************************************************************/
void TJA1101_PHY_Cofigure(uint8_t ModeSelect)
{
	const pin_settings_config_t PHY_RST_PIN={
	  			.base          = PORTA,
	  			.pinPortIdx    = 17u,
	  			.pullConfig    = PORT_INTERNAL_PULL_NOT_ENABLED,
	  			.passiveFilter = false,
	  			.driveSelect   = PORT_HIGH_DRIVE_STRENGTH,
	  			.mux           = PORT_MUX_AS_GPIO,
	  			.pinLock       = false,
	  			.intConfig     = PORT_DMA_INT_DISABLED,
	  			.clearIntFlag  = false,
	  			.gpioBase      = PTA,
	  			.direction     = GPIO_OUTPUT_DIRECTION,
	  			.digitalFilter = false,
	  			.initValue     = 0u,
	    };
	    PINS_Init(&PHY_RST_PIN);/*initialize the R pin*/
	    PINS_DRV_ClearPins(PTA,(1<<17U));
	    const pin_settings_config_t AUTO_OP_PIN={
	  		.base          = PORTD,
	  		.pinPortIdx    = 8u,
	  		.pullConfig    = PORT_INTERNAL_PULL_NOT_ENABLED,
	  		.passiveFilter = false,
	  		.driveSelect   = PORT_LOW_DRIVE_STRENGTH,
	  		.mux           = PORT_MUX_AS_GPIO,
	  		.pinLock       = false,
	  		.intConfig     = PORT_DMA_INT_DISABLED,
	  		.clearIntFlag  = false,
	  		.gpioBase      = PTD,
	  		.direction     = GPIO_OUTPUT_DIRECTION,
	  		.digitalFilter = false,
	  		.initValue     = 1u,
	    };
	    PINS_Init(&AUTO_OP_PIN);/*initialize the Mode Select pin*/
	    const pin_settings_config_t MS_PIN={
	  	.base          = PORTD,
	  	.pinPortIdx    = 9u,
	  	.pullConfig    = PORT_INTERNAL_PULL_NOT_ENABLED,
	  	.passiveFilter = false,
	  	.driveSelect   = PORT_LOW_DRIVE_STRENGTH,
	  	.mux           = PORT_MUX_AS_GPIO,
	  	.pinLock       = false,
	  	.intConfig     = PORT_DMA_INT_DISABLED,
	  	.clearIntFlag  = false,
	  	.gpioBase      = PTD,
	  	.direction     = GPIO_OUTPUT_DIRECTION,
	  	.digitalFilter = false,
	  	.initValue     = ModeSelect,//0:slave mode 1:master mode

	    };
	    PINS_Init(&MS_PIN);/*initialize the Mode Select pin*/
	    vTaskDelay(10);
	    PINS_DRV_SetPins(PTA,(1<<17U));/*reset the TJA1101 transceiver*/
	    vTaskDelay(10);
}
static void mainLoopTask(void* pvParameters)
{
  (void)pvParameters;

  /*
   * configure the ENET transceiver work mode
   */
  TJA1101_PHY_Cofigure(TJA1101_WORK_MODE);
/*
 * initialize GPIO pins
 */
  PINS_DRV_Init(NUM_OF_CONFIGURED_PINS, g_pin_mux_InitConfigArr);

  /* initialize lwIP stack, network interfaces and applications */
#if !NO_SYS
  err_t err;
  sys_sem_t init_sem;
  uint16_t data;

  err = sys_sem_new(&init_sem, 0);/*create  a new semaphore for LwIP stack initialization*/

  LWIP_ASSERT("failed to create init_sem", err == (err_t)ERR_OK);
  LWIP_UNUSED_ARG(err);

  tcpip_init(test_init, (void*)&init_sem);/*Initialize the LwIP stack and tcpip socket*/
  /* we have to wait for initialization to finish before calling update_adapter()! */

  (void)sys_sem_wait(&init_sem);/*check the initialize status*/
  sys_sem_free(&init_sem);
#if (LWIP_SOCKET || LWIP_NETCONN) && LWIP_NETCONN_SEM_PER_THREAD
  netconn_thread_init();
#endif
#else /* !NO_SYS */
  sys_init();
  lwip_init();
  test_init(NULL);
#endif /* !NO_SYS */

  ENET_DRV_EnableMDIO(INST_ETHERNET1,true);
  PHY_FrameworkInit(phyConfig,phyDrivers);
  PHY_Init(INST_ETHERNET1);

  xTaskCreate(tcp_server,"tcp_server",256U,NULL,DEFAULT_THREAD_PRIO,NULL);/*create a task for server communication with remote host*/
  xTaskCreate(tcp_client,"tcp_client",256U,NULL,DEFAULT_THREAD_PRIO,NULL);/*create a task for client communication with remote host*/

  /* MAIN LOOP for driver update (and timers if NO_SYS) */
  while (1) {
#if NO_SYS
    /* handle timers (already done in tcpip.c when NO_SYS=0) */
    sys_check_timeouts();
    (void)enet_poll_interface(&netif);
#else /* NO_SYS */
     vTaskDelay(1000);
    if(PHY_Read(INST_ETHERNET1,23,&data)==STATUS_SUCCESS){
		if(data&0x8000){
			//netif_set_link_up(&netif);
			PINS_DRV_SetPins(PTE,(1<<22U));
			PINS_DRV_ClearPins(PTE,(1<<21U));
		}
		else{
			//netif_set_link_down(&netif);
			PINS_DRV_SetPins(PTE,(1<<21U));
			PINS_DRV_ClearPins(PTE,(1<<22U));
		}
	}else{
		//netif_set_link_down(&netif);
		PINS_DRV_SetPins(PTE,(1<<21U));
		PINS_DRV_ClearPins(PTE,(1<<22U));
	}




#endif /* NO_SYS */
  }

#if (LWIP_SOCKET || LWIP_NETCONN) && LWIP_NETCONN_SEM_PER_THREAD
  netconn_thread_cleanup();
#endif
  /* release the enet interface... */
  enet_ethernetif_shutdown(&netif);
}


void start_example(void)
{
#if defined(USING_OS_FREERTOS)
  BaseType_t ret =xTaskCreate(mainLoopTask, "mainloop", 256U, NULL, DEFAULT_THREAD_PRIO, NULL);
                                     /* Start the tasks and timer running. */
  LWIP_ASSERT("failed to create mainloop", ret == pdPASS);

  vTaskStartScheduler();

  /* If all is well, the scheduler will now be running, and the following
  line will never be reached.  If the following line does execute, then
  there was insufficient FreeRTOS heap memory available for the idle and/or
  timer tasks	to be created.  See the memory management section on the
  FreeRTOS web site for more details. */
  for (;;)
  {
  }
#else
    mainLoopTask(NULL);
#endif /* defined(USING_OS_FREERTOS) */
}

#if defined(USING_OS_FREERTOS)
void vAssertCalled(uint32_t ulLine, const char * const pcFileName)
{
  /* Called if an assertion passed to configASSERT() fails.  See
  www.freertos.org/a00110.html#configASSERT for more information. */

#if PRINTF_SUPPORT
  printf("ASSERT! Line %d, file %s\r\n", ulLine, pcFileName);
#else
  /* Parameters are not used. */
  LWIP_UNUSED_ARG(ulLine);
  LWIP_UNUSED_ARG(pcFileName);
#endif

  taskENTER_CRITICAL();
  {
    LWIP_ASSERT("configASSERT():", 0);
  }
  taskEXIT_CRITICAL();
}
/*-----------------------------------------------------------*/

void vApplicationMallocFailedHook(void)
{
  /* vApplicationMallocFailedHook() will only be called if
  configUSE_MALLOC_FAILED_HOOK is set to 1 in FreeRTOSConfig.h.  It is a hook
  function that will get called if a call to pvPortMalloc() fails.
  pvPortMalloc() is called internally by the kernel whenever a task, queue,
  timer or semaphore is created.  It is also called by various parts of the
  demo application.  If heap_1.c, heap_2.c or heap_4.c is being used, then the
  size of the	heap available to pvPortMalloc() is defined by
  configTOTAL_HEAP_SIZE in FreeRTOSConfig.h, and the xPortGetFreeHeapSize()
  API function can be used to query the size of free heap space that remains
  (although it does not provide information on how the remaining heap might be
  fragmented).  See www.freertos.org/a00111.html for more
  information. */
  vAssertCalled(__LINE__, __FILE__);
}
/*-----------------------------------------------------------*/

void vApplicationStackOverflowHook(TaskHandle_t pxTask, char *pcTaskName)
{
  (void) pcTaskName;
  (void) pxTask;

  /* Run time stack overflow checking is performed if
  configCHECK_FOR_STACK_OVERFLOW is defined to 1 or 2.  This hook
  function is called if a stack overflow is detected.  This function is
  provided as an example only as stack overflow checking does not function
  when running the FreeRTOS Windows port. */
  vAssertCalled(__LINE__, __FILE__);
}
#endif /* defined(USING_OS_FREERTOS) */
