/*
 * sgtl5000.c
 *
 *  Created on: Jun 1, 2018
 *      Author: nxf45772
 */

#include "sgtl5000.h"
#include "sgtl5000_config.h"


static uint8_t tx_buf[128];
static uint8_t rx_buf[128];





static status_t sgtl5000_i2c_write(uint32_t instance,uint16_t RegAddr,uint16_t *txBuff, uint32_t txSize){
	status_t err;
	int i=0;
	int j=0;
	tx_buf[i++]=(RegAddr>>8);
	tx_buf[i++]=(uint8_t)(RegAddr&0x00ff);
	for(j=0;j<txSize;j++){
		tx_buf[i++]=(txBuff[j]>>8);
		tx_buf[i++]=(uint8_t)txBuff[j];
	}
	err=LPI2C_DRV_MasterSendDataBlocking(instance, tx_buf,i, true,100);
	return err;
}
static status_t sgtl5000_i2c_read(uint32_t instance,uint16_t RegAddr,uint16_t *rxBuff, uint32_t rxSize){
	status_t err;
	int i=0;
	uint8_t buf[2];
	//send reg addr
	buf[0]=(RegAddr>>8);
	buf[1]=(uint8_t)(RegAddr&0x00ff);
	err=LPI2C_DRV_MasterSendDataBlocking(instance, buf,2,false,100);
	if(err!=STATUS_SUCCESS){
		LPI2C_DRV_MasterAbortTransferData(instance);
		return err;
	}
	//start read
	err=LPI2C_DRV_MasterReceiveDataBlocking(instance, rx_buf,rxSize*2,true,100);
	if(err!=STATUS_SUCCESS){
		return err;
	}
	for(i=0;i<rxSize;i++){
		rxBuff[i]=(((uint16_t)rx_buf[i*2])<<8)|rx_buf[i*2+1];
	}
	return err;
}
static status_t sgtl5000_write(uint16_t reg,uint16_t data){
	return sgtl5000_i2c_write(INST_LPI2C1,reg,&data,1);
}

static status_t sgtl5000_read(uint16_t reg,uint16_t* data){
	return sgtl5000_i2c_read(INST_LPI2C1,reg,data,1);
}

/*
 * the step is 25mv
 * should to set vdda/2
 * input parameter:
 * 	voltage sholud >800&<1575
 * 	bias
 * 		0:nominal
 * 		1-3=+12.5%
 * 		4=-12.5%
 * 		5=25%
 * 		6=37.5%
 * 		7=-50%
 * 	ramp:
 * 		1:noram vag ramp
 * 		0:slow down vag ramp
 */
static status_t sgtl5000_ref_set(uint16_t voltage_mv,uint16_t bias,uint16_t ramp){
	if(voltage_mv<800)
		voltage_mv=800;
	if(voltage_mv>1575)
		voltage_mv=1575;
	uint16_t res=(voltage_mv-800)/25;
	return sgtl5000_write(CHIP_REF_CTRL,(res<<4)|(bias<<1)|ramp);
}
/*
 * 0:0db
 * 1:+20db
 * 2:+30db
 * 3:+40db
 */
status_t sgtl5000_mic_gain(uint8_t a){
	status_t err;
	uint16_t data;
	err=sgtl5000_read(CHIP_MIC_CTRL,&data);
	if(err!=STATUS_SUCCESS)
		return err;
	data&=0xfffc;
	data|=a;
	return sgtl5000_write(CHIP_MIC_CTRL,data);
}

status_t sgtl5000_mute_dac(bool mute){
	status_t err;
	uint16_t data;
	err=sgtl5000_read(CHIP_ADCDAC_CTRL,&data);
	if(mute){
		data|=(1<<3)|(1<<2);
	}else{
		data&=~((1<<3)|(1<<2));
	}
	err=sgtl5000_write(CHIP_ADCDAC_CTRL,data);
	return err;
}

status_t sgtl5000_dac_vol(uint8_t r_vol,uint8_t l_vol){
	if(r_vol<0x3c)
		r_vol=0x3c;
	if(l_vol<0x3c)
		l_vol=0x3c;
	uint16_t data=((0xff-r_vol)<<8)|(0xff-l_vol);
	return sgtl5000_write(CHIP_ADCDAC_CTRL,data);
}

status_t sgtl5000_pll_init(){
	status_t err;
	uint16_t data;
	uint16_t int_divisor=16;
	uint16_t frac_divisor=786;
	//check id
	err=sgtl5000_read(CHIP_ID,&data);
	if(err!=STATUS_SUCCESS)
		return err;
	if((data&0xff00)!=0xa000)
		return STATUS_UNSUPPORTED;
//	//vddio=3.3v vdda=1.8v,not need charge chump,vddd is external
//	//to save power,only start a necessary part
	err=sgtl5000_write(CHIP_ANA_POWER,DAC_MONO|ADC_MONO|REFTOP_POWER|VAG_POWER\
			|DAC_POWER|HP_POWER|HP_CAPLESS_POWER|ADC_POWER|PLL_POWER|PLL_VCO_POWER);
	//config ref 0.9v,50% bias
	err=sgtl5000_ref_set(1600,7,0);
	err=sgtl5000_write(CHIP_DIG_POWER,ADC_DIG_POWER|DAC_DIG_POWER|I2S_IN_POWER|I2S_OUT_POWER);
	err=sgtl5000_write(CHIP_CLK_CTRL,SYS_FS_48K|MCLK_PLL);
	if(MCLK>17000000U){
		err=sgtl5000_write(CHIP_CLK_TOP_CTRL,(1<<3));
	}else{
		err=sgtl5000_write(CHIP_CLK_TOP_CTRL,(0));
	}

	err=sgtl5000_write(CHIP_PLL_CTRL,(int_divisor<<11)|frac_divisor);
	//set HP volume level
	err=sgtl5000_write(CHIP_MIC_CTRL,BIAS_RES_2K|(5<<4));
	err=sgtl5000_mic_gain(1);
	//input/output path
	err=sgtl5000_write(CHIP_ANA_CTRL,EN_ZCD_HP|EN_ZCD_ADC);
	err=sgtl5000_write(CHIP_SSS_CTRL,DAC_I2S_IN|I2S_OUT_ADC);
	err=sgtl5000_write(CHIP_ANA_HP_CTRL,0);
	err=sgtl5000_write(CHIP_ANA_ADC_CTRL,0x0000);
	err=sgtl5000_write(CHIP_ADCDAC_CTRL,0);
	err=sgtl5000_write(CHIP_DAC_VOL,0xA0a0);
	err=sgtl5000_write(CHIP_I2S_CTRL,I2S_32FS|I2S_16BIT);
	return err;
}

status_t sgtl5000_init(sgtl5000_config_t* config){

	status_t err;

	uint16_t data;

	//check id
	err=sgtl5000_read(CHIP_ID,&data);
	if(err!=STATUS_SUCCESS)
		return err;
	if((data&0xff00)!=0xa000)
		return STATUS_UNSUPPORTED;


	err=sgtl5000_write(CHIP_ANA_POWER,(config->power)->anolog_power);
	err=sgtl5000_write(CHIP_DIG_POWER,(config->power)->digtal_power);
	err=sgtl5000_ref_set((config->chip_ref)->vag_val,(config->chip_ref)->bias_ctrl,(config->chip_ref)->small_pop);
	err=sgtl5000_write(CHIP_CLK_CTRL,((config->clk)->mclk)|((config->clk)->sr));
	if((config->clk)->mclk==MCLK_PLL){
		if(MCLK>17000000U){
			err=sgtl5000_write(CHIP_CLK_TOP_CTRL,(1<<3));
		}else{
			err=sgtl5000_write(CHIP_CLK_TOP_CTRL,(0));
		}
		uint16_t int_divisor=16;
		uint16_t frac_divisor=786;
		err=sgtl5000_write(CHIP_PLL_CTRL,(int_divisor<<11)|frac_divisor);
	}
	//set HP volume level
	err=sgtl5000_write(CHIP_MIC_CTRL,BIAS_RES_2K|(5<<4));
	err=sgtl5000_mic_gain(3);
	//input/output path
	err=sgtl5000_write(CHIP_ANA_CTRL,(config->path)->ana);
	err=sgtl5000_write(CHIP_SSS_CTRL,(config->path)->sss);
	//i2s config
	err=sgtl5000_write(CHIP_I2S_CTRL,(config->i2s)->i2s_dlen|(config->i2s)->i2s_lr_align|\
			(config->i2s)->i2s_lr_pol|(config->i2s)->i2s_master|(config->i2s)->i2s_mode|\
			(config->i2s)->i2s_sclk_fs|(config->i2s)->i2s_slck_inv);
	err=sgtl5000_write(CHIP_ANA_HP_CTRL,0);
	err=sgtl5000_write(CHIP_ANA_ADC_CTRL,0x0000);
	err=sgtl5000_write(CHIP_ADCDAC_CTRL,0);
	err=sgtl5000_write(CHIP_DAC_VOL,0xA0a0);

	return err;
}



