/*
 * Copyright 2017 NXP
 * All rights reserved.
 *
 * THIS SOFTWARE IS PROVIDED BY NXP "AS IS" AND ANY EXPRESSED OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL NXP OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */
/*!
 * @file tss_raw.c
 *
 * @page misra_violations MISRA-C:2012 violations
 *
 * @section [global]
 * Violates MISRA 2012 Advisory Rule 11.4, Conversion between a pointer and integer type.
 * The cast is required to initialize a pointer with an unsigned long define, representing an address.
 *
 * @section [global]
 * Violates MISRA 2012 Required Rule 11.6, Cast from unsigned int to pointer.
 * The cast is required to initialize a pointer with an unsigned long define, representing an address.
 *
 * @section [global]
 * Violates MISRA 2012 Advisory Rule 8.7, External could be made static.
 * Function is defined for usage by application code.
 *
 */

#include "tss_raw.h"
#include <stddef.h>
#include <stdint.h>
#include <stdbool.h>

/*******************************************************************************
 * Variables
 ******************************************************************************/
static tss_raw_state_t * s_rawState;
static const IRQn_Type s_lptmrIRQ[LPTMR_IRQS_ARR_COUNT] = LPTMR_IRQS;
static bool allocatedAdcInstances[ADC_INSTANCE_COUNT] = {false};

/*******************************************************************************
 * Private functions
 ******************************************************************************/
static void TSS_Raw_LptmrIsr(void);

/*FUNCTION**********************************************************************
 *
 * Function Name : TSS_Raw_EquivalentVoltageDigitalization
 * Description   : Equivalent voltage conversion
 *
 * Implements : TSS_Raw_EquivalentVoltageDigitalization_Activity
 *END**************************************************************************/
static inline int16_t TSS_Raw_EquivalentVoltageDigitalization(uint8_t electrodeIndex)
{
	uint16_t result;
	tss_raw_electrode_config_t * electrodePtr = s_rawState->electrodes[electrodeIndex];

	ADC_DRV_ConfigChan(electrodePtr->adcInstance, 0UL, electrodePtr->channelConfig);
	ADC_DRV_WaitConvDone(electrodePtr->adcInstance);
	ADC_DRV_GetChanResult(electrodePtr->adcInstance, 0UL, &result);

	return (int16_t)result;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : TSS_Raw_ChargeDistribution
 * Description   : Charge distribution
 *
 * Implements : TSS_Raw_ChargeDistribution_Activity
 *END**************************************************************************/
static inline void TSS_Raw_ChargeDistribution(uint8_t electrodeIndex)
{
	uint32_t portDirection;
	tss_raw_electrode_config_t * electrodePtr = s_rawState->electrodes[electrodeIndex];

	portDirection = PINS_DRV_GetPinsDirection(electrodePtr->gpioBase);
    /* Drive electrode GPIO low */
    PINS_DRV_ClearPins(electrodePtr->gpioBase, (1 << electrodePtr->pinNumberElec));
    /* Configure electrode pin as GPIO to precharge electrode */
    PINS_DRV_SetMuxModeSel(electrodePtr->portBase, electrodePtr->pinNumberElec, PORT_MUX_AS_GPIO);
    /* Drive Cext GPIO high */
    PINS_DRV_SetPins(electrodePtr->gpioBase, (1 << electrodePtr->pinNumberCext));
    /* Configure Cext pin as GPIO to precharge Cext */
    PINS_DRV_SetMuxModeSel(electrodePtr->portBase, electrodePtr->pinNumberCext, PORT_MUX_AS_GPIO);
    /* Configure Electrode and Cext pins as outputs at the same time */
    PINS_DRV_SetPinsDirection(electrodePtr->gpioBase, portDirection | electrodePtr->portMask);
}

/*FUNCTION**********************************************************************
 *
 * Function Name : TSS_Raw_ChargeRedistribution
 * Description   : Charge redistribution
 *
 * Implements : TSS_Raw_ChargeRedistribution_Activity
 *END**************************************************************************/
static inline void TSS_Raw_ChargeRedistribution(uint8_t electrodeIndex)
{
	uint32_t portDirection;
	tss_raw_electrode_config_t * electrodePtr = s_rawState->electrodes[electrodeIndex];

	portDirection = PINS_DRV_GetPinsDirection(electrodePtr->gpioBase);
	/* Configure Electrode and Cext pins as inputs at the same time */
	PINS_DRV_SetPinsDirection(electrodePtr->gpioBase, portDirection & (~electrodePtr->portMask));
	/* Configure electrode pin as analog input */
	PINS_DRV_SetMuxModeSel(electrodePtr->portBase, electrodePtr->pinNumberElec, PORT_PIN_DISABLED);
}

/*******************************************************************************
 * Public functions
 ******************************************************************************/

/*FUNCTION**********************************************************************
 *
 * Function Name : TSS_Raw_Init
 * Description   : Initialization function for the sampling layer of the
 * Touch Sense solution.
 * It will initialize the internal state structures, ADC instance
 * which are configured for use with electrodes and the timer which
 * will provide an interrupt used for sampling.
 * The application must ensure the that clocks which used by
 * the allocated peripherals are enabled.
 *
 * Implements : TSS_Raw_Init_Activity
 *END**************************************************************************/
void TSS_Raw_Init(tss_raw_config_t * configPtr,
		          tss_raw_state_t * statePtr,
				  tss_raw_electrode_state_t ** electrodeStateArr)
{
	DEV_ASSERT(configPtr != NULL);
	DEV_ASSERT(statePtr != NULL);
	DEV_ASSERT(s_rawState == NULL);
	DEV_ASSERT(electrodeStateArr != NULL);

	uint8_t electrode;

	/* Init state structure */
	s_rawState = statePtr;
	s_rawState->lptmrConfig = configPtr->lptmrConfig;
	s_rawState->lptmrInstance = configPtr->lptmrInstance;
	s_rawState->electrodes = configPtr->electrodes;
	s_rawState->numberOfElectrodes = configPtr->numberOfElectrodes;
	s_rawState->sensingCycles = configPtr->sensingCycles;
	s_rawState->sensingPrecycles = configPtr->sensingPrecycles;
	s_rawState->electrodeState = electrodeStateArr;
	s_rawState->sampleBuffer = configPtr->sampleBuffer;
	s_rawState->timerCallback = configPtr->timerCallback;
	s_rawState->chargeDistributionPeriod = configPtr->chargeDistributionPeriod;

	/* See which ADC instances are required for initialization */
	for(electrode = 0U; electrode < s_rawState->numberOfElectrodes; electrode++)
	{
		if(allocatedAdcInstances[s_rawState->electrodes[electrode]->adcInstance] == false)
		{
			DEV_ASSERT(s_rawState->electrodes[electrode]->adcInstance < ADC_INSTANCE_COUNT);
			ADC_DRV_ConfigConverter(s_rawState->electrodes[electrode]->adcInstance, configPtr->adcConfig);
			allocatedAdcInstances[s_rawState->electrodes[electrode]->adcInstance] = true;
		}
	}

	/* Initialize LPTMR */
	LPTMR_DRV_Deinit(s_rawState->lptmrInstance);
	INT_SYS_ClearPending(s_lptmrIRQ[s_rawState->lptmrInstance]);
	LPTMR_DRV_Init(s_rawState->lptmrInstance, s_rawState->lptmrConfig, false);
	INT_SYS_InstallHandler(s_lptmrIRQ[s_rawState->lptmrInstance], TSS_Raw_LptmrIsr, (isr_t*) NULL);
	INT_SYS_EnableIRQ(s_lptmrIRQ[s_rawState->lptmrInstance]);

	/* Initialize electrodes state structure */
	for(electrode = 0U; electrode < s_rawState->numberOfElectrodes; electrode++)
	{
		s_rawState->electrodeState[electrode]->dischargeBuffer = 0;
		s_rawState->electrodeState[electrode]->dischargeBufferCounter = 0UL;
		s_rawState->electrodeState[electrode]->dischargeRaw = 0;
	}

}

/*FUNCTION**********************************************************************
 *
 * Function Name : TSS_Raw_Deinit
 * Description   : Deinitializes the TSS Raw layer by freeing the used structures
 * and peripherals.
 *
 * Implements : TSS_Raw_Deinit_Activity
 *END**************************************************************************/
void TSS_Raw_Deinit(void)
{
	uint8_t electrode;

	/* See which ADC instances are required for deinitialization */
	for(electrode = 0U; electrode < s_rawState->numberOfElectrodes; electrode++)
	{
		if(allocatedAdcInstances[s_rawState->electrodes[electrode]->adcInstance] == true)
		{
			ADC_DRV_Reset(s_rawState->electrodes[electrode]->adcInstance);
			allocatedAdcInstances[s_rawState->electrodes[electrode]->adcInstance] = false;
		}
	}

	/* Deinitialize state structure */
	s_rawState = NULL;
	/* Deinitialize timer */
	LPTMR_DRV_Deinit(s_rawState->lptmrInstance);
}

/*FUNCTION**********************************************************************
 *
 * Function Name : TSS_Raw_GetElectrodeValue
 * Description   : Convert electrode capacitance to equivalent voltage
 *
 * Implements : TSS_Raw_GetElectrodeValue_Activity
 *END**************************************************************************/
int16_t TSS_Raw_GetElectrodeValue(uint8_t electrodeIndex)
{
	uint32_t sampleNumber;
	uint32_t chargeDistributionPeriodTmp;
	tss_raw_electrode_state_t * electrodeState = s_rawState->electrodeState[electrodeIndex];
	uint32_t sensingCycles = s_rawState->sensingCycles;
	uint32_t sensingPrecycles = s_rawState->sensingPrecycles;
	int16_t * sampleBuffer = s_rawState->sampleBuffer;

	for (sampleNumber = 0UL; sampleNumber < (sensingCycles + sensingPrecycles); sampleNumber++)
	{
		TSS_Raw_ChargeDistribution(electrodeIndex);
		chargeDistributionPeriodTmp = s_rawState->chargeDistributionPeriod;
		while(chargeDistributionPeriodTmp)
		{
			chargeDistributionPeriodTmp--;
		}
		TSS_Raw_ChargeRedistribution(electrodeIndex);
		sampleBuffer[sampleNumber] = TSS_Raw_EquivalentVoltageDigitalization(electrodeIndex);
	}

	electrodeState->dischargeRaw = 0;
	for (sampleNumber = sensingPrecycles; sampleNumber < (sensingCycles + sensingPrecycles); sampleNumber++)
	{
		electrodeState->dischargeRaw += sampleBuffer[sampleNumber];
	}
	electrodeState->dischargeRaw = (int16_t)(electrodeState->dischargeRaw / sensingCycles);

	return electrodeState->dischargeRaw;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : TSS_Raw_ElectrodeGnd
 * Description   : This function will set the value of the electrode
 * to ground level.
 *
 * Implements : TSS_Raw_ElectrodeGnd_Activity
 *END**************************************************************************/
void TSS_Raw_ElectrodeGnd(uint8_t electrodeIndex)
{
	tss_raw_electrode_config_t * electrodePtr = s_rawState->electrodes[electrodeIndex];
	/* Drive electrode GPIO low */
	PINS_DRV_ClearPins(electrodePtr->gpioBase, (1 << electrodePtr->pinNumberElec));
	/* Configure electrode pin as GPIO */
	PINS_DRV_SetMuxModeSel(electrodePtr->portBase, electrodePtr->pinNumberElec, PORT_MUX_AS_GPIO);
	/* Drive Cext GPIO low */
	PINS_DRV_ClearPins(electrodePtr->gpioBase, (1 << electrodePtr->pinNumberCext));
	/* Configure Cext pin as GPIO */
	PINS_DRV_SetMuxModeSel(electrodePtr->portBase, electrodePtr->pinNumberCext, PORT_MUX_AS_GPIO);
	/* Configure Electrode and Cext pins as outputs at the same time */
	PINS_DRV_SetPinsDirection(electrodePtr->gpioBase, (PINS_DRV_GetPinsDirection(electrodePtr->gpioBase) | electrodePtr->portMask));
}

/*FUNCTION**********************************************************************
 *
 * Function Name : TSS_Raw_ElectrodeFloat
 * Description   : This function will set the value of the electrode
 * to high impedance.
 *
 * Implements : TSS_Raw_ElectrodeFloat_Activity
 *END**************************************************************************/
void TSS_Raw_ElectrodeFloat(uint8_t electrodeIndex)
{
	tss_raw_electrode_config_t * electrodePtr = s_rawState->electrodes[electrodeIndex];
	PINS_DRV_SetPinsDirection(electrodePtr->gpioBase, (PINS_DRV_GetPinsDirection(electrodePtr->gpioBase) & ~(electrodePtr->portMask)));
}

/*FUNCTION**********************************************************************
 *
 * Function Name : TSS_Raw_StartTimer
 * Description   : Start allocated timer module with a compare value
 * defined by 'ticks' parameter.
 *
 * Implements : TSS_Raw_StartTimer_Activity
 *END**************************************************************************/
void TSS_Raw_StartTimer(uint32_t ticks)
{
	LPTMR_DRV_StopCounter(s_rawState->lptmrInstance);
	(void)LPTMR_DRV_SetCompareValueByCount(s_rawState->lptmrInstance, (uint16_t)ticks);
	LPTMR_DRV_StartCounter(s_rawState->lptmrInstance);

}

/*FUNCTION**********************************************************************
 *
 * Function Name : TSS_Raw_StopTimer
 * Description   : Stop allocated timer module.
 *
 * Implements : TSS_Raw_StopTimer_Activity
 *END**************************************************************************/
void TSS_Raw_StopTimer(void)
{
	LPTMR_DRV_StopCounter(s_rawState->lptmrInstance);
}

/*FUNCTION**********************************************************************
 *
 * Function Name : TSS_Raw_InstallTimerCallback
 * Description   : Installs a callback function that will be called
 * when the allocated timer raises an interrupt due to the
 * compare match event. Prior to this callback, timer compare
 * flags will be cleared.
 * The callback should have the following signature:
 *  void aTimerCallback(void * param);
 * Where the raw layer state structure will be passed as the
 * param argument.
 *
 * Implements : TSS_Raw_InstallTimerCallback_Activity
 *END**************************************************************************/
void TSS_Raw_InstallTimerCallback(void (*callback)(void *))
{
	s_rawState->timerCallback = callback;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : TSS_Raw_LptmrIsr
 * Description   : LPTIMER Interrupt Service Routine
 *
 * Implements : TSS_Raw_LptmrIsr_Activity
 *END**************************************************************************/
static void TSS_Raw_LptmrIsr(void)
{
	LPTMR_DRV_ClearCompareFlag(s_rawState->lptmrInstance);
	s_rawState->timerCallback(s_rawState);
}

