/*
 * PCA85063A_RTC.c
 *
 *  Created on: Jul 18, 2018
 *      Author: nxa07657
 */

#include "PCA85063A_RTC.h"

#include "S32K148_TBOX_BSP.h"
#include "S32K148_TBOX_BSP_test_Config.h"

/*
 * convert the BCD code number to decimal number
 */
#define  BCD_to_DEC(x) ((x&0xF)+((x&0xF0)>>4)*10)
/*
 * convert the decimal number to BCD code number
 */
#define  DEC_to_BCD(x) ((x%10)+((x/10)<<4))

/*************************************************************************
 * initialize I2C interface for the PCA85063A RTC module
 *
 * send command to soft reset the chip
 *
 * initialize the RTC calendar
 *
 * set the RTC calendar alarm and verify result
 ************************************************************************/
status_t PCA85063A_RTC_Init(void)
{
	status_t Status = STATUS_SUCCESS;

	RTC_Calendar_t Calendar_Init={
			.Second = 0,
			.Minute = 0,
			.Hour = 0,
			.Day = 15,
			.Weekday = 3,
			.Month = 8,
			.Year = 2018
	};

	RTC_Calendar_t Calendar_Alarm_Set={
			.Second = 0,
			.Minute = 00,
			.Hour = 12,
			.Day = 19,
			.Weekday = 4
	};

	RTC_Calendar_t Calendar_Alarm_Get;/*used for RTC calendar read and verify*/

	uint8_t masterTxBuffer[10] = {0};/*buffer used to send the I2C message frame to PCA85063A*/

	/*initialize the I2C interface for PCA85063A RTC communication*/
	I2C_MasterSetSlaveAddress(RTC_I2C,i2c1_MasterConfig0.slaveAddress, i2c1_MasterConfig0.is10bitAddr);/* set the salve address */

    masterTxBuffer[0] = 0x00;/*set the Register Control_1 address (0x00) */
    masterTxBuffer[1] = 0x58;/*configure Register Control_1 = 0x58 for soft reset */

    Status = I2C_MasterSendDataBlocking(RTC_I2C, masterTxBuffer, 2, true, 0xFF);/*soft reset the RTC */

    Status = PCA85063A_RTC_Set_Calendar(Calendar_Init);

    Status = PCA85063A_RTC_Set_Calendar_Alarm(Calendar_Alarm_Set);
    Status = PCA85063A_RTC_Get_Calendar_Alarm(&Calendar_Alarm_Get);

    return Status;
}
/**********************************************************************************
 * set the PCA85063A RTC with current Calendar
 **********************************************************************************/
status_t PCA85063A_RTC_Set_Calendar(RTC_Calendar_t Calendar)
{
	status_t Status = STATUS_SUCCESS;
	uint32_t Buffer_Index = 0;
	uint8_t masterTxBuffer[10] = {0};/*buffer used to send the I2C message frame to PCA85063A*/

	 masterTxBuffer[Buffer_Index++] = 0x04;/*set the write start address as 0x04 */
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar.Second);
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar.Minute);
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar.Hour);
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar.Day);
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar.Weekday);
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar.Month);
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar.Year%100);/*can only set the  offset with 100* year*/

	 /*transmit salve address + write and  0x04 as read start address as well as 8 Bytes configuration data*/
	 Status = I2C_MasterSendDataBlocking(RTC_I2C, masterTxBuffer, Buffer_Index, true, 0xFF);

	 return Status;
}
/**********************************************************************************
 * get current Calendar from the PCA85063A RTC
 *********************************************************************************/
status_t PCA85063A_RTC_Get_Calendar(RTC_Calendar_t *Calendar)
{
	status_t Status = STATUS_SUCCESS;

	uint8_t masterTxBuffer[10] = {0};/*buffer used to send the I2C message frame to PCA85063A*/
	uint8_t masterRxBuffer[10] = {0};/*buffer used to receive the I2C message frame to PCA85063A*/

	masterTxBuffer[0] = 0x04;/*set the read start address as 0x04 */

	/*transmit salve address + write and  0x04 as read start address*/
	Status = I2C_MasterSendDataBlocking(RTC_I2C, masterTxBuffer, 1, true, 0xFF);

	/*transmit salve address + read and read back Time and date registers address from 0x04 to 0x0A*/
	Status = I2C_MasterReceiveDataBlocking(RTC_I2C, masterRxBuffer, 7, true, 0xFF);

	Calendar->Second = BCD_to_DEC((masterRxBuffer[0]&0x7F));/*and 0x7F to remove the OS(oscillator stop) bit influence */
	Calendar->Minute = BCD_to_DEC(masterRxBuffer[1]);
	Calendar->Hour = BCD_to_DEC(masterRxBuffer[2]);
	Calendar->Day = BCD_to_DEC(masterRxBuffer[3]);
	Calendar->Weekday = BCD_to_DEC(masterRxBuffer[4]);
	Calendar->Month = BCD_to_DEC(masterRxBuffer[5]);
	Calendar->Year = 2000 + BCD_to_DEC(masterRxBuffer[6]);/*add the year base--2000 year*/


	if(masterRxBuffer[0]&CLK_OK_MASK)/*check whether the RTC clock integrity is guaranteed */
	{
		Status = STATUS_BUSY;
	}
	else
		Status = STATUS_SUCCESS;

	return Status;
}

/**********************************************************************************
 * set the PCA85063A RTC with Calendar_Alarm
 *********************************************************************************/
status_t PCA85063A_RTC_Set_Calendar_Alarm(RTC_Calendar_t Calendar_Alarm)
{
	status_t Status = STATUS_SUCCESS;
	uint32_t Buffer_Index = 0;

	uint8_t masterTxBuffer[10] = {0};/*buffer used to send the I2C message frame to PCA85063A*/

	 masterTxBuffer[Buffer_Index++] = 0x0B;/*set the write start address as 0x0B */
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar_Alarm.Second);
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar_Alarm.Minute);
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar_Alarm.Hour);
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar_Alarm.Day);
	 masterTxBuffer[Buffer_Index++] = DEC_to_BCD(Calendar_Alarm.Weekday);

	 /*transmit salve address + write and  0x04 as read start address as well as 8 Bytes configuration data*/
	 Status = I2C_MasterSendDataBlocking(RTC_I2C,  masterTxBuffer, Buffer_Index++, true, 0xFF);

	 return Status;
}

/**********************************************************************************
 * get the PCA85063A RTC current Calendar_Alarm configuration
 *********************************************************************************/
status_t PCA85063A_RTC_Get_Calendar_Alarm(RTC_Calendar_t *Calendar_Alarm)
{
	status_t Status = STATUS_SUCCESS;

	uint8_t masterTxBuffer[10] = {0};/*buffer used to send the I2C message frame to PCA85063A*/
	uint8_t masterRxBuffer[10] = {0};/*buffer used to receive the I2C message frame to PCA85063A*/

	masterTxBuffer[0] = 0x0B;/*set the read start address as 0x0B */

	/*transmit salve address + write and  0x04 as read start address*/
	Status = I2C_MasterSendDataBlocking(RTC_I2C,  masterTxBuffer, 1, true, 0xFF);

	/*transmit salve address + read and read back Time and date registers address from 0x04 to 0x0A*/
	Status = I2C_MasterReceiveDataBlocking(RTC_I2C,  masterRxBuffer, 5, true, 0xFF);

	if(masterRxBuffer[0]&Alarm_Disabled_MASK)/*check whether the alarm enabled */
		Status = STATUS_BUSY;  /*alarm disabled*/
	else
	{
		Calendar_Alarm->Second = BCD_to_DEC(masterRxBuffer[0]);
		Calendar_Alarm->Minute = BCD_to_DEC(masterRxBuffer[1]);
		Calendar_Alarm->Hour = BCD_to_DEC(masterRxBuffer[2]);
		Calendar_Alarm->Day = BCD_to_DEC(masterRxBuffer[3]);
		Calendar_Alarm->Weekday = BCD_to_DEC(masterRxBuffer[4]);

		Status = STATUS_SUCCESS;
	}

	return Status;
}

/*********************************************************************************
 * call this function to output the RTC calendar via printf
 ********************************************************************************/
void RTC_Time_Print(RTC_Calendar_t *RTC_Time)
{
	printf("Current RTC time from PCA85063A chip--YYYY/MM/WW/DD--HH:MM:SS is:\n");

	printf("%d/%d/%d/%d--%d:%d:%d\n\r", RTC_Time->Year,\
			RTC_Time->Month,\
			RTC_Time->Day,\
			RTC_Time->Weekday,\
			RTC_Time->Hour,\
			RTC_Time->Minute,\
			RTC_Time->Second);
}
/*********************************************************************************
 * call this function to test the RTC calendar via printf the current calendar time
 ********************************************************************************/
status_t PCA85063A_RTC_Test(void)
{
	status_t Status = STATUS_SUCCESS;

	RTC_Calendar_t Calendar_Current;/*used to get current RTC calendar*/

	/*******************************************************************************************
	 * use test MMA845x_ACC at the same time, it's a must to re-configure the I2C slave address
	 * as they are using the same I2C but with different salve address
	 * ****************************************************************************************/
#ifdef MMA845x_ACC_test
	/* set the salve address */
	I2C_MasterSetSlaveAddress(RTC_I2C,i2c1_MasterConfig0.slaveAddress, i2c1_MasterConfig0.is10bitAddr);
#endif

	Status = PCA85063A_RTC_Get_Calendar(&Calendar_Current);/*get current calendar time*/

	if(Status==STATUS_SUCCESS)/*check whether the RTC timer is correct and usable*/
		RTC_Time_Print(&Calendar_Current);/*output it by UART print*/

	return Status;
}


