/*****************************************************************************
 *   pwm.c:  PWM module file for NXP LPC23xx/24xx Family Microprocessors
 *
 *   Copyright(C) 2007, NXP Semiconductor
 *   All rights reserved.
 *
 *	 Authur: C.J.M.Jansen
 *   History
 *   2007.06.22  ver 1.00    Prelimnary version, first Release
 *
******************************************************************************/
#include "LPC23xx.h"                        /* LPC23xx/24xx definitions */
#include "type.h"
#include "irq.h"
//#include "timer.h"
#include "pwm.h"

volatile DWORD match_counter0, match_counter1, variable;

static DWORD MatchTable [3][2]= {{0,0},	// cycle time (match 0),duty cycle time
								 {0,0},	// match for upper switch
								 {0,0}};// match for lower switch

static DWORD OffsetTable [2][2]= {{0, 	// upper switch set
								   0},	// upper switch reset
								  {0,   // lower switch set
								   0}	// lower switch reset
								 };

/******************************************************************************
** Function name:		PWM0Handler
**
** Descriptions:		PWM Match0 interrupt handler
**						For now, it deals with PWM1 match 0 and PWM0 match 0
**
** parameters:			None
** Returned value:		None
** 
******************************************************************************/
void PWMHandler (void) __irq 
{
    DWORD regVal;
    IENABLE;				/* handles nested interrupt */
	regVal = PWM0IR;
    if ( regVal & MR0_INT )
    {
		match_counter0++;	
        		
	}
	PWM0IR |= regVal;		/* clear interrupt flag on match 0 */
	regVal = PWM1IR;
    if ( regVal & MR0_INT )
    {
		match_counter1++;	
	    
	}

	PWM1IR |= regVal;		/* clear interrupt flag on match 0 */

    IDISABLE;
    VICVectAddr = 0;		/* Acknowledge Interrupt */
	return;
}	

/******************************************************************************
** Function name:		PWM_Init
**
** Descriptions:		PWM initialization, setup all GPIOs to PWM0~6,
**				reset counter, all latches are enabled, interrupt
**				on PWMMR0, install PWM interrupt to the VIC table.
**
** parameters:			ChannelNum, Duty cycle
** Returned value:		true or fase, if VIC table is full, return false
** 
******************************************************************************/
DWORD PWM_Init(void )
{
     	match_counter0 = 0;	   //used in the ISR

// *** PWM 0 (lower switch)****

		PINSEL2|= 0x00C030C0; // P1.3,P1.6,P1.11 are PWM0.2,PWM0.4 and PWM0.6
		  						
		PWM0TCR = TCR_Master_EN | TCR_CNT_EN | TCR_PWM_EN; /* Use syncro between
															PWM0/1, 
															Enable counter for PWM0
															Enable PWM0*/
		
		PWM0PR = 0x00;			/* count frequency:Fpclk (prescalar) */

/*   	PWM0MCR = PWMMR0I | PWMMR0R;	interrupt on PWM0MR0, 
		                                   reset on PWM0MR0, 
										   reset TC if PWM0MR0 matches */				
 	  	PWM0MCR =  PWMMR0R;	/* no interrupt on PWM0MR0, reset TC if  match*/

		PWM0PCR = 0x0000005454; //PWM0.2,PWM0.4 and PWM0.6 are generated

 		match_counter1 = 0;		// used in the ISR

// *** PWM1 (upper switch)*****

		PINSEL4 |= 0x00000444;	//set P2.1,P2.3 and P2.5 to PWM1.2,PWM1.4 and PWM1.6
 //		PINSEL3 |= 0x00208200;  // set P1.20,P1.23 and P1,6 to PWM1.2,PWM1.4 and PWM1.6


		PWM1TCR = TCR_CNT_EN | TCR_PWM_EN; /* Enable COunter for PWM1,
	                                      Enable PWM1*/

    	PWM1PR = 0x00;			/* count frequency:Fpclk */
    
		PWM1MCR = PWMMR0I | PWMMR0R; /* interrupt on PWM1MR0, 
		                                   reset on PWM1MR0, 
										   reset TC if PWM1MR0 matches */
 		
		PWM1PCR = 0x0000005454;	  //PWM1.2,PWM1.4 and PWM1.6 are generated
//		PWM1PCR = 0x0000005654;	  //PWM1.1 PWM1.2,PWM1.4 and PWM1.6 are generated
	
    	/* all PWM latch enabled */
 
		PWM_set_deadtime(20,40,60);		// set offset values
		PWM_set_duty_cycle(500);		// set  duty cycle+ offset values in MatchTable	
				
		PWM_switch(0);					// All outputs are switched off
		PWM_set_cycle(1000);			// set and activate cycle

	  	PWM0TCR = PWM0TCR & 0xFFFFFFEF;  //enable both PWMs in the same time

	if ( install_irq( PWM0_1_INT, (void *)PWMHandler, HIGHEST_PRIORITY ) == FALSE )
    {
		return ( FALSE );
    }
    return (TRUE);
}

/******************************************************************************
** Function name:		PWM_doubleEdge_off
**
** Descriptions:		put channel output on low ("0")level
						double edge :
							0 = all channels
						    1 = PWM1 channel 2
							2 = PWM1 channel 4
							3 = PWM1 channel 6
							4 = PWM0 channel 2
							5 = PWM0 channel 4
							6 = PWM0 channel 6
**						
** parameters:			channel number 
** Returned value:		None
** 
** Remark:				This function is not latching the values in shadow 
**						registers of PWM module
**                      This will be done with PWM_activate_match
******************************************************************************/
void PWM_doubleEdge_off( DWORD channelNum,DWORD outputInverse)
{	
	if (outputInverse == TRUE)
	  { 
    	switch (channelNum)
			{
			  case 0 :  {PWM1MR1= 0x0;PWM1MR2=0xffffffff;		 // all channels inverse
			  			 PWM1MR3= 0x0;PWM1MR4=0xffffffff;
			  			 PWM1MR5= 0x0;PWM1MR6=0xffffffff;
			  			 PWM0MR1= 0x0;PWM0MR2=0xffffffff;
			  			 PWM0MR3= 0x0;PWM0MR4=0xffffffff;
			  			 PWM0MR5= 0x0;PWM0MR6=0xffffffff; 	  
			  										      break;}	
			  case 1 :  {PWM1MR1= 0x0;PWM1MR2=0xffffffff; break;}//upper switch	inverse
			  case 2 :  {PWM1MR3= 0x0;PWM1MR4=0xffffffff; break;}	
			  case 3 :  {PWM1MR5= 0x0;PWM1MR6=0xffffffff; break;}
			  case 4 :  {PWM0MR1= 0x0;PWM0MR2=0xffffffff; break;}//lower switch	inverse
			  case 5 : 	{PWM0MR3= 0x0;PWM0MR4=0xffffffff; break;}
			  case 6 : 	{PWM0MR5= 0x0;PWM0MR6=0xffffffff; break;}
			  default: return;
	   		}
	  }
	else
	   {
   		switch (channelNum)
			{
			  case 0 :  {PWM1MR1= 0xffffffff;PWM1MR2=0x0;		// all channels
			  			 PWM1MR3= 0xffffffff;PWM1MR4=0x0;
			  			 PWM1MR5= 0xffffffff;PWM1MR6=0x0;
			  			 PWM0MR1= 0xffffffff;PWM0MR2=0x0;
			  			 PWM0MR3= 0xffffffff;PWM0MR4=0x0;
			  			 PWM0MR5= 0xffffffff;PWM0MR6=0x0; 	  
			  										      break;}	
			  case 1 :  {PWM1MR1= 0xffffffff;PWM1MR2=0x0; break;}//upper switch	
			  case 2 :  {PWM1MR3= 0xffffffff;PWM1MR4=0x0; break;}	
			  case 3 :  {PWM1MR5= 0xffffffff;PWM1MR6=0x0; break;}
			  case 4 :  {PWM0MR1= 0xffffffff;PWM0MR2=0x0; break;}//lower switch
			  case 5 : 	{PWM0MR3= 0xffffffff;PWM0MR4=0x0; break;}
			  case 6 : 	{PWM0MR5= 0xffffffff;PWM0MR6=0x0; break;}
			  default: return;
	   		}
		}
    return;
}
/******************************************************************************
** Function name:		PWM_doubleEdge_on
**
** Descriptions:		put channel "on" 
						double edge :
						    0 = cycle time of PWM0 and PWM1
						    1 = PWM1 channel 2	upper switch
							2 = PWM1 channel 4
							3 = PWM1 channel 6
							4 = PWM0 channel 2	lower switch
							5 = PWM0 channel 4
							6 = PWM0 channel 6
**						
** parameters:			channel number 	(0 = cycle time),invers output (0= normal,not 0=inverse)
** Returned value:		None
** Remark:				This function is not latching the values in shadow 
**                      registers of PWM module
**                      This will be done with PWM_activate_match
******************************************************************************/
void PWM_doubleEdge_on( DWORD channelNum,DWORD outputInverse)
{
	if (outputInverse == TRUE)
	  {
      switch (channelNum) //PWM channel becomes"low"
	             {	
				   case 0 :  {PWM1MR0= MatchTable[0][0]; // set cycle time,no inversion
				  			 PWM0MR0= MatchTable[0][0];
				 			 break;}	
				  case 1 :  {PWM1MR2= MatchTable[1][0];	// set upper switch
				  			 PWM1MR1= MatchTable[1][1];
							 break;}
				  case 2 :  {PWM1MR4= MatchTable[2][0];	// lower switch
				  			 PWM1MR3= MatchTable[2][1];
							 break;}// 
				  case 3 :  {PWM1MR6= MatchTable[2][0];	// lower switch
				  			 PWM1MR5= MatchTable[2][1];
							 break;}
				  case 4 :  {PWM0MR2= MatchTable[2][0];	// lower switch
				  			 PWM0MR1= MatchTable[2][1];
							 break;}	
				  case 5 : 	{PWM0MR4= MatchTable[2][0];
				  			 PWM0MR3= MatchTable[2][1];
							 break;}
				  case 6 : 	{PWM0MR6= MatchTable[2][0];
				  			 PWM0MR5= MatchTable[2][1];
							 break;}
				  default: return;
				 }
		}
	else
	  {
      	 switch (channelNum) //PWM channel becomes"high"
				{
				  case 0 :  {PWM1MR0= MatchTable[0][0]; // set cycle time
				  			 PWM0MR0= MatchTable[0][0];
				 			 break;}	
				  case 1 :  {PWM1MR1= MatchTable[1][0];	// set upper switch
				  			 PWM1MR2= MatchTable[1][1];
							 break;}
				  case 2 :  {PWM1MR3= MatchTable[1][0];
							 PWM1MR4= MatchTable[1][1];
							 break;}
 				  case 3 :  {PWM1MR5= MatchTable[1][0];
				  			 PWM1MR6= MatchTable[1][1];
							 break;}
				  case 4 :  {PWM0MR1= MatchTable[2][0];	// lower switch
				  			 PWM0MR2= MatchTable[2][1];
							 break;}	
				  case 5 : 	{PWM0MR3= MatchTable[2][0];
				  			 PWM0MR4= MatchTable[2][1];
							 break;}
				  case 6 : 	{PWM0MR5= MatchTable[2][0];
				  			 PWM0MR6= MatchTable[2][1];
							 break;}
 				  default: return;
			 }
      
	  }	 
 
     return;
}
/******************************************************************************
** Function name:		PWM_doubleEdge_on_test
**
** Descriptions:		put channel "on" 
						double edge :
						    0 = cycle time of PWM0 and PWM1
						    1 = PWM1 channel 2	upper switch
							2 = PWM1 channel 4	lower switch
							3 = PWM1 channel 6	lower switch
**						
** parameters:			channel number 	(0 = cycle time),invers output (0= normal,not 0=inverse)
** Returned value:		None
** Remark:				This function is not latching the values in shadow 
**                      registers of PWM module
**                      This will be done with PWM_activate_match
******************************************************************************/
void PWM_doubleEdge_on_test( DWORD channelNum,DWORD outputInverse)
{	
	if (outputInverse == TRUE)
	  {
      switch (channelNum) //PWM channel becomes"low"
	             {	
				   case 0 :  {PWM1MR0= MatchTable[0][0]; // set cycle time,no inversion
				  			 PWM0MR0= MatchTable[0][0];
				 			 break;}	
				  case 1 :  {PWM1MR2= MatchTable[1][0];	// set upper switch
				  			 PWM1MR1= MatchTable[1][1];
							 break;}
				  case 2 :  {PWM1MR4= MatchTable[2][0];	// lower switch
				  			 PWM1MR3= MatchTable[2][1];
							 break;}// 
				  case 3 :  {PWM1MR6= MatchTable[2][0];	// lower switch
				  			 PWM1MR5= MatchTable[2][1];
							 break;}
				  default: return;
				 }
		}
	else
	  {
      switch (channelNum) //PWM channel becomes"high"
				{
				  case 0 :  {PWM1MR0= MatchTable[0][0]; // set cycle time
				  			 PWM0MR0= MatchTable[0][0];
				 			 break;}	
				  case 1 :  {PWM1MR1= MatchTable[1][0];	// set upper switch
				  			 PWM1MR2= MatchTable[1][1];
							 break;}
				  case 2 :  {PWM1MR3= MatchTable[2][0];	// lower switch
				  			 PWM1MR4= MatchTable[2][1];
							 break;}// 
				  case 3 :  {PWM1MR5= MatchTable[2][0];	// lower switch
				  			 PWM1MR6= MatchTable[2][1];
							 break;}
				  default: return;
				 }
      
	  }
     return;
}
/******************************************************************************
** Function name:		PWM_activate_match
**
** Descriptions:		the new match values of PWM0 and PWM1 module will be 
                        latched in shadow registers
						Is used in PWM_switch
**						
** parameters:			None 
** Returned value:		None
** Remark:				
**                      
******************************************************************************/
void PWM_activate_match()
{
//	PWM0TCR = PWM0TCR | 0x00000010;  //disable both PWMs in the same time
		 
     	/* The LER will be cleared when the Match 0 takes place, in order to
    	load and execute the new value of match registers, all the PWMLERs need 
		to reloaded. all PWM latch enabled */
//    	PWM0LER = LER0_EN | LER1_EN | LER2_EN | LER3_EN | LER4_EN | LER5_EN | LER6_EN;
    	PWM0LER = LER1_EN | LER2_EN | LER3_EN | LER4_EN | LER5_EN | LER6_EN;

    	/* The LER will be cleared when the Match 1 takes place, in order to
    	load and execute the new value of match registers, all the PWMLERs need to
    	reloaded. all PWM latch enabled */
//    	PWM1LER = LER0_EN | LER1_EN | LER2_EN | LER3_EN | LER4_EN | LER5_EN | LER6_EN;
    	PWM1LER = LER1_EN | LER2_EN | LER3_EN | LER4_EN | LER5_EN | LER6_EN;

// 	PWM0TCR = PWM0TCR & 0xFFFFFFEF;  //enable both PWMs in the same time

     return;
}
/******************************************************************************
** Function name:		PWM_set_deadtime
**
** Descriptions:		calculates from the delay,rise-delay and fall-delay
					    the upper and lower switch match values and puts it in 
						the Offset table 
				
**						
** parameters:			delay,rise_delay, fall_delay 
** Returned value:		None
** Remark:				
**                      
******************************************************************************/
void PWM_set_deadtime(DWORD delay,DWORD rise_delay,DWORD fall_delay)
{
	OffsetTable[0][0]= delay + rise_delay; 			// table is for changing duty cycle 
	OffsetTable[0][1]= delay + rise_delay; 			// upper
	OffsetTable[1][0]= delay;						// lower
	OffsetTable[1][1]= delay + fall_delay;			// lower
     return;
}
/******************************************************************************
** Function name:		PWM_set_duty_cycle
**
** Descriptions:		calculates the new value for MatchTable
						Is used in PWM_double_edge_on
**						
** parameters:			duty cycle 
** Returned value:		None
** Remark:				
**                      
******************************************************************************/
void PWM_set_duty_cycle(DWORD duty_cycle)
{
 DWORD temp;
	// if duty_cycle > cycle - delay - risetime - falltime --> duty_cycle =
	// cycle - delay - risetime - falltime
	temp= MatchTable[0][0]-OffsetTable[0][1]- OffsetTable[1][1]+ OffsetTable[1][0];
	if (duty_cycle > temp) duty_cycle= temp;

	MatchTable[1][0]= OffsetTable[0][0];	 		 // upper switch set 
	MatchTable[1][1]= OffsetTable[0][1]+ duty_cycle; // upper switch reset 
 	MatchTable[2][0]= OffsetTable[1][0];	  		 // lower switch set 
	MatchTable[2][1]= OffsetTable[1][1]+ duty_cycle; // lower switch reset 
	MatchTable[0][1]= duty_cycle;  					 // store duty cycle
     return;
}
/******************************************************************************
** Function name:		PWM_set_cycle
**
** Descriptions:		put new value in MatchTable and activates (new) cycle
				
**						
** parameters:			cycle 
** Returned value:		None
** Remark:				
**                      
******************************************************************************/
void PWM_set_cycle(DWORD cycle)
{
 	MatchTable[0][0]= cycle;										 // update cycle
	PWM_doubleEdge_on(0,NormalOutput);								 // 0= duty cycle for Match0.0 and Match1.0
	PWM0LER = LER0_EN ;  PWM1LER = LER0_EN ; 						 // latch into Match0 register (activate)
     return;
}
/******************************************************************************
** Function name:		PWM_switch
**
** Descriptions:		switch on the upper and lower switch for the bridge
**						
** parameters	
** Returned value:		None
** Remark:				
**                      
******************************************************************************/
void PWM_switch(DWORD i)
{
	 switch (i)
	  {
		 case 0 :  {PWM_doubleEdge_off(0,0); 						// all channel off
					PWM_activate_match();
		 			break;}		
		 case 1 :  {PWM_doubleEdge_off(6,NormalOutput);
		 			PWM_doubleEdge_on(1,NormalOutput);
					PWM_doubleEdge_on(5,NormalOutput);
					PWM_activate_match();
		 			break;}	
		 case 2 :  {PWM_doubleEdge_off(1,NormalOutput);
		 			PWM_doubleEdge_on(3,NormalOutput);PWM_doubleEdge_on(5,NormalOutput);
					PWM_activate_match();
		 			break;}	
		 case 3 :  {PWM_doubleEdge_off(5,NormalOutput);
		 			PWM_doubleEdge_on(3,NormalOutput);PWM_doubleEdge_on(4,NormalOutput);
					PWM_activate_match();
		 			break;}
		 case 4 :  {PWM_doubleEdge_off(3,NormalOutput);
		 			PWM_doubleEdge_on(2,NormalOutput);PWM_doubleEdge_on(4,NormalOutput);
					PWM_activate_match();
		 			break;}
		 case 5 :  	{PWM_doubleEdge_off(4,NormalOutput);
		 			PWM_doubleEdge_on(2,NormalOutput);PWM_doubleEdge_on(6,NormalOutput);
					PWM_activate_match();
		 			break;}
		 case 6 :  {PWM_doubleEdge_off(2,NormalOutput);
		 			PWM_doubleEdge_on(1,NormalOutput);PWM_doubleEdge_on(6,NormalOutput);
					PWM_activate_match();
		 			break;}
 		 default: return;
	  }

     return;
}
/******************************************************************************
** Function name:		PWM_switch_test
**
** Descriptions:		switch on the upper and lower switch for the bridge
**						Only the PWM1 is used  For testing on a LPC23xx
** parameters	
** Returned value:		None
** Remark:				
**                      
******************************************************************************/
void PWM_switch_test(DWORD i)
{
	 switch (i)
	  {
		 case 0 :  {PWM_doubleEdge_off(0,NormalOutput); 						// all channel off
					PWM_activate_match();
		 			break;}		
		 case 1 :  {PWM_doubleEdge_on_test(1,NormalOutput);
					PWM_doubleEdge_on_test(2,NormalOutput);
					PWM_doubleEdge_off(3,NormalOutput);
					PWM_activate_match();
		 			break;}	
		 case 2 :  {PWM_doubleEdge_off(2,NormalOutput);
		 			PWM_doubleEdge_on_test(3,NormalOutput);
					PWM_activate_match();
		 			break;}	
		 case 3 :  {PWM_doubleEdge_off(1,NormalOutput);
		 			PWM_doubleEdge_off(3,NormalOutput);
					PWM_activate_match();
		 			break;}	
		  default: return;
	  }

     return;
}
