/******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2011-2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
***************************************************************************//*!
*
* @file main.c
*
* @author a13984
*
* @version 0.0.1
*
* @date Jul-15-2011
*
* @brief 
*       Hall-sensor Bipolar
      
  \n Copyright (c) 2011 Freescale Semiconductor
  \n Freescale Confidential Proprietary
  
  \brief        Motor control using Hall Sensors Hands on                        
  \n            The firmware was developed on CodeWarrior 10.1                                                                                       
  \n            Description: The firmware initialize 6 PWM to work in            
  \n            center-aligned modeputs in order to drive six transistors of the 
  \n            3-phase power stage. A startup                                   
  \n            routine was implemented to get out the motor from its stand still
  \n            position. Speed measurement is implemented in this Lab. This Lab  
  \n            uses two touch pads, one push button (SW2) and various LEDs. The motor can
  \n            spin in clockwise and counter clockwise direction. Also can be   
  \n            stop it at any moment using the push button (SW2). Can control speeds 
  \n            from 200 rpm to 5000 rpm. The PWM used is bipolar                
  
 \ maxon EC motor  EC32 200187 Flat motor 
                 
                  motor runs on CW as specified on data sheet of motor
                  if voltage is positive  PWM% >50%  (500 - 850)
                                                        
 Notes: The global variables and local variables declared within   
 this code are written in lowercase. The constant used in this     
 project are written in uppercase                                  
                                                                                                             

 PT60 Tower board TWRI-to-APMOTOR56F800 connection:
 
 1.PWM Connection:	
 PT60              APMOTOR56F800               TWRPI pin
 =============     ==================        =================
 FTM2 Channel0 --> Phase A  Q1_TOP_A         right pin14
 FTM2 Channel1 --> Phase A  Q4_BOTTOM_A      right pin15
 FTM2 Channel2 --> Phase B  Q2_TOP_B         right pin16
 FTM2 Channel3 --> Phase B  Q5_BOTTOM_B      right pin18
 FTM2 Channel4 --> Phase C  Q3_TOP_C         right pin19
 FTM2 Channel5 --> Phase C  Q6_BOTTOM_C      right pin20
 
 2. Power connection:
    PT60              APMOTOR56F800               TWRPI pin
 =============     ==================        =================
   3.3V                  pin 1                 right pin2
   GND				     pin 3                 right pin3
 
 3. Hall Connection:
    PT60              APMOTOR56F800               TWRPI pin
 =============     ==================        =================
 PTF6/ADC1        --> T0 (Hall Sensor 1)       left pin8
 PTF5/ADC2        --> T1 (Hall Sensor 2)       left pin11
 PTF4/ADC3        --> T2 (Hall Sensor 3)       left pin12                           
 
 Usage:
  1. Press SW2 on tower board to RUN/STOP motor
  2. Push  PAD1 on tower board to increase the speed or set clockwise direction
  3. Push  PAD2 on tower board to decrease the speed or set counter clockwise direction                                  
*
*******************************************************************************
*
* provide a demo for how to initialize the PT60, output messages via SCI, 
* flash operations, etc.
* NOTE:
*	printf call may occupy a lot of memory (around 1924 bytes), so please
*	consider your code size before using printf.
******************************************************************************/

#include <hidef.h> /* for EnableInterrupts macro */
#include "derivative.h" /* include peripheral declarations */
#include "common.h"
#include "ics.h"
#include "ftm.h"
#include "mtim.h"
#include "printf.h"
#include "tsi.h"


/******************************************************************************
* External objects
******************************************************************************/

/******************************************************************************
* Constants and macros
******************************************************************************/
/**************************************************************************************************/
/*   3-PHASE POWER STAGE                                                                          */
/*                                                                                                */
/*   Q1_TOP_A = Q1        Q2_TOP_B = Q2          Q3_TOP_C = Q3                                    */
/*                                                                                                */
/*       |-                    |-                      |-                                         */
/*     -|                    -|                      -|                                           */
/*       |-                    |-                      |-                                         */
/* -----------------------------------------------------------------                              */
/*  Q4_BOTTOM_A = Q4      Q5_BOTTOM_B = Q5       Q6_BOTTOM_C = Q6                                 */
/*                                                                                                */
/*       |-                    |-                      |-                                         */
/*     -|                    -|                      -|                                           */
/*       |-                    |-                      |-                                         */
/*                                                                                                */
/**************************************************************************************************/
/*Transistors pin control Qx Pin assignation   */
#define Q1_TOP_A    PORT_PTCD_PTCD0   /* Q1   FTM2CH0 pwm0*/
#define Q4_BOTTOM_A PORT_PTCD_PTCD1   /* Q4   FTM2CH1 pwm1*/
#define Q2_TOP_B    PORT_PTCD_PTCD2   /* Q2   FTM2CH2 pwm2*/
#define Q5_BOTTOM_B PORT_PTCD_PTCD3   /* Q5   FTM2CH3 pwm3*/
#define Q3_TOP_C    PORT_PTBD_PTBD4   /* Q3   FTM2CH4 pwm4*/
#define Q6_BOTTOM_C PORT_PTBD_PTBD5   /* Q6   FTM2CH5 pwm5*/

/* control direction to output */
#define Q1_TOP_A_DD    PORT_PTCOE_PTCOE0   /* Q1 */
#define Q4_BOTTOM_A_DD PORT_PTCOE_PTCOE1   /* Q4 */
#define Q2_TOP_B_DD    PORT_PTCOE_PTCOE2   /* Q2 */
#define Q5_BOTTOM_B_DD PORT_PTCOE_PTCOE3   /* Q5 */
#define Q3_TOP_C_DD    PORT_PTBOE_PTBOE4   /* Q3 */
#define Q6_BOTTOM_C_DD PORT_PTBOE_PTBOE5   /* Q6 */


/*Hall Sensor inputs */
#define T0_HALL_1       PORT_PTFD_PTFD6    /* T0 Hall Sensor 1 */      
#define T1_HALL_2       PORT_PTFD_PTFD5    /* T1 Hall Sensor 2 */      
#define T2_HALL_3       PORT_PTFD_PTFD4    /* T2 Hall Sensor 3 */      


/*User interfase input/outputs */
#define SW_RUN_STOP     PORT_PTDD_PTDD5  /* This is used to put the motor state in Run/Stop: SW2 on tower board  */
#define PUSH_BUTTON1    bPAD1  /* PAD1 is touched on tower board*/
#define PUSH_BUTTON2    bPAD2  /* PAD2 is touched on tower board*/

#define Q1_TOP_A_PWM(pwm_value)       FTM2_C0V = pwm_value      /* Q1 */
#define Q4_BOTTOM_A_PWM(pwm_value)    FTM2_C1V = pwm_value      /* Q4 */
#define Q2_TOP_B_PWM(pwm_value)       FTM2_C2V = pwm_value      /* Q2 */
#define Q5_BOTTOM_B_PWM(pwm_value)    FTM2_C3V = pwm_value      /* Q5 */
#define Q3_TOP_C_PWM(pwm_value)       FTM2_C4V = pwm_value      /* Q3 */
#define Q6_BOTTOM_C_PWM(pwm_value)    FTM2_C5V = pwm_value      /* Q6 */

                                                          
/*Qc */
#define Q1_TOP_A_HIGH_TRUE_PULSES     FTM2_C0SC_ELSx = 2     // Ac - Q1
#define Q4_BOTTOM_A_HIGH_TRUE_PULSES  FTM2_C1SC_ELSx = 2     // An - Q4    
#define Q2_TOP_B_HIGH_TRUE_PULSES     FTM2_C2SC_ELSx = 2     // Bn - Q2
#define Q5_BOTTOM_B_HIGH_TRUE_PULSES  FTM2_C3SC_ELSx = 2     // Bc - Q5    
#define Q3_TOP_C_HIGH_TRUE_PULSES     FTM2_C4SC_ELSx = 2     // Cn - Q3
#define Q6_BOTTOM_C_HIGH_TRUE_PULSES  FTM2_C5SC_ELSx = 2     // Cc - Q6    


#define Q1_TOP_A_LOW_TRUE_PULSES      FTM2_C0SC_ELSx = 1      // An - Q1 
#define Q4_BOTTOM_A_LOW_TRUE_PULSES   FTM2_C1SC_ELSx = 1      // Ac - Q4
#define Q2_TOP_B_LOW_TRUE_PULSES      FTM2_C2SC_ELSx = 1      // Bn - Q2
#define Q5_BOTTOM_B_LOW_TRUE_PULSES   FTM2_C3SC_ELSx = 1      // Bc - Q5 
#define Q3_TOP_C_LOW_TRUE_PULSES      FTM2_C4SC_ELSx = 1      // Cn - Q3
#define Q6_BOTTOM_C_LOW_TRUE_PULSES   FTM2_C5SC_ELSx = 1      // Cc - Q6 

/* The following parameter should be changed accordingly 
 * 
 */
#define TIMER_FREQ  ((BUS_CLK_HZ/32/2) * 60)      /* This is the timer frequency. 
                                         This value is obtained by dividing 
                                        * bus clock frequency / 32, 
                                        * refer to FTM1 timing setting in FTM1_Init_Capt()
                                        * */

#define MOTOR_STATE_STANDBY       0
#define MOTOR_STATE_STARTUP       1
#define MOTOR_STATE_FAULT         2
#define MOTOR_STATE_RUN           3
#define DIRECTION_CW              0         /* Clockwise direction */
#define DIRECTION_CCW             1         /* Counter Clockwise direction */

#define PWM_MODULO_VALUE          1000      /* This is the value that will be loaded into PWM modulo register */  
#define PWM_MINIMUN_VALUE         550       /* This is the minimum value for a PWM */
#define PWM_MAXIMUN_VALUE         900       /* This is the maximum value for a PWM */

#define DEAD_TIME                 10        /* Dead time used to avoid short circuits in the same phase transistors */ 


/* Speed parameters */
#define MAX_SPEED   5000       /* This is the maximum speed in rpms permited. Motor limitations */
#define MIN_SPEED   200        /* This is the minimum speed in rpms permited. Motor limitations */
#define STEP        200        /* This is the increment or decrement value when a push button is pressed */  


/* LEDs */
#define LED1   PORT_PTGD_PTGD0    // HALL 1 LED
#define LED2   PORT_PTGD_PTGD1    // HALL 2 LED
#define LED3   PORT_PTGD_PTGD2    // HSW - Run/Stop
#define LED4   PORT_PTGD_PTGD3    // motor spin direction

/******************************************************************************
* Local types
******************************************************************************/

/******************************************************************************
* Local function prototypes
******************************************************************************/
void delay(unsigned int time);
void FTM1_Init_Capt(void);
void FTM2_Init_PWM(void);
unsigned char get_motor_position(void);
void commutation(unsigned char value, unsigned int time_on);
void speed_control(void);
interrupt VectorNumber_Vkbi1 void KBI1_ISR(void);
interrupt VectorNumber_Vftm1ch1 void  FTM1_CH1_ISR (void);
interrupt VectorNumber_Vftm2ovf void FMT2_ISR(void);

void KEY1_Processing( void );
void KEY2_Processing( void );
void KEY3_Processing( void );
void KEY4_Processing( void );

void GPIO_Init(void) ;

/******************************************************************************
* Local variables
******************************************************************************/

/******************************************************************************
* Global variables
******************************************************************************/
volatile unsigned char motor_state; 
volatile unsigned char direction; 

/*control parameters */
int max_error =  50;    /* This is the maximum error of diference between desired speed (speed_sp) and actual speed (speed) */
unsigned char error_div = 30;   /* This variable is used to tune the system */
unsigned int timer_control_val = 1; /* This variable is used to tune the system. This variable sets the base time for speed control to execute */
volatile unsigned int timer_control = 1;     /* This variable sets the base time for speed control to execute */
int error;        /* This variable shows the error between speed_sp and speed variables */
volatile int speed_sp = 0;     /* This is the desired speed */
int pwm_out = 1000;            /* Control desired output 50% = 0 RPM */


/*PWM parameters */
int pwm_modulo = PWM_MODULO_VALUE;    /* PWM period = 2 * PWM_MODULO_VALUE */
int pwm_min    = PWM_MINIMUN_VALUE;   
int pwm_max    = PWM_MAXIMUN_VALUE;



/*conmutation parameter*/
unsigned char new_motor_position;
unsigned char motor_position;
unsigned int  pwm_value = 1000;            /* This is the amount of energy in a PWM signal */

/*motor parameters*/
unsigned int period = 1;                /* This variable is used to store the period of Hall-sensor 0 every time is active */ 
int speed = 0;                       /* speed variable is signed because the operations to calculate it and the speed loop */
                                     /* control may use negative speeds */

/* motor control flags */
unsigned char stop_motor;           /* Variable used in the state machine to stop the motor */
unsigned char new_position, last_position;
unsigned int led_blink = 0;         /* This variable is used as a timer to make a LED to blink */  

/* keypad states */
unsigned char bPAD1=0,bPAD2=0;
unsigned char bPAD3=0,bPAD4 =0;
/******************************************************************************
* Local functions
******************************************************************************/
void GPIO_Init(void) 
{ 
  
/* Enable Switch as outputs*/
  Q4_BOTTOM_A_DD = 1;
  Q5_BOTTOM_B_DD = 1;
  Q6_BOTTOM_C_DD = 1;
  
  PORT_PTFOE_PTFOE6 = 0;
    PORT_PTFOE_PTFOE5 = 0;
    PORT_PTFOE_PTFOE4 = 0;
    PORT_PTFIE_PTFIE6 = 1;
    PORT_PTFIE_PTFIE5 = 1;
    PORT_PTFIE_PTFIE4 = 1;  
}


unsigned int m_uiTemp;
/******************************************************************************
* Global functions
******************************************************************************/
void main(void) {
    
  /* 
   * Initialize system including clocks
   */
  Sys_Init();
  GPIO_Init();
  /* Initialize user LEDs */
  LED0_Init();
  LED1_Init();
  LED2_Init();
  LED3_Init();  
  
  /*
   * Call your routines
   */
  // printf("Welcome to PT60 BLDC Demo!\n");
  
  /* Now it is safe to enable interrupts 
   *  
   */
  /* Initialize buttons */
  Button_Init();
  FTM1_Init_Capt();
  FTM2_Init_PWM();
  
  /* Initialize TSI */
  TSI_Init();
  
  // FreeMASTER initialization
  //FMSTR_Init();
 
  //EnableInterrupts; /* enable interrupts */
  direction == DIRECTION_CW ;
  stop_motor = 1;
  timer_control =0;
  speed_sp = MIN_SPEED;
  motor_state = MOTOR_STATE_STANDBY; 
  

  for(;;) 
  {
  // The FreeMASTER poll call must be called in the main application loop
	// to handle the communication interface and protocol. 
	// In LONG_INTR FreeMASTER interrupt mode, all the processing is done 
	// during the communication interrupt routine and the FMSTR_Poll() is 
	// compiled empty.
	//FMSTR_Poll();
  TSI_Processing(); /* must be called periodically to detect touch pad states */
  
  if (SW_RUN_STOP == 0)  
    motor_state = MOTOR_STATE_STANDBY;   /* If the switch SW5 is in OFF postion then the motor state machine goes to Standby state */ 
                                                               
  switch (motor_state)                                         /* Execute the the CASE depending on motor state value */ 
  {
    
    case MOTOR_STATE_STANDBY:               /* STANDBY STATE. Verify which push button is pressed first */
      stop_motor = 1;     
      commutation(0,100);
      speed_sp = MIN_SPEED;
      if (PUSH_BUTTON1)      /* If push button 1 was pressed first, then the motor will sipin in clockwise direction */
      {
    	bPAD1 = 0;
        direction = DIRECTION_CW;
        motor_state = MOTOR_STATE_STARTUP;  /* Put the motor state into Run state */
      }
       
      if (PUSH_BUTTON2)      /* If push button 2 was pressed first, then the motor will sipin in counter clockwise direction */
      {
    	bPAD2 = 0;
        direction = DIRECTION_CCW;
        motor_state = MOTOR_STATE_STARTUP;  /* Put the motor state into Run state */
      }
      speed = 0;
      break;
      
    case MOTOR_STATE_STARTUP:               /* STARTUP STATE */
      speed = 0;
      stop_motor = 0;
      pwm_out = 1000;                       /* This is to avoid that the first time speed control is executed pwm_value be zero */
      motor_state = MOTOR_STATE_RUN;        /* Put the motor state into Run state */
      break;
                                            /* RUN STATE */
    case MOTOR_STATE_RUN:
      speed_control();                      /* Execute the speed control algorithm */
      new_position = get_motor_position();  /* Get the new motor position */ 
      if (new_position != last_position)    /* If the new position is different than old position make a new commutation */
      {                                     /* else do nothing */
        commutation(new_position,2);   
        last_position = new_position;
      }      
      speed  = (TIMER_FREQ / period >> 2);  /* Speed Calculation */
      break;
                                            
    case MOTOR_STATE_FAULT:                 /* FAULT STATE */
      stop_motor = 1;     
      commutation(0,10);
      speed = 0;
     break;
  }
   
  
  LED1 = T0_HALL_1;           /* The LED will turn on if the Hall sensor 1 is active */
  LED2 = T1_HALL_2;           /* The LED will turn on if the Hall sensor 2 is active */
  
  if (direction == DIRECTION_CW) 
  {
    
    LED3 = 0;                 /* If motor spins in clockwise direction the LED4 will turn on */
  }
  
  else if ((led_blink==0) && (direction == DIRECTION_CCW)) 
  {
    
    LED3 ^= 1;          /* If motor spins in counter clockwise direction the LED4 will blink */
    led_blink = 1000;         /* Counter used as a delay for blinkink LED */
  } 
  
  else if (stop_motor) 
    LED3 = 0; /* If motor stops the LED6 will turn off */

  else led_blink--;           /* Decrement counter */
  
  } /* loop forever */
   
  /* please make sure that you never leave main */
}


/*****************************************************************************//*!
+FUNCTION----------------------------------------------------------------
* @function name: FTM2_Init_PWM
*
* @brief initialize FTM2 to center aligned which is used to control motor
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/

void FTM2_Init_PWM(void) 
{
  //FTM2_MODE_FTMEN = 1;					/* use enhance feature */
  FTM2_C0SC = FTM2_C0SC_ELSB_MASK;      /* FTM2 channel 0 is configured to work in Center-aligned PWM mode (High true pulses). Transistor Q4_BOTOM_A_PWM */
  FTM2_C0V = 0x0000;                    /* Counter value is 0 */
  FTM2_C1SC = FTM2_C1SC_ELSA_MASK;      /* FTM2 channel 1 is configured to work in Center-aligned PWM mode (Low true pulses). Transistor Q1_TOP_A_PWM */
  FTM2_C1V = 0x0000;                    /* Counter value is 0 */
  FTM2_C2SC = FTM2_C2SC_ELSB_MASK;      /* FTM2 channel 2 is configured to work in Center-aligned PWM mode (High true pulses). Transistor Q2_TOP_B_PWM */
  FTM2_C2V = 0x0000;                    /* Counter value is 0 */
  FTM2_C3SC = FTM2_C3SC_ELSA_MASK;      /* FTM2 channel 3 is configured to work in Center-aligned PWM mode (Low true pulses). Transistor Q5_BOTOM_B_PWM */
  FTM2_C3V = 0x0000;                    /* Counter value is 0 */
  FTM2_C4SC = FTM2_C4SC_ELSB_MASK;      /* FTM2 channel 4 is configured to work in Center-aligned PWM mode (Low true pulses). Transistor Q5_BOTOM_B_PWM */
  FTM2_C4V = 0x0000;                    /* Counter value is 0 */
  FTM2_C5SC = FTM2_C5SC_ELSA_MASK;      /* FTM2 channel 5 is configured to work in Center-aligned PWM mode (Low true pulses). Transistor Q5_BOTOM_B_PWM */
  FTM2_C5V = 0x0000;                    /* Counter value is 0 */
  
  FTM2_MOD = pwm_modulo;                /* Initialize Timer modulo with 2500 */ 
  FTM2_SC = (FTM2_SC_CLKS0_MASK|FTM2_SC_CPWMS_MASK)
          |  FTM2_SC_TOIE_MASK ;         /* TPM clock source is bus clock. All channels will operate in center-aligned mode. Timer overflow enabled */
}



/**
 * \brief   This is the FTM1 initialization module. FTM1 channel 1 is configured to work in input capture
 * \n       mode (detects falling edge only). 
 * \author  William Jiang
 */

void FTM1_Init_Capt(void) 
{
  
  FTM1_C1SC = FTM1_C1SC_ELSB_MASK | FTM1_C1SC_CHIE_MASK; /* Input capture mode enabled and Channel 1 interrupt enabled */
  FTM1_SC = (FTM1_SC_PS2_MASK | FTM1_SC_PS0_MASK|          /* FTM clock source is bus clock divided by 32 */
            FTM1_SC_CLKS0_MASK );                       /* bus clock /32 of bus frequency */
}


                                    
/**
 * \brief   This is a delay function. The function receives an unsigned int value.
 * \author  William Jiang
 * \param   time - This is the final count value (from 0 to time)
 */

void delay (unsigned int time) 
{

  unsigned int a;
  for (a = 0; a <= time; a ++);
}

                                  
/**
 * \brief   The function gets the motor position, reading the hall-sensors. The function returns the motor position
 * \n       The hall sensor reading can not be 0 or 7, at least 1 sensor is always on, but not the three of it
 * \author  William Jiang
 * \return  motor_position - Motor position according to hall sensor.
 */

unsigned char get_motor_position(void) 
{

  unsigned char motor_position = 0;
  if(T0_HALL_1) 
    motor_position = motor_position |0x04;  /* If Hall sensor is active, store 4 in motor_position variable */
  if(T1_HALL_2) 
    motor_position = motor_position |0x02;  /* If Hall sensor is active, store 2 in motor_position variable */
  if(T2_HALL_3) 
    motor_position = motor_position |0x01;  /* If Hall sensor is active, store 1 in motor_position variable */

  if (direction == DIRECTION_CCW)
    motor_position = (~motor_position) & 0x07;

  return motor_position;
}


/**
 * \brief   The function is used to commutate between six transistors.
 * \author  William Jiang
 * \param   valor - This is the value of the step to be commutated. There are six steps for commutation
 * \n       Time_on - This parameter is used in delay function
 */    

void commutation(unsigned char value, unsigned int time_on) 
{

  Q1_TOP_A_HIGH_TRUE_PULSES;
  Q2_TOP_B_HIGH_TRUE_PULSES; 
  Q3_TOP_C_HIGH_TRUE_PULSES;
  Q4_BOTTOM_A_HIGH_TRUE_PULSES;
  Q5_BOTTOM_B_HIGH_TRUE_PULSES;          
  Q6_BOTTOM_C_HIGH_TRUE_PULSES;
  
  Q3_TOP_C_PWM    (0);           /* Turn off transistors */
  Q6_BOTTOM_C_PWM (0);           /* Turn off transistors */
  Q2_TOP_B_PWM    (0);           /* Turn off transistors */
  Q5_BOTTOM_B_PWM (0);           /* Turn off transistors */
  Q1_TOP_A_PWM    (0);           /* Turn off transistors */
  Q4_BOTTOM_A_PWM (0);           /* Turn off transistors */
      
                                         
  value &= 7;
  
  switch (value) 
  {
    case 5:              /* If motor position is 5 then turn Q1 and Q5 transistors on */ 
      Q1_TOP_A_HIGH_TRUE_PULSES;      /* Configure PWM in low true pulses */ 
      Q5_BOTTOM_B_HIGH_TRUE_PULSES;   /* Configure PWM in low true pulses */
      Q4_BOTTOM_A_LOW_TRUE_PULSES;   /* Configure PWM in high true pulses */
      Q2_TOP_B_LOW_TRUE_PULSES;      /* Configure PWM in high true pulses */ 
      Q3_TOP_C_PWM    (0);           /* Turn off transistors */
      Q6_BOTTOM_C_PWM (0);           /* Turn off transistors */
      Q1_TOP_A_PWM    (pwm_value);
      Q4_BOTTOM_A_PWM (pwm_value + DEAD_TIME);
      Q2_TOP_B_PWM    (pwm_value + DEAD_TIME);
      Q5_BOTTOM_B_PWM (pwm_value);
      break;
    
    case 4:              /* If motor position is 4 then turn Q1 and Q6 transistors on */ 
      Q1_TOP_A_HIGH_TRUE_PULSES;      /* Configure PWM in low true pulses */ 
      Q6_BOTTOM_C_HIGH_TRUE_PULSES;   /* Configure PWM in low true pulses */
      Q4_BOTTOM_A_LOW_TRUE_PULSES;   /* Configure PWM in high true pulses */ 
      Q3_TOP_C_LOW_TRUE_PULSES;      /* Configure PWM in high true pulses */
      Q2_TOP_B_PWM    (0);           /* Turn off transistors */
      Q5_BOTTOM_B_PWM (0);           /* Turn off transistors */
      Q1_TOP_A_PWM    (pwm_value);
      Q4_BOTTOM_A_PWM (pwm_value + DEAD_TIME);
      Q3_TOP_C_PWM    (pwm_value + DEAD_TIME);
      Q6_BOTTOM_C_PWM (pwm_value);   
      break;
      
    case 6:              /* If motor position is 6 then turn Q2 and Q6 transistors on */
      Q2_TOP_B_HIGH_TRUE_PULSES;      /* Configure PWM in low true pulses */ 
      Q6_BOTTOM_C_HIGH_TRUE_PULSES;   /* Configure PWM in low true pulses */
      Q5_BOTTOM_B_LOW_TRUE_PULSES;   /* Configure PWM in high true pulses */ 
      Q3_TOP_C_LOW_TRUE_PULSES;      /* Configure PWM in high true pulses */
      Q1_TOP_A_PWM    (0);           /* Turn off transistors */
      Q4_BOTTOM_A_PWM (0);           /* Turn off transistors */
      Q2_TOP_B_PWM    (pwm_value);
      Q5_BOTTOM_B_PWM (pwm_value + DEAD_TIME);
      Q3_TOP_C_PWM    (pwm_value + DEAD_TIME);
      Q6_BOTTOM_C_PWM (pwm_value);   
      break;
      
    case 2:              /* If motor position is 2 then turn Q2 and Q4 transistors on */ 
      Q2_TOP_B_HIGH_TRUE_PULSES;      /* Configure PWM in low true pulses */      
      Q4_BOTTOM_A_HIGH_TRUE_PULSES;   /* Configure PWM in low true pulses */ 
      Q1_TOP_A_LOW_TRUE_PULSES;      /* Configure PWM in high true pulses */ 
      Q5_BOTTOM_B_LOW_TRUE_PULSES;   /* Configure PWM in high true pulses */
      Q3_TOP_C_PWM    (0);           /* Turn off transistors */
      Q6_BOTTOM_C_PWM (0);           /* Turn off transistors */             
      Q1_TOP_A_PWM    (pwm_value + DEAD_TIME);                              
      Q4_BOTTOM_A_PWM (pwm_value);                                          
      Q2_TOP_B_PWM    (pwm_value);                                          
      Q5_BOTTOM_B_PWM (pwm_value + DEAD_TIME);                               
      break;
      
    case 3:              /* If motor position is 3 then turn Q3 and Q4 transistors on */ 
      Q3_TOP_C_HIGH_TRUE_PULSES;      /* Configure PWM in low true pulses */
      Q4_BOTTOM_A_HIGH_TRUE_PULSES;   /* Configure PWM in high true pulses */
      Q1_TOP_A_LOW_TRUE_PULSES;      /* Configure PWM in high true pulses */ 
      Q6_BOTTOM_C_LOW_TRUE_PULSES;   /* Configure PWM in low true pulses */ 
      Q2_TOP_B_PWM    (0);           /* Turn off transistors */
      Q5_BOTTOM_B_PWM (0);           /* Turn off transistors */             
      Q1_TOP_A_PWM    (pwm_value + DEAD_TIME);                              
      Q4_BOTTOM_A_PWM (pwm_value);                                          
      Q3_TOP_C_PWM    (pwm_value);                                          
      Q6_BOTTOM_C_PWM (pwm_value + DEAD_TIME);                               
      break;
      
    case 1:              /* If motor position is 1 then turn Q3 and Q5 transistors on */ 
      Q3_TOP_C_HIGH_TRUE_PULSES;      /* Configure PWM in low true pulses */ 
      Q5_BOTTOM_B_HIGH_TRUE_PULSES;   /* Configure PWM in low true pulses */
      Q2_TOP_B_LOW_TRUE_PULSES;      /* Configure PWM in high true pulses */ 
      Q6_BOTTOM_C_LOW_TRUE_PULSES;   /* Configure PWM in high true pulses */
      Q1_TOP_A_PWM    (0);           /* Turn off transistors */
      Q4_BOTTOM_A_PWM (0);           /* Turn off transistors */
      Q2_TOP_B_PWM    (pwm_value + DEAD_TIME);
      Q5_BOTTOM_B_PWM (pwm_value);
      Q3_TOP_C_PWM    (pwm_value);
      Q6_BOTTOM_C_PWM (pwm_value + DEAD_TIME);
      break;
  }
  
  delay(time_on);
}




/** * \brief   This routine does the speed control loop
 * \n       
 * \author  William Jiang
 *    pwm_out  = pwm_out + limited(error/x)
 */
 
void speed_control(void) 
{ 
  if (timer_control == 0) 
    {
      error = (speed_sp - speed) / error_div;
      if (error > max_error) 
        error = max_error;
      if (error < -max_error) 
        error = -max_error;     
      
      pwm_out = pwm_out + error;     
      
      if (pwm_out > pwm_max) 
        pwm_out = pwm_max;
      if (pwm_out <= pwm_min) 
        pwm_out = pwm_min;
    
     pwm_value = pwm_out;
     timer_control = timer_control_val;
    }
}


/**
 * \brief   The keyboard interrupt service routine verify if the speed set point will increase or decrease.
 * \n       The number 38 is the vector number associated to KBI module. 
 * \author  William
 */


interrupt VectorNumber_Vkbi1 void KBI1_ISR(void) 
{
  KBI1_SC_KBACK = 1;   /* Clear the KBI interrupt flag */
}


/**
 * \brief   Within the FTM1 channel 1 interrupt service routine, the channel gets the number of input captures
 * \n       that are used to calculate the period. The number 6 is the vector number associated to FTM1 module
 * \n       channel 1. For further information please refer to Chapter 5 of the Reference Manual
 * \author  Ulises Corrales, Luis Puebla
 */

interrupt VectorNumber_Vftm1ch1 void  FTM1_CH1_ISR (void) 
{
  
  static unsigned int TICsz = 0;   /* This variable is used to store the number of input captures */

  (void)FTM1_C1SC;   
  FTM1_C1SC_CHF = 0;        /* Clear TPM3 channel 0 interrupt flag */
  period = FTM1_C1V - TICsz; /* period variable is used to calculate the speed in rpms. The period is the */
  TICsz = FTM1_C1V;          /* number of input captures minus the last input capture */    
}

/**
 * \brief   The TPM1 interrupt service routine decrements timer_control varible. When timer_control variable
 * \n       is 0, the speed control routine is executed. The number 11 is the vector number associated to timer overflow of the TPM1 module. 
 * \n       For further information please refer to Chapter 5 of the Reference Manual
 * \author  Ulises Corrales, Luis Puebla
 */
interrupt VectorNumber_Vftm2ovf void FMT2_ISR(void) 
{  
  
  FTM2_SC_TOF = 0;                       /* Clear Timer overflow flag */
  if (timer_control) 
    timer_control--;   /* Decrement timer_control variable */
}



/*****************************************************************************//*!
+FUNCTION----------------------------------------------------------------
* @function name: KEY1_Processing
*
* @brief handles touch pad1 event. When the touch pad1 is touched, increase/decrease
* 	the motor speed by a step depending on the motor rotation direction.
* 	it also toggles LED3.
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/


void KEY1_Processing( void )
{
	LED3_Toggle();
	//you can add code to handle corresponding function
	bPAD1 = 1;
	 if (direction == DIRECTION_CW) 
	  {
		timer_control =0;
		if (speed_sp >= MAX_SPEED) 
		  speed_sp = MAX_SPEED;
		else 
		  speed_sp = speed_sp + STEP;
	  }	  
	  else
	  {                  
		  timer_control =0;
		  if (speed_sp <= MIN_SPEED){    
			direction == DIRECTION_CW;       
			speed_sp = MIN_SPEED;
			motor_state = MOTOR_STATE_STANDBY;
		  }
		  else
		  {
		    speed_sp = speed_sp - STEP; 
		  }
	  }		
}

/*****************************************************************************//*!
+FUNCTION----------------------------------------------------------------
* @function name: KEY2_Processing
*
* @brief handles touch pad2 event. When the touch pad is touched, decrease/increase
* 	the motor speed by a step depending on the motor rotation direction. Its function
* 	is reverse of the touch pad1.
* 	it also toggles LED3.
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/

void KEY2_Processing( void )
{
	LED3_Toggle();
	//you can add code to handle corresponding function
	bPAD2 = 1;
	 if (direction == DIRECTION_CW) 
	  { 
	      timer_control =0;
	      if (speed_sp <= MIN_SPEED){
	        direction == DIRECTION_CCW;       
	        speed_sp = MIN_SPEED;
	        motor_state = MOTOR_STATE_STANDBY; 
	      }
	      else 
	        speed_sp = speed_sp - STEP;	    
	  }	  
	  else
	  {
	    timer_control =0;
	    
	    if (speed_sp >= MAX_SPEED) 
	      speed_sp = MAX_SPEED;
	    else 
	      speed_sp = speed_sp + STEP;
	  }		
}

/*****************************************************************************//*!
+FUNCTION----------------------------------------------------------------
* @function name: KEY3_Processing
*
* @brief handles touch pad3 event. no job to do...
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/

void KEY3_Processing( void )
{
	//you can add code to handle corresponding function
	 bPAD3 = 1;
}

/*****************************************************************************//*!
+FUNCTION----------------------------------------------------------------
* @function name: KEY4_Processing
*
* @brief handles touch pad2 event. no job to do.
*        
* @param  none
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/

void KEY4_Processing( void )
{
	//you can add code to handle corresponding function
	 bPAD4 = 1;
}
