/******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2011-2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
***************************************************************************//*!
*
* @file SPI.c
*
* @author B37531
*
* @version 0.0.1
*
* @date Aug 23, 2011
*
* @brief application entry point which performs application specific tasks. 
*
******************************************************************************/
#include "common.h"
#include "spi.h"
#include "string.h"
#include "printf.h"


/******************************************************************************
* Local variables
******************************************************************************/
static unsigned char m_ucSPI0_TxBuff[16];
static unsigned char m_ucSPI0_RxBuff[16];
static unsigned char m_ucSPI1_TxBuff[16];
static unsigned char m_ucSPI1_RxBuff[16];


/******************************************************************************
* Local functions
*****************************************************************************/
void SPI0_RdWrByte( unsigned char* pWrBuff,unsigned char* pRdBuff,unsigned short uiLength );

/******************************************************************************
* Local function prototypes
******************************************************************************/


/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: SPI0_Init
   *
   * @brief initialize SPI0 to master mode
   *        
   * @param  none 
   *
   * @return none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void SPI0_Init( void )
{
	volatile char dummy;
	
	CS_PIN_Init();
	// SPI0 initial
	//SYS_SOPT1_SPI0PS  = 1;     // SPI0 SPSCK0, MOSI0, MISO0, and SS0 are mapped on PTE0, PTE1, PTE2, and PTE3.
	SPI0_C1_MSTR = 1;            // master
	SPI0_BR = 0x34;              // Baud rate divisor is 2,Baud rate prescaler divisor is 1                              
	dummy = SPI0_D;
	SPI0_C1_SPE  = 1;            // en
}

/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: SPI1_Init
   *
   * @brief initialize SPI1 to slave mode,enable FIFO
   *        
   * @param  none 
   *
   * @return none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void SPI1_Init( void )
{
	unsigned char i;
	volatile char dummy;
	// SPI1 initial
	SPI1_C1_MSTR = 0;            // slave
	SPI1_C2_SPIMODE = 0;         // 8 bit, SPI hardware match interrupt disable,polling
	//SPI1_BR = 0x78;            // Baud rate divisor is 2,Baud rate prescaler divisor is 1
	SPI1_C3_FIFOMODE = 1;        // FIFO enabled
	SPI1_S |= SPI1_S_SPTEF_MASK;
	dummy = SPI1_DL;
	dummy = SPI1_DH;
	SPI1_C1_SPE = 1;             // enable
	
	
}

/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: SPI0_RdWrByte
   *
   * @brief implement read/write data from SPI0
   *        
   * @param  *pWrBuff   -- write data buffer pointer 
   *         *pRdBuff   -- read  data buffer pointer
   *         uiLength   -- read/write data length
   *
   * @return none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void SPI0_RdWrByte( unsigned char* pWrBuff,unsigned char* pRdBuff,unsigned short uiLength )
{
	unsigned short i;
	for( i=0;i<uiLength;i++)
	{
		SPI0_S |= SPI0_S_SPTEF_MASK;
		SPI0_D = pWrBuff[i];
		while(!SPI0_S_SPRF);
		pRdBuff[i] = SPI0_D;
	}
	
	
}
/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: Test_SPI
   *
   * @brief implement SPI test, SPI0 as master, SPI1 as slave
   *        SPI0 write a frame data  to SPI1, verify if correct
   *        SPI0 read a frame data from SPI1,verify if correct
   *        
   * @param  none 
   *
   * @return none
   *
   * @ Pass/ Fail criteria: return test status
   *                        0x01 --- write fail
   *                        0x02 --- read fail
   *                        0x03 --- read and write fail
   *                        0x00 --- test success
   *****************************************************************************/
unsigned char Test_SPI( void )
{
	unsigned char i;
	unsigned char bTestResult = 0;
	unsigned short uiCount;
	
	for( i = 0; i< 16; i++ )
	{
		m_ucSPI0_TxBuff[i] = i + 5;
		m_ucSPI1_TxBuff[i] = i + 7;
	}
	CS_LOW();
	SPI0_RdWrByte(&m_ucSPI0_TxBuff[0],&m_ucSPI0_RxBuff[0],8);
	
	uiCount = 0xffff;
	while((!SPI1_S_SPRF)&&(--uiCount ));
	if( uiCount == 0 )
	{
		bTestResult |= 0x02;
	}
	else
	{
		i = 0;
		while(!SPI1_S_RFIFOEF)
		{
			m_ucSPI1_RxBuff[i++] = SPI1_DL;
		}
		
		// write data to SPI1 FIFO,wait SPI0 read data
		i = 0;
		while( !SPI1_S_TXFULLF )
		{
			SPI1_DL = m_ucSPI1_TxBuff[i++];
		}
		
		SPI0_RdWrByte(&m_ucSPI0_TxBuff[0],&m_ucSPI0_RxBuff[0],8);
		
		bTestResult = 0;
		if( memcmp(&m_ucSPI0_TxBuff[0],&m_ucSPI1_RxBuff[0], 8 ) )
		{
			
			bTestResult |= 0x01;
		}
		if( memcmp(&m_ucSPI1_TxBuff[0],&m_ucSPI0_RxBuff[0], 8 ) )
		{
			bTestResult |= 0x02;
		}
	}
	
	CS_HIGH();
	switch( bTestResult )
	{
		case 0x01:
		{
			printf("Error: SPI0 failed to write data to SPI1!\n ");
			LED1_Toggle();
			LED2_Off();
			LED3_Off();
			LED0_Off();
		}
		break;
		case 0x02:
		{
			printf("Error: SPI0 failed to read data from SPI1!\n ");
			LED2_Toggle();
			LED1_Off();
			LED3_Off();
			LED0_Off();
		}
		break;
		case 0x03:
		{
			printf("Error: SPI0 failed to communicate with SPI1!\n ");
			LED2_Toggle();
			LED1_Toggle();
			LED3_Off();
			LED0_Off();
		}
		break;
		case 0x00:
		{
			printf("SPI communication is success!\n");
					
			LED3_Toggle();
			LED0_Off();
			LED1_Off();
			LED2_Off();
		}
		break;
		default:
			break;
	}
	
	return bTestResult;
}
