/******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2011-2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
***************************************************************************//*!
*
* @file tsi.c
*
* @author B37531
*
* @version 0.0.1
*
* @date Aug 23, 2011
*
* @brief application entry point which performs application specific tasks. 
*
*******************************************************************************
*
* provide TSI driver,complete electrode scan and processing,support stop mode
******************************************************************************/
#include <hidef.h> /* for EnableInterrupts macro */
#include "derivative.h" /* include peripheral declarations */
#include "common.h"
#include "tsi.h"
#include "printf.h"
#include "mtim.h"
/******************************************************************************
* Constants and macros
******************************************************************************/

/******************************************************************************
* Global variables
******************************************************************************/
unsigned char g_KeyCheckMode;
unsigned long g_ulCurrentTimeCount = 0;
/******************************************************************************
* Local variables
******************************************************************************/
static ELECTRODE_WORK_INFO  m_ElecWorkInfo[ELECTRODE_NUM];
static unsigned short m_uiScanValue[ELECTRODE_NUM];
static unsigned char m_ucSignalChangeFlag;
static unsigned char m_bScanCompleteFlag;
static unsigned char m_ucChannelIndex;
static unsigned char m_ucTSI_WorkState;
static unsigned char m_ucMtim0IntCount;
static unsigned long m_ulIdleTimeCount = 0;
void (* const KeyHandle[ELECTRODE_NUM])(void) =
{
	KEY1_Processing,	/* call back functions defined externally */
	KEY2_Processing,
	KEY3_Processing,
	KEY4_Processing,
};
/*****************************************************************************
* Local types
******************************************************************************/



/******************************************************************************
* Local variables
******************************************************************************/

/******************************************************************************
* Local functions
*****************************************************************************/
void TSI_VaribleInit( void );
void TSI_Init( void );
void TSI_Scan( unsigned short *pResult );
void TSI_Processing( void );
void TSI_CheckState( void );
void RTC_Init( void );
/******************************************************************************
* Global functions
******************************************************************************/
void TSI_Init( void );
void TSI_Processing( void );
void KEY1_Processing( void );
void KEY2_Processing( void );
void KEY3_Processing( void );
void KEY4_Processing( void );
void TSI_Check( void );
void TSI_EnterIntoStopMode( void );
void MCU_EnterStopMode( void );
/******************************************************************************
* Local function prototypes
******************************************************************************/
/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: TSI_VaribleInit
   *
   * @brief initialize TSI variables
   *        
   * @param  none 
   *
   * @return none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void TSI_VaribleInit( void )
{
	unsigned char i;
	m_ucChannelIndex = 0;
	m_ucTSI_WorkState = TSI_WORK_STATE_NULL;
	g_KeyCheckMode = KEY_CHECK_MODE_NORMAL;
	for(i=0;i<ELECTRODE_NUM;i++)
	{
		m_ElecWorkInfo[i].State = KEY_RELEASED;
#if( BOARD_TYPE == EVB_BOARD )
		m_ElecWorkInfo[i].Channel = i;          //enable channel 0,1,2,3
#endif
		m_ElecWorkInfo[i].Count = 0;
	}
#if( BOARD_TYPE == TOWER_BOARD )
		m_ElecWorkInfo[0].Channel = 0x0f;          
		m_ElecWorkInfo[1].Channel = 0x0e;  
		m_ElecWorkInfo[2].Channel = 0x01;  
		m_ElecWorkInfo[3].Channel = 0x00;  
#endif
}
/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: TSI_Init
   *
   * @brief TSI module initialization
   *        
   * @param  none 
   *
   * @return none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void TSI_Init( void )
{
	unsigned char i,j;
	
	TSI_VaribleInit();
	//
	TSI_CS1_PS = TSI_CS1_PS_16;
	TSI_CS1_NSCN = TSI_CS1_NSCN_32;
	TSI_CS2 = TSI_CS2_REFCHRG_32uA | TSI_CS2_DVOLT_00 | TSI_CS2_EXTCHRG_32uA;
	
	// enable channel
	TSI_PEN0 = TSI_PEN0_VALUE;
	TSI_PEN1 = TSI_PEN1_VALUE;
	
	// enable stop mode
	TSI_CS0_STPE = 1;
	// enable interrupt
#if( TSI_WORK_MODE != TSI_WORK_MODE_POLLING)

	TSI_CS0_TSIIEN = 1;
	// init RTC
	RTC_Init();
	MTIM0_Init();
	m_ucChannelIndex = 0;
	//select current channel 
	TSI_CS3_TSICH = m_ElecWorkInfo[m_ucChannelIndex].Channel;
#endif
	// enable TSI
	TSI_CS0_TSIEN = 1;

#if 0	
#if( BOARD_TYPE == TOWER_BOARD )
	  LED0_Init();
	  LED1_Init();
	  LED2_Init();
	  LED3_Init(); 
#endif
#endif	
	// Get baseline
#if( TSI_WORK_MODE == TSI_WORK_MODE_POLLING)
	  TSI_Scan( &m_uiScanValue[0] );
	  for(i=0;i<ELECTRODE_NUM;i++)
		{
			m_ElecWorkInfo[i].BaseLine = m_uiScanValue[i];
		}
		for(j=0;j<10;j++)
		{
			TSI_Scan( &m_uiScanValue[0] );
			for(i=0;i<ELECTRODE_NUM;i++)
			{
				m_ElecWorkInfo[i].BaseLine = (m_uiScanValue[i]+m_ElecWorkInfo[i].BaseLine)>>1;
			}
		}
#else

#endif
	//
}
/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: TSI_Scan
   *
   * @brief polling method to complete scanning all of electrode
   *        
   * @param  none 
   *
   * @return none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void TSI_Scan( unsigned short *pResult )
{
	unsigned char i;
	unsigned int uiScanValue;
	for(i=0;i<ELECTRODE_NUM;i++)
	{
		// specify the scanning channel	
		TSI_CS3_TSICH = m_ElecWorkInfo[i].Channel;
		// start scanning
		TSI_CS0 |= 0x01;
		// wait scan complete
		while(!TSI_CS0_EOSF);
		TSI_CS0_EOSF = 1;
		pResult[i] = TSI_CNT;
	
	}
}
/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: TSI_Processing
   *
   * @brief process the scanning data to determine the key status
   *        
   * @param  none 
   *
   * @return none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void TSI_Processing( void )
{
	unsigned char i;
	unsigned short uiDelta;
	//start scan
#if( TSI_WORK_MODE == TSI_WORK_MODE_POLLING )
	TSI_Scan( &m_uiScanValue[0] );
	// calculate delta
	for(i=0;i<ELECTRODE_NUM;i++)
	{
		m_ElecWorkInfo[i].Delta = m_uiScanValue[i] - m_ElecWorkInfo[i].BaseLine;
	}
#endif	
	// determine key state
	for(i=0;i<ELECTRODE_NUM;i++)
	{
		if( m_ElecWorkInfo[i].Delta > 0 )
		{
			uiDelta = m_ElecWorkInfo[i].Delta;
		}
		else
		{
			uiDelta = 0;
		}
		if( uiDelta > KEY_ON_SENSITIVITY )
		{
			if( m_ElecWorkInfo[i].State == KEY_RELEASED )
			{
				if(m_ElecWorkInfo[i].Count > KEY_OVERTIME)
				{
					m_ElecWorkInfo[i].State = KEY_TOUCHED;
					m_ucSignalChangeFlag |= (0x01<<i);
					m_ElecWorkInfo[i].Count = 0;
				}
				else
				{
					m_ElecWorkInfo[i].Count ++;
				}
			}
			else if( m_ElecWorkInfo[i].State == KEY_TOUCHED )
			{
				//repeat touch
				//
			}
		}
		else if( uiDelta < KEY_OFF_SENSITIVITY )
		{
			if( m_ElecWorkInfo[i].State == KEY_TOUCHED )
			{
				if(m_ElecWorkInfo[i].Count > KEY_OVERTIME)
				{
					m_ElecWorkInfo[i].State = KEY_RELEASED;
					m_ucSignalChangeFlag |= (0x01<<i);
					m_ElecWorkInfo[i].Count = 0;
				}
				else
				{
					m_ElecWorkInfo[i].Count ++;
				}
			}
			else if( m_ElecWorkInfo[i].State == KEY_RELEASED )
			{
				// always released
				//
			}
		}
		else
		{
			// is noise or other
			//
		}
	}
	// check key state,processing
	TSI_CheckState();
}
/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: TSI_CheckState
   *
   * @brief check the change of the key status and implement the corresponding function
   *        
   * @param  none 
   *
   * @return none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void TSI_CheckState( void )
{
	unsigned char i;
	if( m_ucSignalChangeFlag != 0 )
	{
		// update data idle time 
		m_ulIdleTimeCount = g_ulCurrentTimeCount;
		//key state occur change,handle corresponding key
		for(i=0;i<ELECTRODE_NUM;i++)
		{
			if( m_ucSignalChangeFlag&(0x01<<i) )
			{
				//key 0 release or touched
				if( m_ElecWorkInfo[i].State == KEY_TOUCHED)
				{
					//
					KeyHandle[i]();
				}
				//clear flag
				m_ucSignalChangeFlag &= (~(0x01<<i)); 
			}
		}
	}
	else
	{

	}
}

/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: TSI_ISR
   *
   * @brief TSI interrupt service routine 
   *        
   * @param  none 
   *
   * @return none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
interrupt VectorNumber_Vtsi void TSI_ISR ( void )
{
	if( TSI_CS0_EOSF )
	{
		TSI_CS0_EOSF = 1;
		m_uiScanValue[m_ucChannelIndex++] = TSI_CNT;
		if( m_ucChannelIndex < ELECTRODE_NUM )
		{
			//select next channel scan
			TSI_CS3_TSICH = m_ElecWorkInfo[m_ucChannelIndex].Channel;
			m_bScanCompleteFlag = 0;
			// change mode to soft triggle
			TSI_CS0_STM = 0;
			// start next channel scanning
			TSI_CS0_SWTS = 1;
		}
		else
		{
			//all of channels scan completed,setting flag
			m_bScanCompleteFlag = 1;
			
			// wait the next time hardware triggle
			m_ucChannelIndex = 0;
			//select next channel scan
			TSI_CS3_TSICH = m_ElecWorkInfo[m_ucChannelIndex].Channel;
		}
		
	}
}

/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: RTC_ISR
   *
   * @brief RTC interrupt service routine to wake MCU and as hardware trigger source to TSI 
   *        
   * @param  none 
   *
   * @return none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
interrupt VectorNumber_Vrtc void RTC_ISR ( void )
{
	if( RTC_SC1_RTIF )
	{
		RTC_SC1_RTIF = 1;
		
		g_ulCurrentTimeCount ++;
		
		//LED0_Toggle();
	}
}
/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: RTC_Init
   *
   * @brief RTC initialization function 
   *        
   * @param  none 
   *
   * @return none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void RTC_Init( void )
{
	RTC_MOD			= 10;         // 1S
	RTC_SC1_RTIE = 1;			 // enable RTC interrupt
	RTC_SC2_RTCLKS = 0x01;        //select clock source is 1KHZ
	RTC_SC2_RTCPS = 0x06;          // clock prescaler is 100
}
/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: TSI_Check
   *
   * @brief check TSI scan state and process touched action
   *        when on touch last some times,make MCU enter stop mode.
   *        when check there are touch action occur, restore to normal mode
   * 
   *        
   * @param  none 
   *
   * @return none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void TSI_Check( void )
{
	unsigned char i;
	unsigned char bKeyTouchFlag;
	if( !m_bScanCompleteFlag )
	{
		// enter into stop mode or wait mode
		if( g_KeyCheckMode != KEY_CHECK_MODE_NORMAL)
		{
			MCU_EnterStopMode();
		}
		return;
	}
	m_bScanCompleteFlag = false;
	if( m_ucTSI_WorkState != TSI_WORK_STATE_NORMAL )
	{
		for(i=0;i<ELECTRODE_NUM;i++)
		{
			if( m_ucTSI_WorkState == TSI_WORK_STATE_NULL )
			{
				m_ElecWorkInfo[i].BaseLine = m_uiScanValue[i];
			}
			else
			{
				m_ElecWorkInfo[i].BaseLine = (m_uiScanValue[i] + m_ElecWorkInfo[i].BaseLine)>>1;
			}
		}
		m_ucTSI_WorkState ++;
		if( m_ucTSI_WorkState > 3)
		{
			m_ucTSI_WorkState = TSI_WORK_STATE_NORMAL;
			
			m_ulIdleTimeCount = g_ulCurrentTimeCount;
		}
		else
		{
			return;
		}
	}
		
	bKeyTouchFlag = 0;
	for(i=0;i<ELECTRODE_NUM;i++)
	{
		m_ElecWorkInfo[i].Delta = m_uiScanValue[i] - m_ElecWorkInfo[i].BaseLine;
		
		if( ( m_ElecWorkInfo[i].Delta > 0 ) && 
			( m_ElecWorkInfo[i].Delta > KEY_ON_SENSITIVITY ) ||
			(m_ElecWorkInfo[i].State != KEY_RELEASED) )
		{
			bKeyTouchFlag = 1;
		}
	}
	if( bKeyTouchFlag )
	{
		if( g_KeyCheckMode != KEY_CHECK_MODE_NORMAL )
		{
			printf("\nwake PT60 from stop mode!\n");
			printf("restore PT60 to normal mode!\n");
			g_KeyCheckMode = KEY_CHECK_MODE_NORMAL;
		}
		TSI_Processing();
		bKeyTouchFlag = 0;
	}
	else
	{
#if( TSI_WORK_MODE == TSI_WORK_MODE_STOP )
		if( g_KeyCheckMode ==  KEY_CHECK_MODE_NORMAL )
		{
			if( (g_ulCurrentTimeCount - m_ulIdleTimeCount) >= TSI_IDLE_TIME )
			{
				printf("PT60 will enter into stop mode!\n");
				printf("if touch will wake up PT60!\n");
				m_ulIdleTimeCount = g_ulCurrentTimeCount;
				g_KeyCheckMode = KEY_CHECK_MODE_STANDBY;
				TSI_EnterIntoStopMode();
				MCU_EnterStopMode();
	
			}
		}
		else
		{
			TSI_EnterIntoStopMode();
			MCU_EnterStopMode();
	
		}
		// no touch
		
#endif
	}
}

/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: MTIMO_ISR
   *
   * @brief MTIM0 interrupt service routine, 
   *        when running in normal mode,periodically to trigger TSI to start scan.
   * 		periodically feed watchdog
   *        
   * @param  none 
   *
   * @return none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/

interrupt VectorNumber_Vmtim0 void MTIMO_ISR( void )
{
	if( MTIM0_SC_TOF )
	{
		MTIM0_SC_TOF = 0;
		m_ucMtim0IntCount ++;
		if( m_ucMtim0IntCount >= 7 ) // interval 14ms to triggle a TSI scaning 
		{
			m_ucMtim0IntCount = 0;
			if( !TSI_CS0_SCNIP )
			{
				m_ucMtim0IntCount = 0;
				if( g_KeyCheckMode == KEY_CHECK_MODE_NORMAL)
				{
					m_ucChannelIndex = 0;
					TSI_CS3_TSICH = m_ElecWorkInfo[m_ucChannelIndex].Channel;
					// start next channel scanning
					TSI_CS0_SWTS = 1;
				}
			}
		}
	}
	
	// feed watchdog
	__RESET_WATCHDOG();
}
/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: TSI_EnterIntoStopMode
   *
   * @brief when enable TSI to enter stop mode,first to set TSI_CS0_STM
   *        then wait RTC overflow to trigger scan.
   *        
   * @param  none 
   *
   * @return none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/
void TSI_EnterIntoStopMode( void )
{
	// wait TSI is idle,
	while(TSI_CS0_SCNIP);
	// change mode to hardware trigger
	TSI_CS0_STM = 1;
	m_ucChannelIndex = 0;
	TSI_CS3_TSICH = m_ElecWorkInfo[m_ucChannelIndex].Channel;
}
/*****************************************************************************//*!
   +FUNCTION----------------------------------------------------------------
   * @function name: MCU_EnterStopMode
   *
   * @brief when implement instruction "stop",with some interrupt source can't work
   *        in stop mode,need do some things to avoid them generate interrupt during 
   *        stop.
   *        when want MCU to enter stop mode,please call this function 
   *        
   * @param  none 
   *
   * @return none
   *
   * @ Pass/ Fail criteria: none
   *****************************************************************************/

void MCU_EnterStopMode( void )
{
	//@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
	_Stop;
	// delay 1.2mS to wait acquire Fll
	DelayUS(100);  
}


