//#define DISABLE_INTERRUPT
/******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2011-2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
***************************************************************************//*!
*
* @file main.c
*
* @author a13984
*
* @version 0.0.1
*
* @date Jul-15-2011
*
* @brief 
*       TWR-S08PT60 tower board test code  
*
*******************************************************************************
*
* provide a demo for how to initialize the PT60, output messages via SCI, 
* flash operations, etc.
* NOTE:
*	printf call may occupy a lot of memory (around 1924 bytes), so please
*	consider your code size before using printf.
******************************************************************************/

#include <hidef.h>      /* for EnableInterrupts macro */
#include "derivative.h" /* include peripheral declarations */
#include "common.h"
#include "ics.h"
#include "ftm.h"
#include "mtim.h"
#include "printf.h"
#include "tsi.h"
#include "ir.h"
#include "iic.h"
#include "mma8451.h"
#include "pt60_tower.h"
#include "string.h"
#include "crc.h"
/******************************************************************************
* External objects
******************************************************************************/


/******************************************************************************
* Constants and macros
******************************************************************************/
const enum {
	TEST_STATE_IDLE,
	TEST_STATE_TOUCH_PAD,			// SW2
	TEST_STATE_IRDA,				// SW2
	TEST_STATE_POTENTIOMETER,		// SW2
	TEST_STATE_ACCELEROMETER,		// SW3
	TEST_STATE_UART,				// SW3
	TEST_STATE_FAT,					// SW2
	TEST_STATE_FCC,					// SW3
	TEST_STATE_WAIT,
	TEST_STATE_IR,                  // SW3 
	TEST_STATE_SPI,
	TEST_STATE_DONE                 // SW3 
	
};
/* LEDs */
#define LED1   PORT_PTGD_PTGD0    // HALL 1 LED
#define LED2   PORT_PTGD_PTGD1    // HALL 2 LED
#define LED3   PORT_PTGD_PTGD2    // HSW - Run/Stop
#define LED4   PORT_PTGD_PTGD3    // motor spin direction

#undef	Button2_Is_Pressed()
#undef  Button3_Is_Pressed()

#define Button2_Is_Pressed()	!(buttonStates & (1<<5))
#define Button3_Is_Pressed()	!(buttonStates & (1<<4))

/******************************************************************************
* Local types
******************************************************************************/

/******************************************************************************
* Local function prototypes
******************************************************************************/

void Globals_Init(void);
void Modules_Init(void);
void IdleTask(void);
void TestTSI(void);
void TestPotentiometer(void);
void TestAccelerometer(void);
void TestUART(void);
void WaitForTestMode(void);
uint8_t LAB_StateMachine(void);
void DelayUS(word timeUS);
interrupt VectorNumber_Vkbi1 void KBI1_ISR(void);

/* TSI functions */
void KEY1_Processing( void );
void KEY2_Processing( void );
void KEY3_Processing( void );
void KEY4_Processing( void );

/*
 * accelerometer functions
 */
void MMA8451_PT60Tower_Test(void);

/******************************************************************************
* Local variables
******************************************************************************/

/******************************************************************************
* Global variables
******************************************************************************/
uint8_t   g_ucPrintfFlag =1;
/* keypad states */
unsigned char bPAD1=0,bPAD2=0;
unsigned char bPAD3=0,bPAD4 =0;

/* Test statemachine state */
uint8_t iTestState = TEST_STATE_FAT;

/* Test variables */
uint8_t  iErrCount;

/* Variables for accelerometer */ 
extern unsigned int resultx, resulty, resultz;

/******************************************************************************
* Local functions
******************************************************************************/


/******************************************************************************
* Global functions
******************************************************************************/

void main(void) {
    
  /* 
   * Initialize system including clocks
   */
  Sys_Init();
  
  /* Initialize all peripheral modules
   * 
   */
  Modules_Init();
						
  Globals_Init();
  
  EnableInterrupts; /* enable interrupts */

  // enable watchdog
  wdog_enable();
  // Endless loop to check test mode and handle test
  
  for(;;)
  {
	  LAB_StateMachine();
										
  }
  
}
/*****************************************************************************//*!
+FUNCTION----------------------------------------------------------------
* @function name: Globals_Init
*
* @brief initializes all global variables. 
*        
* @param  none 
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/

void Globals_Init(void)
{
	iTestState = TEST_STATE_FAT;
}
/*****************************************************************************//*!
+FUNCTION----------------------------------------------------------------
* @function name: Modules_Init
*
* @brief initializes all related peripheral and functional modules. 
*        
* @param  none 
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/

void Modules_Init(void)
{
	/* Initialize user LEDs */
	LED0_Init();
	LED1_Init();
	LED2_Init();
	LED3_Init();  
	 
	/* Initialize buttons */
	Button_Init();
	
	/* Enable button interrupts */
	KBI1_SC_KBIE = 1;
	
	/* Initialize TSI */
	TSI_Init();
	
	// init IR
	IR_Init();
	  
	// Init ADC
	ADC_Init();
	
	/* Initialize I2C and the accelerometer */
	init_I2C();
	MMA8451_Active();	
	
	/* Iniitialize UART (connected to SCI1) */
	Init_SCI1(BUS_CLK_HZ);
	
	SPI0_Init();
	
	SPI1_Init();

}

/*****************************************************************************//*!
+FUNCTION----------------------------------------------------------------
* @function name: KBI1_ISR
*
* @brief handles KBI1 events, i.e., SW2 and SW3 button events.
* 		 It changes test state accordingly. 
*        
* @param  none 
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
static unsigned char buttonStates;	
interrupt VectorNumber_Vkbi1 void KBI1_ISR(void) 
{
   KBI1_SC_KBACK = 1;   /* Clear the KBI interrupt flag */
   // NOTE: disable KBI pin function before reading PORT data register
   KBI1_PE = 0;
   buttonStates = PORT_PTDD;
   // NOTE: restore KBI pin function after PORT data register
   KBI1_PE = KBI1_PE_KBIPE4_MASK | KBI1_PE_KBIPE5_MASK; /* enable KBI pin */
   KBI1_SC_KBACK = 1;	
   g_ucPrintfFlag = 1;
   switch (iTestState )
  {
  case TEST_STATE_IDLE:	
	  if(Button2_Is_Pressed())
	  {
		  // move to the FAT state
		  iTestState = TEST_STATE_FAT;
	  }
	  break;
  case TEST_STATE_TOUCH_PAD:
	  if(Button2_Is_Pressed())
	  {
		  // move to potentiometer test state
		  iTestState = TEST_STATE_POTENTIOMETER;
	  }
	  break;
  case TEST_STATE_POTENTIOMETER:
	  if(Button3_Is_Pressed())
	  {
		  // move to accelerometer test state
		  iTestState = TEST_STATE_ACCELEROMETER;
	  }
	  break;
  case TEST_STATE_ACCELEROMETER:
	  if(Button3_Is_Pressed())
	  {
		  // move to UART test state
		  iTestState = TEST_STATE_UART;	  
	  }
	  break;
  case TEST_STATE_UART:
	  // stay in this last state
	  if(Button3_Is_Pressed())
	 	  {
	 		  // move to UART test state
	 		  iTestState = TEST_STATE_IR;	  
	 	  }
	  break;

  case TEST_STATE_IR:
	  if(Button3_Is_Pressed())
	  	  {
	  		  // move to UART test state
	  		  iTestState = TEST_STATE_SPI;	  
	  	  }
       break;
  case TEST_STATE_SPI:
  	  if(Button3_Is_Pressed())
  	  	  {
  	  		  // move to UART test state
  	  		  iTestState = TEST_STATE_TOUCH_PAD;	  
  	  	  }
         break;
  default:
	  break;
  }
}

/*****************************************************************************//*!
+FUNCTION----------------------------------------------------------------
* @function name: KEY1_Processing
*
* @brief handles touch pad 1 on board
*        
* @param  none 
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/

void KEY1_Processing( void )
{
	LED3_Toggle();
	//you can add code to handle corresponding function
	bPAD1 = 1;
}
/*****************************************************************************//*!
+FUNCTION----------------------------------------------------------------
* @function name: KEY2_Processing
*
* @brief handles touch pad 2 on board
*        
* @param  none 
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/

void KEY2_Processing( void )
{
	LED2_Toggle();
	//you can add code to handle corresponding function
	bPAD2 = 1;		
}

/*****************************************************************************//*!
+FUNCTION----------------------------------------------------------------
* @function name: KEY3_Processing
*
* @brief handles touch pad 3 on board
*        
* @param  none 
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/

void KEY3_Processing( void )
{
	LED1_Toggle();
	//you can add code to handle corresponding function
	 bPAD3 = 1;
}

/*****************************************************************************//*!
+FUNCTION----------------------------------------------------------------
* @function name: KEY4_Processing
*
* @brief handles touch pad 4 on board
*        
* @param  none 
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/

void KEY4_Processing( void )
{
	LED0_Toggle();
	//you can add code to handle corresponding function
	 bPAD4 = 1;
}
/*****************************************************************************//*!
+FUNCTION----------------------------------------------------------------
* @function name: LAB_StateMachine
*
* @brief state machine for LAB demo
*        
* @param  none 
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/

uint8_t LAB_StateMachine(void)
{
	uint8_t status = 0;
    uint8_t bFlag;
	
	for(;;)
	{
	  DisableInterrupts;
	  status = iTestState;
	  bFlag = g_ucPrintfFlag;
	  g_ucPrintfFlag = 0;
	  EnableInterrupts;
	  
	  switch ( status )
	  {
	  case TEST_STATE_IDLE:	
		  // Hanlde IDLE state
		  break;
	  case TEST_STATE_FAT:
		  // Show test banner on Terminal
		  if( bFlag )
		  {
			  printf("Welcome to TWR-S08PT60 demo show!\n");	
		  }
		  // Move to touch pad test state
		  iTestState = TEST_STATE_TOUCH_PAD;
		  g_ucPrintfFlag = 1;
		  break;
	  case TEST_STATE_TOUCH_PAD:
		  if( bFlag )
		  {
			  printf("\n");
			  printf("Test touch pad, touch any pad and see its LED.\n");
			  printf("If no touch lasting for 5 seconds, MCU will enter into stop mode!\n");
			  printf("When touch occurs, will wake up MCU and restore normal running!\n");
			  printf("Press and release SW2 to enter potentiometer demo.\n");
			  
			  g_KeyCheckMode = KEY_CHECK_MODE_NORMAL;
		  }
		  // Handle touch pad events
		  TestTSI(); /* must be called periodically to detect touch pad states */
		  break;
	  case TEST_STATE_POTENTIOMETER:
		  if( bFlag )
		  {
			  printf("\n");
			  printf("Test potentiometer, rotate potentiometer and see LED changes.\n");
			  printf("Press and release SW3 to enter accelerometer demo.\n");
			  
		  }
		  // Handle Potentiometer measurement
		  TestPotentiometer();
		  break;
	  case TEST_STATE_ACCELEROMETER:
		  if( bFlag )
		  {
			  printf("\n");
			  printf("Test accelerometer, rotate board and see LED changes.\n");
			  printf("Press and release SW3 to enter Uart demo.\n");

		  }
		  // Handle Accelerometer
		  TestAccelerometer();
		  break;
	  case TEST_STATE_UART:
		  if( bFlag )
		  {
			  printf("\n");
			  printf("Test UART, if LED flashes, it indicates error.\n");
			  printf("Press and release SW3 to enter IR demo.\n");
		  }
		  // Handle UART
		  TestUART();
		  break;
	  case TEST_STATE_IR:
		  if( bFlag )
		  {
			  printf("\n");
			  printf("Test IR, please use two boards and keep them close!\n");
			  printf("Press and release SW3 to enter into SPI demo.\n");
		  }	  
	  	  IR_test();
	
		  break;
	  case TEST_STATE_SPI:
	  if( bFlag )
	  {
		  printf("\n");
		  printf("Test SPI communication,SPI0 is the master and SPI1 is the slave with FIFO!\n");
		  printf("If success, Green light flashes;\n");
		  printf("Otherwise,Red light flashes.\n");
		  printf("Press and release SW3 to enter into TSI demo.\n");
	  }	  
	  Test_SPI();

	  break;
		  
	  case TEST_STATE_DONE:
		  if( bFlag )
		  {
			  printf("Demo completed successfully!\n");
			  iTestState = TEST_STATE_IDLE;
		  }
		  break;
	  default:
		  break;
	  }
	}
	return (status);
}

/*****************************************************************************//*!
+FUNCTION----------------------------------------------------------------
* @function name: TestTSI
*
* @brief handle TSI events
*        
* @param  none 
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
void TestTSI(void)
{
	TSI_Check();
}

/*****************************************************************************//*!
+FUNCTION----------------------------------------------------------------
* @function name: TestPotentiometer
*
* @brief test potentiometer and display corresponding LEDs
*        
* @param  none 
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
void TestPotentiometer(void)
{
	  ADC_test(); 
}

/*****************************************************************************//*!
+FUNCTION----------------------------------------------------------------
* @function name: TestAccelerometer
*
* @brief test accelerometer via IIC and display corresponding LEDs
*        
* @param  none 
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
void TestAccelerometer(void)
{
	MMA8451_PT60Tower_Test();
}

/*****************************************************************************//*!
+FUNCTION----------------------------------------------------------------
* @function name: TestUART
*
* @brief test UART connection and display corresponding LEDs
*        
* @param  none 
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
static unsigned char ucTxBuff[34];
static unsigned char ucRxBuff[34];
void TestUART(void)
{
	unsigned char i;
	unsigned char *pPoint;
	unsigned short uiCrcValue;
	unsigned short uiCount;
	volatile unsigned char dummy;
	for( i = 0;i<32;i++ )
	{
		ucTxBuff[i] = i;
		ucRxBuff[i] = 0x00;
	}

	uiCrcValue = Crc16Calculate( &ucTxBuff[0],32 );
	pPoint = (unsigned char *)&uiCrcValue;
	ucTxBuff[32] = pPoint[0];
	ucTxBuff[33] = pPoint[1];
	for(i = 0; i < 34; i++)
	{
		TERMIO_PutChar2(ucTxBuff[i]);
		uiCount = 0xffff;
		 while(( ( SCI1_S1 & SCI1_S1_RDRF_MASK) != SCI1_S1_RDRF_MASK )&&
			   ( --uiCount ) );
		 if( uiCount == 0 )
		 {
			 break;
		 }
		 dummy = SCI1_S1;
		 ucRxBuff[i] = SCI1_D;
	}
	uiCrcValue = Crc16Calculate( &ucRxBuff[0],34 );
	if( uiCrcValue != 0 )
	{
		printf("Uart communicate with CRC verify fail!\n");
		/* Error: Flash all four LEDs */
		LED0_Toggle();
		LED1_Toggle();
		LED2_Toggle();
		LED3_Toggle();
		DelayUS(1000);
			
	}
	else
	{
		/* Success: Turn off all LEDs and then ON */
		LED0_On();
		LED1_On();
		LED2_On();
		LED3_On();	
		
		printf("Uart communicate with CRC verify success!\n");
	}
}
/*****************************************************************************//*!
+FUNCTION----------------------------------------------------------------
* @function name: MMA8451_PT60Tower_Test
*
* @brief accelerometer test function 
*        
* @param  none 
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
void MMA8451_PT60Tower_Test(void)
{	
    if((u8MMA8451ReadRegister(DATA_STATUS_REG) & 0x0f) != 0)
	{
	    MMA8451_Read_3axis_data();   

	  if(resultz < 9000)  // when board in front side 
	  {
	  if(resultz >= 4100 )  // board on the flat position 4100-4300
	  {
	      LED0_On();
	      LED1_On();
	      LED2_On();
	      LED3_On();
	  }
	  else if(resultz >= 3400 && resultz < 3900)   
	  {	     
	      LED3_Off();
	  }
	  else if(resultz >= 2800 && resultz < 3300)
	  {	     
	      LED2_Off();
	  }
	  else if(resultz >= 2300 && resultz < 2700)
	  {	     
	      LED1_Off();
	  }
	  else if(resultz >= 0 && resultz < 2200)
	  {	     
	      LED0_Off();
	  }  
    }	 
	}
    else
    {
    	LED0_Off();
    	LED1_Off();
    	LED2_Off();
    	LED3_Off();
    }
    DelayUS(1000);  			
}
/*****************************************************************************//*!
+FUNCTION----------------------------------------------------------------
* @function name: DelayUS
*
* @brief delay function 
*        
* @param  timeUS -- counter for delay 
*
* @return none
*
* @ Pass/ Fail criteria: none
*****************************************************************************/
void DelayUS(word timeUS)
{
	word i,j;
	for(i = 0; i < timeUS; i++)
	{
		for(j = 0; j < 16; j++)	/* delay 1 us */
		{
			asm(nop);
		}
	}
}




