/*
** #################################################################################################################
**
**     Filename  : arch.H
**
**     Family    : Freescale DSP
**
**     Compiler  : Freescale DSP C Compiler
**
**     Abstract  :
**
**         This module implements core-specific (architecture-specific) macros.
**
**
**     (c) Freescale Semiconductor
**     2012 All Rights Reserved
**
**     http      : www.freescale.com
**
** #################################################################################################################
*/

#ifndef __PE_arch
#define __PE_arch

/*lint -save  -esym(961,19.13) Disable MISRA rule (19.13) checking. Assembler access macros use '#' symbol to specify constant. */
/*lint -save  -e14 -esym(960,8.5) Disable MISRA rule (5.4,8.5) checking. This header file contains definitions of core-specific (architecture-specific) macros implemented as inline assembler functions. */
/*lint -esym(765,archGetLimitBit,archGetSetSaturationMode,archDelay,shl2,shr2,impyuu,impysu) Disable MISRA rule (8.10) checking for symbols (archGetLimitBit,archGetSetSaturationMode,archDelay,shl2,shr2,impyuu,impysu). These symbols are used as inline functions. */

/* Inline functions declarations */
inline Word16 archGetLimitBit(void);
inline bool archGetSetSaturationMode (register bool bSatMode);
inline void archDelay(register UWord16 ticks);
inline Word16 shl2(register Word16 num, register Word16 shift);
inline Word16 shr2(register Word16 num, register Word16 shift);
inline UWord32 impyuu(register UWord16 unsigA, register UWord16 unsigB);
inline Word32 impysu(register Word16 sigA, register UWord16 unsigB);

/* Get limit bit */
inline Word16 archGetLimitBit(void)
{
    register Word16 ret;
    /*lint -save  -e586 Disable MISRA rule (2.1) checking. Optimized assembler implementation used. */
    asm {
        move.w SR,ret
        bfclr 0xffbf,ret
    }
    /*lint -restore Enable MISRA rule (2.1) checking. */
    /*lint -save  -e530 Disable MISRA rule (9.1) checking. Value to variable is assigned in assembler code. */
    return ret;
    /*lint -restore Enable MISRA rule (9.1) checking. */
}

/* Clear sticky limit bit */
#define archResetLimitBit() do { asm(bfclr #0x40,SR); asm(nop); asm(nop); } while(0)

/* Clear saturation mode */
#define archSetNoSat() do { asm(bfclr #0x10,OMR); asm(nop); asm(nop); } while(0)

/* Set saturation mode */
#define archSetSat32() do { asm(bfset #0x10,OMR); asm(nop); asm(nop); } while(0)

/* Set saturation mode and return it previous value */
inline bool archGetSetSaturationMode (register bool bSatMode)
{
    register Word16 ret;
    /*lint -save  -e586 Disable MISRA rule (2.1) checking. Optimized assembler implementation used. */
    asm {
        move.w OMR,ret
        bfclr 0xffef,ret
    }
    if(bSatMode)                /* using "if" enables compile-time optimization */
    {
        asm { bfset 0x10,OMR; nop; nop; }
    }
    else
    {
        asm { bfclr 0x10,OMR; nop; nop; }
    }
    /*lint -restore Enable MISRA rule (2.1) checking. */
    /*lint -save  -e530 Disable MISRA rule (9.1) checking. Value to variable is assigned in assembler code. */
    return (bool)ret;
    /*lint -restore Enable MISRA rule (9.1) checking. */
}

/* Set two's-complement rounding */
#define archSet2CompRound() do { asm(bfset #0x20,OMR); asm(nop); asm(nop); } while(0)

/* Set convergent rounding */
#define archSetConvRound() do { asm(bfclr #0x20,OMR); asm(nop); asm(nop); } while(0)

/* STOP */
#define archStop() asm(STOP)

/* Software Interrupt */
#define archTrap() asm(SWI)

/* WAIT */
#define archWait() asm(WAIT)

/* Debug Halt */
#define archDebugHalt() asm(DEBUGHLT)

/* Enable Interrupts on certain levels */
#define archEnableInt() asm(bfclr #0x0300,SR)
/*lint -save  -esym(960,19.12) Disable MISRA rule (19.12) checking. Assembler access macros use '#' symbol to specify constant. */
#define archEnableIntLvl123() do { asm(bfset #0x0300,SR); asm(bfclr #0x0200,SR); } while(0)
#define archEnableIntLvl23() do { asm(bfset #0x0300,SR); asm(bfclr #0x0100,SR); } while(0)
/*lint -restore  +esym(960,19.12) Enable MISRA rule (19.12) checking. */

/* Disable Interrupts */
#define archDisableInt() do { asm(bfset #0x0300,SR); asm(nop); asm(nop); asm(nop); asm(nop); asm(nop); asm(nop); } while(0)

/* Push SR on HWS and disable interrupts  */
#define archPushIntAndDisable() do { asm(moveu.w SR,HWS); asm(nop); asm(nop); asm(bfset #0x0300,SR); asm(nop); asm(nop); asm(nop); asm(nop); asm(nop); } while(0)

/* Pop SR from HWS (renew the state before calling archPushIntAndDisable() */
#define archPopIntMask() do { asm(moveu.w HWS,SR); } while(0)

/* Save SR into register (in C declare as "register UWord16")  */
#define archSaveIntAndDisable(intsave) do { asm(move.w SR,intsave); asm(bfset #0x0300,SR); asm(nop); asm(nop); asm(nop); asm(nop); asm(nop); } while(0)

/* Restore SR from register */
#define archRestoreIntMask(intsave) do { asm(moveu.w intsave,SR); } while(0)

/* Delay for requested number of ticks */
/*lint -esym(522,archDelay) Disable MISRA rule (14.2) checking for symbols (archDelay). Inline function implements simple "for" loop without any side effects. */
inline void archDelay(register UWord16 ticks)
{
    /*lint -save  -e586 Disable MISRA rule (2.1) checking. Optimized assembler implementation used. */
    asm { 
        nop
        nop
        do ticks,archDelay1
           nop
archDelay1:
    }
    /*lint -restore Enable MISRA rule (2.1) checking. */
}

/* Shift left */
inline Word16 shl2(register Word16 num, register Word16 shift)
{
    /*lint -save  -e586 Disable MISRA rule (2.1) checking. Optimized assembler implementation used. */
    asm { asll.w shift,num }
    /*lint -restore Enable MISRA rule (2.1) checking. */
    return num;
}

/* Shift right */
inline Word16 shr2(register Word16 num, register Word16 shift)
{
    /*lint -save  -e586 Disable MISRA rule (2.1) checking. Optimized assembler implementation used. */
    asm { asrr.w shift,num }
    /*lint -restore Enable MISRA rule (2.1) checking. */
    return num;
}

/* Integer multiply unsigned 16b x unsigned 16b */
inline UWord32 impyuu(register UWord16 unsigA, register UWord16 unsigB)
{
    register UWord32 l;
    /*lint -save  -e586 Disable MISRA rule (2.1) checking. Optimized assembler implementation used. */
    asm {

        .iasm_reg2regsetcopyflag off
        
        move.w unsigA,A0
        move.w unsigB,B0
        Impyuu A0,B0,l        /* thanks to impyuu F0,F0,FFF, we can use 'l' as destination */
        
        .iasm_reg2regsetcopyflag on
    }
    /*lint -restore Enable MISRA rule (2.1) checking. */
    /*lint -save  -e530 Disable MISRA rule (9.1) checking. Value to variable is assigned in assembler code. */
    return l;
    /*lint -restore Enable MISRA rule (9.1) checking. */
}

/* Integer multiply signed 16b x unsigned 16b */
inline Word32 impysu(register Word16 sigA, register UWord16 unsigB)
{
    register Word32 l;
    /*lint -save  -e586 Disable MISRA rule (2.1) checking. Optimized assembler implementation used. */
    asm {
        .iasm_reg2regsetcopyflag off
        
        move.w sigA,A1
        move.w unsigB,B0
        Impysu A1,B0,Y        /* unlike impyuu, there is no impysu F0,F0,FFF (must use Y) */
        nop                    /* let Y propagate through pipeline */
        
        .iasm_reg2regsetcopyflag on
        
        tfr Y,l                /* transfer Y to return register */
    }
    /*lint -restore Enable MISRA rule (2.1) checking. */
    /*lint -save  -e530 Disable MISRA rule (9.1) checking. Value to variable is assigned in assembler code. */
    return l;
    /*lint -restore Enable MISRA rule (9.1) checking. */
}

#define archMemRead(Local, Remote, Bytes) (*(Local) = *(Remote))
#define archMemWrite(Remote, Local, Bytes) (*(Remote) = *(Local))

#define archCoreRegisterBitSet(Mask, Reg)     asm(bfset   Mask,Reg)
#define archCoreRegisterBitClear(Mask, Reg)    asm(bfclr   Mask,Reg)
#define archCoreRegisterBitChange(Mask, Reg)   asm(bfchg   Mask,Reg)
#define archCoreRegisterBitTestHigh(Mask, Reg) asm(bftsth  Mask,Reg)
#define archCoreRegisterBitTestLow(Mask, Reg)  asm(bftstl  Mask,Reg)

#define archMemBitSet(Mask, Addr)          asm(bfset   Mask,Addr)
#define archMemBitClear(Mask, Addr)         asm(bfclr   Mask,Addr)
#define archMemBitChange(Mask, Addr)        asm(bfchg   Mask,Addr)
#define archMemBitTestHigh(Mask, Addr)       asm(bftsth  Mask,Addr)
#define archMemBitTestLow(Mask, Addr)        asm(bftstl  Mask,Addr)

/*lint -restore  +esym(960,8.5) Enable MISRA rule (5.4,8.5) checking. */
/*lint -restore  +esym(961,19.13) Enable MISRA rule (19.13) checking. */

#endif
